/**********************************************************************
 ** Verb - contains methods and attributes for a verb. It stores the
 **        attributes needed to call a command based on a verb and the
 **        methods to manipulate it all
 **
 **
 **    
 ** Last reviewed: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef VERB_C
#define VERB_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "verb.h"
#include "objtype.h"
#include "player.h"

/***********************************************************************
 ** _Verb (constructor) - loads verb name and initializes attributes
 **
 ** Parameters: verb_name - this verb name 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Verb::Verb(char *verb_name)
{
   the_verb = verb_name;
   set_name(verb_name);
   set_area("verbs");

   if (the_verb.str_len() > MAX_VERB_LEN)
      the_verb.truncate(MAX_VERB_LEN);

   the_command = NULL;
   obj_type = OBJ_TYPE_VERB;

   num_executes = max_execute = avg_execute = 0;
   min_execute = 10000000;
}


/***********************************************************************
 ** ~_Verb (destructor) - cleans up for destruction
 **
 ** Parameters: None 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Verb::~Verb(void)
{
}

/***********************************************************************
 ** get_verb_str - gets the verb's entire string name
 **
 ** Parameters: None 
 **
 ** Returns: a pointer to the verb string
 **
 ***********************************************************************/

char *Verb::get_verb_str()
{
   return get_name();
}


/***********************************************************************
 ** get_verb_format - returns the format of the verb
 **
 ** Parameters: None 
 **
 ** Returns: a number defining the format
 **
 ***********************************************************************/

int Verb::get_verb_format()
{
   return format;
}

/***********************************************************************
 ** get_verb_comflag - returns the comflag required to use this command
 **
 ** Parameters: None 
 **
 ** Returns: a number defining the comflag
 **
 ***********************************************************************/

int Verb::get_verb_comflag()
{
   return comflag;
}


/***********************************************************************
 ** set_verb_format - assigns the format for the verb
 **
 ** Parameters: the_format - the format to set
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Verb::set_verb_format(int the_format)
{
   format = the_format;
}


/***********************************************************************
 ** set_verb_format - assigns the comflag needed to use the verb
 **
 ** Parameters: the_format - the format to set
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Verb::set_verb_comflag(int the_comflag)
{
   comflag = the_comflag;
}


/***********************************************************************
 ** set_verb_format - assigns the format for the verb
 **
 ** Parameters: command_str - 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Verb::set_verb_command(Cmd_Func in_command)
{
   the_command = in_command;
}


/***********************************************************************
 ** get_command - returns a command Cmd_Func
 **
 ** Parameters: None 
 **
 ** Returns: the command
 **
 ***********************************************************************/

Cmd_Func Verb::get_command()
{
   return the_command;
}


/***********************************************************************
 ** add_new_execute - adds new timestats for this command execution
 **
 ** Parameters: timespan - how many milliseconds it took to execute
 **
 ** Returns: 1 for success
 **
 **
 ***********************************************************************/

int Verb::add_new_execute(long timespan)
{
   num_executes++;
   if (timespan > max_execute)
      max_execute = timespan;

   if (timespan < min_execute)
      min_execute = timespan;

   avg_execute = ((avg_execute*(num_executes-1)) + timespan) / num_executes;

   return 1;
}


/***********************************************************************
 ** display_data - displays verb database statistics
 **                
 **
 ** Parameters: the_player - who we send the messages to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Verb::display_data(Player *the_player)
{
   char verbname[100];
   Strings the_color;

   sprintf(verbname, "%-25s", the_verb.str_show());
   if (min_execute == 10000000)
      the_player->send_plr("&+g%s &*  Not Used Yet\n", verbname);
   else 
   {
      if (avg_execute < 3000)
         the_color = "&+G";
      else if (avg_execute < 6000)
         the_color = "&+Y";
      else
         the_color = "&+R";

      the_player->send_plr("&+g%s &+m%ld\t&+c%ld\t%ld\t%s%ld&*\n", 
                 verbname, num_executes,  
                 min_execute, max_execute, the_color.str_show(), avg_execute);
   }
   return 1;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Verb::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Verb::get_mem_size_dynamic()
{
   int  size = 0;

   size += the_verb.get_mem_size_dynamic();
   
   size += get_mem_size_entity();

   return size;
}

#endif


