// ========================================================================
// copyright (C) 1999-2003 by Tobias Erbsland <te@profzone.ch>
// ------------------------------------------------------------------------
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2 of the License, or (at your
// option) any later version.
// ========================================================================

#ifndef REPORT_H
#define REPORT_H
#ifdef _MSC_VER
#include "win32/config_win32.h"
#endif
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "parameter.h"
#include <string>
#include <ctime>

/** The is the abstract report class. Every report is itherited from this
  * class.
  *@short The abstract report class.
  *@author Tobias Erbsland
  */
class Report : public Parameter
{

private:

	std::string dateformat;
	int datesize;
	void checkDateformat( void );

public:
	/// All report types.
	enum reportType_e { TABLE };
	/// All column types.
	enum colTyp_e { COUNTER, STRING, INTEGER, FLOAT, PERCENT, DATE };
	/// Styles for every column
	enum colStyle_e { NONE, ROUND, RECT, QUOTE };
	/// Aligments for every column.
	enum colAlign_e { LEFT, RIGHT };
	/// Addiotional Flag for special views.
	enum { SUM = 1 }; // Flags;

protected:

	/** This is the main struct holding all parameters for every
	  * column.
	  */
	struct col_t
	{
		int Size;
		colTyp_e Typ;
		colAlign_e Align;
		colStyle_e Style;
		std::string Title;
		bool Sum;

		float floatSum;
		long longSum;
	};

	/** Pointer to the default Output stream.
	  */
	std::ostream * os;

	/** This is the list used for column declarations.
	  */
	std::list< col_t > col_list;

	/** Pointer to the current column.
	  */
	std::list< col_t >::iterator ci;

	/** Linecounter.
	  */
	long lc;

	/** The string for the Report title.
	  */
	std::string Title;

	/** The report Type.
	  */
	reportType_e Type;

	/** Convert a unix timestamp to a string.
	  */
	std::string getDateStr( time_t t );

	/** Check how many chars the datestring need.
	  */
	int getDateLenght( void );


public:

	/** The constructor sets the output stream pointer default to 0. Please
	  * check this in every function implementation, for security reasons.
	  */
	Report();

	/** This function must be called first, before any other function is called.
	  * With this, important parameters would be set.
	  *@param title Set the title of the following report.
	  *@param reportType Set the next report type.
	  */
	virtual void startNewReport( const std::string &title, const reportType_e reportType ) = 0;

	/** Is the report type a "table", after starting a new Report (call startNewReport),
	  * you must add every column of the table with this function.
	  *@param colSize For a fixed width column this is a positive integer, with the
	  *Value -1 you can create a dynamic sized column. This value means only the minimal
	  *width of the column.
	  *@param colTyp Set here the Type of the column.
	  *@param colAlign Set here the Align of the column.
	  *@param colTitle Every column is labled with a Title. Set the Title here. Make
	  *sure the Title is noch Bigger than the colSize.
	  *@param colStyle Here you can set special formating options, like (xxx) or [xxx].
	  *@param colFlags This parameter is optional for special Flags like SUM or others.
	  */
	virtual void addCol( const int colSize, const colTyp_e colTyp, const colAlign_e colAlign,
	                     const std::string &colTitle, const colStyle_e colStyle, const unsigned long colFlags = 0 );

	/** Gererate a page Header for one or more single Reports
	  */
	virtual void printPageHeader( void ) = 0;

	/** After setup all columns with addCol, you must call this
	  * function. It's possible this function creates a header for the Report.
	  */
	virtual void printHeader( void ) = 0;

	/** If you create a Table, after setup all columns and calling printHeader, you
	  * must fill every Field auf the table (except autofilled-colums) with the
	  * right value from left to right, top to bottom.
	  */
	virtual void printField( const std::string &field ) = 0;
	virtual void printField( const long field ) = 0;
	virtual void printField( const double field ) = 0;

	/** If the Field empty, call this function. You can choose between a realline
	  * or not. See the examples in the current analyse modules.
	  */
	virtual void skipField( const bool realline = true ) = 0;

	/** After the last line of the table, call this function to create perhaps
	  * a footer.
	  */
	virtual void printFooter( void ) = 0;

	/** Generate a final Footer before closing a stream
	  */
	virtual void printPageFooter( void ) = 0;

	/** This function is only used by some Plugins. Don't use it in analyse
	  * modules!
	  */
	virtual void setOstream( std::ostream * newos ) { os = newos; };

	/** This function is only used by some Plugins. Don't use it in analyse
	  * modules!
	  */
	virtual std::ostream * getOstream( void ) { return os; };

	/** The virtual deconstructor.
	  */
	virtual ~Report();
};

#endif
