// ========================================================================
// copyright (C) 1999-2003 by Tobias Erbsland <te@profzone.ch>
// ------------------------------------------------------------------------
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2 of the License, or (at your
// option) any later version.
// ========================================================================

#include "syslogfilter.h"
#include "plugin.h"
#include <ctime>

//#define DEBUG

syslogFilter::syslogFilter() :
		strPrefix( "syslog" ),
		strName( "Standart Syslog Filter" ),
		strVersion( "V0.1" ),
		strAuthor( "Tobias Erbsland <te@profzone.ch>" ),
		strHelp( "Filter for default syslog files." ),
		intern_mail_suffix( "@profzone.ch" )
{
	addParam( "year", "Set the start year of the syslog file.", "Year with 4 digits.", STRING, "now" );
	addParam( "counter", "Set this flag to display the linecounter.", "This is a Flag.", FLAG, "0" );
}

bool syslogFilter::run( void )
{
	long lc = 0;
	logline_t lgln;
	bool linecounter = getFlagParam( "counter" );

	if ( getStringParam( "year" ) == "now" )
	{
		// Set year to the current time.
		time_t now = time( 0 ); // now.
		struct tm tm_now = *localtime( &now ); // as tm struct.
		intYear = tm_now.tm_year; // Get year.
	}
	else
	{ // is the year valid?
		if ( getIntParam( "year" ) >= 1900 && getIntParam( "year" ) <= 9999 )
		{
			intYear = getIntParam( "year" ) - 1900;
		}
		else
		{
			std::cerr << "Error: Parameter --syslog-year=" << getStringParam( "year" ) << " is not a valid year!" << std::endl;
			return false; // Abort.
		}
	}

	while ( ( *is ) )
	{
		try
		{
			lgln = getLogLine( ( *is ) );
			lc++;
		}
		catch ( endOfLog ) { break; }

		if ( lgln.command == "sendmail" )
		{
			sendmail( lgln.time, lgln.message );
		};
		if ( !( lc % 5000 ) && linecounter ) std::cerr << "  line " << lc << " ..." << std::endl;
	}
	return true; // immer ok...
}

int syslogFilter::readTwoChars( std::istream &streamin )
{
	int newInt;

	newInt = ( streamin.get() - '0' ) * 10;
	newInt += streamin.get() - '0';

	return newInt;
}

syslogFilter::logline_t syslogFilter::getLogLine( std::istream &streamin )
{
	logline_t logLine;
	struct tm tmTime;
	std::string strMonth, strPid, strProcess;
	static char buffer[ 1024 ];

	// Zeit und monat in ein universelles Format wandeln.
	if ( !( streamin >> strMonth ) ) throw endOfLog();
	tmTime.tm_mon = strToMonth( strMonth ) - 1;
	if ( !( streamin >> tmTime.tm_mday ) ) throw endOfLog();
	if ( !( streamin.ignore() ) ) throw endOfLog();
	tmTime.tm_hour = readTwoChars( streamin );
	if ( !( streamin.ignore() ) ) throw endOfLog();
	tmTime.tm_min = readTwoChars( streamin );
	if ( !( streamin.ignore() ) ) throw endOfLog();
	tmTime.tm_sec = readTwoChars( streamin );
	// Host kommando und pid konvertieren.
	if ( !( streamin >> logLine.host ) ) throw endOfLog();
	if ( !( streamin >> strProcess ) ) throw endOfLog();
	logLine.command = strProcess.substr( 0, strProcess.find( "[" ) );
	logLine.pid = atoi( strProcess.substr( strProcess.find( "[" ) + 1, strProcess.find( "[" ) - strProcess.find( "]" ) - 1 ).c_str() );
	tmTime.tm_year = intYear; // Set Year.
	logLine.time = mktime( &tmTime );
	streamin.ignore();
	// Restliche Zeile lesen.
	if ( !( streamin.getline( buffer, 1024 ) ) ) throw endOfLog();
	logLine.message = buffer;
	return logLine;
}

std::string syslogFilter::translate( const std::string &email )
{
	std::string a = email;
	if ( afp->addrFilter( a ) )  // Try change, changed!?
		return a;
	else
		return email;
}

syslogFilter::sendmail_t syslogFilter::splitup( std::string line )
{
	static std::map< std::string, std::string > linemap;
	sendmail_t sm;
	sm.size = 0;
	sm.ok = false; // Alle Werte ruecksetzen.
#ifdef DEBUG
	std::string fullline = line;
#endif
	linemap.clear();
	if ( line.find( ":" ) == std::string::npos )
	{
		return sm; // Error, no ":" -> Abort.
	};
	sm.id = line.substr( 0, line.find( ":" ) );
	line.erase( 0, line.find( ":" ) + 2 );
	if ( line.find( "=" ) == std::string::npos )
	{
		return sm; // Error, no "=" -> Abort.
	};
	while( line.find( ", " ) != std::string::npos )
	{
		linemap[ line.substr( 0, line.find( "=" ) ) ] = line.substr( line.find( "=" ) + 1, line.find( ", " ) - line.find( "=" ) - 1 );
		line.erase( 0, line.find( ", " ) + 2 );
	}
	linemap[ line.substr( 0, line.find( "=" ) ) ] = line.substr( line.find( "=" ) + 1 );

	if( linemap.find( "from" ) != linemap.end() )
	{
#ifdef DEBUG
		std::cout << "Line \"" << fullline << "\"." << std::endl;
		std::cout << "  From before translation: \"" << linemap["from"] << "\"." << std::endl;
#endif
		sm.from = translate( linemap[ "from" ] ); // Einfuellen in Struktur.
#ifdef DEBUG
		std::cout << "  From after translation: \"" << sm.from << "\"." << std::endl;
#endif
		sm.size = atoi( linemap[ "size" ].c_str() );
		sm.ok = true;
	}
	if ( linemap.find( "to" ) != linemap.end() )
	{
		// Empfaenger nur eintragen, wenn Nachricht auch angekommen.
		if ( linemap[ "stat" ].substr( 0, 4 ) == "Sent" ) sm.to = translate( linemap[ "to" ] );
		sm.ok = true;
	};
	return sm;
}

void syslogFilter::sendmail( const time_t time, const std::string &line )
{
	sendmail_t sm;
	Analyse::unity_t empty_me;
	empty_me.size = 0; // Leeres Mailentry erzeugen.
	empty_me.date = time;
	empty_me.client = "unknown";

	sm = splitup( line );
	if( sm.ok )
	{
		if ( unity_map.find( sm.id ) == unity_map.end() )
		{
			unity_map[sm.id] = empty_me;
			unity_map[sm.id].message_id = sm.id;
		};
		if ( sm.from.length() > 0 )
		{
			unity_map[ sm.id ].from = sm.from;
			unity_map[ sm.id ].size = sm.size;
		}
		else if ( sm.to.length() > 0 )
		{
			if ( sm.to.find( "@" ) == std::string::npos )  // Intern oder Extern?
			{
				unity_map[ sm.id ].to_intern.push_back( sm.to );
			}
			else
			{ // dann extern...
				unity_map[ sm.id ].to_extern.push_back( sm.to );
			};
		}
	}
}

