/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// modified for Armagetron my Manuel Moos

// net_wins.c

//#include "quakedef.h"
//#include "winquake.h"
#include "tSysTime.h"

#ifdef WIN32
#include <windows.h>
#include <winsock.h>
#include <sys/timeb.h> 
#endif
#include <stdlib.h> 
#include <stdio.h> 

#include "tString.h"
#include "nNet.h"

//extern tString hostname;

#define MAXHOSTNAMELEN		256

unsigned int net_hostport=4534;
tString my_tcpip_address;
bool tcpipAvailable=false;
tString hostname="UNNAMED";
//tString& hostname=real_hostname;

int net_acceptsocket = -1;		// socket for fielding new connections
static int net_controlsocket;
static int net_broadcastsocket = 0;
static struct sockaddr broadcastaddr;

static unsigned long myAddr;

bool	winsock_lib_initialized;

int (PASCAL FAR *pWSAStartup)(WORD wVersionRequired, LPWSADATA lpWSAData);
int (PASCAL FAR *pWSACleanup)(void);
int (PASCAL FAR *pWSAGetLastError)(void);
SOCKET (PASCAL FAR *psocket)(int af, int type, int protocol);
int (PASCAL FAR *pioctlsocket)(SOCKET s, long cmd, u_long FAR *argp);
int (PASCAL FAR *psetsockopt)(SOCKET s, int level, int optname,
							  const char FAR * optval, int optlen);
int (PASCAL FAR *precvfrom)(SOCKET s, char FAR * buf, int len, int flags,
							struct sockaddr FAR *from, int FAR * fromlen);
int (PASCAL FAR *psendto)(SOCKET s, const char FAR * buf, int len, int flags,
						  const struct sockaddr FAR *to, int tolen);
int (PASCAL FAR *pclosesocket)(SOCKET s);
int (PASCAL FAR *pgethostname)(char FAR * name, int namelen);
struct hostent FAR * (PASCAL FAR *pgethostbyname)(const char FAR * name);
struct hostent FAR * (PASCAL FAR *pgethostbyaddr)(const char FAR * addr,
												  int len, int type);
int (PASCAL FAR *pgetsockname)(SOCKET s, struct sockaddr FAR *name,
							   int FAR * namelen);

#include "nNet.h"

int winsock_initialized = 0;
WSADATA		winsockdata;

enum nSocketError
{
	nSocketError_Ignore,				// nothing special happened
	nSocketError_Reset					// socket needs to be reset
};

nSocketError ANET_Error()
{
	tASSERT( pWSAGetLastError );
	int error = pWSAGetLastError();

	switch ( error )
	{
		case WSAENOTSOCK:
			return nSocketError_Reset;
			break;
		case WSAECONNRESET:
			return nSocketError_Reset;
			break;
		case WSAEWOULDBLOCK:
			return nSocketError_Ignore;
			break;
		case WSANOTINITIALISED:
			break;
		case WSAENETDOWN:
			break;
		case WSAEFAULT:
			break;
		case WSAEINTR:
			break;
		case WSAEINVAL:
			break;
		case WSAEISCONN:
			break;
		case WSAENETRESET:
			break;
		case WSAEOPNOTSUPP:
		case WSAEAFNOSUPPORT:
		case WSAEADDRNOTAVAIL:
			return nSocketError_Reset;
			break;
		case WSAESHUTDOWN:
			break;
		case WSAEMSGSIZE:
			break;
		case WSAETIMEDOUT:
			break;
		default:
			st_Breakpoint();
			break;
	}

	st_Breakpoint();
	return nSocketError_Ignore;
};

//=============================================================================

static double blocktime;

int PASCAL FAR BlockingHook(void)  
{ 
    MSG		msg;
    BOOL	ret;
 
	if ((tSysTimeFloat() - blocktime) > 2.0)
	{
		WSACancelBlockingCall();
		return FALSE;
	}

    /* get the next message, if any */ 
    ret = (BOOL) PeekMessage(&msg, NULL, 0, 0, PM_REMOVE); 
 
    /* if we got one, process it */ 
    if (ret) { 
        TranslateMessage(&msg); 
        DispatchMessage(&msg); 
    } 
 
    /* TRUE if we got a message */ 
    return ret; 
} 


void ANET_GetLocalAddress()
{
	struct hostent	*local = NULL;
	char			buff[MAXHOSTNAMELEN];
	unsigned long	addr;

	if (myAddr != INADDR_ANY)
		return;

	if (pgethostname(buff, MAXHOSTNAMELEN) == SOCKET_ERROR)
		return;

	blocktime = tSysTimeFloat();
	WSASetBlockingHook(BlockingHook);
	local = pgethostbyname(buff);
	WSAUnhookBlockingHook();
	if (local == NULL)
		return;

	myAddr = *(int *)local->h_addr_list[0];

	addr = ntohl(myAddr);
	my_tcpip_address.Clear();	
	my_tcpip_address << ((addr >> 24) & 0xff) << '.'
			 << ((addr >> 16) & 0xff) << '.'
			 << ((addr >> 8) & 0xff) << '.'
			 << (addr & 0xff);
}


int ANET_Init (void)
{
	int		i;
	char	buff[MAXHOSTNAMELEN];
	char	*p;
	int		r;
	WORD	wVersionRequested;
	HINSTANCE hInst;

// initialize the Winsock function vectors (we do this instead of statically linking
// so we can run on Win 3.1, where there isn't necessarily Winsock)
    hInst = LoadLibrary("wsock32.dll");
	
	if (hInst == NULL)
	{
		Con_SafePrintf ("Failed to load winsock.dll\n");
		winsock_lib_initialized = false;
		return -1;
	}

	winsock_lib_initialized = true;

    (void *&)pWSAStartup = GetProcAddress(hInst, "WSAStartup");
    (void *&)pWSACleanup = GetProcAddress(hInst, "WSACleanup");
    (void *&)pWSAGetLastError = GetProcAddress(hInst, "WSAGetLastError");
    (void *&)psocket = GetProcAddress(hInst, "socket");
    (void *&)pioctlsocket = GetProcAddress(hInst, "ioctlsocket");
    (void *&)psetsockopt = GetProcAddress(hInst, "setsockopt");
    (void *&)precvfrom = GetProcAddress(hInst, "recvfrom");
    (void *&)psendto = GetProcAddress(hInst, "sendto");
    (void *&)pclosesocket = GetProcAddress(hInst, "closesocket");
    (void *&)pgethostname = GetProcAddress(hInst, "gethostname");
    (void *&)pgethostbyname = GetProcAddress(hInst, "gethostbyname");
    (void *&)pgethostbyaddr = GetProcAddress(hInst, "gethostbyaddr");
    (void *&)pgetsockname = GetProcAddress(hInst, "getsockname");

    if (!pWSAStartup || !pWSACleanup || !pWSAGetLastError ||
		!psocket || !pioctlsocket || !psetsockopt ||
		!precvfrom || !psendto || !pclosesocket ||
		!pgethostname || !pgethostbyname || !pgethostbyaddr ||
		!pgetsockname)
	{
		Con_SafePrintf ("Couldn't GetProcAddress from winsock.dll\n");
		return -1;
	}

    /* Not in Armagetron
	if (COM_CheckParm ("-noudp"))
		return -1;
    */

	if (winsock_initialized == 0)
	{
		wVersionRequested = MAKEWORD(1, 1); 

		r = pWSAStartup (MAKEWORD(1, 1), &winsockdata);

		if (r)
		{
			Con_SafePrintf ("Winsock initialization failed.\n");
			return -1;
		}
	}
	winsock_initialized++;

	// determine my name
	if (pgethostname(buff, MAXHOSTNAMELEN) == SOCKET_ERROR)
	{
		Con_DPrintf ("Winsock TCP/IP Initialization failed.\n");
		if (--winsock_initialized == 0)
			pWSACleanup ();
		return -1;
	}

	// if the quake hostname isn't set, set it to the machine name
	if (strcmp(hostname, "UNNAMED") == 0)
	{
		// see if it's a text IP address (well, close enough)
		for (p = buff; *p; p++)
			if ((*p < '0' || *p > '9') && *p != '.')
				break;

		// if it is a real name, strip off the domain; we only want the host
		if (*p)
		{
			for (i = 0; i < 15; i++)
				if (buff[i] == '.')
					break;
			buff[i] = 0;
		}
		hostname=buff;
	}

	/* Not in Armagetron
	i = COM_CheckParm ("-ip");
	if (i)
	{
		if (i < com_argc-1)
		{
			myAddr = inet_addr(com_argv[i+1]);
			if (myAddr == INADDR_NONE)
				Sys_Error ("%s is not a valid IP address", com_argv[i+1]);
			strcpy(my_tcpip_address, com_argv[i+1]);
		}
		else
		{
			Sys_Error ("NET_Init: you must specify an IP address after -ip");
		}
	}
	else */
	{
		myAddr = INADDR_ANY;
		my_tcpip_address="INADDR_ANY";
	}

	if ((net_controlsocket = ANET_OpenSocket (0)) == -1)
	{
		Con_Printf("ANET_Init: Unable to open control socket\n");
		if (--winsock_initialized == 0)
			pWSACleanup ();
		return -1;
	}

	((struct sockaddr_in *)&broadcastaddr)->sin_family = AF_INET;
	((struct sockaddr_in *)&broadcastaddr)->sin_addr.s_addr = INADDR_BROADCAST;
	((struct sockaddr_in *)&broadcastaddr)->sin_port = htons((unsigned short)net_hostport);

	// Con_Printf("Winsock TCP/IP Initialized\n");
	tcpipAvailable = true;

	return net_controlsocket;
}

//=============================================================================

void ANET_Shutdown (void)
{
	ANET_Listen (false);
	ANET_CloseSocket (net_controlsocket);
	if (--winsock_initialized == 0)
		pWSACleanup ();
}

//=============================================================================
static bool sn_listen = false;

void ANET_Listen (bool state)
{
	sn_listen = state;

	// enable listening
	if (state)
	{
		if (net_acceptsocket != -1)
			return;
		ANET_GetLocalAddress();
		if ((net_acceptsocket = ANET_OpenSocket (net_hostport)) == -1)
			Con_Printf ("\nANET_Listen: Unable to open accept socket on desired port, trying next port...\n\n");
		return;
	}

	// disable listening
	if (net_acceptsocket == -1)
		return;
	ANET_CloseSocket (net_acceptsocket);
	net_acceptsocket = -1;
}

//=============================================================================

int ANET_OpenSocket (int port)
{
	int newsocket;
	struct sockaddr_in address;
	u_long _true = 1;

	if ((newsocket = psocket (PF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1)
		return -1;

	if (pioctlsocket (newsocket, FIONBIO, &_true) == -1)
		goto ErrorReturn;

	address.sin_family = AF_INET;
	address.sin_addr.s_addr = INADDR_ANY;
	address.sin_port = htons((unsigned short)port);
	if( bind (newsocket, (sockaddr *)&address, sizeof(address)) == 0)
		return newsocket;

	Con_Printf ("\nUnable to bind to ");
	Con_Printf (ANET_AddrToString((struct sockaddr *)&address));
	Con_Printf ("\n");
ErrorReturn:
	pclosesocket (newsocket);
	return -1;
}

//=============================================================================

int ANET_CloseSocket (int socket)
{
	if (socket == net_broadcastsocket)
		net_broadcastsocket = 0;
	return pclosesocket (socket);
}


//=============================================================================
/*
============
PartialIPAddress

this lets you type only as much of the net address as required, using
the local network components to fill in the rest
============
*/
static int PartialIPAddress (const char *in, struct sockaddr *hostaddr)
{
	char buff[256];
	char *b;
	int addr;
	int num;
	int mask;
	int run;
	int port;
	
	buff[0] = '.';
	b = buff;
	strcpy(buff+1, in);
	if (buff[1] == '.')
		b++;

	addr = 0;
	mask=-1;
	while (*b == '.')
	{
		b++;
		num = 0;
		run = 0;
		while (!( *b < '0' || *b > '9'))
		{
		  num = num*10 + *b++ - '0';
		  if (++run > 3)
		  	return -1;
		}
		if ((*b < '0' || *b > '9') && *b != '.' && *b != ':' && *b != 0)
			return -1;
		if (num < 0 || num > 255)
			return -1;
		mask<<=8;
		addr = (addr<<8) + num;
	}
	
	if (*b++ == ':')
		port = atoi(b);
	else
		port = net_hostport;

	hostaddr->sa_family = AF_INET;
	((struct sockaddr_in *)hostaddr)->sin_port = htons((short)port);	
	((struct sockaddr_in *)hostaddr)->sin_addr.s_addr = (myAddr & htonl(mask)) | htonl(addr);
	
	return 0;
}
//=============================================================================

int ANET_Connect (int socket, struct sockaddr *addr)
{
	return 0;
}

//=============================================================================

int ANET_CheckNewConnections (void)
{
	char buf[4096];

	if (net_acceptsocket == -1)
		return -1;

	int ret = precvfrom (net_acceptsocket, buf, sizeof(buf), MSG_PEEK, NULL, NULL);
	if ( ret > 0)
	{
		return net_acceptsocket;
	}
	else if ( ret == SOCKET_ERROR )
	{
		switch ( ANET_Error() )
		{
			case nSocketError_Reset:
				ANET_Shutdown();
				ANET_Init();
				ANET_Listen( true );
				break;
			case nSocketError_Ignore:
				break;
		}
	}

	return -1;
}

//=============================================================================

int ANET_Read (int& socket, int8 *buf, int len, struct sockaddr *addr)
{
	int addrlen = sizeof (struct sockaddr);
	int ret;

	ret = precvfrom (socket, buf, len, 0, (struct sockaddr *)addr, &addrlen);

	if ( ret == SOCKET_ERROR )
	{
		switch ( ANET_Error() )
		{
			case nSocketError_Reset:
				{
					bool listen = sn_listen;
//					if ( socket = )
					ANET_Shutdown();
					ANET_Init();
					ANET_Listen( listen );
					socket = net_controlsocket;
					return 0;
				}
				break;
			case nSocketError_Ignore:
				return 0;
				break;
		}
	}

	return ret;
}

//=============================================================================

int ANET_MakeSocketBroadcastCapable (int socket)
{
	int	i = 1;

	// make this socket broadcast capable
	if (psetsockopt(socket, SOL_SOCKET, SO_BROADCAST, (char *)&i, sizeof(i)) < 0)
		return -1;
	net_broadcastsocket = socket;

	return 0;
}


//=============================================================================
int ANET_Broadcast (int& sock, const int8 *buf, int len, unsigned int port)
{
  int ret;

  if (sock != net_broadcastsocket)
    {
      if (net_broadcastsocket != 0)
	Sys_Error("Attempted to use multiple broadcasts sockets\n");
      ret = ANET_MakeSocketBroadcastCapable (sock);
      if (ret == -1)
	{
	  Con_Printf("Unable to make socket broadcast capable\n");
	  return ret;
	}
    }

  ((struct sockaddr_in *)&broadcastaddr)->sin_port = htons((unsigned short)port);

  return ANET_Write (sock, buf, len, &broadcastaddr);
}

//=============================================================================

int ANET_Write (int& socket, const int8 *buf, int len, struct sockaddr *addr)
{
	int ret;

//	tASSERT( socket < 0 || socket != net_acceptsocket );

	ret = psendto (socket, buf, len, 0, (struct sockaddr *)addr, sizeof(struct sockaddr));

	if ( ret == SOCKET_ERROR )
	{
		switch ( ANET_Error() )
		{
			case nSocketError_Reset:
				{
					bool listen = sn_listen;
					ANET_Shutdown();
					ANET_Init();
					ANET_Listen( listen );
					socket = net_controlsocket;
				}
				return 0;
				break;
			case nSocketError_Ignore:
				return 0;
				break;
		}
	}

	return ret;
}

//=============================================================================

char *ANET_AddrToString (const struct sockaddr *addr)
{
	static char buffer[22];
	int haddr;

	haddr = ntohl(((struct sockaddr_in *)addr)->sin_addr.s_addr);
	sprintf(buffer, "%d.%d.%d.%d:%d", (haddr >> 24) & 0xff, (haddr >> 16) & 0xff, (haddr >> 8) & 0xff, haddr & 0xff, ntohs(((struct sockaddr_in *)addr)->sin_port));
	return buffer;
}

//=============================================================================

int ANET_StringToAddr (const char *string, struct sockaddr *addr)
{
	int ha1, ha2, ha3, ha4, hp;
	int ipaddr;

	sscanf(string, "%d.%d.%d.%d:%d", &ha1, &ha2, &ha3, &ha4, &hp);
	ipaddr = (ha1 << 24) | (ha2 << 16) | (ha3 << 8) | ha4;

	addr->sa_family = AF_INET;
	((struct sockaddr_in *)addr)->sin_addr.s_addr = htonl(ipaddr);
	((struct sockaddr_in *)addr)->sin_port = htons((unsigned short)hp);
	return 0;
}

//=============================================================================

int ANET_GetSocketAddr (int socket, struct sockaddr *addr)
{
	int addrlen = sizeof(struct sockaddr);
	unsigned int a;

	memset(addr, 0, sizeof(struct sockaddr));
	pgetsockname(socket, (struct sockaddr *)addr, &addrlen);
	a = ((struct sockaddr_in *)addr)->sin_addr.s_addr;
	if (a == 0 || a == inet_addr("127.0.0.1"))
		((struct sockaddr_in *)addr)->sin_addr.s_addr = myAddr;

	return 0;
}

//=============================================================================

int ANET_GetNameFromAddr (const struct sockaddr *addr, char *name)
{
	struct hostent *hostentry;

	hostentry = pgethostbyaddr ((char *)&((struct sockaddr_in *)addr)->sin_addr, sizeof(struct in_addr), AF_INET);
	if (hostentry)
	{
		strncpy (name, (char *)hostentry->h_name, NET_NAMELEN - 1);
		return 0;
	}

	strcpy (name, ANET_AddrToString (addr));
	return 0;
}

//=============================================================================

int ANET_GetAddrFromName(const char *name, struct sockaddr *addr)
{
	struct hostent *hostentry;

	if (name[0] >= '0' && name[0] <= '9')
		return PartialIPAddress (name, addr);
	
	hostentry = pgethostbyname (name);
	if (!hostentry)
		return -1;

	addr->sa_family = AF_INET;
	((struct sockaddr_in *)addr)->sin_port = htons((unsigned short)net_hostport);	
	((struct sockaddr_in *)addr)->sin_addr.s_addr = *(int *)hostentry->h_addr_list[0];

	return 0;
}

//=============================================================================

int ANET_AddrCompare (struct sockaddr *addr1, struct sockaddr *addr2)
{
	if (addr1->sa_family != addr2->sa_family)
		return -1;

	if (((struct sockaddr_in *)addr1)->sin_addr.s_addr != ((struct sockaddr_in *)addr2)->sin_addr.s_addr)
		return -1;

	if (((struct sockaddr_in *)addr1)->sin_port != ((struct sockaddr_in *)addr2)->sin_port)
		return 1;

	return 0;
}

//=============================================================================

int ANET_GetSocketPort (struct sockaddr *addr)
{
	return ntohs(((struct sockaddr_in *)addr)->sin_port);
}


int ANET_SetSocketPort (struct sockaddr *addr, int port)
{
	((struct sockaddr_in *)addr)->sin_port = htons((unsigned short)port);
	return 0;
}

//=============================================================================
