/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Lookup stuff
 *
 *	by Tony Sideris	(05:40PM Apr 29, 2002)
 *================================================*/
#include "arson.h"

#include <qfileinfo.h>
#include <qtextstream.h>
#include <qfile.h>
#include <qdir.h>

#include <klineeditdlg.h>
#include <klocale.h>
#include <kconfig.h>

#include "mainwnd.h"
#include "lookup.h"
#include "submit.h"
#include "cdindex.h"
#include "cddb.h"
#include "konfig.h"

/*========================================================*/

ArsonLookup::ArsonLookup (uint type)
	: m_type(type)
{
	ARSON_INSTANCE_INCR("lookup");
}

ArsonLookup::~ArsonLookup (void)
{
	ARSON_INSTANCE_DECR("lookup");
}

/*========================================================*/

const QString &ArsonLookup::typeDesc (int type)
{
	static QString types[_lookup_max] = {
		i18n("freedb (http)"),
		i18n("freedb (local)"),
		i18n("cdindex"),
	};

	return types[type];
}

/*========================================================*/

const QString &ArsonLookup::typeDesc (void) const
{
	return typeDesc(m_type);
}

/*========================================================*/

ArsonLookup *ArsonLookup::create (const QString &str)
{
	ArsonLookup *ptr = NULL;
	const int pos = str.find(':');

	if (pos != -1)
	{
		switch (str.left(pos).toInt())
		{
		case freedbHttp:
			ptr = new ArsonFreedbHttpLookup;
			break;

		case freedbLocal:
			ptr = new ArsonFreedbLocalLookup;
			break;

		case cdindex:
			ptr = new ArsonCdindexLookup;
			break;
		}

		if (ptr)
		{
			const QString rest = str.right(str.length() - pos - 1);
			Trace("Building object from: %s\n", rest.latin1());
			ptr->fromString(rest);
		}
	}

	return ptr;
}

/*========================================================*/
/*	General internet lookup
 *========================================================*/

QString ArsonInetLookup::display (void) const
{
	return m_addr + ":" + QString::number(m_port);
}

QString ArsonInetLookup::toString (void) const
{
	return display();
}

void ArsonInetLookup::fromString (const QString &str)
{
	const int pos = str.find(':');

	if (pos != -1)
	{
		setAddr(str.left(pos));
		setPort(str.right(str.length() - pos - 1).toShort());
	}
}

/*========================================================*/
/*	CDINDEX access
 *========================================================*/

bool ArsonCdindexLookup::submit (const ArsonCdInfo &info) const
{
	arsonErrorMsg(
		i18n("cdindex submit is not currently supported..."));
	return false;
}

bool ArsonCdindexLookup::cdInfo (ArsonCdInfo &info) const
{
	ArsonCdindexDlg dlg (info, addr(), port());

	if (dlg.exec() == QDialog::Accepted)
	{
		info.merge(dlg.cdInfo());
		return true;
	}

	return false;
}

/*========================================================*/
/*	Freedb over HTTP
 *========================================================*/

class freedbSocket : public ArsonHttpPostSocket
{
public:
	freedbSocket (const QString &host, short port)
		: ArsonHttpPostSocket(host, port)
	{
		//	Nothing...
	}

private:
	virtual void response (int code, const QString &msg)
	{
		if (code != 200)
			arsonErrorMsg(
				i18n("Freedb server returned unknown or unwanted code %1: %2")
				.arg(code).arg(msg));
	}

	virtual QString path (void) const
	{
		return QString("/~cddb/submit.cgi");
	}
};

/*========================================================*/

bool ArsonFreedbHttpLookup::submit (const ArsonCdInfo &info) const
{
	QString email (ACONFIG.ripper().email());

	if (email != QString::null &&
		ACONFIG.ripper().is(ArsonConfig::RipperCfg::ripperEmailPrompt))
		email = QString::null;

	while (email == QString::null)
	{
		bool ok;
		email = KLineEditDlg::getText(
			i18n("Please enter your email address. This is needed by freedb so\nthey notify you if there is a problem with your submission."),
#ifdef ARSONDBG
			(QString(getenv("USER")) == "tonys111")	//	me == lazy
			? QString(BUGEMAIL) : QString::null
#else
			QString::null
#endif	//	ARSONDBG
			, &ok, ArsonFrame::getFrame());

		if (!ok)
			return false;

		if (email.find('@') < 1 || email.length() < 3)
		{
			arsonErrorMsg(
				i18n("That doesn't appear to be a valid email address, please enter one (or freedb will get mad at *me*)."));
			email = QString::null;
		}
	}

	ACONFIG.ripper().setEmail(email);

	if (email != QString::null)
	{
		freedbSocket *ps = new freedbSocket(addr(), port());
		ArsonFreedbSubmit submit (info, email);
		ArsonCdInfoSubmitDlg dlg (submit, ps);

		return (dlg.exec() == QDialog::Accepted);
	}

	return false;
}

/*========================================================*/

bool ArsonFreedbHttpLookup::cdInfo (ArsonCdInfo &info) const
{
	ArsonCddbDlg dlg (info, addr(), port());

	if (dlg.exec() == QDialog::Accepted)
	{
		info.merge(dlg.cdInfo());
		return true;
	}

	return false;
}

/*========================================================*/
/*	Local Freedb access
 *========================================================*/

QString ArsonFreedbLocalLookup::display (void) const
{
	QString res (m_path);

	if (autosave())
		res.append(i18n(" [auto]"));

	return res;
}

QString ArsonFreedbLocalLookup::toString (void) const
{
	return QString("%1:%2").arg(m_path).arg(m_autosave);
}

void ArsonFreedbLocalLookup::fromString (const QString &str)
{
	const QStringList sl = QStringList::split(":", str);

	m_path = sl[0];

	if (sl.count() > 1)
		m_autosave = sl[1].toInt() ? true : false;
	else
		m_autosave = false;
}

bool ArsonFreedbLocalLookup::submit (const ArsonCdInfo &info) const
{
	const QString dbFile (findFile(info, true));

	if (dbFile != QString::null)
	{
		ArsonFreedbSubmit writer (info, QString::null, false);

		Trace("Writing cddb entry to %s\n", dbFile.latin1());
		return writer.writeFile(dbFile);
	}

	return false;
}

bool ArsonFreedbLocalLookup::cdInfo (ArsonCdInfo &info) const
{
	QString categ;
	const QString dbFile (findFile(info, false, &categ));

	if (dbFile != QString::null)
	{
		QFile file (dbFile);

		Trace("Retrieving local cddb info from: %s\n",
			dbFile.latin1());

		if (file.open(IO_ReadOnly))
		{
			QTextStream ts (&file);
			ArsonCddbReadParser parser (&ts);

			if (parser.parse())
			{
				parser.applyTo(info);
				info.setCateg(categ);
				return true;
			}
		}
	}

	return false;
}

/*========================================================*/
/*	Searches for the given CDDB file in each
 *	category directory.
 *========================================================*/

QString ArsonFreedbLocalLookup::findCateg (const QString &id) const
{
	QDir dir (path());
	QStringList sl = dir.entryList(QDir::Dirs);
	QStringList::ConstIterator it, end;

	for (it = sl.begin(), end = sl.end(); it != end; ++it)
	{
		if (*it == "." || *it == ".." || !dir.cd(*it))
			continue;

		if (dir.exists(id))
			return *it;

		dir.cdUp();
	}

	return QString::null;
}

/*========================================================*/

QString ArsonFreedbLocalLookup::findFile (const ArsonCdInfo &info, bool create, QString *pc) const
{
	const QString cat (create ? info.categ() : findCateg(info.cddbID()));

	if (cat != QString::null)
	{
		QDir dir (path());

		if (create)
			dir.mkdir(cat);	//	It's OK if this fails...

		if (dir.cd(cat))
		{
			QFileInfo fi (dir, info.cddbID());

			if (create || fi.exists())
			{
				if (pc)
					*pc = cat;

				return fi.absFilePath();
			}
		}
		else if (create)
		{
			arsonErrorMsg(
				i18n("Unable to gain access to local freedb directory: %1/%2")
				.arg(dir.canonicalPath())
				.arg(cat));
		}
	}

	return QString::null;
}

/*========================================================*/
/*	Container for lookup objects
 *========================================================*/

ArsonLookupOrder::ArsonLookupOrder (void)
{
	//	Nothing...
}

/*========================================================*/

ArsonLookupOrder &ArsonLookupOrder::operator= (const ArsonLookupOrder &obj)
{
	LOOKUPS::ConstIterator it, end;
	clear();

	for (it = obj.m_lookups.begin(), end = obj.m_lookups.end(); it != end; ++it)
		addLookup((*it)->clone());

	return THIS;
}

/*==========================================================*/

int ArsonLookupOrder::addLookup (ArsonLookup *ptr)
{
	const int index = m_lookups.count();
	m_lookups.append(ptr);
	return index;
}

/*==========================================================*/

void ArsonLookupOrder::delLookup (int index)
{
	delete m_lookups[index];
	m_lookups.remove(m_lookups.at(index));
}

/*==========================================================*/

void ArsonLookupOrder::clear (void)
{
	LOOKUPS::Iterator it, end;

	for (it = m_lookups.begin(), end = m_lookups.end(); it != end; ++it)
		delete (*it);

	m_lookups.clear();
}

/*========================================================*/

void ArsonLookupOrder::autoSubmit (const ArsonCdInfo &info) const
{
	for (int index = 0; index < count(); ++index)
	{
		const ArsonFreedbLocalLookup *pl = (ArsonFreedbLocalLookup *) lookup(index);

		if (pl->type() != ArsonLookup::freedbLocal)
			continue;
		
		if (pl->autosave())
			pl->submit(info);
	}
}

/*==========================================================*/

#define LOOKUP_GROUP	"lookups"
#define LOOKUP_KEY		"lookups"

namespace arson {
	QString save_string (ArsonLookup *ptr)
	{
		return QString("%1:%2")
			.arg(ptr->type())
			.arg(ptr->toString());
	}
};

void ArsonLookupOrder::save (KConfig *pk)
{
	QStringList sl;
	pk->setGroup(LOOKUP_GROUP);

	for (int index = 0, size = count(); index < size; ++index)
		sl.append(arson::save_string(lookup(index)));

//	Trace("writing lookups: %s\n", sl.join(",").latin1());
	pk->writeEntry(LOOKUP_KEY, sl);
}

int ArsonLookupOrder::load (KConfig *pk)
{
	pk->setGroup(LOOKUP_GROUP);
	clear();

	const QStringList sl = pk->readListEntry(LOOKUP_KEY);

	for (QStringList::ConstIterator it = sl.begin(), end = sl.end();
		 it != end; ++it)
	{
		if (ArsonLookup *ptr = ArsonLookup::create(*it))
			addLookup(ptr);
	}

	return count();
}

/*========================================================*/
