/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Main window class implementation for Arson
 *
 *	by Tony Sideris	(05:53AM Jul 31, 2001)
 *================================================*/
#include "arson.h"

#include <kmenubar.h>
#include <kfiledialog.h>
#include <klocale.h>
#include <kapp.h>
#include <kstatusbar.h>
#include <kstdaction.h>
#include <kaction.h>
#include <kedittoolbar.h>
#include <kkeydialog.h>

#include <qlistbox.h>
#include <qtabwidget.h>
#include <qvbox.h>

#include "docwidget.h"

#include "configdlg.h"
#include "mainwnd.h"
#include "konfig.h"
#include "tools.h"
#include "isofs.h"
#include "logwnd.h"
#include "audiodoc.h"
#include "vcddoc.h"
#include "datadoc.h"
#include "cdripper.h"

/*========================================================*/

ArsonFrame *ArsonFrame::spMainFrame = NULL;

#define MAINWNDSETTINGS		"mainwnd-settings"

/*========================================================*/

namespace arson {
	void nameToolbars (KMainWindow *pw)
	{
		struct {
			const char *name;
			const QString lb;
			/*
			 *	Maps toolbars to user-friendly names for them
			 *	(for use in mini-title-bar, menu, etc...
			 */
		}	names[] = {
			{ "mainToolBar", i18n("Main Toolbar") },
			{ "barList", i18n("List Toolbar") },
			{ "barRipper", i18n("Ripper Toolbar") },
			{ "barTools", i18n("Tools Toolbar") },
		};

		for (int index = 0; index < ARRSIZE(names); ++index)
		{
			if (KToolBar *pt = pw->toolBar(names[index].name))
				pt->setLabel(names[index].lb);
#ifdef ARSONDBG
			else
			{ Trace("failed to find toolbar %s\n", names[index].name); }
#endif	//	ARSONDBG
		}
	}
};

/*========================================================*/
/*	Function implementations
 *========================================================*/

#define STARTPAGE		ACONFIG.startDoc()

ArsonFrame::ArsonFrame (const char *name)
	: KMainWindow(0, name),
	m_pTabCtrl(NULL),
	m_pDocActions(NULL),
	m_pCurrentDoc(NULL)
{
	int index;

	memset(&m_pages, 0, sizeof(m_pages));
	
	//	This is the one and only frame
	Assert(spMainFrame == NULL);
	spMainFrame = this;

	//	The collection of document-specific actions
	m_pDocActions = new KActionCollection(this);

	//	Connect app-wide actions
	setupActions();
	
	//	Create the main tab control
	m_pTabCtrl = new QTabWidget(this);
	m_pTabCtrl->setTabShape(QTabWidget::Triangular);

	createDocuments();
	
	//	Setup the rest of the frame
	statusBar()->message(i18n("Ready..."));
	setCentralWidget(m_pTabCtrl);

	QObject::connect(m_pTabCtrl,
		SIGNAL(currentChanged(QWidget*)),
		this,
		SLOT(slotTabSelChange(QWidget*)));

	setCurrentDocument(STARTPAGE);

	createGUI(
		/*	In debug this forces the use of the local
		 *	RC file, in case arson is installed...
		 */
#ifdef ARSONDBG
		ARSON_SRCDIR "/src/arsonui.rc"
#endif	//	ARSONDBG
		);

	arson::nameToolbars(this);
	applyMainWindowSettings(kapp->config());
}

ArsonFrame::~ArsonFrame (void)
{
	spMainFrame = NULL;
}

/*========================================================*/

void ArsonFrame::createDocuments (void)
{
	ArsonDocWidget *docs[_DOCUMENT_MAX] = {
		new ArsonMp3Doc(m_pTabCtrl),
		new ArsonVcdDoc(m_pTabCtrl),
		new ArsonDataDoc(m_pTabCtrl),
		new ArsonRipperDoc(m_pTabCtrl),
	};

	for (int index = 0; index < _DOCUMENT_MAX; ++index)
	{
		ArsonDocWidget *pd = m_pages[index] = docs[index];
		pd->setActionCollection(docActions());

		pd->setStatusBar(statusBar());
		pd->create();
		pd->show();

		m_pTabCtrl->addTab(pd, documentTitle(index));
	}
}

/*========================================================*/

const QString &ArsonFrame::documentTitle (int doc)
{
	static QString tits[] = {		// hehe...
		i18n("&Audio CD"),
		i18n("&VCD"),
		i18n("&Data CD"),
		i18n("CD Ri&pper"),
	};

	return tits[doc];
}

/*========================================================*/
/*	Toggle actions which remember the last activated
 *	action. These are used for otggling the toolbars
 *	and log window.
 *========================================================*/

class toggleAction : public KToggleAction
{
public:
	toggleAction (const QString &text,
		int accel, QObject *recv, const char *slot,
		QObject *parent, const char *name)
		: KToggleAction(text, accel, recv, slot, parent, name)
	{
		//	Nothing...
	}

	virtual void slotActivated (void)
	{
		lastActivated = this;
		KToggleAction::slotActivated();
	}

	static toggleAction *last (void) { return lastActivated; }
	
private:
	static toggleAction *lastActivated;
};

/*========================================================*/

class toggleWidgetBase : public toggleAction
{
public:
	toggleWidgetBase (const QString &text,
		int accel, QObject *recv, const char *slot,
		QObject *parent, const char *name)
		: toggleAction(text, accel, recv, slot, parent, name)
	{
		//	Nothing...
	}

	virtual QWidget *widget (void) = 0;
};

/*========================================================*/

class toggleWidget : public toggleWidgetBase
{
public:
	toggleWidget (const QString &text,
		int accel, QObject *recv, const char *slot,
		QObject *parent, const char *name, QWidget *widget)
		: toggleWidgetBase(text, accel, recv, slot, parent, name), m_widget(widget)
	{
		//	Nothing...
	}

	virtual QWidget *widget (void) { return m_widget; }

protected:
	QWidget *m_widget;
};

/*========================================================*/
//#ifndef ARSON_KDE3
class toggleToolbar : public toggleWidgetBase
{
public:
	toggleToolbar (const QString &text,
		int accel, QObject *recv, const char *slot,
		QObject *parent, const char *name, const char *tbname)
		: toggleWidgetBase(text, accel, recv, slot, parent, name), m_tbname(tbname)
	{
		//	Nothing...
	}

	virtual QWidget *widget (void)
	{
		if (ArsonFrame *pf = ArsonFrame::getFrame())
			return pf->toolBar(m_tbname);

		return NULL;
	}

protected:
	const char *m_tbname;
};
//#endif	//	ARSON_KDE3

toggleAction *toggleAction::lastActivated = NULL;

/*========================================================*/
/*	Create ALL teh application's actions.
 *========================================================*/

#define STDACTION(id,slot)					KStdAction::action((id), this, (slot), actionCollection())
#define ACTION(text,img,accel,slot,name)	new KAction((text), (img), (accel), this, (slot), actionCollection(), (name))

#define STDDOCACTION(id)					KStdAction::action((id), NULL, NULL, docActions())
#define DOCACTION(text,img,accel,name)		new KAction((text), (img), (accel), docActions(), (name))

#define WIDGETACTION(desc,name,widget)		new toggleWidget((desc), 0, this, SLOT(slotToggleVisibility()), actionCollection(), (name), (widget))

//#ifndef ARSON_KDE3
#define TOOLBARACTION(desc,tb)				(new toggleToolbar((desc), 0, this, SLOT(slotToggleVisibility()), actionCollection(), QString("toggle_") + (tb), tb))->setChecked(true)
//#endif	//	ARSON_KDE3

void ArsonFrame::setupActions (void)
{
	typedef QValueList<KAction*> ACTIONS;

	uint index;
	ACTIONS acts;

	STDACTION(KStdAction::Open, SLOT(slotOpen()));
	STDACTION(KStdAction::Quit, SLOT(close()));
	STDACTION(KStdAction::Preferences, SLOT(slotConfig()));
	STDACTION(KStdAction::KeyBindings, SLOT(slotConfigKeys()));
	STDACTION(KStdAction::ConfigureToolbars, SLOT(slotConfigToolbars()));
	KRecentFilesAction *pr = (KRecentFilesAction *) STDACTION(
		KStdAction::OpenRecent, SLOT(slotOpenRecent(const KURL&)));

	ACTION(i18n("&Create CD from Image..."), "writeimg", 0, SLOT(slotWriteImg()), "misc_write_img");
	ACTION(i18n("Co&py CD..."), "cdcopy", 0, SLOT(slotCopyCd()), "misc_copy_cd");
	ACTION(i18n("&Read CD Image to File..."), "readimg", 0, SLOT(slotReadCd()), "misc_read_cd");
	ACTION(i18n("Create &Image from Directory..."), "writedir", 0, SLOT(slotWriteDir()), "misc_write_dir");
	ACTION(i18n("&Unlock Drive..."), "lock", 0, SLOT(slotUnlock()), "misc_unlock");
	ACTION(i18n("Blan&k CDRW..."), "eraser", 0, SLOT(slotBlank()), "misc_blank");
	ACTION(i18n("Fi&xate Disk..."), NULL, 0, SLOT(slotFixate()), "misc_fixate");

	ACTION(i18n("&Import Playlist..."), "fileimport", 0, SLOT(slotImport()), "file_import");

	DOCACTION(i18n("&Text File..."), NULL, 0, "file_export_text");
	DOCACTION(i18n("&M3U File..."), NULL, 0, "file_export_m3u");

#ifdef ARSONDBG
	ACTION("&Debug", NULL, 0, SLOT(slotDebug()), "file_debug");
#endif	//	ARSONDBG

	DOCACTION(i18n("Rescan &Drive"), "cdscan", Qt::CTRL + Qt::Key_R, "ripper_rescan");
	DOCACTION(i18n("&Toggle Checked"), "toggle", Qt::CTRL + Qt::Key_T, "ripper_toggle_checked");
	DOCACTION(i18n("&Rip Tracks..."), "rip", Qt::CTRL + Qt::Key_F5, "ripper_rip");
	DOCACTION(i18n("&Edit Disk Info..."), "edit", 0, "ripper_edit");

	STDDOCACTION(KStdAction::New);
	STDDOCACTION(KStdAction::Save);
	STDDOCACTION(KStdAction::SaveAs);

	DOCACTION(i18n("&Add File(s)..."), "addfile", Qt::Key_Insert, "list_add");
	DOCACTION(i18n("Add Direc&tory..."), "addfolder", Qt::CTRL + Qt::Key_Insert, "list_add_dir");
	DOCACTION(i18n("&Shuffle List"), "shuffle", Qt::CTRL + Qt::Key_F, "list_shuffle");
	DOCACTION(i18n("&Edit CD-Text..."), "edit", 0, "list_edit_info");
	DOCACTION(i18n("Write CD..."), "burn", Qt::Key_F5, "list_write_cd");

	DOCACTION(i18n("&Open File"), "exec", 0, "list_open");
	DOCACTION(i18n("Open &With..."), NULL, 0, "list_open_with");
	DOCACTION(i18n("&View..."), NULL, 0, "list_file_view");

	DOCACTION(i18n("&Remove File(s)"), "editdelete", Qt::Key_Delete, "list_del");
	DOCACTION(i18n("Move &Up"), "up", Qt::CTRL + Qt::Key_Up, "list_up");
	DOCACTION(i18n("Move &Down"), "down", Qt::CTRL + Qt::Key_Down, "list_down");

	DOCACTION(i18n("&Invert Selection"), NULL, 0, "list_invert");

	DOCACTION(i18n("&New Folder"), "folder_new", 0, "list_folder_new");
	DOCACTION(i18n("&Rename"), NULL, Qt::Key_F2, "list_rename");
	
	STDDOCACTION(KStdAction::SelectAll);

//#ifndef ARSON_KDE3
	/*	You get this for free in KDE3.0.1 or so...
	 */
	TOOLBARACTION(i18n("Show &Main Toolbar"), "mainToolBar");
	TOOLBARACTION(i18n("Show &Burner Toolbar"), "barList");
	TOOLBARACTION(i18n("Show &Ripper Toolbar"), "barRipper");
	TOOLBARACTION(i18n("Show &Tools Toolbar"), "barTools");
//#endif	//	ARSON_KDE3
	
	WIDGETACTION(i18n("Show Command Log"), "toggle_log_window", ArsonLogWindow::Wnd());
	
	//	Let the log window know about it's toggle action
	ArsonLogWindow::Wnd()->addUiItem(
		new ArsonLogActionState((KToggleAction *)
			actionCollection()->action("toggle_log_window")));

	/*	Disable all doc-actions, the first
	 *	document will re-enable the ones it needs.
	 */
	for (index = 0, acts = m_pDocActions->actions();
		 index < acts.count(); ++index)
		acts[index]->setEnabled(false);

	acts = m_pDocActions->actions();
	KActionCollection *pac = actionCollection();

	for (ACTIONS::Iterator it = acts.begin(), end = acts.end();
		 it != end; ++it)
		pac->insert(*it);

	pr->loadEntries(kapp->config());
}

/*========================================================*/

ArsonDocWidget *ArsonFrame::findDocument (const QString &doctype)
{
	const QString front ("arson-");
	const QString type = doctype.lower();

	for (int index = 0; index < _DOCUMENT_MAX; ++index)
	{
		if (m_pages[index])
		{
			const QString tmp (front + m_pages[index]->propDocType());

			if (tmp == type)
			{
				setCurrentDocument(index);
				return m_pages[index];
			}
		}
	}

	return NULL;
}

/*========================================================*/
/*	Signal handlers
 *========================================================*/

void ArsonFrame::slotDebug (void)
{
	int index;
	QValueList<KAction*> al = actionCollection()->actions();

	for (index = 0; index < al.count(); index++)
		DBGOUT("Action: %s -> %08x\n",
			al[index]->name(),
			al[index]);

	Trace("=====================================\n");
	ARSON_INSTANCE_DUMP_ALL();

	ArsonProgramHelp help (ArsonConfig::PROGRAM_LAME);

	if (help.valid())
		help.show();
}

/*========================================================*/

void ArsonFrame::slotConfig (void)
{
	ArsonConfigDlg(ACONFIG, ArsonConfigDlg::BURNER, this).exec();
}

/*========================================================*/

void ArsonFrame::slotOpen (void)
{
	const KURL url =
		KFileDialog::getOpenURL(NULL,
			ArsonFileFilter(ArsonDocWidget::ARSON_FILTER,
				ArsonFileFilter::AllFiles).toString());

	if (!url.isEmpty())
		arsonOpenDoc(url);
}

/*========================================================*/

void ArsonFrame::slotTabSelChange (QWidget *pCurrent)
{
	const int index = m_pTabCtrl->currentPageIndex();

	Trace("Tab selection change (index = %d)\n", index);
	setCurrentDocument(index);
}

/*========================================================*/

ArsonDocWidget *ArsonFrame::setCurrentDocument (int index)
{
	if (ArsonDocWidget *pc = currentDocument())
		pc->setActive(false);

	if ((m_pCurrentDoc = m_pages[index]))
	{
		m_pCurrentDoc->setActive(true);
	
		if (m_pTabCtrl->currentPageIndex() != index)
			m_pTabCtrl->setCurrentPage(index);
	}
	
	return currentDocument();
}

/*========================================================*/

void ArsonFrame::slotWriteImg (void)
{
	arsonWriteImg();
}

void ArsonFrame::slotReadCd (void)
{
/*
	const QString filename = KFileDialog::getSaveFileName(QString::null,
		i18n("*.iso|ISO Files (*.iso)\n*|All Files"));

	if (filename != QString::null)
		ArsonReadCdMgr::readCd(filename);
*/
	arsonReadCd();
}

void ArsonFrame::slotCopyCd (void)
{
	arsonCdCopy();
}

void ArsonFrame::slotWriteDir (void)
{
	arsonWriteDir();
}

/*========================================================*/
/*	Toggle toolbar and widget visibility
 *========================================================*/

void ArsonFrame::slotToggleVisibility (void)
{
	if (toggleWidgetBase *ptr = (toggleWidgetBase *) toggleAction::last())
	{
		if (QWidget *widget = ptr->widget())
		{
			if (ptr->isChecked())
				widget->show();
			else
				widget->hide();
		}
	}
}

/*========================================================*/

void ArsonFrame::closeEvent (QCloseEvent *close)
{
	saveMainWindowSettings(kapp->config());

	//	Clear out all documents
	for (int index = 0; index < _DOCUMENT_MAX; ++index)
	{
		if (m_pages[index])
		{
			try { m_pages[index]->deleteContents(); }
			catch (ArsonAbortAction&) {
				//	Cancelled
				return;
			}
		}
	}
	
	ACONFIG.setGeometry(geometry());
	close->accept();

	if (KRecentFilesAction *pr = (KRecentFilesAction *)
		actionCollection()->action("file_open_recent"))
		pr->saveEntries(kapp->config());
}

/*========================================================*/

void ArsonFrame::slotConfigToolbars (void)
{
	KEditToolbar dlg (actionCollection(), xmlFile());

	if (dlg.exec())
		createGUI();
}

/*========================================================*/

void ArsonFrame::slotConfigKeys (void)
{
	KKeyDialog::configureKeys(actionCollection(), xmlFile());
}

/*========================================================*/

void ArsonFrame::slotUnlock (void)
{
	arsonUnlockDrive();
}

/*========================================================*/

void ArsonFrame::slotBlank (void)
{
	arsonBlankCdrw();
}

/*========================================================*/

void ArsonFrame::slotOpenRecent (const KURL &url)
{
	arsonOpenDoc(url);
}

/*========================================================*/

void ArsonFrame::slotFixate (void)
{
	arsonFixate();
}

/*========================================================*/

void ArsonFrame::slotImport (void)
{
	const KURL url = KFileDialog::getOpenURL(QString::null,
		ArsonFileFilter(i18n("*.md5|MD5 Files\n*.m3u|M3U Files")).toString());

	if (!url.isEmpty())
	{
		const QString ext = QFileInfo(url.fileName()).extension().lower();
		ArsonFileListDoc *pd = (ArsonFileListDoc *)
			setCurrentDocument(DOCUMENT_AUDIO);

		Trace("Checking extension %s\n", ext.latin1());
		
		if (ext == "md5")
			pd->openMd5File(url);

		else if (ext == "m3u")
			pd->openM3uFile(url);

		else
			arsonErrorMsg(
				i18n("Unknown file format: %1")
				.arg(url.path()));
	}
}

/*========================================================*/
