/*
 * Copyright (c) 2003 Brian Kreulen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/* Index file creation implementation
 *
 *	by Brian Kreulen (06:49PM Mar 08, 2003)
 *================================================*/

#include "arson.h"
#include <klocale.h>
#include <qfile.h>
#include <qdir.h>
#include <qvaluelist.h>
#include <qlistview.h>

#include "cdinfo.h"
#include "cdripper.h"
#include "audiodoc.h"
#include "playlist.h"

 /*========================================================*/
/*	Playlist File list document
 *========================================================*/

ArsonPlaylistFileListDoc::ArsonPlaylistFileListDoc (QWidget *parent, const char *name)
	: ArsonFileListDoc(parent, name), m_fileMode(Playlist_Overwrite)
{
}

/*=========================================================================
  This method is used to import an existing filelist document into our own
  We assume that this document has called create()
  =========================================================================*/
void ArsonPlaylistFileListDoc::importRipperFileListDoc (ArsonRipperDoc *pflDoc)
{
    int trackNo;
    KURL url;
    ArsonPlaylistListItem *pListItem;
    ArsonRipperDoc::ITEMLIST chkItems = pflDoc->checkedItems();
    QValueListConstIterator<ArsonRipperDoc::ITEMPAIR> it;
    QString plFileStr;
    QString cdiFormat;

    for ( it = chkItems.begin(); it != chkItems.end(); ++it )
    {
        trackNo = (*it).pListItem->text(0).toInt();
        trackNo--; // gotta have it zero-based

        plFileStr = g_CONFIG.ripper().outdir();

        // Make sure last character is a '/'
        if (plFileStr[plFileStr.length()] != '/')
            plFileStr.append('/');

        // Add on our cdiFormat filename
        cdiFormat = g_CONFIG.ripper().cdiFormat();
        if(!pflDoc->info().formatTrackName(trackNo + 1,cdiFormat,cdiFormat))
        {
            Trace("Error in formatting track Name, check your ripper config");
            return;
        }

        plFileStr += cdiFormat;
        url.setPath(plFileStr + pflDoc->outputFormat()->extension);

        // Take this information and add a playlist list item to our doc
        pListItem = createPlaylistItem(url,
            pflDoc->info().track(trackNo).time().totalSeconds(),
            pflDoc->info().track(trackNo).title());
        addItem(pListItem);
    }
}

/*========================================================*/

void ArsonPlaylistFileListDoc::importMp3ListDoc (ArsonMp3Doc *pflDoc)
{
    KURL url;
    ArsonListItem *pListItem;
    ArsonPlaylistListItem *pPlaylistItem;
    const ArsonCdInfo *pcdInfo = pflDoc->cdInfo();
    int cnt = 0;

	for (ItemIterator it(pflDoc); it; ++it)
    {
        pListItem = it.item();
        url.setPath(pListItem->persist());

        // Take this information and add a playlist list item to our doc
        pPlaylistItem = createPlaylistItem(url,
            pcdInfo->track(cnt).time().totalSeconds(),
            pListItem->display());
        addItem(pPlaylistItem);

        ++cnt;
    }    
}

/*========================================================*/
    
ArsonM3UFileWriter *ArsonPlaylistFileListDoc::createM3UFileWriter (const KURL &url, int fileMode)
{
    KURL empty;
    return new ArsonM3UFileWriter(this, url, fileMode, empty);
}

/*========================================================*/

ArsonPlaylistListItem *ArsonPlaylistFileListDoc::createPlaylistItem (const KURL &url, int length, const QString &name) const
{
    ArsonPlaylistListItem *rtrn = new ArsonPlaylistListItem(url);
    rtrn->setSongLength(length);
    rtrn->setSongName(name);
    return rtrn;
}

/*========================================================*/

void ArsonPlaylistFileListDoc::addItem (ArsonPlaylistListItem *ptr, ArsonLvPos *pos)
{
    ArsonListDoc::addItem(ptr, pos);
}

/*========================================================*/

bool ArsonPlaylistFileListDoc::saveDocument (const KURL &url)
{
    ArsonM3UFileWriter *m3uWriter = createM3UFileWriter(url, (m_fileMode == Playlist_Overwrite) ?
        ArsonM3UFileWriter::M3uWriter_Overwrite : ArsonM3UFileWriter::M3uWriter_Append);

    if (m3uWriter)
    {
        writeDocumentFile(*m3uWriter);
        delete m3uWriter;
    }
    else
    {
		arsonErrorMsg(i18n("Error opening file: %1").arg(url.prettyURL()));
        return false;
    }

    return true;
}

/*========================================================*/
/*	Write an m3u file
 *  - We send a new, empty KURL to ArsonFileWriter so it won't copy any
 *  - temp files when we're done.
 *========================================================*/

ArsonM3UFileWriter::ArsonM3UFileWriter (ArsonFileListDoc *pd, const KURL &url, int fileMode, KURL &empty)
	: ArsonFileWriter(pd, empty), m_fileMode(fileMode), m_stream(NULL)
{
    QString fileName = url.prettyURL();
    fileName = fileName.remove(fileName.find("file:"), 5);

    // If the m3u file does NOT exist, switch to overwrite mode, otherwise
    // we might miss the opening line (#EXTM3U)
    if (!QFile::exists(fileName))
        m_fileMode = fileMode = M3uWriter_Overwrite;
    
    verifyDirectory(fileName);
    m_indexFile.setName(fileName);

    m_indexFile.open((fileMode == M3uWriter_Append) ? (IO_ReadWrite | IO_Append) : IO_WriteOnly);
    m_stream = new QTextStream(&m_indexFile);

    if (!m_indexFile.isOpen() || m_stream == NULL)
		arsonErrorMsg(i18n("Error opening file: %1").arg(fileName));
}

/*========================================================*/

ArsonM3UFileWriter::~ArsonM3UFileWriter ()
{
    if (!m_indexFile.isOpen())
        m_indexFile.close();

    if (m_stream != NULL)
        delete m_stream;
}

/*========================================================*/

bool ArsonM3UFileWriter::visit (QListViewItem *pi, ArsonListItem *pl)
{
    if (!m_indexFile.isOpen())
        return false;

    ArsonPlaylistListItem *pf = (ArsonPlaylistListItem *) pl;

    // see http://hanna.pyxidis.org/tech/m3u.html for the m3u specification
    writeM3uLine("#EXTINF:" + QString::number(pf->songLength()) + "," + pf->display());

    QString absFile = pf->persist();
    absFile = absFile.remove(absFile.find("file:"), 5);

    if (g_CONFIG.ripper().pathType() == 0) {
        int pos = absFile.findRev('/');
        Trace("Using Relative Paths: %d : ", pos);
        
        if (pos > 0)
            absFile = absFile.remove(0,pos + 1);
    } else {
        Trace("Using Absolute Paths: ");
    }

    Trace("%s\n", (const char *)absFile);
    
	writeM3uLine(absFile);

	return true;
}

/*========================================================*/

void ArsonM3UFileWriter::beginDocument (void)
{
    if (!m_indexFile.isOpen())
        return;

    // only write the next line if we're writing a fresh index file
    if (m_fileMode == M3uWriter_Overwrite)
        writeM3uLine("#EXTM3U");
}

/*========================================================*/

void ArsonM3UFileWriter::writeM3uLine(const QString &str) {
	textStream() << str << endl;
}

/*========================================================*/

bool ArsonM3UFileWriter::verifyDirectory (const QString &dirName)
{
	const QDir srcdir (QFileInfo(dirName).dir());

	if (!srcdir.exists())
	{
		QDir dir = QDir::root();
		QStringList::ConstIterator it, end;
		const QStringList sl(
			QStringList::split(QChar('/'),
				srcdir.absPath()));

		for (it = sl.begin(), end = sl.end(); it != end; ++it)
		{
			const bool create = dir.mkdir(*it);

			if (!dir.cd(*it))
			{
				if (!create)
					arsonErrorMsg(
						i18n("Failed to create output directory: %1 in %2")
						.arg(*it)
						.arg(dir.absPath()));
				else
					arsonErrorMsg(
						i18n("Failed to access %1 in %2")
						.arg(*it)
						.arg(dir.absPath()));

				return false;
			}
		}
	}

	return true;
}
