<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    file-defs.php                                           */
/* Author:      Paul Waite                                              */
/* Description: Definitions for filesystem access.                      */
/*                                                                      */
/* ******************************************************************** */
/** @package file */

// ----------------------------------------------------------------------
// Definitions
/** Save the file on close */
define("SAVE_FILE",    0);
/** Abandon file on close */
define("ABANDON_FILE", 1);

// ----------------------------------------------------------------------
/**
* Return a random unique filename, with optional prefix and an optional
* extension. The filename is composed from an md5 of microtime(), plus
* the prefix and extension if given.
*
* @param string $prefix  Optional fixed prefix required.
* @param string $extn  Optional fixed extension required.
* @return string Unique filename string
*/
function unique_filename($prefix="", $extn="") {
  $fname = md5( $prefix . microtime() . $extn );
  $fname = $prefix . $fname;
  if ($extn != "") {
    $fname .= ".$extn";
  }
  return $fname;
} // unique_filename

// ----------------------------------------------------------------------
/**
* Make sure path exists
* A utility function which makes sure a given directory path
* is made, using mkdir as required. This function assumes
* that permissions are all as needed.
* @param string $path  The directory path which must exist
*/
function mkpath($path) {
  if (substr($path, 0, 2) != "./" && substr($path, 0, 1) != "/") {
    $path = "./" . $path;
  }
  $result = true;
  clearstatcache();
  if (!is_dir($path)) {
    $dirs = explode("/", $path);
    $testpath = "";
    foreach ($dirs as $dir) {
      if ($dir == "") $testpath = "/";
      elseif ($dir == ".") $testpath = realpath($dir);
      else $testpath .= "/$dir";
      if (!is_dir($testpath)) {
        if (!mkdir($testpath)) {
          $result = false;
          break;
        }
      }
    }
  }
  return $result;
} // mkpath

// ----------------------------------------------------------------------
/**
* Return the file extension from a filepath.
*/
function get_file_extn($path){
  $extn = "";
  if ($path != "") {
    $fname = basename($path);
    $bits = explode(".", $fname);
    $extn = strtolower(trim($bits[ count($bits) - 1 ]));
  }
  return $extn;
} // get_file_extn

/**
* Return the file stem (name without extn) from a filepath.
*/
function get_file_stem($path){
  $stem = "";
  if ($path != "") {
    $fname = basename($path);
    $bits = explode(".", $fname);
    $stem = strtolower(trim($bits[ 0 ]));
  }
  return $stem;
} // get_file_stem

// ----------------------------------------------------------------------
/**
* Returns an array containing full paths to the files in the directory
* given by the path. This can be optionally recursive, in which case we will
* end up with a long list of path/filenames which comprise the whole
* tree, minus the special files '.' and '..'
* @param string $path Path to the directory containing the files to list
* @param boolean $recurse Whether to recurse the tree listing files in subdirs
* @param $regex Full regular expression filename must match, eg: '/^.*?\.jpg$/i'
* @return array List of full paths that comprise files in the directory, or tree
*/
function get_dirlist($path, $recurse=false, $regex="") {
  $dirlist = array();
  if ($dirH = opendir($path)) {
    while (false !== ($filename = readdir($dirH))) {
      if ($filename != "." && $filename != "..") {
        $filetype = filetype("$path/$filename");
        $valid = (($regex == "" || $filetype == "dir") ? true : preg_match($regex, $filename));
        if ($valid) {
          if ($filetype == "dir") {
            if ($recurse) {
              $dirlist += get_dirlist("$path/$filename", $recurse, $regex);
            }
            else {
              $dirlist[] = "$path/$filename";
            }
          }
          else {
            $dirlist[] = "$path/$filename";
          }
        }
      }
    } // while
    closedir($dirH);
  }
  // Return array of paths..
  return $dirlist;
} // get_dirlist

// ----------------------------------------------------------------------
/**
* The inputfile class manages files for input. It opens, reads and closes
* files in input-only mode.
* @package file
*/
class inputfile {
  // Public
  /** The file name */
  var $name = "";
  /** The directory prefix (if any) of the file */
  var $dir = "";
  /** The full path to the file */
  var $datafname = "";
  /** True if file was opened */
  var $opened = false;
  /** Size of opened file in bytes */
  var $filesize = 0;
  /** Content of file after readall() */
  var $content = "";
  // Private
  /** The file pointer
      @access private */
  var $fp;
  // ....................................................................
  /**
  * Constructor
  * Open the given file for input.
  * @param  string  $name The name of the file
  * @param  string  $dir  The directory the file is in
  * @return boolean True if file was opened, else false
  */
  function inputfile($name, $dir="") {
    $this->opened = false;
    if ($dir != "") {
      // Separate filename and directory..
      $this->name = $name;
      $this->dir = $dir;
    }
    else {
      // Combined path/filename..
      $this->dir = dirname($name);
      $this->name = basename($name);
    }

    // Full pathnames..
    $this->datafname = $this->dir . "/" . $this->name;
    clearstatcache();
    if (file_exists($this->datafname) && is_file($this->datafname)) {
      $this->fp = fopen($this->datafname, "r");
      if ($this->fp === false) {
        $this->opened = false;
      }
      else {
        $this->opened = true;
        $this->filesize = filesize($this->datafname);
      }
    }
    return $this->opened;
  } // inputfile
  // ....................................................................
  /**
  * Reads all of the file contents into class variable $content.
  * @return boolean True if the content was successfully read
  */
  function readall() {
    if ($this->opened) {
      $this->content = fread($this->fp, $this->filesize);
      return $this->content;
    }
    else return false;
  }
  // ....................................................................
  /**
  * Reads the next line of the file, up to max chars specified.
  * @param integer $maxchars Maximum chars per line to read
  * @return mixed A string containing the line, or false if failed
  */
  function readln($maxchars=4096) {
    if ($this->opened && !$this->eof()) {
      $buf = fgets($this->fp, $maxchars);
      return rtrim($buf);
    }
    else return false;
  }
  // ....................................................................
  /** Return true if at end-of-file or not opened, else false */
  function eof() {
    $eof = true;
    if ($this->opened) {
      $eof = feof($this->fp);
    }
    return $eof;
  }
  // ....................................................................
  /** Close the file */
  function closefile() {
    if ($this->opened) {
      fclose($this->fp);
    }
  }
} // inputfile class

// ----------------------------------------------------------------------
/**
* CSV Inputfile class
* Manage CSV-formatted files for input.
* Open, close and read CSV formatted files for input.
* @package file
*/
class csv_inputfile extends inputfile {
  /**
  * Constructor
  * Open the given CSV formatted file for input.
  * @param  string  $name The name of the file
  * @param  string  $dir  The directory the file is in
  * @return boolean True if file was opened, else false
  */
  function csv_inputfile($name, $dir="") {
    return $this->inputfile($name, $dir);
  }
  // ....................................................................
  /**
  * Reads the next line of the file in the form of an array of
  * fields as delimited by the given charater. Lines can be of
  * a length up to the specified maximum no. of chars.
  * NB: This properly handles double-quoted elements which
  * contain the delimiter character.
  * @param string  $delimiter CSV delimiter character, default ","
  * @param integer $maxchars  Maximum chars per line to read
  * @return array An array of the CSV fields read in
  */
  function readln($delimiter=",", $maxchars=4096) {
    if ($this->opened && !feof($this->fp)) {
      $csvarray = fgetcsv($this->fp, $maxchars, $delimiter);
      return $csvarray;
    }
    else return false;
  }
} // csv_inputfile class

// ----------------------------------------------------------------------
/**
* Open a file which contains a stylesheet and read in the style
* settings. This is a special case of inputfile.
* @package file
*/
class stylesheet extends inputfile {
  /** @access private */
  var $class;
  // ....................................................................
  /**
  * Constructor
  * Open the given stylesheet file for input.
  * @param  string  $name The name of the file
  * @param  string  $dir  The directory the file is in
  * @return boolean True if file was opened, else false
  */
  function stylesheet($sspath, $dir="") {
    if ($this->inputfile($sspath, $dir)) {
      // We support the DIY readln() and readall() which
      // are provided by inputfile, but we also read the
      // stylesheet into our special structure, so that
      // styles are easily accessible..
      $this->class = array();
      $this->readstyles();
    }
  }
  // ....................................................................
  /**
  * Read in the styles from stylesheet. We have repeated
  * entries of the form: classlist { attrib: def; attrib: def; }
  * NB: this does not, as yet, cope with 'complex' stylesheets
  *     which have inheritance in the name definitions.
  * @access private
  */
  function readstyles() {
    if ($this->opened) {
      if ($this->readall()) {
        // Remove comments, normalise case..
        $content = strtolower(preg_replace("/\/\*.*?\*\//", "", $this->content));
        $entries = explode("}", $content);
        foreach ($entries as $entry) {
          $entrybits = explode("{", $entry);
          $classlist = trim($entrybits[0]);
          $classes = explode(",", $classlist);
          foreach ($classes as $classname) {
            $classname = trim($classname);
            if ($classname != "") {
              if (substr($classname, 0, 1) == ".") {
                $classname = substr($classname, 1);
              }
              if (isset($this->class[$classname])) {
                $classattr = $this->class[$classname];
                //debugbr("ADDING TO CLASS $classname");
              }
              else {
                $classattr = array();
                //debugbr("FIRST TIME FOR CLASS $classname");
              }
              $classdefs = explode(";", $entrybits[1]);
              foreach($classdefs as $classdef) {
                $defbits = explode(":", $classdef);
                $name    = trim($defbits[0]);
                $setting = trim($defbits[1]);
                if ($name != "") {
                  $classattr[$name] = $setting;
                  //debugbr("[$classname][$name] -> [$setting]");
                }
              }
              // Store this class away..
              $this->class[$classname] = $classattr;
              //debugbr("CLASS: [$classname]");
            }
          } // foreach classname
        } // foreach entry
      } // if readall
      $this->closefile();
    } // if opened
  } // readstyles
  // ....................................................................
  /**
  * Return a style setting(s) for given class name in the list format:
  *   "attrib:def;attrib:def; ..."
  * Also allow specification of a particular attribute within the class
  * to return the style setting for.
  * @param  string  $classname The name of the class
  * @param  string  $attrname  The name of the class attribute
  * @return boolean True if file was opened, else false
  */
  function style($classname, $attrname="") {
    $style = "";
    $classname = strtolower($classname);
    $attrname = strtolower($attrname);
    if (isset($this->class[$classname])) {
      $class = $this->class[$classname];
      if ($attrname != "") {
        if (isset($class[$attrname])) {
          $style = $class[$attrname];
        }
      }
      else {
        while (list($name, $setting) = each($class)) {
          $style .= "$name:$setting;";
        }
      }
    }
    return $style;
  }
} // stylesheet class

// ----------------------------------------------------------------------
/**
* The outputfile class manages files for outputting content. This includes
* both appending to an existing file, and creating new files. The method
* used in both cases is to write to a temporary file, and then rename/move
* it onto the final file path when closefile() is called.
* @package file
*/
class outputfile {
  // Public
  /** The file name */
  var $name = "";
  /** The directory prefix (if any) of the file */
  var $dir = "";
  /** The full path to the file */
  var $datafname = "";
  /** Type of file 'text' or 'templated' */
  var $filetype = "text";
  /** True if file was sucessfully opened */
  var $opened = false;
  /** How to save: default "overwrite", or "append" to append. */
  var $savehow = "overwrite";

  // Private
  /** Full pathname of template file
      @access private */
  var $template_fname = "";
  /** Content of template file when read in
       @access private */
  var $template_contents = "";
  /** Full pathname of destination (temporary) file
      @access private */
  var $tempfname = "";
  /** File pointer
      @access private */
  var $fp;
  /** File pointer
      @access private */
  var $template_fp;
  // ....................................................................
  /**
  * Constructor
  * Note that we open a temporary file, write to this file, and only move
  * it to the destination when closed. This means it is less prone to
  * interferring with external processes which might be accessing it when
  * in the process of being built.
  * @param  string  $name The name of the file
  * @param  string  $dir  The directory the file is in
  * @param  string  $savehow "overwrite" (default), or "append"
  * @return boolean True if file was opened, else false
  */
  function outputfile($name, $dir="", $savehow="overwrite") {
    if ($dir != "") {
      // Separate filename and directory..
      $this->name = $name;
      $this->dir = $dir;
    }
    else {
      // Combined path/filename..
      $this->dir = dirname($name);
      $this->name = basename($name);
    }

    // Make sure directory (full path) is there..
    if ($this->dir != "") {
      if (!file_exists($this->dir)) {
        mkpath($this->dir, 0755);
        if (!file_exists($this->dir)) {
          die("");
        }
      }
    }

    // Whether to append content on close, or overwrite (default)
    $this->savehow = $savehow;

    // Full pathnames..
    $this->datafname = $this->dir . "/" . $this->name;
    $this->tempfname = tempnam("/tmp", APP_PREFIX);
    if ($this->tempfname != "") {
      $this->fp = fopen($this->tempfname, "w");
      if (!$this->fp) $this->opened = false;
      else $this->opened = true;
    }
    else {
      $this->opened = false;
    }

    // Return status..
    return $this->opened;
  } // outputfile
  // ....................................................................
  /**
  * Set the file type. Possible types include 'text' (default), 'html',
  * and 'templated'.
  * @param  string $type The file type
  */
  function type($type) {
    $this->filetype = $type;
  } // type
  // ....................................................................
  /**
  * This defines a template for the file content and flags the
  * file as being of "templated" type. The template is expected to
  * be at the $path specified. The template is opened and the entire
  * file contents are put into $this->content.
  * @param  string  $path The path to the template file
  */
  function template($path) {
    $this->filetype = "templated";
    $this->template_fname = $path;
    $this->template_fp = fopen($this->template_fname, "r");
    if ($this->template_fp) {
      $this->template_contents = fread($this->template_fp, filesize($this->template_fname));
      fclose ($this->template_fp);
    }
    return $this;
  } // template
  // ....................................................................
  /**
  * Replace a pattern in the current $this->content with replacement
  * string. This allows us to work on a template, replacing bits of
  * it that represent our dynamic content.
  * @param  string  $tag      The pattern in the template to replace
  * @param  string  $newstuff The replacement content
  */
  function replace($tag, $newstuff) {
    $tmp = str_replace($tag, $newstuff, $this->template_contents);
    $this->template_contents = $tmp;
    return $this;
  } // replace
  // ....................................................................
  /**
  * Flush any buffered content to the stream.
  * @return boolean True if stream was flushed, else false
  */
  function flushbuffer() {
    $flushed = false;
    if ($this->opened) {
      $flushed = fflush($this->fp);
    }
    return $flushed;
  } // flushbuffer
  // ....................................................................
  /**
  * Close the file. We move the temporary file that has been built to
  * the actual file path contained in 'datafname'. If this is called with
  * no argument, then the default mode SAVE_MODE is assumed and the file
  * is saved. However if the argument is set to the constant ABANDON_FILE
  * then nothing will be saved/created.
  * @param integer $mode How to save: SAVE_FILE or ABANDON_FILE
  * @return boolean True if file was saved and closed successfully.
  */
  function closefile($mode=SAVE_FILE) {
    $ok = false;
    if ($this->opened) {
      switch ($this->filetype) {
        case "html":
          // Finish off the HTML file..
          $this->writeln("</body>");
          $this->writeln("</html>");
          break;
        case "templated":
          // Write the templated contents out..
          $this->write($this->template_contents);
          break;
        default:
          // Text files leave as-is
      } // switch

      fclose($this->fp);
      if ($mode == SAVE_FILE) {
        switch ($this->savehow) {
          case "append":
            if (PHP_VERSION >= 4.3) {
              $s = file_get_contents($this->tempfname);
            }
            else {
              $lines = file($this->tempfname);
              $s = implode("", $lines);
            }
            $ofp = fopen($this->datafname, "a");
            $wrote = fwrite($ofp, $s);
            fclose($ofp);
            $ok = ($wrote == strlen($s));
            break;
          default:
            $ok = copy($this->tempfname, $this->datafname);
        } // switch
      }
      else {
        $ok = true;
      }
      // Remove temporary workfile..
      unlink($this->tempfname);
      // Flag as closed..
      $this->opened = false;
    }
    return $ok;
  } // closefile
  // ....................................................................
  /**
  * Write to file
  * Write the given string to the output file.
  * @param string $s The string to write to the file
  * @return bool True if write was successful
  */
  function write($s) {
    $res = fwrite($this->fp, $s);
    return ($res != -1);
  } // write
  // ....................................................................
  /**
  * Write line to file
  * Write the given string to the output file as a line.
  * @param string $s The string to write as a line to the file
  * @return bool True if writeln was successful
  */
  function writeln($s) {
    return $this->write($s . "\n");
  } // writeln
} // outputfile class

// ----------------------------------------------------------------------
/**
* This is a derived class from outputfile and is a quick way of creating
* a file with content in a single hit. You can check the $created
* property afterwards to determine success/failure.
* @package file
*/
class quickfile extends outputfile {
  /** True if quickfile was created successfully */
  var $created = false;
  /**
  * Creates a file containing given content quickly.
  * @param $path Path to file to create
  * @param $content Optional content to put in the file
  */
  function quickfile($path, $content="") {
    $this->outputfile($path);
    if ($this->opened) {
      if ($this->write($content)) {
        $this->created = $this->closefile();
      }
    }
  } // quickfile
} // quickfile class

// ----------------------------------------------------------------------
// FILEUPLOAD CLASS

/** Error code: All ok, no errors */
define("UPLOAD_E_OK",       0);
/** Failed to move file, filesys error (perms?) */
define("UPLOAD_E_MOV",    100);
/** Uploaded file exceeded the given maximum size */
define("UPLOAD_E_MAXSIZ", 101);
/** Uploaded file was not of an allowed mime type */
define("UPLOAD_E_BADTYP", 102);
/** Attempt to move a non-uploaded file. Hacking. */
define("UPLOAD_E_HACK",   103);
/** Destination directory access problem */
define("UPLOAD_E_ACCESS", 104);

/**
* Fileupload class.
* This is a class for handling file uploads. It assumes that one or
* more files have been uploaded and are being actioned by the current
* Php script which instantiates this object.
* @package file
*/
class fileupload {
  /** Array of info on uploaded files */
  var $uploaded;
  /** No. of files uploaded */
  var $uploaded_count = 0;
  /** True if some files were uploaded */
  var $hasfiles = false;
  /** Destination dir to put files */
  var $destination_dir = ".";
  /** Max filesize, else delete (0 = any size) */
  var $max_size = 0;
  /** Array of allowed mimetypes (unset = any types) */
  var $allowed_types;
  /** Upload field name */
  var $fieldname = "";
  /** Original filename */
  var $filename = "";
  /** Mime type of the the file */
  var $mimetype = "";
  /** Physical size of the file (bytes) */
  var $filesize = 0;
  /** Upload path to the file on the server */
  var $filepath = "";
  /** Error code. Check to see if error occurred. */
  var $errcode = UPLOAD_E_OK;
  /** Error message array, goes with errcode. */
  var $errmsg = array();
  // ....................................................................
  /**
  * Constructor. On instantiation this class checks the global
  * variable space for uploaded file info and pre-processes it.
  * NB: You may specify a filesize limit with $maxsize. If you
  *     do then we do not take notice of the MAX_FILE_SIZE submitted
  *     from the form. This is recommended, since the form value
  *     can easily be hacked..
  * @param  integer $maxsize The maximum filesize in bytes
  * @return boolean Status, true if all ok, else false.
  */
  function fileupload($maxsize=0) {
    global $MAX_FILE_SIZE;
    // Over-ride max size with possible submitted form value..
    if (isset($MAX_FILE_SIZE)) $this->max_size = $maxsize;
    // Over-ride with locally-coded value (safest)..
    if (isset($maxsize)) $this->max_size = $maxsize;
    $this->errcode = UPLOAD_E_OK;

    // Always check for uploaded files..
    if (isset($_FILES) && version_compare(phpversion(), "4.1.0", "ge")) {
      $this->uploaded = $this->unpackinfo($_FILES);
    }
    else {
      global $HTTP_POST_FILES;
      if (isset($HTTP_POST_FILES)) {
        $this->uploaded = $this->unpackinfo($HTTP_POST_FILES);
      }
    }
    // Set the statuses, get current file info..
    if (isset($this->uploaded)) {
      $this->uploaded_count = count($this->uploaded);
      if ($this->uploaded_count > 0) {
        $this->hasfiles = true;
        $this->get_first();
      }
    }

    // Return status so far..
    return ($this->errcode == UPLOAD_E_OK);
  } // fileupload
  // ....................................................................
  /**
  * Set the maximum filesize allowed
  * @param  integer $maxsize The maximum filesize in bytes
  */
  function set_maxsize($maxsize=0) {
    $this->max_size = $maxsize;
  } // set_maxsize
  // ....................................................................
  /**
  * Set the allowed list of filetypes. This is specified as a
  * comma-delimited list, and replaces any already defined.
  * @param  string $types The list of allowed mimetypes
  */
  function set_allowed_types($types="") {
    if (isset($this->allowed_types)) {
      unset($this->allowed_types);
    }
    if ($types != "") {
      $this->allowed_types = explode(",", strtolower(trim($types)));
    }
  } // set_allowed_types
  // ....................................................................
  /**
  * Unpacks the array variable which holds the uploaded file info. We
  * also deal with the special case where the user has uploaded a
  * set of multiple files, using the special array syntax for naming
  * the form control.
  * @param array $rawinfo The raw file info details from the upload
  * @access private
  */
  function unpackinfo($rawinfo) {
    $allinfo = array();
    while (list($fieldname, $info) = each($rawinfo)) {
      $repeatcount = count($info["name"]);
      if (is_array($info["name"])) {
        for ($ix=0; $ix < $repeatcount; $ix++) {
          if ($info["name"][$ix] != "") {
            $tmpinfo["fname"]    = $fieldname;
            $tmpinfo["name"]     = $info["name"][$ix];
            $tmpinfo["type"]     = $info["type"][$ix];
            $tmpinfo["size"]     = $info["size"][$ix];
            $tmpinfo["tmp_name"] = $info["tmp_name"][$ix];
            $allinfo[] = $tmpinfo;
          }
        }
      }
      else {
        if ($info["name"] != "") {
          $tmpinfo["fname"]    = $fieldname;
          $tmpinfo["name"]     = $info["name"];
          $tmpinfo["type"]     = $info["type"];
          $tmpinfo["size"]     = $info["size"];
          $tmpinfo["tmp_name"] = $info["tmp_name"];
          $allinfo[] = $tmpinfo;
        }
      }
    }
    return $allinfo;
  }
  // ....................................................................
  /**
  * Acquire the details on the current upload file. This includes the
  * original filename, its mimetype, its size and the full path on
  * the webserver to which the file was uploaded. The details are
  * populated in class variables:
  *   $this->fieldname
  *   $this->filename
  *   $this->mimetype
  *   $this->filesize
  *   $this->filepath
  * And these variables are then available to be read/used by the
  * calling application code.
  * @return boolean True if info was obtained
  */
  function get_current() {
    if ($this->hasfiles) {
      $info = current($this->uploaded);
      if ($info) {
        $this->fieldname = $info["fname"];
        $this->filename  = $info["name"];
        $this->mimetype  = strtolower($info["type"]);
        $this->filesize  = $info["size"];
        $this->filepath  = $info["tmp_name"];
        return true;
      }
    }
    return false;
  } // get_current
  // ....................................................................
  /**
  * Acquire the details on the first upload file.
  * @see get_current
  * @return boolean True if info was obtained
  */
  function get_first() {
    if ($this->hasfiles) {
      reset($this->uploaded);
      return $this->get_current();
    }
    return false;
  } // get_first
  // ....................................................................
  /**
  * Get details on the next file which was uploaded. If there are no
  * more, then this function returns false.
  * @see get_current
  * @return boolean True if info was obtained
  */
  function get_next() {
    if ($this->hasfiles) {
      if (next($this->uploaded)) {
        return $this->get_current();
      }
    }
    return false;
  } // get_next
  // ....................................................................
  /**
  * Get details on the previous file which was uploaded. If there are no
  * more, then this function returns false.
  * @see get_current
  * @return boolean True if info was obtained
  */
  function get_previous() {
    if ($this->hasfiles) {
      if (previous($this->uploaded)) {
        return $this->get_current();
      }
    }
    return false;
  } // get_previous
  // ....................................................................
  /**
  * Get details on the last file which was uploaded. If there are no
  * more, then this function returns false.
  * @see get_current
  * @return boolean True if info was obtained
  */
  function get_last() {
    if ($this->hasfiles) {
      if (last($this->uploaded)) {
        return $this->get_current();
      }
    }
    return false;
  } // get_last
  // ....................................................................
  /**
  * Fatal error ocurred. Log messages, and delete upload file.
  * @param integer $code Error code
  * @param string  $msg  Error message text
  * @access private
  */
  function fatal_error($code, $msg) {
    $emsg = "UPLOAD ERROR[$code]: $msg";
    $this->errmsg[] = $emsg;
    $this->errcode = $code;
    debugbr($emsg);
    log_sys($emsg);
    if (file_exists($this->filepath)) {
      unlink($this->filepath);
    }
  } // fatal_error
  // ....................................................................
  /**
  * Return any error message(s).
  * @return string Error message text, or nullstring if no error.
  */
  function error_message() {
    $s = "";
    foreach ($this->errmsg as $emsg) { $s .= "$emsg "; }
    return $s;
  } // error_message
  // ....................................................................
  /**
  * Store the current upload file. Optionally specify a destination
  * dir, and a filename. This is useful if you want to process each
  * file separately, and need to store the uploaded file in a
  * particular place.
  * @param string   $destdir  Destination directory path
  * @param string   $filename Destination filename (or default to uploaded filename)
  * @return boolean True if all was ok, store succeeded
  */
  function store($destdir="", $filename="") {
    if ($destdir != "") {
      if ( realpath($destdir) !== FALSE ) {
       $this->destination_dir = realpath($destdir);
      } else {
       $this->destination_dir = $destdir;
      }
    }

    if ($filename != "") $this->filename = basename($filename);
    if ($this->hasfiles) {
      if (is_writable($this->destination_dir)) {
        if (is_uploaded_file($this->filepath)) {
          if ($this->max_size == 0 || $this->filesize <= $this->max_size) {
            if (!isset($this->allowed_types) || in_array($this->mimetype, $this->allowed_types)) {
              $dest = "$this->destination_dir/$this->filename";
              if (!move_uploaded_file($this->filepath, $dest)) {
                $this->fatal_error(UPLOAD_E_MOV, "failed to move upload file: '$this->filepath' to '$dest'");
              }
              else {
                // Change mode..
                chmod($dest, 0755);
              }
            }
            else {
              // File didn't have the correct type, so remove..
              $this->fatal_error(UPLOAD_E_BADTYP, "deleted file of incorrect type $this->mimetype: '$this->filename'");
            }
          }
          else {
            // File was too big, so just delete it..
            $this->fatal_error(UPLOAD_E_MAXSIZ, "deleted file bigger than $this->max_size: '$this->filename'");
          }
        }
        else {
          // Naughty - an attempt to hack maybe..
          $this->fatal_error(UPLOAD_E_HACK, "attempt to move a non-upload file: '$this->filepath'");
        }
      }
      else {
        // Destination directory is read-only..
        $this->fatal_error(UPLOAD_E_ACCESS, "destination directory not writable: '$this->destination_dir'");
      }
    }
    // Return status so far..
    return ($this->errcode == UPLOAD_E_OK);
  } // store
  // ....................................................................
  /**
  * Store all files away at destination dir. We use the original
  * names by default.
  * @param string   $destdir  Destination directory path
  * @return boolean True if all was ok, store succeeded
  */
  function storeall($destdir="") {
    if ($destdir != "") $this->destination_dir = realpath($destdir);
    if ($this->hasfiles) {
      $this->get_first();
      do {
        $this->store();
      } while ($this->get_next());
    }
    // Return status so far..
    return ($this->errcode == UPLOAD_E_OK);
  } // storeall
} // fileupload class

// ----------------------------------------------------------------------
?>