/**
 * File:          $RCSfile: linked_list.h,v $
 * Module:        Linked list
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.26 $
 * Last edited:   $Date: 2002/09/30 12:26:20 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef _GAN_LINKED_LIST_H
#define _GAN_LINKED_LIST_H

#include <gandalf/common/misc_defs.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \addtogroup Common
 * \{
 */

/**
 * \addtogroup CommonList
 * \{
 */

/**
 * \brief Doubly-linked list node.
 */
typedef struct Gan_List_Node
{
   void                 *p_data;
   struct Gan_List_Node *p_next;
   struct Gan_List_Node *p_prev;
} Gan_List_Node;



/**
 * \brief Doubly-linked list type.
 */
typedef struct Gan_List
{
   
   Gan_List_Node *p_first;         /**< First node in list */
   Gan_List_Node *p_current;       /**< Current node in list */
   Gan_List_Node *p_last;          /**< Last node in list */
   Gan_List_Node *p_stack_current; /**< Node stored on the stack */
   
   int node_count;                 /**< Number of nodes in list */
   int current_position;           /**< Current position in list */
   int current_position_stack;     /**< Position stored on stack */
   Gan_Bool is_straight;           /**< Whether list is straight or circular */

   /// indicates whether the structure was dynamically allocated
   Gan_Bool alloc;
} Gan_List;


/* Constructor and Destructor */
Gan_List*        gan_list_new ( void );
Gan_List*        gan_list_form(Gan_List *list);
Gan_List*        gan_list_new_from_array( void **array, unsigned n );
void             gan_list_free ( Gan_List *list, void (*free_func)(void *));
Gan_Bool gan_list_insert_first ( Gan_List *list, void *data );
Gan_Bool gan_list_insert_next  ( Gan_List *list, void *data );
Gan_Bool gan_list_insert_prev  ( Gan_List *list, void *data );
Gan_Bool gan_list_insert_last  ( Gan_List *list, void *data );
Gan_Bool gan_list_insert_at ( Gan_List *list, void *data, int pos );
Gan_Bool gan_list_delete_first   ( Gan_List *list, void (*free_func)(void *) );
Gan_Bool gan_list_delete_next    ( Gan_List *list, void (*free_func)(void *) );
Gan_Bool gan_list_delete_current ( Gan_List *list, void (*free_func)(void *) );
Gan_Bool gan_list_delete_prev    ( Gan_List *list, void (*free_func)(void *) );
Gan_Bool gan_list_delete_last    ( Gan_List *list, void (*free_func)(void *) );
Gan_Bool gan_list_delete_at( Gan_List *list, void (*free_func)(void *),
                             int pos );
void gan_list_delete_all ( Gan_List *list, void (*free_func)(void *) );
void gan_list_goto_head ( Gan_List *list);
void gan_list_goto_tail ( Gan_List *list);
Gan_Bool gan_list_goto_pos ( Gan_List *list, int pos );
void *gan_list_set_prev ( Gan_List *list, void *new_data );
void *gan_list_set_current (Gan_List *list, void *new_data);
void *gan_list_set_next ( Gan_List *list, void *new_data );
void gan_list_push_current ( Gan_List *list );
void gan_list_pop_current ( Gan_List *list );
int gan_list_get_pos ( Gan_List *list );
int gan_list_get_size ( Gan_List *list );
void gan_list_process_data( Gan_List *list,
                            void (*process_func)(void *, void *), void *data );
Gan_List* gan_list_concat( Gan_List *list1, Gan_List *list2 );
void gan_list_reverse( Gan_List *list );
int gan_list_to_array ( Gan_List *list, void ***array_ptr );
Gan_List* gan_list_copy( Gan_List *list );
void gan_list_sort_asc ( Gan_List *list,
                         int (*compare)(const void *, const void *) );
void gan_list_make_straight                        ( Gan_List *list );
void gan_list_make_circular                        ( Gan_List *list );
Gan_Bool gan_list_is_circular                        ( Gan_List *list );

Gan_Bool gan_list_contains  ( Gan_List *list, void *data );

#ifdef GAN_GENERATE_DOCUMENTATION
#define TYPE int
#endif

/**
 * \brief Returns previous data pointer, decrements the current position.
 * \param list List pointer
 * \param node_type Type of object in list
 * \return Data pointer.
 *
 * Returns the data pointed to by the previous node and decrements the current
 * position such that gan_list_get_prev() can be called repeatedly in order to
 * iterate through the list. Call gan_list_goto_head() or another
 * gan_goto_...() function to set the current position in the list first. 
 * \a TYPE is the type the data returned should be cast to, e.g. \c int.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
TYPE *gan_list_get_prev ( Gan_List *list, TYPE node_type );
#else
#define gan_list_get_prev(list,TYPE) ((TYPE*) (gan_list_get_prev_node(list)))
#endif

/**
 * \brief Returns the data pointed to by the current node.
 * \param list The linked list
 * \param node_type The type of object the returned pointer points to
 * \return Data pointer.
 *
 * Returns the data pointed to by the current node. \a TYPE is the type the
 * data returned should be cast to, e.g. \c int.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
TYPE *gan_list_get_current ( Gan_List *list, TYPE node_type );
#else
#define gan_list_get_current(list,TYPE)        ((TYPE*) (gan_list_get_current_node(list)))
#endif

/**
 * \brief Macro: Returns the next data pointer, increments the current position.
 * \param list The linked list.
 * \param node_type The type of object the returned pointer points to
 * \return Data pointer.
 *
 * Returns the data pointed to by the next node and increments the current
 * position such that gan_list_get_next() can be called repeatedly in order
 * to iterate through the list. Call gan_list_goto_head() or another
 * gan_goto_...() function to set the current position in the list first.
 * \a TYPE is the type the data returned should be cast to, e.g. \c int.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
TYPE *gan_list_get_next ( Gan_List *list, TYPE node_type );
#else
#define gan_list_get_next(list,TYPE) ((TYPE*) (gan_list_get_next_node(list)))
#endif

#ifdef GAN_GENERATE_DOCUMENTATION
#undef TYPE
#endif

/* not user functions  */
void *gan_list_get_prev_node ( Gan_List *list );
void *gan_list_get_next_node ( Gan_List *list );
void *gan_list_get_current_node ( Gan_List *list );

/**
 * \}
 */

/**
 * \}
 */

#ifdef __cplusplus
}
#endif

#endif /* _GAN_LINKED_LIST_H */
