/*  GkrellmWHO2 plugin
 *  Copyright (C) 2003 Alexander Shishckin
 *
 *  Author:  Alexander Shishckin <shisha@shisha.spb.ru>
 *
 *  This program is free software which I release under the GNU General Public
 *  License. You may redistribute and/or modify this program under the terms
 *  of that license as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  To get a copy of the GNU General Puplic License,  write to the
 *  Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gkrellm2/gkrellm.h>

#include "config.h"

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <sys/stat.h>

#include "gkrellmwho2.h"

/* configuration widgets */
static GtkWidget	*exclude_button;
static GtkWidget	*name_only_button;
static GtkWidget	*status_button;
static GtkWidget	*group_status_button;
static GtkWidget	*sort_name_button;
static GtkWidget	*sort_idle_button;
static GtkWidget	*speed_spin_button;
static GtkWidget	*freq_spin_button;
static GtkWidget	*away_spin_button;
static GtkWidget	*gone_spin_button;
static GtkWidget	*color_dialog = NULL;
static GtkWidget	*status_color_str[3];

static GkrellmMonitor	*monitor;
static GkrellmPanel		*panel;
static GkrellmDecal		*default_decal = NULL;
static GkrellmStyle		*style;
static GkrellmTextstyle *ts, *ts_alt;
static GkrellmTicks		*pGK;
static GtkTooltips		*user_tips = 0;

static gint     style_id;
static int		color_index;
static GdkColor	status_colors[3];
static GdkColormap	*colormap = NULL;

static int	get_logname()
{
	FILE	*p;
	char	str[MAX_STRING_LEN];
	
	if( (log_name = getenv("USER")) )
		return 0;
	else if( (log_name = getenv("LOGNAME")) )
		return 0;
	else if( (p = popen("who am i", "r")) && 
			fscanf( p, "%8s", str ) &&
			!pclose(p) ) {
		log_name = strdup(str);
		return 0;
	} else if( (p = popen("whoami", "r")) && 
			fscanf( p, "%s", str ) &&
			!pclose(p) ) {
		log_name = strdup(str);
		return 0;
	}
	
	return -1;
}

int		idle_status( time_t idle )
{
	if( idle >= away_time && idle < gone_time ) return 1; /* away */
	else if( idle >= gone_time ) return 2; /* gone */
	else return 0; /* act */
}

void	get_color_name( GdkColor color, char *name )
{
	sprintf( name, "#%2.2x%2.2x%2.2x", (color.red >> 8) & 0xff,
			(color.green >> 8) & 0xff, (color.blue >> 8) & 0xff );
}

static void change_color(gchar *color_name, GdkColor *color)
{
    if (!colormap)
        colormap = gtk_widget_get_colormap(gkrellm_get_top_window());
    if (color->red || color->green || color->blue)
        gdk_colormap_free_colors(colormap, color, 1);
    gdk_color_parse(color_name, color);
    gdk_colormap_alloc_color(colormap, color, FALSE, TRUE);
}

static void format_user_str(int user, char *str)
{
	sprintf( str, "%s ", users[user]->name );
	if( !name_only ) {
		if( show_status )
			sprintf( str + strlen(str), "(%d/%s) ", users[user]->count,
					status[ idle_status(users[user]->idle_time) ] );
		else
			sprintf( str + strlen(str), "(%d) ", users[user]->count );
	}
}

static void clear_users()
{
	while( user_count-- > 0 ) {
		free(users[user_count]->name);
		g_free(users[user_count]->usr_str);
		g_free(users[user_count]->text_style);
		free(users[user_count]);
	}
	
    user_count = 0;
}

static gint find_user( char *name )
{
	gint	i;

	for( i = 0; i < user_count; i++ ) {
		if( !strcmp(users[i]->name, name) )
			return i;
	}

	return -1;
}

static int	add_user( char *name, time_t idle_time )
{
	gint	uid;
	
#ifdef DEBUG
		g_print( "gkrellmwho2: adding user (%s) with idle time (%d)\n", name, (int)idle_time );
#endif
	if(	(uid = find_user(name)) < 0 ) {
		/* We have a new user here */
		users[user_count] = malloc(sizeof(WhoUser));
		users[user_count]->name = strdup(name);
		users[user_count]->usr_str = g_strdup("");
		users[user_count]->count = 1;
		users[user_count]->idle_time = idle_time;
		users[user_count++]->text_style = NULL;
		return 1;
	}
	users[uid]->count++;
	if( users[uid]->idle_time >= idle_time )
		users[uid]->idle_time = idle_time;
	return 0;
}

static int	sort_idlestatus( WhoUser **a, WhoUser **b )
{
	return idle_status((*a)->idle_time) - idle_status((*b)->idle_time);
}

static int	sort_name( WhoUser **a, WhoUser **b )
{
	return strcmp( (*a)->name, (*b)->name );
}

static int	sort_idle( WhoUser **a, WhoUser **b )
{
	return (*a)->idle_time - (*b)->idle_time;
}

static void	get_users()
{
	FILE	*p;
	time_t	idle_time = 0;
	int		i;
	int		pos = 0;
	int		w[4] = {0,0,0,0};
	struct stat sbuf;
	char	*str, *cur_str;
	char	user[MAX_STRING_LEN];
	char	line[MAX_STRING_LEN];
	char	tmp[MAX_STRING_LEN];
	char	host[MAX_STRING_LEN];

	clear_users();
	
	if( !( p=popen( COMMAND, "r" )) ) {
		/* strcpy( scroll_text, "COULD NOT OPEN PIPE" ); */
		return;
	}
	
	if( user_tips_text )
		g_free( user_tips_text );
	
	str = malloc( MAX_USERS * MAX_STRING_LEN );
	str[0] = '\0';
	cur_str = str;

    /* read in user names from the pipe */
    while( !feof( p ) && user_count<MAX_USERS )  {
		user[0] = '\0';
		/* get user name */
		fscanf( p, "%[^\n]\n", cur_str );
		sscanf( cur_str, "%9s %8s %16s %*[^\n]",
				user, line, host );

		if( !user[0] ) break; /* if COMMAND returns no users */
		
#ifdef DEBUG
		g_print( "gkrellmwho2: got user (%s), line (%s), host (%s)\n", user, line, host );
#endif

		if( (exclude_me && strncmp(user, log_name, strlen(log_name))) || !exclude_me ) {
			cur_str += strlen( cur_str );
			*cur_str++ = '\n';
			
			/* Get user's tty idle time in unix format */
			if( snprintf( tmp, MAX_STRING_LEN, "%s%s", DEVTTY, line ) &&
				stat(tmp, &sbuf) == 0 )
					idle_time = time(NULL) - sbuf.st_atime;

			/* add it only if it's not there already */
			add_user(user, idle_time);
		}
    }
    pclose( p );
	
	if( !user_count ) {
		strcpy(str, no_users);
		strcpy( default_msg, no_users );
	} else {
		*(--cur_str) = '\0';
		default_msg[0] = '\0';

		/* Colorize users */
		for( i = 0; i < user_count; i++) {
			w[idle_status(users[i]->idle_time)+1]++;
			users[i]->text_style = gkrellm_copy_textstyle(ts);
			users[i]->text_style->color =
				status_colors[ idle_status( users[i]->idle_time ) ];
		}

		/* Let's sort users right here */
		/* First, by idle status */
		if( group_by_status ) {
			qsort( users, user_count, sizeof(WhoUser *), (sortfn *)sort_idlestatus );
			if( sort_by == by_name )
				for( i = 0; i < 3; i++ ) {
					qsort( &users[pos], w[i+1], sizeof(WhoUser *), (sortfn *)sort_name );
					pos += w[i+1];
				}
			else if( sort_by == by_idle )
				for( i = 0; i < 3; i++ ) {
					qsort( &users[pos], w[i+1], sizeof(WhoUser *), (sortfn *)sort_idle );
					pos += w[i+1];
				}
		} else if( sort_by == by_name ) {
			qsort( users, user_count, sizeof(WhoUser *), (sortfn *)sort_name );
		} else if( sort_by == by_idle ) {
			qsort( users, user_count, sizeof(WhoUser *), (sortfn *)sort_idle );
		}
	}
	default_msg_width = gdk_string_width( gkrellm_default_font(1), default_msg );
	
	user_tips_text = g_strdup( str );
	gtk_tooltips_set_tip( GTK_TOOLTIPS(user_tips), panel->drawing_area,
			user_tips_text, NULL );
	free(str);
}

static void toggle_view() {
	if( name_only )
		name_only = FALSE;
	else
		name_only = TRUE;
}

/* gtk callbacks */
static gint panel_expose_event(GtkWidget *widget, GdkEventExpose *ev)
{
	gdk_draw_pixmap(widget->window,
			widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
			panel->pixmap, ev->area.x, ev->area.y, ev->area.x, ev->area.y,
			ev->area.width, ev->area.height);
	return FALSE;
}

static void panel_press_event(GtkWidget *widget, GdkEventButton *ev)
{
	if( ev->button == 3 )
		gkrellm_open_config_window(monitor);
	else if( ev->button == 1 )
		button1_down = TRUE;
}

static void panel_release_event(GtkWidget *widget, GdkEventButton *ev)
{
	if( drag_line )
		drag_line = FALSE;
	if( button1_down ) {
		button1_down = FALSE;
		toggle_view();
	}
}

static void panel_motion_event(GtkWidget *widget, GdkEventButton *ev)
{
	/*gint full_len;*/
	
	if( !(ev->state & GDK_BUTTON1_MASK) ) {
		button1_down = FALSE;
		drag_line = FALSE;
		return;
	}
	if( !button1_down )
		return;

	drag_line = TRUE;
	/* full_len = w - users_xoff;
	x_scroll = ev->x - (unsigned int)(ev->x / users_xoff) * users_xoff - w;*/
}

#ifndef GKRELLM_HAVE_DECAL_TEXT_INSERT

void decal_text_clear(GkrellmDecal *d)
{
	gdk_draw_rectangle(d->stencil, gkrellm_bit_GC(0), TRUE, 0, 0, d->w, d->h);
}

void decal_text_insert( GkrellmDecal *d, gchar *s, GkrellmTextstyle *ts,
               gint x_off, gint y_off )
{
	gint	x, y;

	if (!s || !d)
		return;
	if (!ts)
		ts = &d->text_style;
	x = x_off;
	y = d->y_baseline + y_off;
	gkrellm_draw_string(d->pixmap, ts, x, y, s);
	gdk_draw_string(d->stencil, ts->font, gkrellm_bit_GC(1), x, y, s);
	if (ts->effect)
		gdk_draw_string(d->stencil, ts->font, gkrellm_bit_GC(1),
				x + 1, y + 1, s);
	d->modified = TRUE;
}

#else
/* gkrellm 2.1.11 will have these two decal text functions built in. */
#define gkrellm_decal_text_clear	decal_text_clear
#define gkrellm_decal_text_insert	decal_text_insert

#endif

static void update_plugin()
{
	GkrellmMargin	*margin;
	static gint	freq, i;
	gint	x_off;
	char	str[MAX_STRING_LEN];

    if( pGK->second_tick )  {
		freq++;
		if( freq==frequency )  {
			get_users();
			freq = 0;
		}
    }

	if( !w ) {
		margin = gkrellm_get_style_margins(style);
		w = gkrellm_chart_width() - margin->left - margin->right - 2;
		x_scroll = w;
	}
	
	/* Calculating the whole damn string length */
	users_xoff = -default_msg_width;
	for( i = 0; i < user_count; i++ ) {
		if( !users[i]->text_style )
			return;
		users[i]->offset = -users_xoff;
		format_user_str(i, str);
		if( gkrellm_dup_string(&users[i]->usr_str, str) )
			users[i]->usr_str_width =
				gdk_string_width( gkrellm_default_font(1), str );
		users_xoff -= users[i]->usr_str_width;
	}
	
	if( !button1_down ) {
		if( x_scroll > users_xoff )
			x_scroll -= scroll_speed;
    	else
			x_scroll = w;
    }
	
	decal_text_clear(default_decal);
	
	for( i = 0; i < user_count; i++) {
		users[i]->text_style->color =
			status_colors[ idle_status( users[i]->idle_time ) ];
		x_off = x_scroll + users[i]->offset;
		if( x_off + users[i]->usr_str_width > 0 &&
			x_off < default_decal->w )
			decal_text_insert( default_decal, users[i]->usr_str,
					users[i]->text_style, x_off, 0 );
	}
	if( !user_count )
		decal_text_insert( default_decal, default_msg, NULL, x_scroll, 0 );
	
	gkrellm_draw_panel_layers(panel);
}


static void create_plugin(GtkWidget *vbox, gint first_create)
{
	if( !no_users )
		no_users = strdup( NONE_MSG );

	if (first_create) {
		if( get_logname() < 0 )
			fprintf( stderr, "failed to determine user's login name\n" );
		pGK = gkrellm_ticks();
		panel = gkrellm_panel_new0();
	}

	style = gkrellm_meter_style(style_id);

	ts = gkrellm_copy_textstyle(gkrellm_meter_textstyle(style_id));
	ts_alt = gkrellm_copy_textstyle(gkrellm_meter_textstyle(style_id));

	/* Can call this only after style and ts are initialized */
	get_users();

	default_decal = gkrellm_create_decal_text(panel, "Aly", ts, style, -1, -1, -1);

	gkrellm_panel_configure(panel, NULL, style);
	gkrellm_panel_create(vbox, monitor, panel);

	user_tips = gtk_tooltips_new();
	user_tips_text = g_strdup("WHO gives the funk!");
	gtk_tooltips_set_tip( GTK_TOOLTIPS(user_tips), panel->drawing_area,
			user_tips_text, NULL);
	gtk_tooltips_set_delay(user_tips, 100);

	if (first_create) {
	    g_signal_connect(G_OBJECT (panel->drawing_area), "expose_event",
    	        G_CALLBACK(panel_expose_event), NULL);
	    g_signal_connect(G_OBJECT (panel->drawing_area), "button_press_event",
    	        G_CALLBACK(panel_press_event), NULL);
	    g_signal_connect(G_OBJECT (panel->drawing_area), "button_release_event",
    	        G_CALLBACK(panel_release_event), NULL);
	    g_signal_connect(G_OBJECT (panel->drawing_area), "motion_notify_event",
    	        G_CALLBACK(panel_motion_event), NULL);
	}
}

static void save_plugin_config(FILE *f)
{
	int		i;
	char	str[MAX_STRING_LEN];
	fprintf(f, "%s exclude_me %d\n", MONITOR_CONFIG_KEYWORD, exclude_me);
	fprintf(f, "%s name_only %d\n", MONITOR_CONFIG_KEYWORD, name_only);
	fprintf(f, "%s show_status %d\n", MONITOR_CONFIG_KEYWORD, show_status);
	fprintf(f, "%s group_by_status %d\n", MONITOR_CONFIG_KEYWORD, group_by_status);
	fprintf(f, "%s sort_by %d\n", MONITOR_CONFIG_KEYWORD, sort_by);
	fprintf(f, "%s scroll_speed %d\n", MONITOR_CONFIG_KEYWORD, scroll_speed);
	fprintf(f, "%s frequency %d\n", MONITOR_CONFIG_KEYWORD, frequency);
	fprintf(f, "%s away_time %d\n", MONITOR_CONFIG_KEYWORD, (int)away_time);
	fprintf(f, "%s gone_time %d\n", MONITOR_CONFIG_KEYWORD, (int)gone_time);
	for( i = 0; i < 3; i++ ) {
		get_color_name(status_colors[i], str);
		fprintf(f, "%s %s_color %s\n", MONITOR_CONFIG_KEYWORD, status[i], str );
	}
}

static void load_plugin_config(gchar *config_line)
{
	gchar	config_keyword[32], config_data[CFG_BUFSIZE];
	gint	n, i;
	char	str[MAX_STRING_LEN];

	if ((n = sscanf(config_line, "%31s %[^\n]",
				config_keyword, config_data)) != 2)
		return;
	
	if ( !strcmp(config_keyword, "exclude_me") )
		sscanf(config_data, "%d", &exclude_me);
	if ( !strcmp(config_keyword, "name_only") )
		sscanf(config_data, "%d", &name_only);
	if ( !strcmp(config_keyword, "show_status") )
		sscanf(config_data, "%d", &show_status);
	if ( !strcmp(config_keyword, "group_by_status") )
		sscanf(config_data, "%d", &group_by_status);
	if ( !strcmp(config_keyword, "sort_by") )
		sscanf(config_data, "%d", &sort_by);
	if ( !strcmp(config_keyword, "scroll_speed") )
		sscanf(config_data, "%d", &scroll_speed);
	if ( !strcmp(config_keyword, "frequency") )
		sscanf(config_data, "%d", &frequency);
	if ( !strcmp(config_keyword, "away_time") )
		sscanf(config_data, "%d", (int *)&away_time);
	if ( !strcmp(config_keyword, "gone_time") )
		sscanf(config_data, "%d", (int *)&gone_time);
	for( i = 0; i < 3; i++ ) {
		sprintf( str, "%s_color", status[i] );
		if ( !strcmp(config_keyword, str) )
			sscanf(config_data, "%31s", str);
		change_color(str, &status_colors[i]);
	}
}

static void apply_plugin_config(void)
{
	int		i;
	char	*str;
	
	exclude_me = GTK_TOGGLE_BUTTON(exclude_button)->active;
	name_only = GTK_TOGGLE_BUTTON(name_only_button)->active;
	show_status = GTK_TOGGLE_BUTTON(status_button)->active;
	group_by_status = GTK_TOGGLE_BUTTON(group_status_button)->active;
	if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sort_name_button)) )
		sort_by = by_name;
	else if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sort_idle_button)) )
		sort_by = by_idle;
	else
		sort_by = 0;
	scroll_speed =
		gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON(speed_spin_button) );
	frequency =	gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON(freq_spin_button) );
	away_time =	gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON(away_spin_button) );
	gone_time =	gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON(gone_spin_button) );
	for( i = 0; i < 3; i++ ) {
		str = strdup( gtk_entry_get_text(GTK_ENTRY(status_color_str[i])) );
		change_color( str, &status_colors[i] );
		free(str);
	}
}

static void cb_chosen_color( GtkWidget *widget, gpointer data )
{
	GtkColorSelection	*colorsel;
	GdkColor			color;
	char				str[MAX_STRING_LEN];

	/* Get the ColorSelection widget */
	colorsel = GTK_COLOR_SELECTION (GTK_COLOR_SELECTION_DIALOG (color_dialog)->colorsel);

	gtk_color_selection_get_current_color(colorsel, &color);
	get_color_name(color, str);
	gtk_entry_set_text (GTK_ENTRY (status_color_str[color_index]), str);
}

static void cb_choose_color( GtkWidget *widget, gpointer data )
{
	gchar				*str;
	gint				response;
	GtkColorSelection	*colorsel;
	GdkColor			color;

	str = g_malloc(MAX_STRING_LEN);
	sprintf(str, "Choose color for '%s' users", status[color_index]);
	color_dialog =  gtk_color_selection_dialog_new(str);
	g_free(str);

	/* Get the ColorSelection widget */
	colorsel = GTK_COLOR_SELECTION (GTK_COLOR_SELECTION_DIALOG (color_dialog)->colorsel);

	/* Set the default color to the value from the input line */
	str = g_strdup( gtk_entry_get_text(GTK_ENTRY(status_color_str[color_index])) );
	change_color(str, &color);
	gtk_color_selection_set_current_color(colorsel,	(GdkColor *)&color);

	response = gtk_dialog_run(GTK_DIALOG(color_dialog));

	if(response == GTK_RESPONSE_OK)
		cb_chosen_color(NULL, NULL);

	gtk_widget_hide(color_dialog);
}

static void cb_choose_act_color( GtkWidget *widget, gpointer data )
{
	color_index = 0;
	cb_choose_color( widget, data );
}

static void cb_choose_away_color( GtkWidget *widget, gpointer data )
{
	color_index = 1;
	cb_choose_color( widget, data );
}

static void cb_choose_gone_color( GtkWidget *widget, gpointer data )
{
	color_index = 2;
	cb_choose_color( widget, data );
}

static void create_plugin_tab(GtkWidget *tab_vbox)
{
	GtkWidget	*notebook;
	GtkWidget	*general_tab;
	GtkWidget	*idle_tab;
	GtkWidget	*general_vbox;
	GtkWidget	*sort_vbox;
	GtkWidget	*idle_vbox;
	GtkWidget	*color_hbox;
	GtkWidget	*label;
	GtkWidget	*choose_button;
	GtkWidget	*info_tab;
	GtkWidget	*info_text;
	GtkWidget	*about_tab;
	GtkWidget	*about_text;

	gint		i;
	char		str[MAX_STRING_LEN];

	static void	*cb_choose_color_func[3] = {
		cb_choose_act_color,
		cb_choose_away_color,
		cb_choose_gone_color
		};

	notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), notebook, TRUE, TRUE, 0);
	sprintf( str, "Hide user '%s'", log_name );

	/* -- General options tab -- */
	general_tab = gkrellm_gtk_framed_notebook_page(notebook, "General");
	general_vbox = gkrellm_gtk_framed_vbox(general_tab, "General options",
			4, FALSE, 0, 2);
	gkrellm_gtk_check_button(general_vbox, &exclude_button, exclude_me,
			FALSE, 0, str);
	gkrellm_gtk_check_button(general_vbox, &name_only_button, name_only,
			FALSE, 0, "Show only users' names");
	gkrellm_gtk_spin_button(general_vbox, &speed_spin_button, (gfloat)scroll_speed,
			1.0, 20.0, 1.0, 1.0, 0, 60, NULL, NULL, TRUE, "Scrolling speed: ");
	gkrellm_gtk_spin_button(general_vbox, &freq_spin_button, (gfloat)frequency,
			1.0, 20.0, 1.0, 1.0, 0, 60, NULL, NULL, TRUE, "Update frequency (sec): ");
	
	sort_vbox = gkrellm_gtk_framed_vbox(general_tab, "Sorting options",
			4, FALSE, 0, 2);
	gkrellm_gtk_check_button(sort_vbox, &group_status_button, group_by_status,
			FALSE, 0, "Sort users by idle status");
	sort_name_button = gtk_radio_button_new_with_label(NULL, "by name");
	sort_idle_button = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(sort_name_button), "by idle");

	switch( sort_by ) {
		case by_name:
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(sort_name_button), TRUE );
			break;
		case by_idle:
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(sort_idle_button), TRUE );
			break;
		default:
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(sort_name_button), FALSE );
	}
	
	gtk_box_pack_end( GTK_BOX(sort_vbox), sort_name_button, FALSE, FALSE, 0 );
	gtk_box_pack_end( GTK_BOX(sort_vbox), sort_idle_button, FALSE, FALSE, 0 );
/*	gkrellm_gtk_check_button(sort_vbox, &sort_name_button, sort_by_name,
			FALSE, 0, "Sort users by name"); */

	/* -- Idle status options tab */
	idle_tab = gkrellm_gtk_framed_notebook_page(notebook, "Idle status");
	idle_vbox = gkrellm_gtk_framed_vbox(idle_tab, "Idle status groups options",
			4, FALSE, 0, 2);
	
	gkrellm_gtk_check_button(idle_vbox, &status_button, show_status,
			FALSE, 0, "Show users' idle status");
	gkrellm_gtk_spin_button(idle_vbox, &away_spin_button, (gfloat)away_time,
			1.0, 3600.0, 1.0, 1.0, 0, 60, NULL, NULL, TRUE, "Away time (sec): ");
	gkrellm_gtk_spin_button(idle_vbox, &gone_spin_button, (gfloat)gone_time,
			1.0, 7200.0, 1.0, 1.0, 0, 60, NULL, NULL, TRUE, "Gone time (sec): ");
	
	/* Color input areas */
	for( i = 0; i < 3; i++ ) {
		color_hbox = gtk_hbox_new(FALSE, 0);
		sprintf( str, "Color for '%s' users: ", status[i] );
		label = gtk_label_new (str);

		gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
		gtk_misc_set_alignment (GTK_MISC (label), 0, 0);

		get_color_name(status_colors[i], str);
		
		status_color_str[i] = gtk_entry_new_with_max_length (10);
		gtk_entry_set_text (GTK_ENTRY (status_color_str[i]), str);
		gtk_entry_set_editable (GTK_ENTRY (status_color_str[i]), TRUE);
		
		choose_button = gtk_button_new_with_label("choose");
		gtk_signal_connect(GTK_OBJECT(choose_button), "clicked",
				GTK_SIGNAL_FUNC(cb_choose_color_func[i]), NULL );

		gtk_box_pack_end(GTK_BOX(color_hbox), choose_button, FALSE, FALSE, 0);
		gtk_box_pack_end (GTK_BOX (color_hbox), status_color_str[i], FALSE, FALSE, 0);
		gtk_box_pack_end(GTK_BOX (color_hbox), label, FALSE, FALSE, 0);
		
		gtk_container_add(GTK_CONTAINER(idle_vbox), color_hbox);
	}
	
	/* -- Info tab */
	info_tab = gkrellm_gtk_framed_notebook_page(notebook, "Info");
	info_text = gkrellm_gtk_scrolled_text_view(info_tab, NULL,
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	for (i = 0; i < sizeof(plugin_info_text)/sizeof(gchar *); ++i)
		gkrellm_gtk_text_view_append(info_text, plugin_info_text[i]);

	/* -- About tab */
	about_tab = gtk_label_new( "About" );
	about_text = gtk_label_new( About );
	gtk_notebook_append_page( GTK_NOTEBOOK(notebook), about_text, about_tab );
}


static GkrellmMonitor	plugin_mon	=
	{
	CONFIG_NAME,        /* Name, for config tab.    */
	0,                  /* Id,  0 if a plugin       */
	create_plugin,      /* The create function      */
	update_plugin,      /* The update function      */
	create_plugin_tab, 	/* The config tab create function   */
	apply_plugin_config,/* Apply the config function        */

	save_plugin_config, /* Save user config   */
	load_plugin_config, /* Load user config   */
	MONITOR_CONFIG_KEYWORD,/* config keyword     */

	NULL,               /* Undefined 2  */
	NULL,               /* Undefined 1  */
	NULL,               /* private      */

	MON_MAIL,           /* Insert plugin before this monitor */

	NULL,               /* Handle if a plugin, filled in by GKrellM     */
	NULL                /* path if a plugin, filled in by GKrellM       */
	};


GkrellmMonitor *gkrellm_init_plugin(void)
{
	style_id = gkrellm_add_meter_style(&plugin_mon, STYLE_NAME);
	monitor = &plugin_mon;
	return &plugin_mon;
}

