/* gmoo - a gtk+ based graphical MOO/MUD/MUSH/... client
 * Copyright (C) 1999-2000 Gert Scholten
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>

#include "config.h"
#include "window.h"
#include "world.h"
#include "properties.h"
#include "dialog.h"
#include "settings.h"
#include "help.h"
#include "dproperties.h"
#include "notebook.h"

typedef struct _manager_t manager_t;
struct _manager_t {
    GtkWidget *window;
    GtkWidget *clist;
    GtkWidget *but_new;
    GtkWidget *but_edit;
    GtkWidget *but_copy;
    GtkWidget *but_delete;
    GtkWidget *but_up;
    GtkWidget *but_down;
    GtkWidget *but_close;
    GtkWidget *but_help;
};

#define row_text(list, row) \
row == 0 ? TEMPLATE : \
GTK_CLIST_ROW(g_list_nth(GTK_CLIST(list)->row_list, row))->cell->u.text

void wm_update_sensitivities(manager_t *m) {
    int row;

    if(GTK_CLIST(m->clist)->selection) {
	row = GPOINTER_TO_UINT(GTK_CLIST(m->clist)->selection->data);

	gtk_widget_set_sensitive(m->but_edit, TRUE);
	gtk_widget_set_sensitive(m->but_copy, TRUE);
	gtk_widget_set_sensitive(m->but_delete, (row != 0));
	gtk_widget_set_sensitive(m->but_up,   (row > 1));
	gtk_widget_set_sensitive(m->but_down, (row > 0 &&
					       row < GTK_CLIST(m->clist)->rows
					       -1));
    } else {
	gtk_widget_set_sensitive(m->but_edit,   FALSE);
	gtk_widget_set_sensitive(m->but_copy,   FALSE);
	gtk_widget_set_sensitive(m->but_delete, FALSE);
	gtk_widget_set_sensitive(m->but_up,     FALSE);
	gtk_widget_set_sensitive(m->but_down,   FALSE);
    }
}

void wm_load_world_names(manager_t *m) {
    GtkCList *c = GTK_CLIST(m->clist);
    char *text[1];
    GList *list, *l;
    GdkColor white = {0, 0xffff, 0xffff, 0xffff};
    GdkColor black = {0, 0x0000, 0x0000, 0x0000};

    gtk_clist_clear(c);
    text[0] = "* " TEMPLATE;
    gtk_clist_append(c, text);
#if 0
    gtk_clist_set_foreground(c, 0, &settings->colors[WHITE + BRIGHT]);
    gtk_clist_set_background(c, 0, &settings->colors[BLACK]);
#endif
    gtk_clist_set_foreground(c, 0, &white);
    gtk_clist_set_background(c, 0, &black);

    list = gm_world_get_names();
    for(l = list; l; l = g_list_next(l)) {
	text[0] = l->data;
	gtk_clist_append(c, text);
    }
    g_list_foreach(list, (GFunc) g_free, NULL);
    g_list_free(list);

    wm_update_sensitivities(m);
}

void wm_select_row_cb(GtkCList *clist, int row, int column, GdkEvent *e,
		      manager_t *m) {
    wm_update_sensitivities(m);
}

void wm_unselect_row_cb(GtkCList *clist, int row, int column, GdkEvent *e,
			manager_t *m) {
    wm_update_sensitivities(m);
}

void wm_save_world_order(manager_t *m) {
    GList *order = NULL;
    int row;
    if(debug) printf("Have to save the order of the worlds\n");

    for(row = 1; row < GTK_CLIST(m->clist)->rows; row++) {
	order = g_list_append(order, row_text(m->clist, row));
    }

    gm_world_save_order(order);
    g_list_free(order);
}

void wm_row_moved_cb(GtkCList *clist, int from, int to, manager_t *m) {
    wm_save_world_order(m);
    wm_update_sensitivities(m);
}

void wm_subwindow_destroyed(GtkWidget *window, manager_t *m) {
    wm_load_world_names(m);
}

void wm_new_cb(GtkWidget *but, manager_t *m) {
    d_properties_t *p = gm_dialog_properties_open_new();
    p->open_world = FALSE;
    pc_modal(p->pc, TRUE);
    gtk_signal_connect(GTK_OBJECT(p->pc->window), "destroy",
		       GTK_SIGNAL_FUNC(wm_subwindow_destroyed), m);
}

void wm_edit_cb(GtkWidget *but, manager_t *m) {
    int row = GPOINTER_TO_UINT(GTK_CLIST(m->clist)->selection->data);
    char *name = row_text(m->clist, row);
    d_properties_t *p;
    world *w;

    if(debug) printf("Have to edit world %d: \"%s\"\n", row, name);

    if((w = gm_notebook_find_world(name))) {
	p = gm_dialog_properties_open(w);
    } else {
	p = gm_dialog_properties_open_by_name(name);
    }
    p->open_world = FALSE;
    pc_modal(p->pc, TRUE);
    gtk_signal_connect(GTK_OBJECT(p->pc->window), "destroy",
		       GTK_SIGNAL_FUNC(wm_subwindow_destroyed), m);
}

void wm_copy_cb(GtkWidget *but, manager_t *m) {
    int row = GPOINTER_TO_UINT(GTK_CLIST(m->clist)->selection->data);
    char *name = row_text(m->clist, row);
    char *text = g_strdup_printf(_("Select a new name for world\n"
				   "%s:"), name);
    char *newname;

    while((newname = gm_dialog_text(_("Copy world"), text, name, TRUE))) {
	if(gm_world_copy_by_name(name, newname)) {
	    break;
	}
	g_free(newname);
    }
    if(newname) wm_load_world_names(m);

    g_free(newname);
    g_free(text);
}

void wm_up_cb(GtkWidget *but, manager_t *m) {
    int from = GPOINTER_TO_UINT(GTK_CLIST(m->clist)->selection->data);
    gtk_clist_swap_rows(GTK_CLIST(m->clist), from, from - 1);
}

void wm_down_cb(GtkWidget *but, manager_t *m) {
    int from = GPOINTER_TO_UINT(GTK_CLIST(m->clist)->selection->data);
    gtk_clist_swap_rows(GTK_CLIST(m->clist), from, from + 1);
}

void wm_delete_cb(GtkWidget *but, manager_t *m) {
    int row = GPOINTER_TO_UINT(GTK_CLIST(m->clist)->selection->data);
    char *name = row_text(m->clist, row);
    char *msg;

    msg = g_strdup_printf(_("If you delete world\n"
			    "%s\n"
			    "All settings related to\n"
			    "this world will be lost...\n"
			    "\n"
			    "Are you sure ?"), name);
    if(gm_dialog_popup(_("Warning"), msg, B_YES | B_NO, TRUE) == B_YES) {
	gm_world_remove(name);
	gtk_clist_remove(GTK_CLIST(m->clist), row);
    }
    g_free(msg);
}

void wm_close_cb(GtkWidget *w, manager_t *m) {
    gtk_widget_hide(m->window);

    wm_save_world_order(m);

    gtk_widget_destroy(m->window);
    g_free(m);
}

void wm_window_deleted(GtkWidget *window, GdkEvent *e, manager_t *m) {
    wm_close_cb(window, m);
}

void gm_dialog_manager_open() {
    GtkWidget *vbox;
    GtkWidget *hbox;
    GtkWidget *scrollw;
    GtkWidget *tmp;
    GtkWidget *vbb;
    GtkWidget *hbb;
    manager_t *m = g_malloc(sizeof(manager_t));

    if(debug) printf("Opening world manager dialog\n");

    m->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_transient_for(GTK_WINDOW(m->window),
				 GTK_WINDOW(gm_window_get_window()));
    gtk_window_set_title(GTK_WINDOW(m->window), _("World manager"));
    gtk_window_set_position(GTK_WINDOW(m->window), GTK_WIN_POS_MOUSE);
    gtk_window_set_policy(GTK_WINDOW(m->window), TRUE, TRUE, FALSE);
    gtk_window_set_modal(GTK_WINDOW(m->window), TRUE);
    gtk_window_set_wmclass(GTK_WINDOW(m->window),
                           "gmoo_world_manager", "gmoo");
    gtk_signal_connect(GTK_OBJECT(m->window), "delete_event",
                       GTK_SIGNAL_FUNC(wm_window_deleted), m);
 
    vbox = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox);
    gtk_container_add(GTK_CONTAINER(m->window), vbox);

    hbox = gtk_hbox_new(FALSE, 5);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);

    scrollw = gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_show(scrollw);
    gtk_box_pack_start(GTK_BOX(hbox), scrollw, TRUE, TRUE, 0);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollw),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

    m->clist = gtk_clist_new(1);
    gtk_widget_show(m->clist);
    gtk_container_add(GTK_CONTAINER(scrollw), m->clist);
    gtk_widget_set_usize(m->clist, 150, -1);
    gtk_clist_set_column_auto_resize(GTK_CLIST(m->clist), 0, TRUE);

    gtk_clist_column_titles_show(GTK_CLIST(m->clist));
    gtk_signal_connect(GTK_OBJECT(m->clist), "select_row",
                       GTK_SIGNAL_FUNC(wm_select_row_cb), m);
    gtk_signal_connect(GTK_OBJECT(m->clist), "unselect_row",
                       GTK_SIGNAL_FUNC(wm_select_row_cb), m);
    gtk_signal_connect_after(GTK_OBJECT(m->clist), "row_move",
                             GTK_SIGNAL_FUNC(wm_row_moved_cb), m);

    tmp = gtk_label_new(_("Worlds"));
    gtk_widget_show(tmp);
    gtk_clist_set_column_widget(GTK_CLIST(m->clist), 0, tmp);
    gtk_label_set_justify(GTK_LABEL(tmp), GTK_JUSTIFY_LEFT);
    gtk_misc_set_alignment(GTK_MISC(tmp), 0, 0.5);
    gtk_misc_set_padding(GTK_MISC(tmp), 5, 0);

    vbb = gtk_vbutton_box_new();
    gtk_widget_show(vbb);
    gtk_box_pack_start(GTK_BOX(hbox), vbb, FALSE, TRUE, 0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(vbb), GTK_BUTTONBOX_START);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(vbb), 5);
    gtk_button_box_set_child_size(GTK_BUTTON_BOX(vbb), 0, 0);

    m->but_new = gtk_button_new_with_label(_("New"));
    gtk_widget_show(m->but_new);
    gtk_container_add(GTK_CONTAINER(vbb), m->but_new);
    gtk_signal_connect(GTK_OBJECT(m->but_new), "clicked",
                       GTK_SIGNAL_FUNC(wm_new_cb), m);

    m->but_edit = gtk_button_new_with_label(_("Edit"));
    gtk_widget_show(m->but_edit);
    gtk_container_add(GTK_CONTAINER(vbb), m->but_edit);
    gtk_signal_connect(GTK_OBJECT(m->but_edit), "clicked",
                       GTK_SIGNAL_FUNC(wm_edit_cb), m);

    m->but_copy = gtk_button_new_with_label(_("Copy"));
    gtk_widget_show(m->but_copy);
    gtk_container_add(GTK_CONTAINER(vbb), m->but_copy);
    gtk_signal_connect(GTK_OBJECT(m->but_copy), "clicked",
                       GTK_SIGNAL_FUNC(wm_copy_cb), m);

    m->but_delete = gtk_button_new_with_label(_("Delete"));
    gtk_widget_show(m->but_delete);
    gtk_container_add(GTK_CONTAINER(vbb), m->but_delete);
    gtk_signal_connect(GTK_OBJECT(m->but_delete), "clicked",
                       GTK_SIGNAL_FUNC(wm_delete_cb), m);

    m->but_up = gtk_button_new_with_label(_("Move Up"));
    gtk_widget_show(m->but_up);
    gtk_container_add(GTK_CONTAINER(vbb), m->but_up);
    gtk_signal_connect(GTK_OBJECT(m->but_up), "clicked",
                       GTK_SIGNAL_FUNC(wm_up_cb), m);

    m->but_down = gtk_button_new_with_label(_("Move Down"));
    gtk_widget_show(m->but_down);
    gtk_container_add(GTK_CONTAINER(vbb), m->but_down);
    gtk_signal_connect(GTK_OBJECT(m->but_down), "clicked",
                       GTK_SIGNAL_FUNC(wm_down_cb), m);

    tmp = gtk_hseparator_new();
    gtk_widget_show(tmp);
    gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, TRUE, 0);

    hbb = gtk_hbutton_box_new();
    gtk_widget_show(hbb);
    gtk_box_pack_start(GTK_BOX(vbox), hbb, FALSE, TRUE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(hbb), 5);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(hbb), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbb), 0);
    gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbb), 0, 0);

    m->but_close = gtk_button_new_with_label(_("Close"));
    gtk_widget_show(m->but_close);
    gtk_container_add(GTK_CONTAINER(hbb), m->but_close);
    GTK_WIDGET_SET_FLAGS(m->but_close, GTK_CAN_DEFAULT);
    gtk_widget_grab_default(m->but_close);
    gtk_widget_grab_focus(m->but_close);
    gtk_signal_connect(GTK_OBJECT(m->but_close), "clicked",
                       GTK_SIGNAL_FUNC(wm_close_cb), m);

    m->but_help = gtk_button_new_with_label(_("Help"));
    gtk_widget_show(m->but_help);
    gtk_container_add(GTK_CONTAINER(hbb), m->but_help);
    GTK_WIDGET_SET_FLAGS(m->but_help, GTK_CAN_DEFAULT);
    gtk_signal_connect(GTK_OBJECT(m->but_help), "clicked",
                       GTK_SIGNAL_FUNC(gm_help_show), HELP_DMANAGER);

    wm_load_world_names(m);

    gtk_widget_show(m->window);
}

