/**
 ** playlist.c - playlist loading/saving and UI functions
 **
 ** Copyright (C) 2000 Matthew Pratt <mattpratt@yahoo.com>
 **
 ** This software is licensed under the terms of the GNU General 
 ** Public License (GPL). Please see the file LICENSE for details.
 **/ 

#include "utility.h"
#include "playlist.h"
#include "rule_list.h"
#include "mp3_control.h"
#include "mp3list.h"
#include "xmms_play.h"
#include "dynamic_list.h"

extern  GnomeUIInfo playlist_popup_menu[]; /* from menus.h */

GList *playlist = NULL;

/*
 * Indicate that song is in the playlist by makring its colour in all the other
 * lists. If it is not in the playlist, then set default colours in the other 
 * lists.
 */
void playlist_set_other_row_colours(MP3 *mp3)
{
    GdkColor *fg_color;
    GdkColor *bg_color;

    if(mp3->row_playlist >= 0){
	fg_color = &gnomp3.in_playlist_fg_color;
	bg_color = &gnomp3.in_playlist_bg_color;
    }else{
	fg_color = NULL;
	bg_color = NULL;
    }
    
    if(mp3->row_alllist >= 0){
	gtk_clist_set_background( GTK_CLIST(gnomp3.all_clist),mp3->row_alllist,
				  bg_color);
	gtk_clist_set_foreground( GTK_CLIST(gnomp3.all_clist),mp3->row_alllist,
				  fg_color);
    }
    if(mp3->row_timelist >= 0){
	gtk_clist_set_background(GTK_CLIST(gnomp3.time_clist),mp3->row_timelist,
				 bg_color);
	gtk_clist_set_foreground(GTK_CLIST(gnomp3.time_clist),mp3->row_timelist,
				 fg_color);
    }
    if(mp3->row_dirlist >= 0){
	gtk_clist_set_background(GTK_CLIST(gnomp3.dir_clist), mp3->row_dirlist,
				 bg_color);
	gtk_clist_set_foreground(GTK_CLIST(gnomp3.dir_clist), mp3->row_dirlist,
				 fg_color);
    }
    if(mp3->row_songlist){
	gtk_ctree_node_set_background( GTK_CTREE(gnomp3.song_ctree), 
				       mp3->row_songlist, 
				       bg_color);
	gtk_ctree_node_set_foreground( GTK_CTREE(gnomp3.song_ctree), 
				       mp3->row_songlist, 
				       fg_color);
    }

}

/*
 * Add the given mp3 to the playlist and update it atributes to indicate so.
 * Colour the song in the other lists accordingly.
 */
int playlist_add_song(MP3 *mp3)
{
    char *text[3];
    int n;

    if(mp3->row_playlist != -1)
	return mp3->row_playlist;

    playlist = g_list_append( playlist, mp3 );


    text[0] = "";
    text[1] = mp3->display_name;
    text[2] = mp3list_build_time(mp3);

    n = gtk_clist_append( GTK_CLIST(gnomp3.play_clist), text);
    gtk_clist_set_row_data( GTK_CLIST(gnomp3.play_clist), n,
			    mp3);
    gtk_clist_set_pixmap( GTK_CLIST(gnomp3.play_clist), n, 0, 
			  gnomp3.music_pixmap->pixmap, gnomp3.music_pixmap->mask );
    mp3->row_playlist = n;

    playlist_set_other_row_colours(mp3);

    xmms_play_add_song(mp3);
    
    return n;
}

/*
 * Delete a song from the playlist and mark it so.
 */
void playlist_remove_row(MP3 *mp3)
{
    GList *ptr;
    int i;

    if(mp3->playing){
	mp3_control_play_next();
    }
    gtk_clist_remove( GTK_CLIST(gnomp3.play_clist), mp3->row_playlist );
    xmms_play_del_song(mp3);

    mp3->row_playlist = -1;
    playlist_set_other_row_colours(mp3);
    
    playlist = g_list_remove( playlist, mp3);

    /* scan thru playlist and set row_playlist */
    for(ptr = playlist, i = 0; ptr; ptr = ptr->next, i++){
	MP3_NODE(ptr)->row_playlist = i;
    }
}

/* deletes the selected rows from the playlist. Called from the menu item */
void playlist_delete(gpointer data)
{
    MP3 *mp3;
    GList *selection, *next;
    selection = GTK_CLIST(gnomp3.play_clist)->selection;
    
    gtk_clist_freeze(GTK_CLIST(gnomp3.play_clist));

    while(selection){
	next = selection->next;
	mp3 = gtk_clist_get_row_data ( GTK_CLIST(gnomp3.play_clist), 
				       (int)selection->data );
	playlist_remove_row(mp3);
	selection = next;
    }
    gtk_clist_thaw(GTK_CLIST(gnomp3.play_clist));
}

/* 
 * save the current playlist with the name specified, including dynamic rules.
 * The order that the playlist was created/loaded in is preserved using the 
 * order of the 'playlist' linked list.
 */
void playlist_save(char *directory, char *filename)
{
  FILE *fp;
  char path[MAX_PATH*2];
  char mesg[MAX_PATH];
  GList *ptr;
  MP3 *mp3;

  g_snprintf( path, MAX_PATH*2, "%s/%s", directory, filename );

  fp = fopen( path, "w" );
  if( !fp ){
    g_snprintf(mesg, MAX_PATH, "Could not save song list '%s'", path );
    gnome_ok_dialog(mesg);
    return;
  }

  rule_list_save_rules(fp);
  strncpy( gnomp3.playlist_loaded, filename, MAX_PATH );

  for( ptr = playlist; ptr; ptr = ptr->next ){
      mp3 = ptr->data;

      if( mp3->from_rule )
	  break;
      fprintf( fp, "%s\n", mp3->filename );
  }
  fclose(fp);
}

/*
 * Remove all the songs from the playlist
 */
void playlist_clear()
{
    GList *ptr;
    MP3 *mp3;

    mp3_control_stop();
    gtk_clist_clear(GTK_CLIST(gnomp3.play_clist));
    for(ptr = playlist; ptr; ptr = ptr->next){
	mp3 = ptr->data;
	xmms_play_del_song(mp3);
	mp3->row_playlist = -1;
	playlist_set_other_row_colours(mp3);
    }
    g_list_free(playlist);
    playlist = NULL;
}

/* 
 * loads the playlist with the given filename as the current playlist. Rows
 * begining with a '#' are searched for rules, but otherwise ignored.
 */
void playlist_load(char *directory, char *filename)
{
    FILE *fp;
    char line[1024];
    char path[MAX_PATH*2];
    int length = 0;
    GList *ptr;

    if( !filename || filename[0] == 0 )return;

    gtk_clist_freeze(GTK_CLIST(gnomp3.play_clist));
    playlist_clear();

    /* do copy to path first, as filename may point to gnomp3.playlist_load */
    strncpy( path, filename, MAX_PATH);
    strncpy( gnomp3.playlist_loaded, path, MAX_PATH );
    gtk_entry_set_text( GTK_ENTRY(GTK_COMBO(gnomp3.playlist_combo)->entry), path );

    /* if it is a dynamic playlist then we dont load from a file */
    if(dynamic_list_load(filename)){
	gtk_clist_thaw(GTK_CLIST(gnomp3.play_clist));
	return;
    }

    g_snprintf( path, MAX_PATH*2, "%s/%s", directory, filename );
    printf("Loading %s\n", path );

    
    fp = fopen( path, "r" );
    if( !fp ){
	gnome_ok_dialog("Could not open song list");
	gtk_clist_thaw(GTK_CLIST(gnomp3.play_clist));
	return;
    }
    
    while( fgets( line, 1024, fp ) ){
	g_strstrip( line );

	if( line[0] == '#' ){
	    rule_list_load_rule(line);
	    continue;
	}

	ptr = mp3list_search( mp3list, line );
	if( ptr )
	    playlist_add_song(ptr->data);
	else{
	    ptr = mp3list_search_by_file(line);
	    if( ptr )
		playlist_add_song(ptr->data);
	}

	length++;
    }
    fclose(fp);
    
    g_snprintf( line, 1024, "Playlist \"%s\" loaded ... %d songs", filename, length );
    gnome_appbar_push (GNOME_APPBAR(GNOME_APP(gnomp3.gnomp3)->statusbar),line);

    rule_list_apply();
    gtk_clist_thaw(GTK_CLIST(gnomp3.play_clist));
}

/*
 * Add a row, select it and move to it.
 * Callback for the 'add song' item in the menus attached to the other lists
 */
void playlist_add_row(MP3 *mp3)
{
    int n;
    
    if(!mp3)return;

    n = playlist_add_song( mp3);
    
    gtk_clist_moveto( GTK_CLIST(gnomp3.play_clist), n ,0 , 0.5, 0.5);
    gtk_clist_select_row( GTK_CLIST(gnomp3.play_clist), n , 0);
}


/* 
 * this function assumes that data points to a GTK_CTREE or GTK_CLIST. Any
 * songs that are selected within this list are added to the playlist 
 */
void playlist_add(GtkWidget *w, gpointer data)
{
    GList *selection;
    GtkCListRow *clist_row;
    MP3 *mp3;
    
    selection = GTK_CLIST(data)->selection;
    
    while(selection){
	
	if( GTK_IS_CTREE(data)){
	    clist_row = selection->data;
	    mp3 = GTK_CTREE_ROW(clist_row)->row.data;
	}else if( GTK_IS_CLIST(data) ){
	    clist_row = g_list_nth( GTK_CLIST(data)->row_list, (int)selection->data )->data;
	    mp3 = clist_row->data;
	}
	
	if(!mp3)
	    return;
	playlist_add_song(mp3);
	selection = selection->next;
    }
}

/* 
 * this function adds the whole album/directory that any selected songs in the
 * song ctree come from 
 */
void playlist_add_album(GtkWidget *w, gpointer data)
{
  char *text[2];
  GList *selection;
  GtkCTreeNode *ctree_node;

  selection = GTK_CLIST(gnomp3.song_ctree)->selection;
  
  while(selection){
    
    ctree_node = selection->data;
    
    if( GTK_CTREE_ROW(ctree_node)->parent )
      ctree_node = GTK_CTREE_ROW(ctree_node)->parent;
    
    ctree_node = GTK_CTREE_ROW(ctree_node)->children;

    while( ctree_node ){
      text[0] = GTK_CELL_PIXTEXT(GTK_CTREE_ROW(ctree_node)->row.cell[0])->text;
      text[1] = GTK_CELL_PIXTEXT(GTK_CTREE_ROW(ctree_node)->row.cell[1])->text;
      playlist_add_row( GTK_CTREE_ROW(ctree_node)->row.data);

      ctree_node = GTK_CTREE_ROW(ctree_node)->sibling;
    }
    selection = selection->next;
  }

}

int playlist_cmp_func(GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2)
{
    int r1, r2;

    r1 = g_list_index( playlist, ((GtkCListRow *)ptr1)->data );
    r2 = g_list_index( playlist, ((GtkCListRow *)ptr2)->data );

    return r1 - r2;
}

/* 
 * this is the callback for the 'sort playlist' button. It sorts the list
 * according to either the song name or the order in which the songs were 
 * added 
 */
void playlist_sort_button_toggled(GtkWidget *w)
{
    if (GTK_TOGGLE_BUTTON (w)->active){
	gtk_clist_set_auto_sort(GTK_CLIST(gnomp3.play_clist), FALSE);
	gtk_clist_set_compare_func (GTK_CLIST(gnomp3.play_clist), NULL);
	gtk_clist_set_sort_column(GTK_CLIST(gnomp3.play_clist), 1);
	gtk_clist_set_auto_sort(GTK_CLIST(gnomp3.play_clist), TRUE);

    }else{
	gtk_clist_set_auto_sort(GTK_CLIST(gnomp3.play_clist), FALSE);
	gtk_clist_set_compare_func (GTK_CLIST(gnomp3.play_clist), playlist_cmp_func);
	gtk_clist_set_sort_column(GTK_CLIST(gnomp3.play_clist), 1);
	gtk_clist_set_auto_sort(GTK_CLIST(gnomp3.play_clist), TRUE);
    }
}

/* 
 * marks all the selected rows as having been played. Called as a menu item
 * callback 
 */
void playlist_mark_row_played(GtkWidget *w, gpointer data)
{
    GList *selection = GTK_CLIST(gnomp3.play_clist)->selection;
    MP3 *mp3;
    while(selection){
	mp3 = gtk_clist_get_row_data( GTK_CLIST(gnomp3.play_clist),
				      (int)selection->data );
	mp3->played = TRUE;
	selection = selection->next;
    }
    playlist_update_display();
}

/* 
 * marks all the selected rows as having been unplayed. Called as a menu item
 * callback 
 */
void playlist_mark_row_unplayed(GtkWidget *w, gpointer data)
{
    GList *selection = GTK_CLIST(gnomp3.play_clist)->selection;
    MP3 *mp3;
    while(selection){
	mp3 = gtk_clist_get_row_data( GTK_CLIST(gnomp3.play_clist),
				      (int)selection->data );
	mp3->played = FALSE;
	selection = selection->next;
    }
    playlist_update_display();
}

/* 
 * find the given row in the playlist and update the way it is displayed. 
 * The MP3 node is checked and the icon, and coluor are set according to
 * whether or not it is played/playing/from a rule etc. 
 */
void playlist_update_row_display(int row_num)
{
    MP3 *mp3;

    mp3 = gtk_clist_get_row_data(GTK_CLIST(gnomp3.play_clist), row_num);

    if( mp3->playing ){
	gtk_clist_set_foreground(GTK_CLIST(gnomp3.play_clist), row_num, 
				 &gnomp3.playing_fg_color );
	gtk_clist_set_background(GTK_CLIST(gnomp3.play_clist), row_num, 
				 &gnomp3.playing_bg_color );
	gtk_clist_set_pixmap( GTK_CLIST(gnomp3.play_clist), row_num, 0, 
			      gnomp3.playing_pixmap->pixmap, 
			      gnomp3.playing_pixmap->mask );
	if( !gtk_clist_row_is_visible( GTK_CLIST(gnomp3.play_clist), row_num ))
	    gtk_clist_moveto( GTK_CLIST(gnomp3.play_clist), row_num, 0, 0.5, 0.5 );
	return;
    }
 
    /* icon */
    if( mp3->from_rule ){
	if( mp3->played )
	    gtk_clist_set_pixmap( GTK_CLIST(gnomp3.play_clist), row_num, 0, 
				  gnomp3.finished_rule->pixmap, 
				  gnomp3.finished_rule->mask );
	else
	    gtk_clist_set_pixmap( GTK_CLIST(gnomp3.play_clist), row_num, 0, 
				  gnomp3.rule_pixmap->pixmap, 
				  gnomp3.rule_pixmap->mask );
    }else{
	if( mp3->played )
	    gtk_clist_set_pixmap( GTK_CLIST(gnomp3.play_clist), row_num, 0, 
				  gnomp3.finished_pixmap->pixmap, 
				  gnomp3.finished_pixmap->mask );
	else
	     gtk_clist_set_pixmap( GTK_CLIST(gnomp3.play_clist), row_num, 0, 
				   gnomp3.music_pixmap->pixmap, 
				   gnomp3.music_pixmap->mask );
    }

    /* now for row colours */
    if( mp3->played ){
	gtk_clist_set_foreground(GTK_CLIST(gnomp3.play_clist), row_num, 
				 &gnomp3.played_fg_color );
	gtk_clist_set_background(GTK_CLIST(gnomp3.play_clist), row_num, 
				 &gnomp3.played_bg_color );
    }else{
	gtk_clist_set_foreground(GTK_CLIST(gnomp3.play_clist), row_num, NULL );
	gtk_clist_set_background(GTK_CLIST(gnomp3.play_clist), row_num, NULL );
    }

}

/* update the icons and colours for all the rows in the playlist */
void playlist_update_display()
{
    int i;

    gtk_clist_freeze( GTK_CLIST(gnomp3.play_clist));
    for( i = 0; i < GTK_CLIST(gnomp3.play_clist)->rows; i++){
	playlist_update_row_display(i);
    }
    gtk_clist_thaw( GTK_CLIST(gnomp3.play_clist));
}

void playlist_set_all_unplayed()
{
    GList *ptr;
    for(ptr = playlist; ptr; ptr = ptr->next ){
	MP3_NODE(ptr)->played = FALSE;
    }
}

void playlist_move_row_up(GtkWidget *w, gpointer data)
{
    MP3 *mp3;
    GList *ptr,*selection = GTK_CLIST(gnomp3.play_clist)->selection;
    int first;

    if(!selection)return;
    first = (int)selection->data;

    while(selection){
	mp3 = gtk_clist_get_row_data(GTK_CLIST(gnomp3.play_clist),
				     (int)selection->data);
	ptr = g_list_find(playlist, mp3); 
	playlist = g_list_move_up( playlist, ptr );
	playlist_sort_button_toggled(gnomp3.sort_button);
	selection = selection->next;
    }
    if( !gtk_clist_row_is_visible( GTK_CLIST(gnomp3.play_clist), first))
	gtk_clist_moveto( GTK_CLIST(gnomp3.play_clist), first, 0, 0.5, 0.5 );
}

void playlist_move_row_down(GtkWidget *w, gpointer data)
{
    MP3 *mp3;
    GList *ptr,*selection = GTK_CLIST(gnomp3.play_clist)->selection;
    int first;

    if(!selection)return;
    first = (int)selection->data;

    while(selection){
	mp3 = gtk_clist_get_row_data(GTK_CLIST(gnomp3.play_clist),
				     (int)selection->data);
	ptr = g_list_find(playlist, mp3); 
	playlist = g_list_move_down( playlist, ptr );
	playlist_sort_button_toggled(gnomp3.sort_button);
	selection = selection->next;
    }
    if( !gtk_clist_row_is_visible( GTK_CLIST(gnomp3.play_clist), first))
	gtk_clist_moveto( GTK_CLIST(gnomp3.play_clist), first, 0, 0.5, 0.5 );
}

void playlist_keypress(GtkWidget *w, GdkEventKey *event)
{
    if( event->keyval == GDK_Up && event->state & GDK_CONTROL_MASK){
	playlist_move_row_up(NULL, NULL);
	gtk_signal_emit_stop_by_name (GTK_OBJECT (w), "key_press_event");
    }

    if( event->keyval == GDK_Down && event->state & GDK_CONTROL_MASK){
	playlist_move_row_down(NULL, NULL);
	gtk_signal_emit_stop_by_name (GTK_OBJECT (w), "key_press_event");
	gtk_signal_emit_stop_by_name (GTK_OBJECT (w), "key_press_event");
    }
    if( event->keyval == GDK_Return ){
	mp3_control_play_selected();
    }

    if( event->keyval == GDK_Right ){
	mp3_control_play_next();
    }
    //printf("event: %d %d\n", event->keyval ,event->state);
}

/*
 * Remove all songs generated by a rule.
 */
void playlist_clear_rules()
{
    GList *ptr, *candidates = NULL;
 
    for(ptr = playlist; ptr; ptr = ptr->next){
	if( MP3_NODE(ptr)->from_rule )
	    candidates = g_list_append(candidates, MP3_NODE(ptr));
    }
    g_list_foreach( candidates, (GFunc)playlist_remove_row, NULL );
    g_list_free(candidates);
}

