#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# $Id: GConditions.py 7057 2005-02-23 16:53:58Z johannes $
#

from gnue.common.definitions.GObjects import GObj
from gnue.common.formatting import GTypecast
import mx.DateTime
import types
import string
import sys
import re

class ConditionError (gException):
  pass

class ConditionNotSupported (ConditionError):
  pass

class MalformedConditionTreeError (ConditionError):
  pass

class ArgumentCountError (MalformedConditionTreeError):
  def __init__ (self, element, wanted):
    msg = u_("Conditionelement '%(element)s' was expected to have '%(wanted)d'"
             "arguments, but only has %(real)d'") \
          % {'element': element.__class__.split ('.') [-1],
             'wanted' : wanted,
             'real'   : len (element._children)}
    MalformedConditionTreeError.__init__ (self, msg)

class MissingFieldError (ConditionError):
  def __init__ (self, element):
    msg = u_("The field '%(field)s' has no entry in the given lookup-table") \
          % {'field': element.name }
    ConditionError.__init__ (self, msg)


class UnificationError (gException):
  pass

class ConversionRuleError (UnificationError):
  def __init__ (self, value1, value2):
    msg = u_("No unification rule for combination '%(type1)s' and "
             "'%(type2)s'") \
          % {'type1': type (value1).__name__,
             'type2': type (value2).__name__}
    UnificationError.__init__ (self, msg)

class ConversionError (UnificationError):
  def __init__ (self, value1, value2):
    msg = u_("Value '%(value1)s' of type '%(type1)s' cannot be converted "
             "into type '%(type2)s'") \
          % {'value1': value1,
             'type1' : type (value1).__name__,
             'type2' : type (value2).__name__}
    UnificationError.__init__ (self, msg)



# =============================================================================
# Base condition class; this is class is acts as root node of condition trees
# =============================================================================

class GCondition (GObj):
  """
  A GCondition instance is allways the root node of a condition tree. All
  children of a GCondition node are evaluated and combined using an AND
  condition if not otherwise stated.
  """
  def __init__(self, parent = None, type = "GCCondition"):
    GObj.__init__ (self, parent, type = type)
    self._maxChildren = None


  # ---------------------------------------------------------------------------
  # Make sure an element of the tree has the requested number of children
  # ---------------------------------------------------------------------------

  def _needChildren (self, number):
    """
    This function verifies if a condition element has a given number of
    children. If not an ArgumentCountError will be raised.
    """
    if number is not None and len (self._children) != number:
      raise ArgumentCountError, (self, number)


  # ---------------------------------------------------------------------------
  # Evaluate a condition tree using the given lookup dictionary
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    """
    This function evaluates the condition tree using the dictionary @lookup for
    retrieving field values. All children must evaluate to TRUE; evaluation
    stops on the first false result.
    """
    self.validate ()

    for child in self._children:
      if not child.evaluate (lookup):
        return False

    return True


  # ---------------------------------------------------------------------------
  # Validate an element of a condition tree
  # ---------------------------------------------------------------------------

  def validate (self):
    """
    This function calls validate () on all it's children. Descendants might
    override this function to do integrity checks and things like that.
    """
    self._needChildren (self._maxChildren)

    for child in self._children:
      child.validate ()


  # ---------------------------------------------------------------------------
  # Convert an element into prefix notation
  # ---------------------------------------------------------------------------

  def prefixNotation (self):
    """
    This function returns the prefix notation of an element and all it's
    children.
    """
    result = []
    append = result.extend

    if isinstance (self, GConditionElement):
      result.append (self._type [2:])
      append = result.append
    
    for child in self._children:
      append (child.prefixNotation ())

    return result


  # ---------------------------------------------------------------------------
  # Build an element and all it's children from a prefix notation list
  # ---------------------------------------------------------------------------

  def buildFromList (self, prefixList):
    """
    This function creates a (partial) condition tree from a prefix notation
    list.
    """
    checktype (prefixList, types.ListType)

    if len (prefixList):
      item = prefixList [0]

      # be nice if there's a condition part missing
      offset = 1
      if isinstance (item, types.ListType):
        self.buildFromList (item)
        element = self
      else:
        # automatically map 'field' to 'Field' and 'const' to 'Const'
        if item in ['field', 'const']:
          item = item.title ()

        element = getattr (sys.modules [__name__], "GC%s" % item) (self)

        if item == 'exist':
          (table, masterlink, detaillink) = prefixList [1:4]
          element.table      = table
          element.masterlink = masterlink
          element.detaillink = detaillink
          offset = 4

      for subitem in prefixList [offset:]:
        element.buildFromList (subitem)


  # ---------------------------------------------------------------------------
  # Break up all top-down references
  # ---------------------------------------------------------------------------

  def breakReferences (self):
    """
    This function resolves the reference to the parent instance avoiding
    reference cycles.
    """

    self._parent = None
    for item in self._children:
      item.breakReferences ()


# =============================================================================
# Top level classes 
# =============================================================================

class GConditions(GCondition):
  def __init__(self, parent=None, type="GCConditions"):
    GCondition.__init__(self, parent, type = type)

class GConditionElement (GCondition) :
  def __init__(self, parent=None, type="GConditionElement"):
    GCondition.__init__ (self, parent, type = type)


# -----------------------------------------------------------------------------
# A Field element in the condition tree
# -----------------------------------------------------------------------------

class GCField (GConditionElement):
  def __init__(self, parent, name = None, datatype = "char"):
    GConditionElement.__init__ (self, parent, 'GCCField')
    self.type = datatype
    self.name = name

  # ---------------------------------------------------------------------------
  # Evaluate a field element
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    """
    This function returns the fields value in the given lookup dictionary. If
    this dictionary has no key for the field a MissingFieldError will be
    raised.
    """
    if not lookup.has_key (self.name):
      raise MissingFieldError, (self.name)

    return lookup [self.name]


  # ---------------------------------------------------------------------------
  # A field in prefix notation is a tuple of 'field' and fieldname
  # ---------------------------------------------------------------------------

  def prefixNotation (self):
    """
    The prefix notation of a field element is a tuple of the identifier 'field'
    (acting as operator) and the field's name.
    """
    return ['Field', self.name]


  # ---------------------------------------------------------------------------
  # to complete a field element from a prefix notation set the fieldname
  # ---------------------------------------------------------------------------

  def buildFromList (self, prefixList):
    """
    The single argument to a field 'operator' could be it's name, so this
    method set's the fieldname.
    """
    checktype (prefixList, [types.StringType, types.UnicodeType])
    self.name = prefixList


# -----------------------------------------------------------------------------
# A constant definition in a condition tree
# -----------------------------------------------------------------------------

class GCConst (GConditionElement):
  def __init__ (self, parent, value = None, datatype = "char"):
    GConditionElement.__init__ (self, parent, 'GCCConst')
    self.type  = datatype
    self.value = value


  # ---------------------------------------------------------------------------
  # Evaluate a constant
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    """
    This function returns the constants value
    """
    return self.value


  # ---------------------------------------------------------------------------
  # The prefix notation of a constant is a tuple of identifier and value
  # ---------------------------------------------------------------------------

  def prefixNotation (self):
    """
    The prefix notation of a constant is a tuple of the identifier 'Const' and
    the constant's value.
    """
    return ['Const', self.value]


  # ---------------------------------------------------------------------------
  # Recreate a constant from a prefix notation
  # ---------------------------------------------------------------------------

  def buildFromList (self, prefixList):
    """
    The single argument of a constant 'operator' could be it's value, so this
    function set the constant's value.
    """
    self.value = prefixList


# -----------------------------------------------------------------------------
# Base class for parameter elements in a condition tree
# -----------------------------------------------------------------------------

class GCParam (GConditionElement):
  def __init__ (self, parent, name = None, datatype = "char"):
    GConditionElement.__init__ (self, parent, 'GCCParam')
    self.type = datatype
    self.name = name

  # ---------------------------------------------------------------------------
  # Return the value of a parameter
  # ---------------------------------------------------------------------------

  def getValue(self):
    """
    Descendants override this function to return the value of the parameter.
    """
    return ""


  # ---------------------------------------------------------------------------
  # Evaluate the parameter object
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    """
    A parameter element evaluates to it's value.
    """
    return self.getValue ()


  # ---------------------------------------------------------------------------
  # Return a parameter object in prefix notation
  # ---------------------------------------------------------------------------

  def prefixNotation (self):
    """
    The prefix notation of a parameter object is a 'constant' with the
    parameters' value
    """
    return ['Const', self.getValue ()]


# =============================================================================
# Base classes for unary and binary operations
# =============================================================================

class GUnaryConditionElement (GConditionElement):
  def __init__ (self, parent = None, elementType = ''):
    self._maxChildren = 1
    GConditionElement.__init__ (self, parent, elementType)

class GBinaryConditionElement (GConditionElement):
  def __init__ (self, parent = None, elementType = ''):
    GConditionElement.__init__ (self, parent, elementType)
    self._maxChildren = 2
    self.values       = []

  # ---------------------------------------------------------------------------
  # Evaluating a binary element means evaluation of both children
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    """
    This function evaluates both children of a binary element storing their
    values in the property 'values'. Descendants can use these values for
    further evaluations.
    """
    self._needChildren (self._maxChildren)
    self.values = unify ([child.evaluate (lookup) for child in self._children])


# =============================================================================
# Logical operators
# =============================================================================

# -----------------------------------------------------------------------------
# n-ary operation: AND
# -----------------------------------------------------------------------------

class GCand (GConditionElement):
  def __init__ (self, parent = None):
    GConditionElement.__init__ (self, parent, 'GCand')


# -----------------------------------------------------------------------------
# n-ary operation: OR
# -----------------------------------------------------------------------------

class GCor (GConditionElement):
  def __init__ (self, parent = None):
    GConditionElement.__init__ (self, parent, 'GCor')

  # ---------------------------------------------------------------------------
  # Evaluate an OR tree
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    """
    This function concatenates all children of this element by a logical OR.
    The iteration stops on the first 'true' result.
    """
    for child in self._children:
      if child.evaluate (lookup):
        return True

    return False

# -----------------------------------------------------------------------------
# unary operation: NOT
# -----------------------------------------------------------------------------

class GCnot (GUnaryConditionElement):
  def __init__ (self, parent = None):
    GUnaryConditionElement.__init__ (self, parent, 'GCnot')

  # ---------------------------------------------------------------------------
  # logically Invert the childs result
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    """
    This function logically inverts the child's evaluation
    """
    self._needChildren (self._maxChildren)
    return not self._children [0].evaluate (lookup)



# =============================================================================
# Numeric operations
# =============================================================================

# ---------------------------------------------------------------------------
# n-ary operation: Addition
# ---------------------------------------------------------------------------

class GCadd (GConditionElement):
  def __init__ (self, parent = None):
    GConditionElement.__init__ (self, parent, 'GCadd')

  # ---------------------------------------------------------------------------
  # Evaluate the addition element
  # ---------------------------------------------------------------------------
  def evaluate (self, lookup):
    """
    This function creates the sum of all it's children. A unify is used to
    ensure all children evaluate to a numeric type.
    """
    result = 0
    for child in self._children:
      result += unify ([child.evaluation (lookup), 0]) [0]
    return result


# -----------------------------------------------------------------------------
# n-ary operation: Subtraction
# -----------------------------------------------------------------------------

class GCsub (GConditionElement):
  def __init__ (self, parent = None):
    GConditionElement.__init__ (self, parent, 'GCsub')

  # ---------------------------------------------------------------------------
  # Evaluate the subtraction element
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    result = None

    for child in self._children:
      value = unify ([child.evaluation (lookup), 0]) [0]
      if result is None:
        result = value
      else:
        result -= value

    return result


# -----------------------------------------------------------------------------
# n-ary operation: Multiplication
# -----------------------------------------------------------------------------

class GCmul (GConditionElement):
  def __init__ (self, parent = None):
    GConditionElement.__init__ (self, parent, 'GCmul')

  # ---------------------------------------------------------------------------
  # Evaluate the multiplication
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    result = None

    for child in self._children:
      value = unify ([child.evaluate (lookup), 0]) [0]
      if result is None:
        result = value
      else:
        result *= value

    return result


# -----------------------------------------------------------------------------
# n-ary operation: Division
# -----------------------------------------------------------------------------

class GCdiv (GConditionElement):
  def __init__ (self, parent = None):
    GConditionElement.__init__ (self, parent, 'GCdiv')

  # ---------------------------------------------------------------------------
  # Evaluate the division element
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    result = None

    for child in self._children:
      value = unify ([child.evaluate (lookup), 0]) [0]
      if result is None:
        result = value
      else:
        result /= value

    return result

# -----------------------------------------------------------------------------
# unary operation: numeric negation
# -----------------------------------------------------------------------------

class GCnegate (GUnaryConditionElement):
  def __init__ (self, parent = None):
    GUnaryConditionElement.__init__ (self, parent, 'GCnegate')

  # ---------------------------------------------------------------------------
  # Evaluation of the numeric negation
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    """
    This function does a numeric negation on the child's evaluation result.
    """
    self._needChildren (self._maxChildren)
    return -unify ([self._children [0].evaluate (lookup), 0]) [0]
    

# =============================================================================
# Relational operations
# =============================================================================

# -----------------------------------------------------------------------------
# Equality
# -----------------------------------------------------------------------------

class GCeq (GBinaryConditionElement):
  def __init__ (self, parent = None):
    GBinaryConditionElement.__init__ (self, parent, 'GCeq')

  # ---------------------------------------------------------------------------
  # evaluate EQ relation
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    GBinaryConditionElement.evaluate (self, lookup)
    return self.values [0] == self.values [1]


# -----------------------------------------------------------------------------
# Inequality
# -----------------------------------------------------------------------------

class GCne (GBinaryConditionElement):
  def __init__ (self, parent = None):
    GBinaryConditionElement.__init__ (self, parent, 'GCne')

  # ---------------------------------------------------------------------------
  # evaluate NE relation
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    GBinaryConditionElement.evaluate (self, lookup)
    return self.values [0] != self.values [1]


# -----------------------------------------------------------------------------
# Greater Than
# -----------------------------------------------------------------------------

class GCgt (GBinaryConditionElement):
  def __init__ (self, parent = None):
    GBinaryConditionElement.__init__ (self, parent, 'GCgt')

  # ---------------------------------------------------------------------------
  # evaluate GT relation
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    GBinaryConditionElement.evaluate (self, lookup)
    return self.values [0] > self.values [1]


# -----------------------------------------------------------------------------
# Greater or Equal
# -----------------------------------------------------------------------------

class GCge (GBinaryConditionElement):
  def __init__ (self, parent = None):
    GBinaryConditionElement.__init__ (self, parent, 'GCge')

  # ---------------------------------------------------------------------------
  # evaluate GE relation
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    GBinaryConditionElement.evaluate (self, lookup)
    return self.values [0] >= self.values [1]



# -----------------------------------------------------------------------------
# Less Than
# -----------------------------------------------------------------------------

class GClt (GBinaryConditionElement):
  def __init__ (self, parent = None):
    GBinaryConditionElement.__init__ (self, parent, 'GClt')

  # ---------------------------------------------------------------------------
  # evaluate LT relation
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    GBinaryConditionElement.evaluate (self, lookup)
    return self.values [0] < self.values [1]


# -----------------------------------------------------------------------------
# Less or Equal
# -----------------------------------------------------------------------------

class GCle (GBinaryConditionElement):
  def __init__ (self, parent = None):
    GBinaryConditionElement.__init__ (self, parent, 'GCle')

  # ---------------------------------------------------------------------------
  # evaluate LE relation
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    GBinaryConditionElement.evaluate (self, lookup)
    return self.values [0] <= self.values [1]


# -----------------------------------------------------------------------------
# Like
# -----------------------------------------------------------------------------

class GClike (GBinaryConditionElement):
  def __init__ (self, parent = None):
    GBinaryConditionElement.__init__ (self, parent, 'GClike')

  # ---------------------------------------------------------------------------
  # Evaluate a like condition
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    GBinaryConditionElement.evaluate (self, lookup)
    # None cannot be like something else. You should use 'NULL' or 'NOT NULL'
    # instead
    if self.values [0] is None:
      return False

    strpat = "^%s" % self.values [1]
    strpat = strpat.replace ('?', '.').replace ('%', '.*')
    pattern = re.compile (strpat)
    return pattern.match (self.values [0]) is not None


# -----------------------------------------------------------------------------
# Not Like
# -----------------------------------------------------------------------------

class GCnotlike (GBinaryConditionElement):
  def __init__ (self, parent = None):
    GBinaryConditionElement.__init__ (self, parent, 'GCnotlike')

  # ---------------------------------------------------------------------------
  # Evaluate an inverted like condition
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    GBinaryConditionElement.evaluate (self, lookup)
    strpat = "^%s" % self.values [1]
    strpat = strpat.replace ('?', '.').replace ('%', '.*')
    pattern = re.compile (strpat)
    return pattern.match (self.values [0]) is None


# -----------------------------------------------------------------------------
# Between
# -----------------------------------------------------------------------------

class GCbetween (GConditionElement):
  def __init__ (self, parent = None):
    self._maxChildren = 3
    GConditionElement.__init__ (self, parent, 'GCbetween')

  # ---------------------------------------------------------------------------
  # evaluate beetween relation
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    self._needChildren (self._maxChildren)
    values = unify ([v.evaluate (lookup) for v in self._children])
    return values [1] <= values [0] <= values [2]


# -----------------------------------------------------------------------------
# Not Between
# -----------------------------------------------------------------------------

class GCnotbetween (GConditionElement):
  def __init__ (self, parent = None):
    self._maxChildren = 3
    GConditionElement.__init__ (self, parent, 'GCnotbetween')

  # ---------------------------------------------------------------------------
  # evaluate an inverted beetween relation
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    self._needChildren (self._maxChildren)
    values = unify ([v.evaluate (lookup) for v in self._children])
    return not (values [1] <= values [0] <= values [2])


# -----------------------------------------------------------------------------
# is NULL
# -----------------------------------------------------------------------------

class GCnull (GUnaryConditionElement):
  def __init__ (self, parent = None):
    GUnaryConditionElement.__init__ (self, parent, 'GCnull')

  # ---------------------------------------------------------------------------
  # evaluate if a child is NULL
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    self._needChildren (self._maxChildren)
    return self._children [0].evaluate (lookup) is None


# -----------------------------------------------------------------------------
# is Not NULL
# -----------------------------------------------------------------------------

class GCnotnull (GUnaryConditionElement):
  def __init__ (self, parent = None):
    GUnaryConditionElement.__init__ (self, parent, 'GCnotnull')

  # ---------------------------------------------------------------------------
  # evaluate if a child is not NULL
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    self._needChildren (self._maxChildren)
    return self._children [0].evaluate (lookup) is not None

# -----------------------------------------------------------------------------
# upper  
# -----------------------------------------------------------------------------

class GCupper (GUnaryConditionElement):
  def __init__ (self, parent = None):
    GUnaryConditionElement.__init__ (self, parent, 'GCupper')

  # ---------------------------------------------------------------------------
  # evaluate
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    self._needChildren (self._maxChildren)
    return string.upper (self._children [0].evaluate (lookup))

# -----------------------------------------------------------------------------
# lower  
# -----------------------------------------------------------------------------

class GClower (GUnaryConditionElement):
  def __init__ (self, parent = None):
    GUnaryConditionElement.__init__ (self, parent, 'GClower')

  # ---------------------------------------------------------------------------
  # evaluate
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):
    self._needChildren (self._maxChildren)
    return string.lower (self._children [0].evaluate (lookup))


# -----------------------------------------------------------------------------
# exist
# -----------------------------------------------------------------------------

class GCexist (GConditionElement):
  def __init__ (self, parent = None):
    GConditionElement.__init__ (self, parent, 'GCexist')
    self.callback = None

  # ---------------------------------------------------------------------------
  # Evaluate an exist-condition
  # ---------------------------------------------------------------------------

  def evaluate (self, lookup):

    if self.callback is None:
      raise NotImplementedError

    return self.callback (self, lookup)


  # ---------------------------------------------------------------------------
  # Convert an element into prefix notation
  # ---------------------------------------------------------------------------

  def prefixNotation (self):
    """
    This function returns the prefix notation of an exist element and all it's
    children.
    """
    result = ['exist', self.table, self.masterlink, self.detaillink]
    
    for child in self._children:
      result.append (child.prefixNotation ())

    return result



# =============================================================================
# Return a dictionary of all XML elements available
# =============================================================================

def getXMLelements (updates = {}):
  xmlElements = {
      'conditions':       {
         'BaseClass': GCondition,
         'ParentTags':  ('conditions','and','or','not','negate'),
         'Deprecated': 'Use the <condition> tag instead.',
         },
      'condition':       {
         'BaseClass': GCondition,
         'ParentTags':  ('conditions','and','or','not','negate','exist') },
      'cfield':       {
         'BaseClass': GCField,
         'Description': 'Defines a database table\'s field in a condition.',
         'Attributes': {
            'name':     {
               'Required': True,
               'Typecast': GTypecast.name } },
         'ParentTags':  ('eq','ne','lt','le','gt','ge','add','sub','mul',
                         'div','like','notlike','between','notbetween',
                         'upper', 'lower', 'null', 'notnull') },
      'cparam':       {
         'BaseClass': GCParam,
         'Description': 'Defines a parameter value in a condition.',
         'Attributes': {
            'name':        {
               'Required': True,
               'Unique':   True,
               'Typecast': GTypecast.name } },
         'ParentTags':  ('eq','ne','lt','le','gt','ge','add','sub','mul',
                         'div','like','notlike','between','notbetween') },
      'cconst':       {
         'BaseClass': GCConst,
         'Description': 'Defines a constant value in a condition.',
         'Attributes': {
            'value':     {
               'Required': True,
               'Typecast': GTypecast.text },
            'type':     {
               'Typecast': GTypecast.text } },
         'ParentTags':  ('eq','ne','lt','le','gt','ge','add','sub','mul',
                         'div','like','notlike','between','notbetween') },
      'add':       {
         'BaseClass': GCadd,
         'Description': 'Implements addition.',
         'ParentTags':  ('eq','ne','lt','le','gt','ge','add','sub','mul',
                         'div','like','notlike','between','notbetween') },
      'sub':       {
         'BaseClass': GCsub,
         'Description': 'Implements subtraction.',
         'ParentTags':  ('eq','ne','lt','le','gt','ge','add','sub','mul',
                         'div','like','notlike','between','notbetween') },
      'mul':       {
         'BaseClass': GCmul,
         'Description': 'Implements multiplication.',
         'ParentTags':  ('eq','ne','lt','le','gt','ge','add','sub','mul',
                         'div','like','notlike','between','notbetween') },
      'div':       {
         'BaseClass': GCdiv,
         'Description': 'Implements division.',
         'ParentTags':  ('eq','ne','lt','le','gt','ge','add','sub','mul',
                         'div','like','notlike','between','notbetween') },
      'and':       {
         'BaseClass': GCand,
         'Description': 'Implements logical AND relation.',
         'ParentTags':  ('condition','and','or','not','negate') },
      'or':       {
         'BaseClass': GCor,
         'Description': 'Implements logical OR relation.',
         'ParentTags':  ('condition','and','or','not','negate') },
      'not':       {
         'BaseClass': GCnot,
         'Description': 'Implements logical NOT relation.',
         'ParentTags':  ('condition','and','or','not','negate') },
      'negate':       {
         'BaseClass': GCnegate,
         'Description': 'Implements numerical negation.',
         'ParentTags':  ('condition','and','or','not','negate') },
      'eq':       {
         'BaseClass': GCeq,
         'Description': 'Implements a {field} = {value} condition.',
         'ParentTags':  ('condition','and','or','not','negate') },
      'ne':       {
         'BaseClass': GCne,
         'Description': 'Implements a {field} <> {value} condition.',
         'ParentTags':  ('condition','and','or','not','negate') },
      'gt':       {
         'BaseClass': GCgt,
         'Description': 'Implements a {field} > {value} condition.',
         'ParentTags':  ('condition','and','or','not','negate') },
      'ge':       {
         'BaseClass': GCge,
         'Description': 'Implements a {field} >= {value} condition.',
         'ParentTags':  ('condition','and','or','not','negate') },
      'lt':       {
         'BaseClass': GClt,
         'Description': 'Implements a {field} < {value} condition.',
         'ParentTags':  ('condition','and','or','not','negate') },
      'le':       {
         'BaseClass': GCle,
         'Description': 'Implements a {field} <= {value} condition.',
         'ParentTags':  ('condition','and','or','not','negate') },
      'like':       {
         'BaseClass': GClike,
         'Description': 'Implements a {field} LIKE {value} condition.',
         'ParentTags':  ('condition','and','or','not','negate') },
      'notlike':       {
         'BaseClass': GCnotlike,
         'Description': 'Implements a {field} NOT LIKE {value} condition.',
         'ParentTags':  ('condition','and','or','not','negate') },
      'between':       {
         'BaseClass': GCbetween,
         'Description': 'Implements a {field} BETWEEN {value1} {value2} '
                        'condition.',
         'ParentTags':  ('condition','and','or','not','negate') },
      'notbetween':       {
         'BaseClass': GCnotbetween,
         'Description': 'Implements a {field} NOT BETWEEN {value1} {value2} '
                        'condition.',
         'ParentTags':  ('condition','and','or','not','negate') },
      'null':      {
         'BaseClass': GCnull,
         'Description': 'Implements a {field} IS NULL condition.',
         'ParentTags': ('condition','and','or','not') },
      'notnull':      {
         'BaseClass': GCnotnull,
         'Description': 'Implements a {field} IS NOT NULL condition.',
         'ParentTags': ('condition','and','or','not') },
      'upper':       {
         'BaseClass': GCupper,
         'Description': 'Implements upper({value}).',
         'ParentTags':  ('eq','ne','lt','le','gt','ge',
                         'like','notlike','between','notbetween') },
      'lower':       {
         'BaseClass': GClower,
         'Description': 'Implements lower({value}).',
         'ParentTags':  ('eq','ne','lt','le','gt','ge',
                         'like','notlike','between','notbetween') },
      'exist': {
         'BaseClass': GCexist,
         'Description': 'Implements an exist condition.',
         'Attributes': {
            'table': {
               'Required': True,
               'Description': 'TODO',
               'Typecast': GTypecast.name },
            'masterlink': {
               'Required': True,
               'Description': 'TODO',
               'Typecast': GTypecast.text},
            'detaillink': {
               'Required': True,
               'Description': 'TODO',
               'Typecast': GTypecast.text}},
         'ParentTags':  ('eq','ne','lt','le','gt','ge',
                         'like','notlike','between','notbetween') },
      }

  for alteration in updates.keys():
    xmlElements[alteration].update(updates[alteration])

  return xmlElements

###############################################################################
###############################################################################
####                        Convenience Methods                              ##
###############################################################################
###############################################################################

# =============================================================================
# Create a condition tree from an element sequence in prefix notation
# =============================================================================

def buildTreeFromList (prefixList):
  """
  This function creates a new condition tree from the given element sequence,
  which must be in prefix notation.
  """
  checktype (prefixList, types.ListType)

  result = GCondition ()
  result.buildFromList (prefixList)
  return result


# =============================================================================
# Create a condition tree from a dictionary
# =============================================================================

def buildConditionFromDict (dictionary, comparison = GCeq, logic = GCand):
  """
  This function creates a new condition tree using @comparison as operation
  between keys and values and @logic as concatenation for all keys.
  """
  result = GCondition ()
  lastParent = result

  if len (dictionary.keys ()):
    lastParent = logic (lastParent)

  for key, value in dictionary.items ():
    operation = comparison (lastParent)
    GCField (operation, key)

    if type (value) in [types.IntType, types.FloatType, types.LongType]:
      consttype = 'number'
    else:
      consttype = 'char'

    GCConst (operation, value, consttype)

  return result


# =============================================================================
# Combine two conditions with an AND clause. Side-effect: cond1 will be changed
# =============================================================================

def combineConditions (cond1, cond2):
  """
  This function combines the two condition trees @cond1 and @cond2 using an AND
  clause. Both arguments can be given as conditiontrees or as dictionaries. In
  the latter case they would be converted using buildConditionFromDict (). As a
  side effect of this function, @cond1 would be changed (if neccessary). If
  neither of the conditions has a single top-level element of type GCand a new
  element would be created.
  """
  # check for the trivial cases
  if cond1 is None or cond1 == {}:
    return cond2

  elif cond2 is None or cond2 == {}:
    return cond1

  # make sure both parts are condition trees
  if isinstance (cond1, types.DictType):
    cond1 = buildConditionFromDict (cond1)

  if isinstance (cond2, types.DictType):
    cond2 = buildConditionFromDict (cond2)

  # if the master condition has no elements, assign the second condition's
  # children
  if not len (cond1._children):
    cond1._children = cond2._children
    update = cond1

  elif len (cond2._children):
    # First normalize the master condition. This means we make sure to have a
    # valid logical operator as single top element
    if len (cond1._children) > 1 or not isinstance (cond1._children [0], GCand):
      oldChildren = cond1._children [:]
      cond1._children = []
      parent = GCand (cond1)
      parent._children = oldChildren

    else:
      parent = cond1._children [0]

    # determine where to start in the second condition tree
    buddy = cond2
    while len (buddy._children) == 1 and \
          isinstance (buddy._children [0], GCand):
      buddy = buddy._children [0]

    # and append all of the buddy's children to the top-level-and
    parent._children.extend (buddy._children)

    update = parent
  else:
    update = None

  if update is not None:
    for child in update._children:
      child._parent = update

  return cond1


# =============================================================================
# Unify all elements in values to the same type
# =============================================================================

def unify (values):
  """
  This function converts all values in the sequence @values to the same types
  pushing the results into the sequence @result.
  """
  result = []
  __unify (values, result)
  return result

# -----------------------------------------------------------------------------
# Actual working method for unification
# -----------------------------------------------------------------------------
def __unify (values, result):
  """
  This function does the dirty work of 'unify ()'.
  """

  checktype (values, types.ListType)

  if not len (values):
    return

  elif len (values) == 1:
    result.append (values [0])
    return

  if isinstance (values [0], types.StringType):
    values [0] = unicode (values [0])
  if isinstance (values [1], types.StringType):
    values [1] = unicode (values [1])

  v1 = values [0]
  v2 = values [1]

  if v1 is None or v2 is None:
    result.append (None)
    values.remove (None)
    __unify (values, result)

  elif type (v1) == type (v2):
    result.append (v1)
    values.remove (v1)
    __unify (values, result)

  else:
    # String-Conversions
    if isinstance (v1, types.UnicodeType) or isinstance (v2, types.UnicodeType):
      if isinstance (v1, types.UnicodeType):
        oldValue = v1
        chkValue = v2
      else:
        oldValue = v2
        chkValue = v1

      # String to Boolean
      if hasattr (types, "BooleanType") and \
         isinstance (chkValue, types.BooleanType):
        if oldValue.upper () in ['TRUE', 'T']:
          newValue = True
        elif oldValue.upper () in ['FALSE', 'F']:
          newValue = False
        else:
          raise ConversionError, (oldValue, chkValue)
  
      # String to Integer, Long or Float
      elif isinstance (chkValue, types.IntType) or \
           isinstance (chkValue, types.LongType) or \
           isinstance (chkValue, types.FloatType):
        try:
          if oldValue.upper () in ['TRUE', 'T']:
            newValue = 1
          elif oldValue.upper () in ['FALSE', 'F']:
            newValue = 0
          else:
            newValue = int (oldValue)

        except ValueError:

          try:
            newValue = float (oldValue)
  
          except ValueError:
            raise ConversionError, (oldValue, chkValue)
  
      # String to DateTime
      elif isinstance (chkValue, mx.DateTime.DateTimeType) or \
           isinstance (chkValue, mx.DateTime.DateTimeDeltaType):
  
        try:
          newValue = mx.DateTime.Parser.DateTimeFromString (oldValue)

        except ValueError:
          raise ConversionError, (oldValue, chkValue)
 
      else:
        raise ConversionRuleError, (oldValue, chkValue)
  
    # Boolean conversions
    elif hasattr (types, "BooleanType") and \
         (isinstance (v1, types.BooleanType) or \
          isinstance (v2, types.BooleanType)):
      if isinstance (v1, types.BooleanType):
        oldValue = v1
        chkValue = v2
      else:
        oldValue = v2
        chkValue = v1

      # Boolean to Integer
      if isinstance (chkValue, types.IntType):
        if oldValue:
          newValue = 1
        else:
          newValue = 0
  
      # Boolean to Long
      elif isinstance (chkValue, types.LongType):
        if oldValue:
          newValue = 1L
        else:
          newValue = 0L
  
      else:
        raise ConversionRuleError, (oldValue, chkValue)
  
    # Integer conversions
    elif isinstance (v1, types.IntType) or isinstance (v2, types.IntType):
      if isinstance (v1, types.IntType):
        oldValue = v1
        chkValue = v2
      else:
        oldValue = v2
        chkValue = v1
  
      # Integer to Float
      if isinstance (chkValue, types.FloatType):
        newValue = float (oldValue)

      elif isinstance (chkValue, types.LongType):
        newValue = long (oldValue)
  
      else:
        raise ConversionRuleError, (oldValue, chkValue)
  
    # Long conversions
    elif isinstance (v1, types.LongType) or isinstance (v2, types.LongType):
      if isinstance (v1, types.LongType):
        oldValue = v1
        chkValue = v2
      else:
        oldValue = v2
        chkValue = v1
  
      # Long into Float
      if isinstance (chkValue, types.FloatType):
        newValue = float (oldValue)
      else:
        raise ConversionRuleError, (oldValue, chkValue)
  
    else:
      raise ConversionRuleError, (v1, v2)
  
    values [oldValue == v2] = newValue
    __unify (values, result)


# =============================================================================
# Definition of an impossible condition
# =============================================================================

GCimpossible = buildTreeFromList (['eq', ['const', 1], ['const', 2]])



###############################################################################
###############################################################################
####                        Depreciated Methods                              ##
###############################################################################
###############################################################################

#
# a table with extra information about the different condition types
#  (needed for condition tree <-> prefix table conversion)
#
conditionElements = {
  'add':             (2, 999, GCadd ),
  'sub':             (2, 999, GCsub ),
  'mul':             (2, 999, GCmul ),
  'div':             (2, 999, GCdiv ),
  'and':             (1, 999, GCand ),
  'or':              (2, 999, GCor  ),
  'not':             (1,   1, GCnot ),
  'negate':          (1,   1, GCnegate ),
  'eq':              (2,   2, GCeq  ),
  'ne':              (2,   2, GCne  ),
  'gt':              (2,   2, GCgt  ),
  'ge':              (2,   2, GCge  ),
  'lt':              (2,   2, GClt  ),
  'le':              (2,   2, GCle  ),
  'like':            (2,   2, GClike ),
  'notlike':         (2,   2, GCnotlike ),
  'between':         (3,   3, GCbetween ),
  'null':            (1,   1, GCnull),
  'upper':           (1,   1, GCupper),
  'lower':           (1,   1, GClower),
  }

# creates an GCondition Tree out of an list of tokens in a prefix
# order.

def buildTreeFromPrefix(term):

  # create a GCondition object as top object in the object stack
  parent={0:(GCondition())}

  # GCondition will have only one parameter
  # add paramcount=1 to the parameter count stack
  paramcount={0:1}

  # set start level for stack to zero
  level=0
  for i in term:

    # convert type into an object
    if conditionElements.has_key(i[0]):
      e=conditionElements[i[0]][2](parent[level])
      level=level+1
      # get parameter count
      paramcount[level]=conditionElements[i[0]][0]
      parent[level]=e
    elif i[0]=="field":
      e=GCField(parent[level], i[1])
      paramcount[level]=paramcount[level]-1
      if paramcount[level]==0:
        level=level-1
    elif i[0]=="const":
      e=GCConst(parent[level], i[1])
      paramcount.update({level:(paramcount[level]-1)})
      if paramcount[level]==0:
        level=level-1
#    print "NAME: %s  VALUE: %s  LEVEL: %s PCOUNT: %s" % \
#          (i[0],i[1],level,paramcount[level])

  return parent[0];



def buildPrefixFromTree(conditionTree):
  if type(conditionTree) != types.InstanceType:
    tmsg = u_("No valid condition tree")
    raise ConditionError, tmsg
  else:
    otype = string.lower(conditionTree._type[2:])

    #
    #  care for objects without children
    #
    if otype == 'cfield':
      return [('field',"%s" % conditionTree.name)]

    elif otype == 'cconst':
      return [('const',conditionTree.value)]

    elif otype == 'cparam':
      return [('const', conditionTree.getValue())]

    #
    #  if its an conditional object, then process it's children
    #
    elif conditionElements.has_key(otype):
      result=[]

      # first add operator to the list
      result.append((otype,''));  #  ,None));


      # change operations with more than there minimal element no into
      # multiple operations with minimal elements
      # reason: to prevent a b c d AND OR being not well defined
      # because it can be a"a b c d AND AND OR" or "a b c d AND OR OR"
      paramcount=len(conditionTree._children)
      while (paramcount > \
             conditionElements[otype][0]):
        paramcount=paramcount-1
        result.append((otype,''));


      # then add children
      for i in range(0, len(conditionTree._children)):
        result = result + \
                 buildPrefixFromTree(conditionTree._children[i])

      #
      #  check for integrity of condition
      #
      if len(conditionTree._children) < conditionElements[otype][0]:
        tmsg = u_('Condition element "%(element)s" expects at most '
                  '%(expected)s arguments; found %(found)s') \
               % {'element' : otype,
                  'expected': conditionElements[otype][0],
                  'found'   : len (conditionTree._children)}
        raise ConditionError, tmsg

      if len(conditionTree._children) > conditionElements[otype][1]:
        tmsg = u_('Condition element "%(element)s" expects at most '
                  '%(expected)s arguments; found %(found)s') \
               % {'element' : otype,
                  'expected': conditionElements[otype][1],
                  'found'   : len (conditionTree._children)}
        raise ConditionError, tmsg


      # return combination
      return result;

    else:
      tmsg = u_('Condition clause "%s" is not supported '
                'by the condition to prefix table conversion.') % otype
      raise ConditionNotSupported, tmsg

#
# Combine two conditions with an and clause.
# NOTE: This modifies cond1 (and also returns it)
#
def __Original__combineConditions (cond1, cond2):
  if cond1 == None or cond1 == {}:
    return cond2
  elif cond2 == None or cond2 == {}:
    return cond1

  if type(cond1) == type({}):
    cond1 = buildConditionFromDict(cond1)
  if type(cond2) == type({}):
    cond2 = buildConditionFromDict(cond2)

  if not len(cond1._children):
    cond1._children = cond2._children
    return cond1
  elif len(cond2._children):
    children = cond1._children[:]
    cond1._children = []
    _and = GCand(cond1)
    _and._children = children
    if len(cond2._children) > 1:
      _and2 = GCand(cond1)
      _and2._children = cond2._children[:]
    else:
      cond1._children.append(cond2._children[0])

  return cond1



# =============================================================================
# Module self test code
# =============================================================================

if __name__ == '__main__':
  import sys
  from mx.DateTime import *

  # ---------------------------------------------------------------------------
  # self-test-function: unify two arguments
  # ---------------------------------------------------------------------------

  def _check_unify (values):
    args = string.join (["%s (%s)" % (v, type (v).__name__) for v in values])
    print "\nUnify:", args

    try:
      res = string.join (["%s (%s)" % (r, type (r).__name__) \
                          for r in unify (values)])
      print "  RES=", res

    except:
      print sys.exc_info () [0], sys.exc_info () [1]


  _check_unify ([u'17', 5])
  _check_unify ([5, '18'])
  _check_unify ([now (), '2004-08-15 14:00'])
  _check_unify (['13.0', 12])
  _check_unify (['15', ''])
  _check_unify ([7L, '12'])
  _check_unify ([7L, '12.2'])
  _check_unify ([1, 'True'])
  _check_unify ([])
  _check_unify ([1])
  _check_unify ([None])
  _check_unify (['5.43', 12, 18L])
  _check_unify ([None, 'foo'])
  _check_unify (['foo', None])
  _check_unify ([5, None, 2, None, '10'])

  print "unification sequence complete"
  raw_input ()


  # ---------------------------------------------------------------------------
  # Self-test-function: Construct conditions and evaluate them
  # ---------------------------------------------------------------------------

  def _check_construction (source, lookup):
    if isinstance (source, types.DictType):
      cond = buildConditionFromDict (source)
    else:
      cond = buildTreeFromList (source)

    print
    print "Source:", source
    print "Prefix:", cond.prefixNotation ()
    print "Lookup:", lookup
    print "Eval  :", cond.evaluate (lookup)


  print "Condition tree transformation test"

  prefix = ['and', ['eq', ['field', u'foo'], ['const', 'bar']],
                   ['ne', ['field', u'bar'], ['const', 'foobar']]]

  lookup = {'foo': 'bar',
            'bar': 'foobarX'}
  _check_construction (prefix, lookup)


  prefix = ['or', ['between', ['field', u'foo'],
                        ['const', 'bar'], ['const', 'char']],
            ['not', ['null', ['field', u'price']]]]

  lookup = {'foo': 'capo', 'price': None}
  _check_construction (prefix, lookup)

  prefix = ['like', ['field', u'foo'], ['const', 'bar%']]
  cond = buildTreeFromList (prefix)
  lookup = {'foo': 'Barrel'}
  _check_construction (prefix, lookup)

  lookup = {'foo': 'bar is running'}
  _check_construction (prefix, lookup)

  source = {'foo': 'bar', 'bar': 'trash'}
  lookup = {'foo': 'bar', 'bar': 'trash'}
  _check_construction (source, lookup)

  prefix = [['eq', ['field', 'foo'], ['const', 'bar']],
            ['lt', ['field', 'bar'], ['const', 10]]]
  lookup = {'foo': 'bar', 'bar': 5.6}
  _check_construction (prefix, lookup)

  prefix = ['eq', ['upper', ['field', 'nfoo']], ['upper', ['const', 'baR']]]
  lookup = {'nfoo': 'bAr'}
  _check_construction (prefix, lookup)

  prefix = ['eq', ['lower', ['field', 'nfoo']], ['const', 'bar']]
  lookup = {'nfoo': 'BaR'}
  _check_construction (prefix, lookup)

  print "end of construction test sequence"
  raw_input ()

  # ---------------------------------------------------------------------------
  # Self-test-function: Combine and evaluate condition trees
  # ---------------------------------------------------------------------------

  def _check_combineConditions (cond1, cond2):
    rc1 = cond1
    rc2 = cond2
    if not isinstance (cond1, types.DictType):
      rc1 = cond1.prefixNotation ()
    if not isinstance (cond2, types.DictType):
      rc2 = cond2.prefixNotation ()

    print "\nCombination of:"
    print "     a:", rc1
    print "     b:", rc2
    res = combineConditions (cond1, cond2)
    print "   RES=", res.prefixNotation ()


  cond1 = GCondition ()
  cond2 = GCondition ()

  _check_combineConditions (cond1, cond2)

  prefix = ['eq', ['field', 'foo'], ['const', 'bar']]
  cond2 = buildTreeFromList (prefix)

  _check_combineConditions (cond1, cond2)

  cond1 = buildTreeFromList (prefix)
  cond2 = GCondition ()
  _check_combineConditions (cond1, cond2)

  prefix2 = ['and', ['null', ['field', 'nfoo']],
                    ['notnull', ['field', 'nbar']]]
  cond2 = buildTreeFromList (prefix2)
  _check_combineConditions (cond1, cond2)

  cond1 = cond2
  cond2 = buildTreeFromList (prefix)
  _check_combineConditions (cond1, cond2)

  prefix3 = ['null', ['field', 'nfoo']]
  cond1 = buildTreeFromList (prefix)
  cond2 = buildTreeFromList (prefix3)
  _check_combineConditions (cond1, cond2)

  prefix4 = [['ne', ['field', 'foo'], ['const', 'bar']],
             ['lt', ['field', 'price'], ['const', 150.0]]]
  cond2 = buildTreeFromList (prefix4)
  _check_combineConditions (cond1, cond2)

  cond1 = buildTreeFromList (prefix)
  _check_combineConditions (cond2, cond1)

  prefix  = ['or', ['null', ['field', 'nfoo']], ['eq', ['field', 'bar']]]
  prefix2 = ['or', ['notnull', ['field', 'nbar']], ['gt', ['field', 'foo']]]
  cond1 = buildTreeFromList (prefix)
  cond2 = buildTreeFromList (prefix2)
  _check_combineConditions (cond1, cond2)

  cond1 = buildTreeFromList (['and'])
  cond2 = buildTreeFromList (prefix)
  _check_combineConditions (cond1, cond2)

  print "start off"
  prefix = ['exist', 'customer', 'gnue_id', 'invoice.customer',
             ['or', ['null'], ['field', 'nfoo']]]
  cond = buildTreeFromList (prefix)
  for i in cond.findChildrenOfType ('GCexist', True, True):
    print i.table, i.masterlink, i.detaillink
    print "Children:"
    for c in i._children:
      print "C:", c.prefixNotation ()
  
  print "\n\nImpossible condition:", GCimpossible.prefixNotation ()
