#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# $Id: Creation.py 6851 2005-01-03 20:59:28Z jcater $

import os
from gnue.common.datasources.drivers.DBSIG2.Schema.Creation import \
    Creation as Base


# =============================================================================
# Class implementing schema creation for PostgreSQL
# =============================================================================

class Creation (Base.Creation):

  MAX_NAME_LENGTH = 31
  ALTER_MULTIPLE  = False


  # ---------------------------------------------------------------------------
  # Create a new database
  # ---------------------------------------------------------------------------

  def createDatabase (self):
    """
    This function creates the requested user and database using the tools
    'createuser', 'createdb' and 'dropuser'. Of course this function should
    better make use of the template1 database using a connection object. 
    """
    dbname   = self.connection.parameters.get ('dbname', None)
    username = self.connection.parameters.get ('username', 'gnue')
    password = self.connection.parameters.get ('password', None)
    host     = self.connection.parameters.get ('host', None)
    port     = self.connection.parameters.get ('port', None)

    site = ""
    if host is not None:
      site += " --host=%s" % host
    if port is not None:
      site += " --port=%s" % port

    # TODO: use a connection object to the template1 database instead of the
    # shell-scripts. Note: CREATE DATABASE statements must NOT run within a
    # transaction block, so we cannot use the default connection mechanisms.

    try:
      os.system (u"dropuser %s%s 2>/dev/null" % (username, site))

    except:
      pass

    try:
      createuser = u"createuser %s --createdb --adduser %s" % (site, username)
      os.system (createuser)
    except:
      pass

    createdb = u"createdb %s --owner=%s --encoding=UNICODE %s" \
        % (site, username, dbname)
    if os.system (createdb):
      raise gException, ("Database creation failed")


    if password is not None and password:
      self.connection.manager.loginToConnection (self.connection)
      alterUser = u"ALTER USER %s WITH PASSWORD '%s';" % (username, password)
      self.connection.makecursor (alterUser)
      self.connection.commit ()


  # ---------------------------------------------------------------------------
  # Handle special defaults
  # ---------------------------------------------------------------------------

  def _defaultwith (self, code, tableName, fieldDef, forAlter):
    """
    This function creates a sequence for 'serials' and sets the default for
    'timestamps' to 'now ()'

    @param code: code-tuple to merge the result in
    @param tableName: name of the table
    @param fieldDef: dictionary describing the field with the default
    @param forAlter: TRUE if the definition is used in a table modification
    """
    if fieldDef ['defaultwith'] == 'serial':
      seq = self._getSequenceName (tableName, fieldDef)
      code [0].append (u"CREATE SEQUENCE %s%s" % (seq, self.END_COMMAND))
      fieldDef ['default'] = "nextval ('%s')" % seq

    elif fieldDef ['defaultwith'] == 'timestamp':
      fieldDef ['default'] = "now()"



  # ---------------------------------------------------------------------------
  # A key is an integer
  # ---------------------------------------------------------------------------

  def key (self, fieldDefinition):
    """
    Native datatype for a 'key'-field is 'int8'

    @param fieldDefinition: dictionary describing the field
    @return: string with the native datatype 'int8'
    """
    return "int8"


  # ---------------------------------------------------------------------------
  # Create an apropriate type for a number
  # ---------------------------------------------------------------------------

  def number (self, fieldDefinition):
    """
    This function returns an apropriate type for a number according to the
    given length and precision.

    @param fieldDefinition: dictionary describing the field
    @return: string with the native datatype
    """
    scale  = 0
    length = 0

    if fieldDefinition.has_key ('precision'):
      scale = fieldDefinition ['precision']
    if fieldDefinition.has_key ('length'):
      length = fieldDefinition ['length']

    if scale == 0:
      if length <= 4:
        return "smallint"
      elif length <= 9:
        return "integer"
      elif length <= 18:
        return "bigint"
      else:
        return "numeric (%s,0)" % length
    else:
      return "numeric (%s,%s)" % (length, scale)


  # ---------------------------------------------------------------------------
  # Native datatype for boolean is boolean
  # ---------------------------------------------------------------------------

  def boolean (self, fieldDefinition):
    """
    This funciton returns the native data type for a boolean, which is
    'boolean'

    @param fieldDefinition: dictionary describing the field
    @return: 'boolean'
    """
    return "boolean"


  # ---------------------------------------------------------------------------
  # Native datatype for datetime
  # ---------------------------------------------------------------------------

  def datetime (self, fieldDefinition):
    """
    This function returns the native type for a datetime value

    @param fieldDefinition: dictionary describing the field
    @return: 'timestamp without time zone'
    """
    return "timestamp without time zone"

