/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 * Licensed under the GNU GPL.
 * Absolutely no warranty.
 */

#include <glib.h>
#include <config.h>
#include "blocklist.h"
#include "mem.h"
#include "snd.h"
#include "block.h"

#define USE_NEW_INSERT   1
#define USE_BCACHE       1
#define USE_EXTRA_CHECKS 0

AFframecount
blocklist_blocks_frame_count(GList *l) {
    AFframecount fc = 0;
    for(; l; l = l->next) 
        fc += ((block *)l->data)->frame_count;
    return fc;    
}

#if USE_EXTRA_CHECKS
#define VERIFY_FRAME_COUNT(blocklist) \
  if((blocklist)->frame_count != blocklist_blocks_frame_count((blocklist)->l)) { \
    FAIL("bl->frame_count(%ld) != blocklist_frame_count(%ld)!\n", \
            (blocklist)->frame_count, blocklist_blocks_frame_count((blocklist)->l)); \
    blocklist_dump((blocklist)); \
    abort(); \
  } 
#else
#define VERIFY_FRAME_COUNT(blocklist) 
#endif

AFframecount
blocklist_frame_count(struct blocklist *bl) {
    VERIFY_FRAME_COUNT(bl);
    return bl->frame_count;
}

void
blocklist_bcache_insert_adjust(struct blocklist *bl,
                               AFframecount frame_offset,
                               AFframecount frame_count) {
    int i;
    struct bcache *bc;
    for(i = 0; i < MAX_BCACHE_ENTRIES; i++) {
        bc = &bl->bc[i];

        if(!bc->is_valid)
            continue;

        if(bc->frame_offset < frame_offset)
            continue;

        if(bc->frame_offset < frame_offset &&
           bc->frame_offset + ((block *)bc->block)->frame_count > frame_offset) {
            continue;
        }

        if(bc->frame_offset >= frame_offset &&
           bc->frame_offset + ((block *)bc->block)->frame_count <= frame_offset + frame_count) {
            bc->frame_offset += frame_count;
            continue;
        }

        if(bc->frame_offset >= frame_offset &&
           bc->frame_offset < frame_offset + frame_count &&
           bc->frame_offset + ((block *)bc->block)->frame_count > frame_offset + frame_count) {
            bc->frame_offset += frame_count;
            continue;
        }

        if(bc->frame_offset > frame_offset) {
            bc->frame_offset += frame_count;
            continue;
        }

        DEBUG("all cases missed?\n");
    }
}

void
blocklist_bcache_delete_adjust(struct blocklist *bl,
                               AFframecount frame_offset,
                               AFframecount frame_count) {
    int i;
    struct bcache *bc;
    for(i = 0; i < MAX_BCACHE_ENTRIES; i++) {
        bc = &bl->bc[i];

        if(!bc->is_valid)
            continue;

        if(bc->frame_offset < frame_offset)
            continue;

        if(bc->frame_offset < frame_offset &&
           bc->frame_offset + ((block *)bc->block)->frame_count <= frame_offset + frame_count)
            continue;

        if(bc->frame_offset >= frame_offset && 
           bc->frame_offset + ((block *)bc->block)->frame_count <= frame_offset + frame_count) { 
            bc->is_valid = 0;
            continue;
        }

        if(bc->frame_offset >= frame_offset && 
           bc->frame_offset + ((block *)bc->block)->frame_count > frame_offset + frame_count) {
            bc->is_valid = 0;
            continue;
        }

        if(bc->frame_offset > frame_offset + frame_count) {
            bc->frame_offset -= frame_count;
            continue;
        }

        DEBUG("missed all cases?\n");
    }
}

GList *
blocklist_bcache_find(struct blocklist *bl,
                      AFframecount *frame_offset) {
    int i;
    struct bcache *bc;

    /* FIXME: Right now we just look at each block in the cache to see
       whether the requested offset is within any of them. If it
       isn't, we fail. It can be done smarter by inferring the
       requested block from the information in the cache even if there
       is not an exact match. This avoids having to search the entire
       list. */

    for(i = 0; i < MAX_BCACHE_ENTRIES; i++) {
        bc = &bl->bc[i];
        if(bc->is_valid) {
            if(*frame_offset > bc->frame_offset &&
               *frame_offset < bc->frame_offset + ((block *)(bc->block->data))->frame_count) {
                *frame_offset -= bc->frame_offset;
                bc->hits++;
                //                DEBUG("frame_offset: %ld, bc->frame_offset: %ld, bc->block->frame_count: %ld, returning %d\n",
                //                      *frame_offset, bc->frame_offset, ((block *)(bc->block->data))->frame_count, i);
                return bc->block;
            }
        }
    }
    return NULL;
}

void
blocklist_bcache_fill(struct blocklist *bl,
                      AFframecount frame_offset,
                      GList *l) {
    int i, lfu = 0, lowest_hits = -1;
    struct bcache *bc;
    for(i = 0; i < MAX_BCACHE_ENTRIES; i++) {
        bc = &bl->bc[i];
        if(!bc->is_valid) {
            bc->frame_offset = frame_offset;
            bc->hits = 0;
            bc->block = l;
            bc->is_valid = 1;
            return;
        }
        if(bc->frame_offset == frame_offset)
            return;
        if(lowest_hits == -1 || bc->hits <= lowest_hits) {
            lowest_hits = bc->hits;
            lfu = i;
        }
    }
    
    /* No empty, throw away LFU. */

    bc->frame_offset = frame_offset;
    bc->hits = 0;
    bc->block = l;
    bc->is_valid = 1;
}

/* Finds a block given an offset. Returns 0 for success. The b
 * argument receives the last block looked at; in case the offset was
 * out of bounds, it receives the last block in the list, and the
 * return value is 1. If successfull, frame_offset receives the
 * relative offset in the block.
 */

int
blocklist_block_find(struct blocklist *bl,
                     AFframecount *frame_offset,
                     GList **b) {
    GList *l;
#if USE_BCACHE
    AFframecount requested_frame_offset = *frame_offset;

    l = blocklist_bcache_find(bl,
                              frame_offset);
    
    if(l) {
        *b = l;
        return 0;
    }
#endif
    for(l = bl->l; l; l = l->next) {
        *b = l;
        if(((block *)l->data)->frame_count > *frame_offset) {
#if USE_BCACHE 
            blocklist_bcache_fill(bl,
                                  requested_frame_offset - *frame_offset,
                                  l);
#endif
            return 0;
        }
        *frame_offset -= ((block *)l->data)->frame_count;
    }
    return 1;
}
                
/*
 * Removes frame_count frames at frame_offset and returns a list of
 * blocks containing the deleted frames.  
 */

GList *
blocklist_delete(struct blocklist *bl,
                 AFframecount frame_offset,
                 AFframecount frame_count) {
    int err;
    block *blk_split;
    GList *l, *l_next, *del_blks = NULL;
    AFframecount frame_count_old = frame_count;
    AFframecount deleted_frame_count = 0, requested_frame_offset = frame_offset;
    gpointer data;

    /*
     * The general idea:
     *
     * 1. Find the block that the offset resides in. This translates
     *    offset into a local offset relative to the block.
     *
     * 2. If the deletion count extends beyond block[idx], and does
     *    not start at offset 0 in block[lpos], then block[lpos] is split
     *    at the offset and we proceed to the next block.
     *
     * 3. As long as the deletion count exceeds or equals the number
     *    of frames in block[idx], we subtract the number of frames in
     *    the block from the deletion count and delete the block.
     *
     * 4. At this point if the deletion count is still non-zero, it is
     *    guaranteed to be smaller than one block. So if there is still a
     *    block left, we shrink it.
     *
     * Returns NULL (not enough memory to rearrange blocks or offset out
     * of range), or the start of the deleted block list. This function
     * may delete less frames than were requested if we run out of memory 
     * or frame_count exceeds the track frame count. FIXME: maybe better
     * to promise either full success or full failure.
     */

    /* Step 1. */

    err = blocklist_block_find(bl, &frame_offset, &l);

    if(err) 
        return NULL;

    /* Step 2. */

    if(frame_offset && 
       frame_offset + frame_count > ((block *)l->data)->frame_count) {
        frame_count -= ((block *)l->data)->frame_count - frame_offset;
        DEBUG("splitting block...\n");
        blk_split = block_split(((block *)l->data), frame_offset);
        
        if(!blk_split) {

            /* FIXME: uh, this error gets lost completely. */

            FAIL("cannot split block for deletion buffer\n");
            return NULL;
        }

        del_blks = g_list_append(del_blks, blk_split);
        deleted_frame_count += blk_split->frame_count;
        frame_offset = 0;
        l = l->next;
    }

    /* Step 3. */

    while(l && frame_count >= ((block *)l->data)->frame_count) {
        deleted_frame_count += ((block *)l->data)->frame_count;
        frame_count -= ((block *)l->data)->frame_count;
        del_blks = g_list_append(del_blks, l->data);
        data = l->next ? l->next->data : NULL;
        l_next = l->next;
        bl->l = g_list_remove_link(bl->l, l);
        g_list_free(l);
        l = l_next; //data ? g_list_find(bl->l, data) : NULL;
    }

    /* Count is nonzero but no blocks left. */

    if(!l && frame_count) {
        FAIL("unable to delete some frames: req: %ld, satisfied: %ld\n",
             frame_count_old, (frame_count_old - frame_count));
        bl->frame_count -= deleted_frame_count;
        blocklist_bcache_delete_adjust(bl, 
                                       requested_frame_offset,
                                       deleted_frame_count);
        VERIFY_FRAME_COUNT(bl);
        return del_blks;
    }

    if(frame_count == 0) {
        bl->frame_count -= deleted_frame_count;
        blocklist_bcache_delete_adjust(bl, 
                                       requested_frame_offset,
                                       deleted_frame_count);
        VERIFY_FRAME_COUNT(bl);
        return del_blks;
    }

    /* Step 4. */

    /* Partial clone. */

    blk_split = block_clone(((block *)l->data),
                            frame_offset,
                            frame_count);

    if(!blk_split) {
        FAIL("cannot split final block for deletion buffer\n");
        bl->frame_count -= deleted_frame_count;
        blocklist_bcache_delete_adjust(bl, 
                                       requested_frame_offset,
                                       deleted_frame_count);
        VERIFY_FRAME_COUNT(bl);
        return del_blks;
    }

    del_blks = g_list_append(del_blks, blk_split);
    deleted_frame_count += blk_split->frame_count;

    block_move((block *)l->data, frame_offset, 
               frame_offset + frame_count, 
               ((block *)l->data)->frame_count - (frame_offset + frame_count));
    block_resize((block *)l->data,
                 frame_offset + (((block *)l->data)->frame_count - 
                                 (frame_offset + frame_count)));

#if USE_EXTRA_CHECKS
    if(deleted_frame_count != blocklist_blocks_frame_count(del_blks)) {
        FAIL("calculated deleted frame count: %ld, actual: %ld\n", 
             deleted_frame_count, blocklist_blocks_frame_count(del_blks));
        abort();
    }
#endif

    bl->frame_count -= deleted_frame_count;
    blocklist_bcache_delete_adjust(bl, 
                                   requested_frame_offset,
                                   deleted_frame_count);
    VERIFY_FRAME_COUNT(bl);
    return del_blks;
}

void
list_print(GList *l) {
    for(; l; l = l->next) 
        INFO("%p%s", 
             l, l->next ? " -> " : "");
    INFO("\n");
}

/*
 * Inserts the given block list into the target track at the specified
 * offset. The given blocks are not actually copied, instead their
 * reference count is increased and they are added to the target block
 * list. The source block list should be destroyed by the caller.
 *
 * If the given offset equals the count for this track, then insert
 * becomes append and the source block list is appended. 
 *
 * Returns 0 for success.
 */

#if USE_NEW_INSERT
int
blocklist_blocks_insert(struct blocklist *bl,
                        GList *source,
                        AFframecount frame_offset) {
    block *blk_split = NULL;
    AFframecount inserted_frame_count = 0, requested_frame_offset = frame_offset;
    GList *l = NULL, *src_ptr = NULL, *dst_ptr = NULL;
    int err;

    if(frame_offset > blocklist_frame_count(bl)) {
        FAIL("frame_offset(%ld) > blocklist_frame_count(%ld)\n",
             frame_offset, blocklist_frame_count(bl));
        abort();
    }

    /* Blocklist is empty, no need to insert. */

    if(!bl->l) {
        bl->l = g_list_copy(source);
        if(!bl->l) {
            FAIL("could not copy list\n");
            return 1;
        }
        for(src_ptr = bl->l; src_ptr; src_ptr = src_ptr->next) 
            block_addref((block *)src_ptr->data);
        blocklist_blocks_set(bl, bl->l);
        return 0;
    }

    /* Otherwise find the block to insert into/behind. */

    err = blocklist_block_find(bl, &frame_offset, &dst_ptr);
    
    if(err && !dst_ptr) {
        FAIL("didn't find frame_offset %ld but also did not get last block\n", requested_frame_offset);
        abort();
    }

    if(err)
        frame_offset = ((block *)dst_ptr->data)->frame_count;
    //    DEBUG("inserting at %ld yields block %p, list before frobbing:\n",
    //          frame_offset, dst_ptr);
    //    list_print(bl->l);
    if(!err && frame_offset && frame_offset < ((block *)dst_ptr->data)->frame_count) {
        DEBUG("splitting block %p at %ld\n", dst_ptr, frame_offset);
        blk_split = block_split((block *)dst_ptr->data, frame_offset);
        if(!blk_split) {
            FAIL("insertion at %ld failed: could not split block\n", 
                 frame_offset);
            return 1;
        }
        src_ptr = g_list_append(l, blk_split);
        if(dst_ptr->next)
            dst_ptr->next->prev = src_ptr;
        src_ptr->prev = dst_ptr;
        src_ptr->next = dst_ptr->next;
        dst_ptr->next = src_ptr;
        /*
          DEBUG("list after split:\n");
          list_print(bl->l);
        */
    }

    for(src_ptr = source; src_ptr; src_ptr = src_ptr->next) {
        block_addref((block *)src_ptr->data);
        l = g_list_append(l, (block *)src_ptr->data);
        /*
        DEBUG("list before insert block %p:\n", l);
        list_print(bl->l);
        */
        inserted_frame_count += ((block *)src_ptr->data)->frame_count;

        /* Either insert before or append after. */

        if(!frame_offset) {
            l->prev = dst_ptr->prev;
            l->next = dst_ptr;
            if(l->prev) 
                l->prev->next = l;
            dst_ptr->prev = l;

            /* This block is new start of list. */

            if(dst_ptr == bl->l)
                bl->l = l;
            l = NULL;
        } else {
            l->next = dst_ptr->next;
            if(dst_ptr->next)
                dst_ptr->next->prev = l;
            l->prev = dst_ptr;
            dst_ptr->next = l;
            dst_ptr = dst_ptr->next;
            l = NULL;
        }
        /*
        DEBUG("list after insert block:\n");
        list_print(bl->l);
        */
    }
    //    DEBUG("inserted_frame_count: %ld\n", inserted_frame_count);
    bl->frame_count += inserted_frame_count;
    blocklist_bcache_insert_adjust(bl, 
                                   requested_frame_offset,
                                   inserted_frame_count);
    //    blocklist_dump(bl);
    VERIFY_FRAME_COUNT(bl);

    return 0;
}

#else

/* Old insert, relies on block position which is not provided by the
 * new blocklist_block_find() function. Slow but works. 
 */

GList *
blocklist_block_find_old(struct blocklist *bl,
                         AFframecount *frame_offset,
                         int *lpos) {
    GList *l;
    if(lpos)
        *lpos = 0;
    for(l = bl->l; l; l = l->next) {
        if(((block *)l->data)->frame_count > *frame_offset) 
            return l;
        if(lpos)
            (*lpos)++;
        *frame_offset -= ((block *)l->data)->frame_count;
    }
    return NULL;
}

int
blocklist_blocks_insert(struct blocklist *bl,
                 GList *source,
                 AFframecount frame_offset) {
    block *blk_split = NULL;
    AFframecount inserted_frame_count = 0, requested_frame_offset = frame_offset;
    GList *l1 = NULL, *l2;
    int lpos;
    
    if(frame_offset > blocklist_frame_count(bl)) {
        FAIL("frame_offset(%ld) > blocklist_frame_count(%ld)\n",
             frame_offset, blocklist_frame_count(bl));
        abort();
    }

    if(frame_offset == blocklist_frame_count(bl)) {

        /* Append, go to last block. */

        lpos = g_list_length(bl->l);
        frame_offset = 0;

    } else {

        /* Otherwise find the block to insert into. */

        l1 = blocklist_block_find_old(bl, &frame_offset, &lpos);
        if(frame_offset) {
            blk_split = block_split(((block *)l1->data), frame_offset);
            if(!blk_split) {
                FAIL("insertion at %ld failed: could not split block %d\n",
                     frame_offset, lpos);
                return 1;
            }
        }
    }

    for(l2 = source; l2; l2 = l2->next, lpos++) {
        block_addref((block *)l2->data);
        inserted_frame_count += ((block *)l2->data)->frame_count;
        bl->l = g_list_insert(bl->l, l2->data,
                              !blk_split ? lpos : lpos + 1);
    }

    if(blk_split)
        bl->l = g_list_insert(bl->l, blk_split, lpos + 1);

    bl->frame_count += inserted_frame_count;
    blocklist_bcache_insert_adjust(bl,
                                   requested_frame_offset,
                                   inserted_frame_count);
    VERIFY_FRAME_COUNT(bl);

    return 0;
}
#endif /* USE_NEW_INSERT */

int
blocklist_buffer_insert(struct blocklist *bl,
                        frame_bits_t frame_bits,
                        int frame_width,
                        AFframecount frame_offset,
                        AFframecount frame_count) {
    GList *l = NULL;
    block *b;
    int r;

    if(frame_count <= 0) {
        FAIL("frame_count: %ld\n", frame_count);
        abort();
    }

    if(frame_offset < 0) {
        FAIL("frame_offset: %ld\n", frame_offset);
        abort();
    }
        
    b = block_new(CACHE_REAL, frame_width, GRAPH_BITS_HRES, frame_count);
    if(!b) {
        FAIL("could not create new block for put, maybe out of memory\n");
        return 1;
    }
    
    cache_fill(b->frame_cache, frame_bits, 0,
               frame_count * frame_width);

    /* FIXME: Calculate peaks for the frame data and stuff them
       directly into the graph cache. This breaks when the cache is
       not a memory cache. We do it like this because otherwise we
       need another buffer and another memcpy(). We might also be lazy
       and postpone peak calculation until a subsequent
       track_graph_get() but that can result in a sudden very
       long calculation at that time. */
    
    snd_frames_buffer_to_graph_buffer(b->graph_cache_low->data,
                                      b->graph_cache_high->data,
                                      frame_bits,
                                      frame_count,
                                      b->frame_width,
                                      b->graph_hres);
    b->graph_cache_low->high = b->graph_cache_low->sz;
    b->graph_cache_high->high = b->graph_cache_high->sz;

    l = g_list_append(l, b);
    r = blocklist_blocks_insert(bl, l, frame_offset);
    g_list_free(l);
    block_unref(b);
    return r;
}

/*
void
blocklist_compact(struct blocklist *bl) {
    block *blk1 = NULL, *blk2 = NULL;
    GList *l = bl->l, *l2, *l3 = NULL;
    AFframecount original_frame_count = blocklist_blocks_frame_count(l);

    for(l2 = l; l2->next; l2 = l2->next) {
        if(((block *)(l2->data))->ref > 1) {
            blk1 = NULL;
            continue;
        }
        if(!blk1) {
            blk1 = (block *)(l2->data);
            continue;
        }
        blk2 = (block *)l2->data;
        if(blk1->frame_count + blk2->frame_count >= DEF_BLOCK_SIZE) {
            blk1 = NULL;
            continue;
        }

        blk1 = block_join(blk1, blk2);
        if(!blk1)
            continue;
        l3 = l2->prev;
        l = g_list_remove_link(l, l2);
        block_unref(blk2);
        if(l2->data != blk2) {
            FAIL("bug: l2->data: %p != blk2: %p\n", l2->data, blk2);
            abort();
        }
        l2->data = NULL;
        g_list_free_1(l2);
        l2 = l3;
    }
    
    DEBUG("original frame count: %ld, new frame count after join: %ld\n", 
          original_frame_count, blocklist_blocks_frame_count(l));

}
*/

/* Clones the block list. */

struct blocklist *
blocklist_clone(struct blocklist *bl) {
    GList *l1 = NULL, *l2 = NULL, *l3;
    block *b;
    struct blocklist *bl_clone;
    int fail = 0;

    bl_clone = blocklist_new(NULL);

    for(l1 = bl->l; l1; l1 = l1->next) {
        //        DEBUG("cloning block (%ld frames)\n", ((block *)l1->data)->frame_count);
        b = block_clone(((block *)l1->data),
                        0,
                        ((block *)l1->data)->frame_count);

        if(!b) {
            fail = 1;
            break;
        }

        l3 = g_list_append(l2, b);

        if(!l3) {
            fail = 1;
            break;
        }

        l2 = l3;
    }

    if(fail) {
        FAIL("failed to clone block list\n");
        blocklist_blocks_destroy(l2);
        l2 = NULL;
    }

    blocklist_blocks_set(bl_clone, l2);
    return bl_clone;
}

/* Dumps blocklist to stdout. */

void
blocklist_dump(struct blocklist *bl) {
    GList *l;
    int i = 0;
    AFframecount c = 0;

    DEBUG("blocks:\n");
    for(l = bl->l; l; l = l->next) {
        c += ((block *)l->data)->frame_count;
        INFO("%4d: %10ld [%10ld] %p\n     fr              ", 
             i++, c, ((block *)l->data)->frame_count, l);
        cache_dump(((block *)l->data)->frame_cache);
        INFO("     gr              ");
        cache_dump(((block *)l->data)->graph_cache_low);
        cache_dump(((block *)l->data)->graph_cache_high);
    }
    DEBUG("bcache:\n");
    for(i = 0; i < MAX_BCACHE_ENTRIES; i++) {
        DEBUG("[%03d] is_valid: %d, frame_offset: %ld, hits: %d\n",
              i,
              bl->bc[i].is_valid, 
              bl->bc[i].frame_offset,
              bl->bc[i].hits);
    }
}

void
blocklist_block_release(gpointer data, 
                        gpointer user_data) {
    block_unref((block *) data);
}

void
blocklist_blocks_destroy(GList *l) {
    g_list_foreach(l, blocklist_block_release, NULL);
    g_list_free(l);
}
    
void
blocklist_destroy(struct blocklist *bl) {
    blocklist_blocks_destroy(bl->l);
    bl->l = NULL;
    //    DEBUG("freeing blocklist: %p\n", bl);
    mem_free(bl);
}

void
blocklist_blocks_set(struct blocklist *bl, 
                     GList *l) {
    bl->l = l;
    bl->frame_count = blocklist_blocks_frame_count(l);
}

struct blocklist *
blocklist_new(GList *l) {
    int i;
    struct blocklist *bl = mem_alloc(sizeof(struct blocklist));
    if(!bl) {
        FAIL("not enough memory for blocklist struct (%"CONVSPEC_SIZE_T" bytes)\n",
             sizeof(struct blocklist));
        return NULL;
    }
    for(i = 0; i < MAX_BCACHE_ENTRIES; i++) 
        bl->bc[i].is_valid = 0;
    bl->frame_count = 0;
    blocklist_blocks_set(bl, l);
    //    DEBUG("allocated blocklist: %p\n", bl);
    return bl;
}

