/* $Revision: 1.1 $ */
/*
 * "gpgp" Gnome/GTK Front for PGP
 * Copyright (C) 1998  Max Valianskiy
 * Copyright (C) 1999 tftp@netscape.net
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 */
#include "../config.h"

#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

#include "gpgp.h"

struct struct_Undo {
    char *buf;
    int  len;
};
static struct struct_Undo undo = { NULL, 0 };

static int is_editor_page_on_top(void);
static int SourceBuffer_CreateFromFile(const char *filename, char **cpp);
static int SourceBuffer_CreateFromIntegratedEditor(char **cpp);
static int SourceBuffer_Dispose(char *buffer, int size);

static void ProcessedData_PutIntoFile(const char *fname, const char *buf, int size);

static int Undo_Get(const char **buf, int *len);
static void Undo_Set(char *buf, int len);

void unempl_cb (GtkWidget *widget, void *data);
void sign_cb (GtkWidget *widget, void *data);
void sign2_cb (GtkWidget *widget, void* data);
static void generic_cancel_cb(GtkWidget *widget, void *data);
void prepare_app();
void lib_properties_cb (GtkWidget *widget, void *data);
void report_error(GtkWidget* widget, void* data);
void CreateAndInsertKeyringPage_cb(gpointer data, gpointer user_data);
void sign_destroy_cb (GnomePgpSign* widget, char* data);
void destroy_cb (GtkWidget *widget, gpointer data);
void unsign_cb (GtkWidget *widget, void *data);
void unsign2_cb (GtkWidget *widget, void* data);
void unsign_destroy_cb (GnomePgpUnSign* widget, char* data);

/* Menu, toolbar callbacks */

static void about_cb (GtkWidget *widget, void *data);
void close_cb (GtkWidget *widget, void *data);

static void editor_copy_cb(GtkWidget *widget, void *data);
static void editor_cut_cb(GtkWidget *widget, void *data);
static void editor_paste_cb(GtkWidget *widget, void *data);
static void editor_selectall_cb(GtkWidget *widget, void *data);
static void editor_undo_cb(GtkWidget *widget, void *data);
static void editor_DupToUndoBuffer(void);
static int editor_GetData(char **x_buf, int *size);
static void editor_SetData(const char *buf, int size);

void new_cb (GtkWidget *widget, void *data);
void open_cb (GtkWidget *widget, void *data);
void quit_cb (GtkWidget *widget, void *data);
void refresh_cb (GtkWidget *widget, void *data);

static int save_state      (GnomeClient        *client,
                            gint                phase,
                            GnomeRestartStyle   save_style,
                            gint                shutdown,
                            GnomeInteractStyle  interact_style,
                            gint                fast,
                            gpointer            client_data);
static void connect_client (GnomeClient *client,
                            gint         was_restarted,
                            gpointer     client_data);

void discard_session (gchar *id);

struct options
{
    GList* rings; /* rings to open */
    int x,y,h,w; /* window geometry */
} options;

GnomeApp *app;

int restarted = 0;

/* True if parsing determined that all the work is already done.  */
int just_exit = 0;

/* These are the arguments that our application supports. */
#define DISCARD_KEY -1
#if 0
struct argp_option
{
};
static struct argp_option arguments[] =
{
  { "discard-session", DISCARD_KEY, N_("ID"), 0, N_("Discard session"), 1 },
  { NULL, 0, NULL, 0, NULL, 0 }
};

static struct argp parser =
{
  arguments,                    /* Options.  */
  parse_an_arg,                 /* The parser function.  */
  NULL,                         /* Some docs.  */
  NULL,                         /* Some more docs.  */
  NULL,                         /* Child arguments -- gnome_init fills
                                   this in for us.  */
  NULL,                         /* Help filter.  */
  NULL                          /* Translation domain; for the app it
                                   can always be NULL.  */
};
static error_t parse_an_arg (int key, char *arg, struct argp_state *state); 
void parse_args (int argc, char *argv[]);
#endif

GnomeUIInfo helpmenu[] =
{
    {
        GNOME_APP_UI_ITEM, 
        N_("_About..."),
        N_("Info about this program"),
        about_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_ABOUT,
        0, 0,
        NULL
    },
    GNOMEUIINFO_END
};

GnomeUIInfo filemenu[] =
{
    {
        GNOME_APP_UI_ITEM, 
        N_("New keyring..."),
        N_("Create a new keyring"),
        new_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW,
        0, 0, NULL
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("Open keyring..."),
        N_("Open keyring"),
        open_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN,
        0, 0, NULL
    },
    {
        GNOME_APP_UI_ITEM, 
        /* WARNING! This menu item should be #2 (CLOSE) */
        N_("Close keyring"),
        N_("Close keyring"),
        close_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE,
        0, 0, NULL
    },
    GNOMEUIINFO_SEPARATOR,
    {
        GNOME_APP_UI_ITEM, 
        N_("Refresh All"),
        N_("Refresh All"),
        refresh_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REFRESH,
        0, 0, NULL
    },
    GNOMEUIINFO_SEPARATOR,
    {
        GNOME_APP_UI_ITEM, 
        N_("GnomePgpLib Properties..."),
        N_("GnomePgpLib Properties"),
        lib_properties_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PROP,
        0, 0, NULL
    },
    GNOMEUIINFO_SEPARATOR,
    {
        GNOME_APP_UI_ITEM, 
        N_("Quit"),
        N_("Exit " GPGP_OFFICIAL_NAME),
        quit_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_QUIT,
        0, 0, NULL
    },
    GNOMEUIINFO_END
};

GnomeUIInfo editmenu[] =
{
    {
        GNOME_APP_UI_ITEM, 
        N_("_Copy"),
        N_("Copy"),
        editor_copy_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_COPY,
        GDK_C, GDK_CONTROL_MASK,
        NULL
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("Cu_t"),
        N_("Cut"),
        editor_cut_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CUT,
        GDK_X, GDK_CONTROL_MASK,
        NULL
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("_Paste"),
        N_("Paste"),
        editor_paste_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PASTE,
        GDK_V, GDK_CONTROL_MASK,
        NULL
    },
    {
        GNOME_APP_UI_SEPARATOR
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("_Select All"),
        N_("Select All"),
        editor_selectall_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CONVERT,
        0, 0,
        NULL
    },
    {
        GNOME_APP_UI_SEPARATOR
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("_Undo"),
        N_("Undo"),
        editor_undo_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_UNDO,
        0, 0,
        NULL
    },
    GNOMEUIINFO_END
};

GnomeUIInfo pgpmenu[] =
{
    {
        GNOME_APP_UI_ITEM, 
        N_("_Encrypt only..."),
        N_("Encrypt only"),
        sign_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_FORWARD,
        0, 0, NULL
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("_Sign only..."),
        N_("Sign only"),
        sign_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_FORWARD,
        0, 0, NULL
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("_Both sign and encrypt..."),
        N_("Both sign and encrypt"),
        sign_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_FORWARD,
        0, 0, NULL
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("_Decrypt or check signature..."),
        N_("Decrypt or check signature"),
        unsign_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BACK,
        0, 0, NULL
    },
    GNOMEUIINFO_END
};


GnomeUIInfo toolbar[] =
{
#if 0
    {
        GNOME_APP_UI_ITEM, 
        N_("New"),
        N_("Create a new keyring"),
        new_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_NEW,
        0, 0,
        NULL
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("Open"), N_("Open keyring"),
        open_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_OPEN,
        0, 0, NULL
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("Close"), N_("Close key ring"), /* #2 CLOSE2 */
        close_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_CLOSE,
        0, 0, NULL
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("Refresh"), N_("Refresh all keyrings"),
        refresh_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_REFRESH,
        0, 0, NULL
    },
    {
        GNOME_APP_UI_SEPARATOR
    },
#endif
    {
        GNOME_APP_UI_ITEM, 
        N_("_Copy"),
        N_("Copy"),
        editor_copy_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_COPY,
        GDK_C, GDK_CONTROL_MASK,
        NULL
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("Cu_t"),
        N_("Cut"),
        editor_cut_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_CUT,
        GDK_X, GDK_CONTROL_MASK,
        NULL
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("_Paste"),
        N_("Paste"),
        editor_paste_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_PASTE,
        GDK_V, GDK_CONTROL_MASK,
        NULL
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("Select All"),
        N_("Select All"),
        editor_selectall_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_CONVERT,
        0, 0,
        NULL
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("_Undo"),
        N_("Undo"),
        editor_undo_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_UNDO,
        0, 0,
        NULL
    },
    {
        GNOME_APP_UI_SEPARATOR
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("Sign/Encrypt"), N_("Sign and encrypt"),
        sign_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_FORWARD,
        0, 0, NULL
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("Decrypt/Check"), N_("Decrypt and check signature"),
        unsign_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_BACK,
        0, 0, NULL
    },
    {
        GNOME_APP_UI_SEPARATOR
    },
    {
        GNOME_APP_UI_ITEM, 
        N_("Quit"), N_("Exit " GPGP_OFFICIAL_NAME),
        quit_cb, NULL, NULL, 
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_QUIT,
        0, 0, NULL
    },
    GNOMEUIINFO_END
};

GnomeUIInfo mainmenu[] =
{
    GNOMEUIINFO_SUBTREE(N_("_File"), filemenu),
    GNOMEUIINFO_SUBTREE(N_("_Edit"), editmenu),
    GNOMEUIINFO_SUBTREE(N_("_Tools"), pgpmenu),
    GNOMEUIINFO_SUBTREE(N_("_Help"), helpmenu),
    GNOMEUIINFO_END
};

struct struct_notebook_text
{
    GtkBox *box;
    GtkText *text;
    GtkWidget *scrollbar;
    gint pageNo;
};

struct mainwin
{
    GtkNotebook *notebook;
    gint curPage;
    struct struct_notebook_text editor;
    struct struct_notebook_text dlogger;
} mainwin;

int error=0;

struct unsign_destroy_pack
{
   char* buf;
   int   bufsize;
   GtkFileSelection* filesel;
};
void unsign_destroy_2_cb(GtkWidget* widget, struct unsign_destroy_pack* pack);
void unsign_destroy_cancel_cb(GtkWidget* widget, struct unsign_destroy_pack* pack);

int main(int argc, char *argv[])
{
    GnomeClient *client;   

  /* argp_program_version = VERSION; */

    options.rings = g_list_alloc();

    /* i18n */
#if 1
    bindtextdomain (PACKAGE, LOCALEDIR);
    textdomain (PACKAGE);
#endif

    /* init gnome and parse params */
    gnome_init("gpgp", VERSION, argc, argv);

    client = gnome_client_new();

    gtk_signal_connect (
        GTK_OBJECT (client),
        "save_yourself",
        GTK_SIGNAL_FUNC(save_state),
        (gpointer) argv[0]);
    gtk_signal_connect (
        GTK_OBJECT (client),
        "connect",
        GTK_SIGNAL_FUNC(connect_client),
        NULL);

    /*
     * prepare_app() makes all the gtk calls necessary to set up a
     * minimal Gnome application
     */
    if (!just_exit)
    {
        prepare_app ();
        gtk_main ();
    }
    return 0;
}

static void CreateNotebookPageWithTextWidget(struct struct_notebook_text *np)
{
    assert(np != NULL);
    np->box = GTK_BOX(gtk_hbox_new(FALSE, 0));
    np->text = GTK_TEXT(gtk_text_new(NULL, NULL));
    gtk_box_pack_start(np->box,GTK_WIDGET(np->text), TRUE, TRUE, 0);

    /* Create a vertical scrollbar for the text widget editor */
    np->scrollbar = gtk_vscrollbar_new(np->text->vadj);
    gtk_box_pack_start(np->box, np->scrollbar, FALSE, FALSE, 0);

    /* Now mark all components as visible */
    gtk_widget_show(GTK_WIDGET(np->text));
    gtk_widget_show(GTK_WIDGET(np->scrollbar));
    gtk_widget_show(GTK_WIDGET(np->box));
}

static void SetSensitivity_TextWidget(GtkText *text, gint sens)
{
    gtk_widget_set_sensitive(GTK_WIDGET(text), sens);
    gtk_text_set_editable(text, sens);
}

static void InsertPageIntoNotebook(
    struct struct_notebook_text *np,
    const char *tabText)
{
    if (np->pageNo < 0)
    {
        np->pageNo = mainwin.curPage++;
    }
    gtk_notebook_insert_page(
        mainwin.notebook,
        GTK_WIDGET(np->box),
        (tabText != NULL) ? gtk_label_new(tabText) : NULL,
        np->pageNo);
}

void LogTextMessage(const char *msg)
{
    if (msg != NULL)
    {
        if (mainwin.dlogger.text != NULL)
        {
            gint len, tlen;
            len = gtk_text_get_length(mainwin.dlogger.text);
            tlen = strlen(msg);
            if (tlen > 0)
            {
                static const char cr_string[] = "\n";
                const int cr_string_len = sizeof(cr_string)/sizeof(char) - 1;

                gtk_text_set_point(mainwin.dlogger.text, len);
                gtk_text_insert(
                    mainwin.dlogger.text, NULL, NULL, NULL, msg, tlen);
                gtk_text_insert(
                    mainwin.dlogger.text, NULL, NULL, NULL, cr_string, cr_string_len);
            }
            return;
        }
        printf(GPGP_OFFICIAL_NAME ": %s\n", msg);
    }
}

/*
 * prepare app()
 *
 */
void prepare_app()
{
    GtkWidget* mainbox;

    app = GNOME_APP(gnome_app_new (GPGP_ABBREVIATED_NAME,GPGP_OFFICIAL_NAME));

    /* GnomePgp init */
    gpgp_init(app, 1);
  
    mainwin.curPage = 0;
    gtk_widget_set_events(
        GTK_WIDGET(app),
        GDK_KEY_PRESS_MASK | GDK_BUTTON_PRESS_MASK);
    gtk_widget_realize (GTK_WIDGET(app));
    gtk_signal_connect (
        GTK_OBJECT (app),
        "delete_event",
        GTK_SIGNAL_FUNC (quit_cb),
        NULL);
    if (restarted) 
    {
        gtk_widget_set_uposition (GTK_WIDGET(app), options.x, options.y);
        gtk_widget_set_usize     (GTK_WIDGET(app), options.w, options.h);
    }
    else
    {
        void* s;
        char* key;
        char* value;
  
        /* create rings section */
        if (!gnome_config_has_section(GPGP_CONFIG_RINGS))
        {
            gnome_config_set_string(GPGP_CONFIG_PUBRING, gpgp_options.defpub);
            gnome_config_set_string(GPGP_CONFIG_SECRING, gpgp_options.defsec);
        }

        s = gnome_config_init_iterator(GPGP_CONFIG_RINGS);
        while (gnome_config_iterator_next(s, &key, &value))
            options.rings = g_list_append(options.rings,value);
    }

    /* Create menus */
    gnome_app_create_menus(app, mainmenu);
    gtk_widget_set_sensitive( filemenu[2].widget, FALSE ); /* CANCEL */

    /* Create statusbar */
    gnome_app_set_statusbar(
        app, gnome_appbar_new(TRUE, TRUE, GNOME_PREFERENCES_USER));

    /* Set toolbar at left side by default */
    gnome_config_set_string(
        "gpgp/Placement/Toolbar",
        gnome_config_get_string("gpgp/Placement/Toolbar=left"));

    /* Create toolbar */
    gnome_app_create_toolbar(app, toolbar);

    /* Create main box */  
    mainbox = gtk_vbox_new (FALSE, 0);
    gnome_app_set_contents (GNOME_APP(app), mainbox);

    /* create notebook */
    mainwin.notebook = GTK_NOTEBOOK(gtk_notebook_new());
    gtk_notebook_set_tab_pos(mainwin.notebook, GTK_POS_TOP);
    gtk_widget_set_usize(GTK_WIDGET(mainwin.notebook), 600, 200); /* FIXME */

    gtk_box_pack_start(
        GTK_BOX(mainbox),
        GTK_WIDGET(mainwin.notebook),
        TRUE, TRUE, 0);

    mainwin.dlogger.pageNo = 0;
    CreateNotebookPageWithTextWidget(&mainwin.dlogger);
    SetSensitivity_TextWidget(mainwin.dlogger.text, TRUE /* FALSE */);
    InsertPageIntoNotebook(&mainwin.dlogger, _("Activity"));

    /* Create keyrings page */
    g_list_foreach(options.rings, CreateAndInsertKeyringPage_cb, NULL);

    mainwin.editor.pageNo = 0;
    CreateNotebookPageWithTextWidget(&mainwin.editor);
    SetSensitivity_TextWidget(mainwin.editor.text, TRUE);
    InsertPageIntoNotebook(&mainwin.editor, _("Editor"));

    gtk_notebook_set_page(mainwin.notebook, mainwin.editor.pageNo);
    gtk_widget_show_all(GTK_WIDGET(app));
}

static int is_editor_page_on_top(void)
{
    if (mainwin.editor.text == NULL)
        return FALSE;
    else
    {
        gint cur_page;
        cur_page = gtk_notebook_get_current_page(mainwin.notebook);
        return (cur_page == mainwin.editor.pageNo);
    }
}

static void editor_cut_cb (GtkWidget *widget, void *data)
{
    if (is_editor_page_on_top())
        gtk_editable_cut_clipboard(GTK_EDITABLE(mainwin.editor.text));
}

static void editor_copy_cb (GtkWidget *widget, void *data)
{
    if (is_editor_page_on_top())
        gtk_editable_copy_clipboard(GTK_EDITABLE(mainwin.editor.text));
}

static int editor_GetData(char **x_buf, int *size)
{
    int len, allocSize;

    if (mainwin.editor.text == NULL)
        return 0;

    len = gtk_text_get_length(GTK_TEXT(mainwin.editor.text));

    if (x_buf != NULL)
    {
        char *buf;
        const char *bufc;

        bufc = gtk_editable_get_chars(
            GTK_EDITABLE(mainwin.editor.text), 0, len);
        if (bufc == NULL)
            len = 0;

        allocSize = len + 1;
        buf = (char *) g_malloc(allocSize);
        if (buf == NULL)
        {
            gpgp_MessageInsufficientMemory("sign_cb", allocSize);
            return 0;
        }
        memmove(buf, bufc, len);
        buf[len] = '\0';
        *x_buf = buf;
    }
    if (size != NULL)
        *size = len;
    return 1;
}

static void editor_paste_cb (GtkWidget *widget, void *data)
{
    if (is_editor_page_on_top())
        gtk_editable_paste_clipboard(GTK_EDITABLE(mainwin.editor.text));
}

static void editor_selectall_cb (GtkWidget *widget, void *data)
{
    if (is_editor_page_on_top())
        gtk_editable_select_region(GTK_EDITABLE(mainwin.editor.text), 0, -1);
}

static void editor_DupToUndoBuffer(void)
{
    int u_len;

    if (editor_GetData(NULL, &u_len))
    {
        if (u_len > 0)
        {
            char *u_buf;
            u_buf = (char *) g_malloc(u_len+1);
            if (u_buf != NULL)
            {
                if (editor_GetData(&u_buf, &u_len))
                    Undo_Set(u_buf, u_len);
                else
                    g_free(u_buf);
            }
        }
    }
}

static void editor_SetData(const char *buf, int size)
{
    assert(mainwin.editor.text != NULL);

    editor_DupToUndoBuffer();

    gtk_editable_delete_text(GTK_EDITABLE(mainwin.editor.text), 0, -1);
    if ((size > 0) && (buf != NULL))
    {
        gtk_text_insert(
            GTK_TEXT(mainwin.editor.text),
            NULL,
            NULL,
            NULL,
            buf,
            size);
    }
}

static void editor_undo_cb(GtkWidget *widget, void *data)
{
    const char *buf;
    int len;

    /*
     * Undo data blocks are created by user and auto-deleted
     * by undo code when needed. We get the data in read-only mode.
     */
    if (Undo_Get(&buf, &len))
    {
        if (len > 0)
        {
            editor_SetData(buf, len);
        }
    }
}

/*
 * CreateAndInsertKeyringPage_cb()
 *
 * Procedure creates new GNOME_PGP_KEYRING object, populates
 * it with key data and adds it to the keyring notebook control.
 *
 */
void CreateAndInsertKeyringPage_cb(gpointer data, gpointer user_data)
{
    if (data != NULL)
    {
        GtkWidget* keyringpage;
        auto char buf[256];

        keyringpage = gnome_pgp_keyring_new();
        if (keyringpage != NULL)
        {
            gtk_signal_connect(
                GTK_OBJECT (keyringpage),
                "error",
                GTK_SIGNAL_FUNC (report_error),
                NULL);
            error=0;
            gnome_pgp_keyring_select(GNOME_PGP_KEYRING(keyringpage), data);
            if (error == 0)
            {
                gtk_notebook_insert_page(
                    GTK_NOTEBOOK(mainwin.notebook),
                    keyringpage,
                    gtk_label_new(g_basename(data)),
                    mainwin.curPage++);
                gtk_widget_show (keyringpage);
                return;
            }
            else
                gtk_widget_destroy(GTK_WIDGET(keyringpage));
        }
        g_snprintf(buf,sizeof(buf),_("Failed to set up keyring '%s'"),data);
        LogTextMessage(buf);
    }
}

/************************/
/*  Error handling      */
/************************/

void err_ok_cb(GtkWidget* widget, GtkWidget* data)
{
    gtk_widget_destroy(data);
}

void report_error(GtkWidget* widget, void* data)
{
    char buf[1000];
    error=1;
    if (IS_GNOME_PGP_KEYRING(widget))
    {
        strncpy(buf,GNOME_PGP_KEYRING(widget)->error_info,sizeof(buf));
        GNOME_PGP_KEYRING(widget)->error_info=NULL;
    }

    if (IS_GNOME_PGP_SIGN(widget))
    {
        strncpy(buf,GNOME_PGP_SIGN(widget)->error_info,sizeof(buf));
        GNOME_PGP_SIGN(widget)->error_info=NULL;
    }

    if (IS_GNOME_PGP_UNSIGN(widget))
    {
        strncpy(buf,GNOME_PGP_UNSIGN(widget)->error_info,sizeof(buf));
        GNOME_PGP_UNSIGN(widget)->error_info=NULL;
    }
    LogTextMessage(buf);
}

/* Callbacks functions */

/******************/
/*  Quit          */
/******************/

void quit_cb (GtkWidget *widget, void *data)
{
    gtk_main_quit ();
    return;
}

/******************/
/*  Refresh       */
/******************/

void for_each_cb (GtkWidget *widget, void* data)
{
    gnome_pgp_keyring_refresh(GNOME_PGP_KEYRING(widget));
}

void refresh_cb (GtkWidget *widget, void *data)
{
    gtk_container_foreach(
        GTK_CONTAINER(mainwin.notebook),
        (GtkCallback) for_each_cb,
        NULL);
}

/*******************/
/*  Unimplemented  */
/*******************/

void unempl_cb (GtkWidget *widget, void *data)
{
    LogTextMessage(_("This feature is not implemented yet!"));
}

/*
 * Sign/Encrypt
 */

static void CreateSignBox(char *buf, int st_size, char *filename)
{
    GtkWidget* signbox;

    signbox = gnome_pgp_sign_new(
        buf, st_size,
        (filename != NULL) ? g_basename(filename) : NULL);
    gtk_signal_connect(
        GTK_OBJECT(signbox),
        "error",
        GTK_SIGNAL_FUNC(report_error),
        NULL);
    gtk_signal_connect(
        GTK_OBJECT(signbox),
        "cancel",
        GTK_SIGNAL_FUNC(generic_cancel_cb),
        NULL);
    gtk_signal_connect(
        GTK_OBJECT(signbox),
        "destroy",
        GTK_SIGNAL_FUNC(sign_destroy_cb),
        filename);
    gtk_widget_show(signbox);
}

/*
 * Read from integrated editor
 */
static int SourceBuffer_CreateFromIntegratedEditor(char **cpp)
{
    char *buf;
    int len;

    if (editor_GetData(&buf, &len))
    {
        if (cpp != NULL)
            *cpp = buf;
        return len;
    }
    else
        return 0;
}

/*
 * SourceBuffer_CreateFromFile()
 *
 * Procedure allocates a buffer and fills it with data from
 * specified file. Returns number of bytes in that file (buffer).
 *
 * History:
 * 22-May-99 tftp       Replaced shared memory with g_malloc()
 */
static int SourceBuffer_CreateFromFile(const char *filename, char **cpp)
{
    int fd, nRead;
    struct stat stats;
    char *buf;

    assert(cpp != NULL);
    fd = open(filename, O_RDONLY);
    if (fd == -1)
    {
        LogTextMessage(g_strerror(errno));
        return 0;
    }

    if ( fstat(fd, &stats)==-1)
    {
        LogTextMessage(g_strerror(errno));
        return 0;
    }
 
    /*buf = mmap(NULL, stats.st_size, PROT_READ, MAP_SHARED, fd, 0);*/
    buf = (char *) g_malloc(stats.st_size);
    if (buf == NULL)
    {
        gpgp_MessageInsufficientMemory("sign2_cb", stats.st_size);
        return 0;
    }
    nRead = read(fd, buf, stats.st_size);
    if (nRead != stats.st_size)
    {
        gpgp_MessageBox(
            _("Warning"),
            _("Failed to read complete file."));
    }
    *cpp = buf;
    return nRead;
}

/*
 * SourceBuffer_Dispose()
 *
 * Procedure releases (frees) the buffer that we previously
 * allocated and stuffed with source data.
 *
 * History:
 * 22-May-99 tftp       Replaced shared memory with g_malloc()
 */
static int SourceBuffer_Dispose(char *buffer, int size)
{
    if (buffer != NULL)
    {
        /* munmap(buffer, size); */
        g_free(buffer);
    }
    return TRUE; /* Always successful */
}

/*
 * sign_cb()
 *
 * Procedure gets called whenever we want to read and encrypt/sign
 * contents of a file or an integrated editor. Here we create a file
 * selection widget and let loose.
 *
 * History:
 * 15-May-99 tftp       Added handling of integrated editor.
 */
void sign_cb (GtkWidget *widget, void *data)
{
    if (is_editor_page_on_top())
    {
        int len;
        char *buf;
        len = SourceBuffer_CreateFromIntegratedEditor(&buf);
        if ((len > 0) && (buf != NULL))
            CreateSignBox(buf, len, NULL);
    }
    else /* Show the file selection dialog, attach signals to proceed */
    {
        GtkWidget* filesel;

        filesel = gtk_file_selection_new(_("Choose file"));
        gtk_signal_connect(
            GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
            "clicked",
            GTK_SIGNAL_FUNC(destroy_cb),
            filesel);
        gtk_signal_connect(
            GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
            "clicked",
            GTK_SIGNAL_FUNC(sign2_cb),
            filesel); 
        gtk_widget_show(filesel);
    }
}
 
void gpgp_MessageInsufficientMemory(const char *module, int size)
{
    static const char *msg =
        N_("Failed to allocate memory.\nProcedure:%s Size:%d");
    char *buf;
    int length;

    length = strlen(msg);
    if (module != NULL)
        length += strlen(module);
    else
        module = "";
    length += 20;       /* Room for the integer */
    buf = (char *) g_malloc(length);
    if (buf != NULL)
    {
        g_snprintf(buf, length, msg, module, size);
        gpgp_MessageBox(NULL, buf);
        g_free(buf);
    }
    else
        gnome_app_flash(gpgp_options.app, msg);
}

/*
 * sign2_cb()
 *
 * This code handles OK button in file selection dialog.
 *
 * History:
 */
void sign2_cb (GtkWidget *widget, void* data)
{
    char *buf, *filename;
    int len;

    filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(data));
    filename=strdup(filename);
    destroy_cb(widget, data); /* delete file selector */

    len = SourceBuffer_CreateFromFile(filename, &buf);
    if ((len > 0) && (buf != NULL))
        CreateSignBox(buf, len, filename);
}

static void generic_cancel_cb (GtkWidget *widget, void *data)
{
    gnome_app_flash(gpgp_options.app, _("Canceled by user"));
}

/*
 * sign_destroy_cb()
 *
 * This procedure is called when signing is complete.
 * It should save somewhere the processed data - either in
 * the integrated editor or into a file.
 *
 * Parameters:
 * widget       - The GnomePgpSign object; it contains the
 *                processed data.
 * data         - The name of the source file or NULL if
 *                source data was in the integrated editor.
 *
 * History:
 */
void sign_destroy_cb(GnomePgpSign *widget, char *data)
{
    if (widget != NULL)
    {
        int have_data;

        /* Free the source buffer: we are through with it. */
        SourceBuffer_Dispose(widget->buf, widget->bufsize);

        have_data = (widget->obuf != NULL) && (widget->obufsize > 0);
        if (have_data)
        {
            if (data == NULL)   /* Put data into the integrated editor */
            {
                editor_SetData(widget->obuf, widget->obufsize);
            }
            else /* Put data into a file named after source file */
            {
                char *buf;
                static const char *ext = ".pgp";

                buf = g_malloc(strlen(data)+strlen(ext)+1);
                assert (buf != NULL);
                strcpy(buf, data);
                strcat(buf, ext);
                ProcessedData_PutIntoFile(buf, widget->obuf,widget->obufsize);
                g_free(buf);
            }
        }
        else
        {
            gpgp_MessageBox(
                _("Warning"),
                _("No data was generated!"));
        }
        if (widget->obuf)
            free(widget->obuf);
    }
} 

/*******************/
/* UnSign          */
/*******************/

void CreateUnsignBox(char *buf, int size, char *filename)
{
    GtkWidget* unsignbox;

    unsignbox = gnome_pgp_unsign_new(buf, size, filename);
    gtk_signal_connect(
        GTK_OBJECT(unsignbox),
        "error",
        GTK_SIGNAL_FUNC(report_error),
        NULL);
    gtk_signal_connect(
        GTK_OBJECT(unsignbox),
        "cancel",
        GTK_SIGNAL_FUNC(generic_cancel_cb),
        NULL);
    gtk_signal_connect(
        GTK_OBJECT(unsignbox),
        "destroy",
        GTK_SIGNAL_FUNC(unsign_destroy_cb),
        filename);
 
    gtk_widget_show(unsignbox);
    gnome_pgp_unsign_action(GNOME_PGP_UNSIGN(unsignbox));
}

static void CreateFileSelectionBox(char *title, GtkSignalFunc cb)
{
    static GtkWidget *filesel;

    filesel = gtk_file_selection_new(title);
    gtk_signal_connect(
        GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
        "clicked",
        GTK_SIGNAL_FUNC(destroy_cb),
        filesel);
    gtk_signal_connect(
        GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
        "clicked",
        cb,
        filesel); 
    gtk_widget_show(filesel);
}

/*
 * unsign_cb()
 *
 * This is the callback procedure for "Decrypt" menu item, toolbar.
 * Here we either grab data from the integrated editor or initiate
 * file selection procedure. After data has been acquired we create
 * "unsign" widget, pass to it the data buffer, size and the file
 * name (if we got data from file). Then the "unsign" widget starts
 * working, eventually calling its own callbacks.
 *
 * History:
 */
void unsign_cb (GtkWidget *widget, void *data)
{
    if (is_editor_page_on_top()) /* Use editor */
    {
        int len;
        char *buf;
        len = SourceBuffer_CreateFromIntegratedEditor(&buf);
        if ((len > 0) && (buf != NULL))
        {
            CreateUnsignBox(buf, len, NULL);
        }
    }
    else   /* Create file dialog and wait until we are called back */
    {
        CreateFileSelectionBox(_("Choose file"), GTK_SIGNAL_FUNC(unsign2_cb));
    }
}
 
/*
 * unsign2_cb()
 *
 * Procedure gets called by file selection widget when user
 * have selected a file to decrypt and pressed OK button.
 * Here we read the file into a buffer and create the "unsign"
 * widget which should do all the rest for us.
 *
 * History:
 */
void unsign2_cb (GtkWidget *widget, void* data)
{
    char *buf, *filename;
    int len;

    filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(data));
    filename = strdup(filename);
    destroy_cb(widget, data); /* delete file selector */

    len = SourceBuffer_CreateFromFile(filename, &buf);
    if ((len > 0) && (buf != NULL))
    {
        CreateUnsignBox(buf, len, filename);
    }
}

/*
 * unsign_destroy_cb()
 *
 * Procedure gets called when decryption ends. Here we dispose
 * of source data buffer and save decrypted data as appropriate.
 * If source data was in integrated editor then we put processed
 * data into the editor. If the source data was in file then we
 * initiate file selection dialog to let user to enter the file
 * name where to store decrypted data.
 *
 * History:
 */
void unsign_destroy_cb(GnomePgpUnSign *widget, char *data)
{
    struct unsign_destroy_pack *pack;

    SourceBuffer_Dispose(widget->buf, widget->bufsize);

    if (widget->obuf == NULL)
    {
        gpgp_MessageBox(NULL, _("Output buffer == NULL"));
        return;
    }

    pack = g_malloc(sizeof(struct unsign_destroy_pack));
    assert(pack != NULL);

    if (widget->filename == NULL)
    {
        editor_SetData(widget->obuf,widget->obufsize);
    }
    else
    {
        char *fname;
        static const char fname_ext[] = ".txt";

        pack->buf = widget->obuf;
        pack->bufsize = widget->obufsize;
        pack->filesel = GTK_FILE_SELECTION(
            gtk_file_selection_new(_("Save to...")));

        fname = (char *) g_malloc(strlen(widget->filename)+sizeof(fname_ext)+1);
        assert(fname != NULL);
        strcpy(fname, widget->filename);
        strcat(fname, fname_ext);
        gtk_file_selection_set_filename(pack->filesel, fname);
        g_free(fname);

        gtk_signal_connect(
            GTK_OBJECT(pack->filesel->ok_button),
            "clicked",
            GTK_SIGNAL_FUNC(unsign_destroy_2_cb),
            pack);
        gtk_signal_connect(
            GTK_OBJECT(pack->filesel->cancel_button),
            "clicked",
            GTK_SIGNAL_FUNC(unsign_destroy_cancel_cb),
            pack);
        gtk_widget_show(GTK_WIDGET(pack->filesel));
    }
}

void unsign_destroy_2_cb(GtkWidget* widget, struct unsign_destroy_pack* pack)
{
    char *filename;
    int badName = FALSE;

    assert(widget != NULL);
    assert(pack != NULL);

    filename = gtk_file_selection_get_filename(pack->filesel);

    /*
     * Make sure that filename exists, is not blank and
     * does not end as directory (with '/' or '\\' character).
     * Latter occurs if file name is erased; then GNOME returns
     * just directory name.
     */
    if (filename == NULL)
        ++badName;
    else
    {
        int len;
        len = strlen(filename);
        if (len <= 0)
            ++badName;
        else
        {
            if ((filename[len-1] == '/') ||
                (filename[len-1] == '\\'))
            {
                ++badName;
            }
        }
    }
    if (badName)
        gpgp_MessageBox(NULL,_("No file chosen!"));
    else
    {
        ProcessedData_PutIntoFile(filename, pack->buf, pack->bufsize);
        unsign_destroy_cancel_cb(widget, pack);
    }
} 

void unsign_destroy_cancel_cb(GtkWidget* widget, struct unsign_destroy_pack* pack)
{
    if (pack->buf != NULL)
        g_free(pack->buf);
    gtk_widget_destroy(GTK_WIDGET(pack->filesel));
    g_free(pack);
}

/******************/
/* About          */
/******************/

static void about_cb (GtkWidget *widget, void *data)
{
    GtkWidget *about;
    gchar *authors[] =
    {
        GPGP_AUTHOR_0,
        NULL
    };
    about = gnome_about_new(
        GPGP_OFFICIAL_NAME,
        VERSION,
        GPGP_COPYRIGHT_NOTICE,  /* Copyright notice */
        (const gchar**) authors,
        GPGP_WEBSITE,           /* Other comments */
        NULL                    /* Pixmap logo */
        ); 
  gtk_widget_show (about);
}

/***********************/
/*  Close              */
/***********************/

void close_cb (GtkWidget *widget, void *data)
{
    GList *children;
    int num=0;

    /* Authors of GTK was lazy to save current page as a number */
    children = mainwin.notebook->children;
    while (children)
    {
        if (mainwin.notebook->cur_page == children->data)
            break;
        children = children->next;
        num++;
    }
    gtk_notebook_remove_page(mainwin.notebook, num);

    if (!mainwin.notebook->cur_page)  /* CLOSE CLOSE2 */
    {
        gtk_widget_set_sensitive( filemenu[2].widget, FALSE );
        gtk_widget_set_sensitive( toolbar[2].widget, FALSE );
    }
}

/**************************/
/* New                    */
/**************************/

struct new_pack
{
    GnomePgpKeyring* page;
    GtkFileSelection* w;
    GnomeDialog* di;
};

static void new2_cancel_cb (GtkWidget *widget, struct new_pack* data)
{
    if (data->di)
        gtk_widget_destroy(GTK_WIDGET(data->di));
    gtk_widget_destroy(GTK_WIDGET(data->w));
    g_free(data);
}

void new2_append_cb(GtkWidget *widget, struct new_pack* data)
{
    GnomePgpKeyring* keyringpage;
 
    FILE* fp = fopen(gtk_file_selection_get_filename(data->w),"w");
    if (!fp)
    {
        LogTextMessage(g_strerror(errno));
        return;
    }
    fclose(fp);

    keyringpage = GNOME_PGP_KEYRING(gnome_pgp_keyring_new());
    gtk_signal_connect(
        GTK_OBJECT(keyringpage),
        "error",
        GTK_SIGNAL_FUNC(report_error),
        NULL);
    gnome_pgp_keyring_select(
        GNOME_PGP_KEYRING(keyringpage),
        gtk_file_selection_get_filename(data->w));
    gtk_notebook_append_page(
        GTK_NOTEBOOK(mainwin.notebook),
        GTK_WIDGET(keyringpage),
        gtk_label_new(gtk_file_selection_get_filename(data->w)));
    gtk_widget_show(GTK_WIDGET(keyringpage));

    refresh_cb(NULL, NULL);
    new2_cancel_cb(widget, data);
}

void new2_over_cb (GtkWidget *widget, struct new_pack* data)
{
    if (unlink(gtk_file_selection_get_filename(data->w))==-1)
    {
        LogTextMessage(g_strerror(errno));
        return;
    }
    new2_append_cb(widget,data);
}
 
void
new2_cb (GtkWidget *widget, struct new_pack* data)
{
 if (g_file_exists(gtk_file_selection_get_filename(data->w)))
  {
   GtkWidget* msgbox;
   GtkWidget* bcancel;
   GtkWidget* bover;
 
   /* create exist dialog */
   msgbox = gnome_message_box_new (_("File already exist"),
	             GNOME_MESSAGE_BOX_QUESTION,
		     NULL);

   bover = gtk_button_new_with_label(_("Overwrite"));
   bcancel = gnome_stock_button(GNOME_STOCK_BUTTON_CANCEL);

   gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(msgbox)->action_area),bover,
                              FALSE, TRUE, GNOME_PAD_SMALL);
   gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(msgbox)->action_area),bcancel,
                              FALSE, TRUE, GNOME_PAD_SMALL);

   data->di=GNOME_DIALOG(msgbox);

   gtk_signal_connect (GTK_OBJECT (bover), "clicked",GTK_SIGNAL_FUNC (new2_over_cb), data);
   gtk_signal_connect (GTK_OBJECT (bcancel), "clicked",GTK_SIGNAL_FUNC (new2_cancel_cb), data);
   
   gtk_widget_show(bover);
   gtk_widget_show(bcancel);
  
   gtk_widget_hide(GTK_WIDGET(data->w));
   
   gtk_widget_show(msgbox); 
   return;
  } 

  new2_append_cb(widget,data);
}

void destroy_cb(GtkWidget *widget, gpointer data)
{
    gtk_widget_destroy(GTK_WIDGET(data));
}

void new_cb (GtkWidget *widget, void *data)
{
 GtkWidget* filesel;
 struct new_pack* ep=g_malloc(sizeof(struct new_pack));

 filesel=gtk_file_selection_new(_("Choose file..."));
 gtk_file_selection_set_filename( GTK_FILE_SELECTION(filesel), gpgp_options.pgppath );

 ep->w=GTK_FILE_SELECTION(filesel);
 ep->page=(GnomePgpKeyring*) data;
 ep->di=NULL;
 
 gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button), "clicked",GTK_SIGNAL_FUNC(destroy_cb),filesel); 
 gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button), "clicked",GTK_SIGNAL_FUNC(new2_cb),ep);
 gtk_widget_show(filesel);
}

/**********************/
/* open               */
/**********************/
void open2_cb (GtkWidget *widget, struct new_pack *data);

void
open_cb (GtkWidget *widget, void *data)
{
 GtkWidget* filesel;
 struct new_pack* ep=g_malloc(sizeof(struct new_pack));

 filesel=gtk_file_selection_new(_("Choose file..."));
 gtk_file_selection_set_filename(GTK_FILE_SELECTION(filesel),gpgp_options.pgppath);

 ep->w=GTK_FILE_SELECTION(filesel);
 ep->page=(GnomePgpKeyring*) data;
 ep->di=NULL;
 
 gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button), "clicked",GTK_SIGNAL_FUNC(destroy_cb),filesel); 
 gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button), "clicked",GTK_SIGNAL_FUNC(open2_cb),ep);
 gtk_widget_show(filesel);
}

void
open2_cb (GtkWidget *widget, struct new_pack *data)
{
 GnomePgpKeyring* keyringpage;
 
 keyringpage = GNOME_PGP_KEYRING(gnome_pgp_keyring_new());
 gtk_signal_connect (GTK_OBJECT (keyringpage), "error",GTK_SIGNAL_FUNC (report_error),NULL);
 error=0;
 gnome_pgp_keyring_select(GNOME_PGP_KEYRING(keyringpage), gtk_file_selection_get_filename(data->w));
 if (!error)
   {
    gtk_notebook_prepend_page(
        GTK_NOTEBOOK(mainwin.notebook),
        GTK_WIDGET(keyringpage),
        gtk_label_new(gtk_file_selection_get_filename(data->w)));
    gtk_notebook_set_page(mainwin.notebook, 0);
    gtk_widget_set_sensitive(filemenu[4].widget, TRUE);
    gtk_widget_show(GTK_WIDGET(keyringpage));
   }
   else
    gtk_widget_destroy(GTK_WIDGET(keyringpage));

 new2_cancel_cb(widget, data);
}

/****************************/
/* Session management       */
/****************************/

int savering_n=0;

void savering(GtkNotebookPage* page, char* name)
{
    char buf[1000];
    g_snprintf(buf,sizeof(buf),"%s/%d", name, savering_n++);
    printf("%s=%s\n", buf, GNOME_PGP_KEYRING(page->child)->name);
    gnome_config_set_string(buf, GNOME_PGP_KEYRING(page->child)->name);
}

static int save_state(
    GnomeClient        *client,
    gint                phase,
    GnomeRestartStyle   save_style,
    gint                shutdown,
    GnomeInteractStyle  interact_style,
    gint                fast,
    gpointer            client_data)
{
  gchar *session_id;
  gchar *sess;
  gchar *buf;
  gchar *argv[3];
  gint x, y, w, h;

  session_id=gnome_client_get_id (client);

  /* The only state that gnome-hello has is the window geometry.
     Get it. */
  gdk_window_get_geometry (GTK_WIDGET(app)->window, &x, &y, &w, &h, NULL);

  /* Save the state using gnome-config stuff. */
  sess = g_copy_strings ("/gpgp-app/Saved-Session-",
                         session_id,
                         NULL);

  buf = g_copy_strings ( sess, "/x", NULL);
  gnome_config_set_int (buf, x);
  g_free(buf);
  buf = g_copy_strings ( sess, "/y", NULL);

  gnome_config_set_int (buf, y);
  g_free(buf);
  buf = g_copy_strings ( sess, "/w", NULL);
  gnome_config_set_int (buf, w);
  g_free(buf);
  buf = g_copy_strings ( sess, "/h", NULL);
  gnome_config_set_int (buf, h);
  g_free(buf);

  g_free(sess);

/* rings */
  savering_n=0;

  sess = g_copy_strings ("/gpgp-app/rings-Saved-Session-",
                         session_id,
                         NULL);

  g_list_foreach(mainwin.notebook->children, (GFunc) savering, sess);

  g_free(sess);

  gnome_config_sync();
 
  /*
   * Here is the real SM code. We set the argv to the parameters needed
   * to restart/discard the session that we've just saved and call
   * the gnome_session_set_*_command to tell the session manager it.
   */
  argv[0] = (char*) client_data;
  argv[1] = "--discard-session";
  argv[2] = session_id;
  gnome_client_set_discard_command (client, 3, argv);

  /* Set commands to clone and restart this application.  Note that w
e
     use the same values for both -- the session management code will
     automatically add whatever magic option is required to set the
     session id on startup.  */
  gnome_client_set_clone_command (client, 1, argv);
  gnome_client_set_restart_command (client, 1, argv);

  return TRUE;
}

/*
 * Connected to session manager. If restarted from a former session:
 * reads the state of the previous session. Sets os_* (prepare_app
 * uses them)
 */
void connect_client(
    GnomeClient *client,
    gint was_restarted,
    gpointer client_data)
{
    gchar *session_id;

    /*
     * Note that information is stored according to our _old_
     * session id.  The id can change across sessions.
     */
    session_id = gnome_client_get_previous_id (client);
    if (was_restarted && session_id != NULL)
    {
        gchar *sess;
        gchar *buf;
        void  *s;
        char* key;
        char* value;
 
        restarted = 1;

        sess = g_copy_strings ("/gpgp-app/Saved-Session-", session_id, NULL);

        buf = g_copy_strings ( sess, "/x", NULL);
        options.x = gnome_config_get_int (buf);
        g_free(buf);

        buf = g_copy_strings ( sess, "/y", NULL);
        options.y = gnome_config_get_int (buf);
        g_free(buf);

        buf = g_copy_strings ( sess, "/w", NULL);
        options.w = gnome_config_get_int (buf);
        g_free(buf);

        buf = g_copy_strings ( sess, "/h", NULL);
        options.h = gnome_config_get_int (buf);
        g_free(buf);

        g_free(sess);

        /* Rings */
        sess = g_copy_strings("/gpgp-app/rings-Saved-Session-", session_id, NULL);
      
        s = gnome_config_init_iterator(sess);
        while (gnome_config_iterator_next(s, &key, &value))
        {
            options.rings = g_list_append(options.rings,value);
        }
        g_free(sess);
    }

    /* If we had an old session, we clean up after ourselves.  */
    if (session_id != NULL)
        discard_session (session_id);
    return;
}

void discard_session (gchar *id)
{
    gchar *sess;

    sess = g_copy_strings ("/gpgp-app/Saved-Session-", id, NULL);

    /*
     * We use the gnome_config_get_* to work around a
     * bug in gnome-config (it's going under a redesign/rewrite,
     * so i didn't correct it) [Max]
     */
    gnome_config_get_int ("/gpgp-app/Bug/work-around=0");
    gnome_config_clean_section (sess);
    g_free (sess);

    sess = g_copy_strings ("/gpgp-app/rings-Saved-Session-", id, NULL);
    gnome_config_get_int ("/gpgp-app/Bug/work-around=0");
    gnome_config_clean_section (sess);
    g_free (sess);
  
    gnome_config_sync ();

    return;
}

/*
 * Options parsing
 */
#if 0
static error_t
parse_an_arg (int key, char *arg, struct argp_state *state)
{
  if (key == DISCARD_KEY)
    {
      discard_session (arg);
      just_exit = 1;
      return 0;
    }

  /* We didn't recognize it.  */
  return ARGP_ERR_UNKNOWN;
}
#endif

void lib_properties_cb (GtkWidget *widget, void *data)
{
    GtkWidget *properties;
    gnome_app_warning(app, _("This feature is under construction")); 
    properties = gnome_pgp_lib_properties_new();
    gtk_widget_show(properties);
}

static void ProcessedData_PutIntoFile(
    const char *fname,
    const char *obuf,
    int obufsize)
{
    int fd;

    /* TODO: ask for Overwrite and etc here */ 
    fd = open(fname, O_CREAT | O_TRUNC | O_WRONLY, 0666);
    if (fd == -1)
    {
        LogTextMessage(g_strerror(errno));
    }
    else /* File opened OK */
    {
        if ((obufsize > 0) && (obuf != NULL))
        {
            int nWrote;

            nWrote = write(fd, obuf, obufsize);
            if (nWrote != obufsize)
            {
                LogTextMessage(g_strerror(errno));
            }
        }
        close(fd);
    }
}

static int Undo_Get(const char **buf, int *len)
{
    if ((undo.buf == NULL) || (buf == NULL) || (len == NULL))
        return 0;
    else
    {
        *buf = undo.buf;
        *len = undo.len;
        undo.buf = NULL;
        return 1;
    }
}

static void Undo_Set(char *buf, int len)
{
    if (undo.buf != NULL)
    {
        g_free(undo.buf);
        undo.buf = NULL;
    }
    undo.buf = buf;
    undo.len = len;
}
