/*
 *  PGFileNotificationCenter.m
 *  graphviz
 *
 *  Created by Glen Low on Wed Feb 18 2004.
 *  Copyright (c) 2003, Pixelglow Software. All rights reserved.
 *  http://www.pixelglow.com/graphviz/
 *  graphviz@pixelglow.com
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted
 *  provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions
 *    and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
 *    and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *  * Neither the name of Pixelglow Software nor the names of its contributors may be used to endorse or
 *    promote products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
 
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/event.h>

#import "PGFileNotificationCenter.h"

// weak link these functions so that this won't fail on 10.2
int kqueue (void) AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER;
int kevent (int kq, const struct kevent *changelist, int nchanges,
		    struct kevent *eventlist, int nevents,
		    const struct timespec *timeout) AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER;
			
@interface PGFileNotificationRecord: NSObject
	{
		unsigned int sequence_;
		int fileDescriptor_;
		NSString* path_;
		id observer_;
		SEL selector_;
	}
	
+ (PGFileNotificationRecord*) sharedSequence: (int) sequence;
+ (PGFileNotificationRecord*) sharedObserver: (id) observer path: (NSString*) path;

- (id) initWithObserver: (id) observer selector: (SEL) selector path: (NSString*) path;

- (BOOL) isEqualToSequence: (PGFileNotificationRecord*) record;
- (BOOL) isEqualToObserverPath: (PGFileNotificationRecord*) record;

- (unsigned int) hashForSequence;
- (unsigned int) hashForObserverPath;

- (BOOL) notifyObserverWithResponseFlags: (unsigned int) flags writer: (NSFileHandle*) writer;
- (void) updateWorker: (NSFileHandle*) writer;

@end

typedef struct 
	{
		unsigned int sequence;
		int fileDescriptor;
	} PGFileNotificationRequest;

@interface PGFileNotificationResponse: NSObject
	{
		@private
		
		unsigned int sequence_;
		unsigned int flags_;
	}

+ (PGFileNotificationResponse*) responseWithSequence: (unsigned int) sequence flags: (unsigned int) flags;

- (id) initWithSequence: (unsigned int) sequence flags: (unsigned int) flags;

- (unsigned int) sequence;
- (unsigned int) flags;

@end


static const void* RetainCallBack (CFAllocatorRef allocator, const void* value)
	{
		return CFRetain (value);
	}

static void ReleaseCallBack (CFAllocatorRef allocator, const void* value)
	{
		CFRelease (value);
	}
	
static CFStringRef CopyDescriptionCallBack (const void* value)
	{
		return CFCopyDescription (value);
	}
	
static Boolean EqualSequenceCallBack (const void* value1, const void* value2)
	{
		return [((PGFileNotificationRecord*) value1) isEqualToSequence: ((PGFileNotificationRecord*) value2)];
	}

static Boolean EqualObserverPathCallBack (const void* value1, const void* value2)
	{
		return [((PGFileNotificationRecord*) value1) isEqualToObserverPath: ((PGFileNotificationRecord*) value2)];
	}
	
static CFHashCode HashCodeSequenceCallBack (const void* value)
	{
		return [((PGFileNotificationRecord*) value) hashForSequence];
	}

static CFHashCode HashCodeObserverPathCallBack (const void* value)
	{
		return [((PGFileNotificationRecord*) value) hashForObserverPath];
	}
	
static const CFSetCallBacks sequenceCallBacks_ =
	{
		0,
		RetainCallBack,
		ReleaseCallBack,
		CopyDescriptionCallBack,
		EqualSequenceCallBack,
		HashCodeSequenceCallBack
	};

static const CFSetCallBacks observerPathCallBacks_ =
	{
		0,
		RetainCallBack,
		ReleaseCallBack,
		CopyDescriptionCallBack,
		EqualObserverPathCallBack,
		HashCodeObserverPathCallBack
	};

@implementation PGFileNotificationCenter

PGFileNotificationCenter* defaultCenter_;

+ (void) initialize
	{
		defaultCenter_ = [[PGFileNotificationCenter alloc] init];
	}
	
+ (PGFileNotificationCenter*) defaultCenter
	{
		return defaultCenter_;
	}

- (void) notifyObserverWithResponse: (PGFileNotificationResponse*) response
	{
		PGFileNotificationRecord* recordToNotify = (PGFileNotificationRecord*)
			CFSetGetValue (sequenceRecords_, [PGFileNotificationRecord sharedSequence: [response sequence]]);
			
		if (recordToNotify
			&& ![recordToNotify notifyObserverWithResponseFlags: [response flags] writer: writer_])
			{
				CFSetRemoveValue (sequenceRecords_, recordToNotify);
				CFSetRemoveValue (observerPathRecords_, recordToNotify);
			}

	}

- (void) working: (NSFileHandle*) reader
	{
		struct kevent change;
		struct kevent event;
		
		PGFileNotificationRequest request;
				
		int kq = kqueue ();
		
		// register to receive messages through the pipe
		EV_SET (&change, [reader fileDescriptor], EVFILT_READ, EV_ADD | EV_ENABLE, 0, 0, 0);
		kevent (kq, &change, 1, NULL, 0, NULL);
		
		while (true)
			{
				// wait for next event
				kevent (kq, NULL, 0, &event, 1, NULL);
				
				NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
				switch (event.filter)
					{
						case EVFILT_READ:
							// message sent by main thread: read in the action we are to perform
							[[reader readDataOfLength: sizeof (PGFileNotificationRequest)] getBytes: &request];
							EV_SET (&change, request.fileDescriptor, EVFILT_VNODE, EV_ADD | EV_ENABLE | EV_CLEAR,
								NOTE_DELETE | NOTE_WRITE | NOTE_EXTEND, 0, (void*) request.sequence);
							kevent (kq, &change, 1, NULL, 0, NULL);
							break;
							
						case EVFILT_VNODE:
							// file had changed: notify main thread							
							[self performSelectorOnMainThread: @selector (notifyObserverWithResponse:)
								withObject: [PGFileNotificationResponse responseWithSequence: (unsigned int) event.udata flags: event.fflags]
								waitUntilDone: NO];
							break;
					}
				[pool release];
			}
	}

- (id) init
	{
		if ((self = [super init]))
			{
				if (kqueue && kevent)
					{
						NSPipe* pipe = [[NSPipe pipe] retain];
						
						writer_ = [pipe fileHandleForWriting];
						[NSThread detachNewThreadSelector: @selector (working:) toTarget: self withObject: [pipe fileHandleForReading]];
					
						sequenceRecords_ = CFSetCreateMutable (kCFAllocatorDefault, 0, &sequenceCallBacks_);
						observerPathRecords_ = CFSetCreateMutable (kCFAllocatorDefault, 0, &observerPathCallBacks_);
					}
				else
					{
						[self autorelease];
						return nil;
					}
			}
		return self;
	}

- (void) addObserver: (id) observer selector: (SEL) selector path: (NSString*) path
	{
		PGFileNotificationRecord* recordToAdd =
			[[PGFileNotificationRecord alloc] initWithObserver: observer selector: selector path: path];
			
		if (recordToAdd)
			{
				CFSetAddValue (sequenceRecords_, recordToAdd);
				CFSetAddValue (observerPathRecords_, recordToAdd);
				
				[recordToAdd updateWorker: writer_];
				[recordToAdd release];	
			}
	}
	
- (void) removeObserver: (id) observer path: (NSString*) path
	{
		PGFileNotificationRecord* recordToRemove = (PGFileNotificationRecord*)
			CFSetGetValue (observerPathRecords_, [PGFileNotificationRecord sharedObserver: observer path: path]);
			
		if (recordToRemove)
			{
				CFSetRemoveValue (sequenceRecords_, recordToRemove);
				CFSetRemoveValue (observerPathRecords_, recordToRemove);
			}
	}
	
	
@end

static unsigned int sequential_ = 0;
static PGFileNotificationRecord* sharedSequence_;
static PGFileNotificationRecord* sharedObserverPath_;

@implementation PGFileNotificationRecord

+ (void) initialize
	{
		sharedSequence_ = [[PGFileNotificationRecord alloc] init];
		sharedObserverPath_ = [[PGFileNotificationRecord alloc] init];
	}

+ (PGFileNotificationRecord*) sharedSequence: (int) sequence
	{
		sharedSequence_->sequence_ = sequence;
		return sharedSequence_;
	}
	
+ (PGFileNotificationRecord*) sharedObserver: (id) observer path: (NSString*) path
	{
		sharedObserverPath_->observer_ = observer;
		if (sharedObserverPath_->path_ != path)
			{
				[sharedObserverPath_->path_ release];
				sharedObserverPath_->path_ = [path copy];
			}
		return sharedObserverPath_;
	}

- (id) init
	{
		if ((self = [super init]))
			{
				sequence_ = 0;
				fileDescriptor_ = 0;
				path_ = nil;
				observer_ = nil;
				selector_ = nil;			
			}
		return self;	
	}
	
- (id) initWithObserver: (id) observer selector: (SEL) selector path: (NSString*) path
	{
		if ((self = [super init]))
			{
				if ((fileDescriptor_ = open ([path fileSystemRepresentation], O_RDONLY, 0)) != -1)
					{
						sequence_ = ++sequential_;
						path_ = [path copy];
						observer_ = observer;
						selector_ = selector;
					}
				else
					{
						[self autorelease];
						return nil;
					}
			}
		return self;	
	}

- (BOOL) isEqualToSequence: (PGFileNotificationRecord*) record
	{
		return sequence_ == record->sequence_;
	}
	
- (BOOL) isEqualToObserverPath: (PGFileNotificationRecord*) record
	{
		return observer_ == record->observer_ && [path_ isEqualToString: record->path_];
	}

- (unsigned int) hashForSequence
	{
		return (unsigned int) sequence_;
	}
	
- (unsigned int) hashForObserverPath
	{
		return (unsigned int) observer_ ^ [path_ hash];
	}
	
- (BOOL) notifyObserverWithResponseFlags: (unsigned int) flags writer: (NSFileHandle*) writer
	{
		[observer_ performSelector: selector_ withObject: path_];
		
		// if file was deleted, try to reopen it
		if (flags & NOTE_DELETE)
			{
				close (fileDescriptor_);
				fileDescriptor_ = open ([path_ fileSystemRepresentation], O_RDONLY, 0);
				if (fileDescriptor_ == -1)
					return NO;
				else
					[self updateWorker: writer];
			}
		return YES;
	}
	
- (void) updateWorker: (NSFileHandle*) writer
	{
		PGFileNotificationRequest request;
		request.sequence = sequence_;
		request.fileDescriptor = fileDescriptor_;
		[writer writeData: [NSData dataWithBytes: &request length: sizeof (PGFileNotificationRequest)]];
	}

- (void) dealloc
	{
		[path_ release];
		if (fileDescriptor_ == -1)
			close (fileDescriptor_);	
	}

@end

@implementation PGFileNotificationResponse

+ (PGFileNotificationResponse*) responseWithSequence: (unsigned int) sequence flags: (unsigned int) flags
	{
		return [[[PGFileNotificationResponse alloc] initWithSequence: sequence flags: flags] autorelease];
	}

- (id) initWithSequence: (unsigned int) sequence flags: (unsigned int) flags
	{
		if ((self = [super init]))
			{
				sequence_ = sequence;
				flags_ = flags;		
			}
		return self;	
	}

- (unsigned int) sequence
	{
		return sequence_;
	}
	
- (unsigned int) flags
	{
		return flags_;
	}

@end

