/*
 *  PGGraphvizWindowController.m
 *  graphviz
 *
 *  Created by Glen Low on Tue Jun 22 2004.
 *  Copyright (c) 2004, Pixelglow Software. All rights reserved.
 *  http://www.pixelglow.com/graphviz/
 *  graphviz@pixelglow.com
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted
 *  provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions
 *    and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
 *    and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *  * Neither the name of Pixelglow Software nor the names of its contributors may be used to endorse or
 *    promote products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import "PGGraphvizWindowController.h"
#import "PGGraphvizDocument.h"
#import "PGOpenPanelApplicationFilter.h"
#import "PGZoomImageView.h"

static NSString* editTitle_ = nil;
static NSString* applicationDirectory_ = nil;

@implementation PGGraphvizWindowController

+ (void) initialize
	{
		// look for a suitable editor application...
		
		NSArray* applicationDirectories = NSSearchPathForDirectoriesInDomains (
			NSApplicationDirectory, NSSystemDomainMask, YES);
		if ([applicationDirectories count] > 0)
			applicationDirectory_ = [[applicationDirectories objectAtIndex: 0] retain];
	}

- (IBAction) toggleSettings: (id) sender
	{
		[settings toggle: sender];
	}

- (IBAction) actualSizeImage: (id) sender
	{
		[imager setZoom: 1.0];
	}

- (IBAction) zoomInImage: (id) sender
	{
		[imager setZoom: [imager zoom] * 2.0];
	}
	
- (IBAction) zoomOutImage: (id) sender
	{
		[imager setZoom: [imager zoom] * 0.5];
	}

- (IBAction) zoomToFitImage: (id) sender
	{
		NSSize imageSize = [[imager image] size];
		NSSize imagerSize = [[imager enclosingScrollView] documentVisibleRect].size;
		double widthRatio = imagerSize.width / imageSize.width;
		double heightRatio = imagerSize.height / imageSize.height;
		
		[imager center];
		[imager setZoom: (widthRatio < heightRatio ? widthRatio : heightRatio)];
	}
	
- (NSRect) windowWillUseStandardFrame: (NSWindow*) sender defaultFrame: (NSRect) defaultFrame
	{
		NSRect windowFrame = [sender frame];
		NSScrollView* scrollView = [imager enclosingScrollView];
		NSSize imageSize = [[imager image] size];
		double zoom = [imager zoom];
		NSRect imagerFrame = [scrollView convertRect: [scrollView documentVisibleRect] toView: nil];
		
		// window chrome == window frame - imager frame
		// standard frame == window chrome + zoomed image size
		float defaultFrameTop = NSMaxY (defaultFrame);
		defaultFrame.size = NSMakeSize (
			windowFrame.size.width - imagerFrame.size.width + imageSize.width * zoom,
			windowFrame.size.height - imagerFrame.size.height + imageSize.height * zoom);
		defaultFrame.origin.y = defaultFrameTop - defaultFrame.size.height;
		
		[imager center];
		return defaultFrame;
	}
	
- (BOOL) validateToolbarItem: (NSToolbarItem*) item
	{
		if ([[item itemIdentifier] isEqualToString: @"DotEdit"])
			{
				if (editTitle_)
					[item setToolTip: editTitle_];
				return [document respondsToSelector: @selector (editWithApplication:)];
			}
		else
			return YES;
	}
	
- (IBAction) editDocument: (id) sender
	{
		NSUserDefaults* userDefaults = [NSUserDefaults standardUserDefaults];
		NSString* editor = [userDefaults stringForKey: @"editor"];
		BOOL editOK;
		
		if (!editor || ([[NSApp currentEvent] modifierFlags] & NSAlternateKeyMask))
			{
				NSString* editorPath;
				CFURLRef editorURL;
				
				// if there's an editor in the defaults, the editor path is its full application path
				// else if we can find a suitable program, the editor path is the path to that program
				if (editor)
					editorPath = [[NSWorkspace sharedWorkspace] fullPathForApplication: editor];
				else if (LSCopyApplicationForMIMEType (CFSTR ("text/x-graphviz"), kLSRolesEditor, &editorURL) == noErr
					|| LSGetApplicationForInfo ('****', kLSUnknownCreator, CFSTR ("*"), kLSRolesEditor, NULL, &editorURL) == noErr)
					{
						editorPath = [((NSString*) CFURLCopyPath (editorURL)) autorelease];
						CFRelease (editorURL);
					}
				else
					editorPath = nil;
					
				// open panel with only applications selectable, and starting with the editor path
				NSOpenPanel *openPanel = [NSOpenPanel openPanel];
				PGOpenPanelApplicationFilter *filter = [[PGOpenPanelApplicationFilter alloc] initWithSource: [document fileName] roles: kLSRolesEditor];
				[openPanel setDelegate: filter];
				if ([openPanel
					runModalForDirectory: (editorPath ? [editorPath stringByDeletingLastPathComponent] : applicationDirectory_)
					file: [editorPath lastPathComponent]
					types: nil] == NSOKButton)
					{
						// set the selected editor as default (don't bother with the directory, since we search for it everytime we want it)
						editorPath = [openPanel filename];
						editor = [editorPath lastPathComponent];
						[userDefaults setObject: editor forKey: @"editor"];
		
						// now that we've changed the editor, update the edit title for tooltips etc.
						[editTitle_ release];
						editTitle_ = [[NSString stringWithFormat:
							[[[NSBundle mainBundle] localizedStringForKey: @"Edit.title" value: nil table: nil] retain],
							[[NSFileManager defaultManager] displayNameAtPath: editorPath]] retain];
						editOK = YES;
					}
				else
					editOK = NO;
					
				[openPanel setDelegate: nil];
				[filter release];
			}
		else
			editOK = YES;

		if (editOK)
			[document performSelector: @selector (editWithApplication:) withObject: editor];
	}

@end
