/* This file is for showing dependency informations
 * Gnometoaster depends on a variety of other programs to accomplish it's work.
 * Those programs are referenced by the respective filenames of their
 * executables. While this scheme works perfectly well, it is sometimes
 * difficult for a normal user to find out how to resolve a dependency
 * as chances aren't bad he won't even know what a specific executable
 * Gnometoaster needs is actually doing.
 * To assist the user in that matter, dependencies.h/.c will maintain
 * a small database containing informations about the client programs
 * Gnometoaster needs. Those informations will contain an URL where
 * a specific program can be found and other useful informations.
 * For a start, we'll pop up those informations whenever something goes
 * wrong. In the future, we might have a browseable dependency database
 * or a help shortcut bringing up the desired informations depending on the
 * screen context */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "config.h"
#include "int.h"
#include "dependencies.h"
#include "dialog.h"
#include "wheel.h"
#ifdef HAVE_GNOME
# include <gnome.h>
#else
# include <gtk/gtk.h>
#endif

char *dependencies_distrib_debian_urls[]=
{ "ftp://ftp.debian.org/dists/potato",
     "ftp://ftp.de.debian.org/dists/potato",
     "ftp://ftp.hu.debian.org/dists/potato",
     "ftp://ftp.fi.debian.org/dists/potato",
     "ftp://ftp.us.debian.org/dists/potato",
     NULL
};

dependencies_distrib dependencies_distrib_debian=
{
   N_("Debian/GNU Linux"),
   N_("The Debian Project"),
   N_("potato"),
   dependencies_distrib_debian_urls,
     N_("The ultimate, free Linux distribution. Not too easy for beginners, though.")
};

dependencies_distrib *dependencies_client_mpg123_supported_distribs[]=
{
   &dependencies_distrib_debian,
     NULL
};

char *dependencies_client_mpg123_urls[]=
{
   "http://www.mpg123.de",
     "http://ftp.tu-clausthal.de/pub/unix/audio/mpg123",
     NULL
};

dependencies_client dependencies_client_mpg123=
{
   "mpg123",
     N_("MPEG123 MPEG Audio Player"),
   N_("Michael Hipp"),
   N_("0.59r (1999/Jun/15)"),
   dependencies_client_mpg123_urls,
     dependencies_client_mpg123_supported_distribs,
     NULL,
     N_("MPG123 is used by Gnometoaster to decode MP3 files for both playing and "
	"recording. "
	"Essentially mpg123 is a commandline mpeg audio player capable of handling all sorts "
	"of mpeg audio streams (MPEG1+2, Layers 1 through 3).")
};

dependencies_distrib *dependencies_client_sox_supported_distribs[]=
{
   &dependencies_distrib_debian,
     NULL
};

char *dependencies_client_sox_urls[]=
{
   "http://www.spies.com/Sox/",
     "http://home.sprynet.com/~cbagwell/sox.html",
     NULL
};

dependencies_client dependencies_client_sox=
{
   "sox",
     N_("Sound Exchange"),
   N_("Lance Norskog"),
   N_("12.16"),
   dependencies_client_sox_urls,
     dependencies_client_sox_supported_distribs,
     NULL,
     N_("Gnometoaster is using sox to convert various types of .wav files into "
	"cd digital audio streams suitable for direct recording.\n"
	"SoX is a sound file format converter for Unix and DOS PCs written by Lance Norskog and other invaluable contributors. It also does sample rate conversion and some sound "
	"effects. It's the swiss army knife of sound tools: the interface isn't great, but it does almost everything. "
	"SoX uses file suffices to determine the nature of a sound sample file. If it finds the suffix in its list, it uses the appropriate read or write handler to deal with that file. SoX has an "
	"auto-detect feature that attempts to figure out the nature of an unmarked sound sample. It works very well. This is the 'auto' file format. "
	"(Source: http://www.spies.com/Sox/).")
};

dependencies_distrib *dependencies_client_mkisofs_supported_distribs[]=
{
   &dependencies_distrib_debian,
     NULL
};

char *dependencies_client_mkisofs_urls[]=
{
   "http://www.fokus.gmd.de/research/cc/glone/employees/joerg.schilling/private/cdrecord.html",
     NULL
};

dependencies_client dependencies_client_mkisofs=
{
   "mkisofs",
     N_("mkisofs"),
   N_("Eric Youngdale/Yggdrasil Computing Inc."),
   N_("1.13"),
   dependencies_client_mkisofs_urls,
     dependencies_client_mkisofs_supported_distribs,
     NULL,
     N_("Gnometoaster is using 'mkisofs' to create ISO9660 filesystems.\n"
	"That's what you usually do when creating a data CD (CDROM) with\n"
	"files on it.\n"
	"mkisofs is effectively a pre-mastering program to generate the iso9660 filesystem - it takes a snapshot of a\n"
	"given directory tree, and generates a binary image which will correspond to an iso9660 filesystem when written to a\n"
	"block device.(Source:mkisofs manpage (man 8 mkisofs))\n")
};

dependencies_distrib *dependencies_client_cdda2wav_supported_distribs[]=
{
   &dependencies_distrib_debian,
     NULL
};

char *dependencies_client_cdda2wav_urls[]=
{
   "http://www.fokus.gmd.de/research/cc/glone/employees/joerg.schilling/private/cdrecord.html",
     NULL
};

dependencies_client dependencies_client_cdda2wav=
{
   "cdda2wav",
     N_("cdda2wav"),
   N_("Heiko Eissfeldt (heiko@colossus.escape.de)"),
   N_("1.9"),
   dependencies_client_cdda2wav_urls,
     dependencies_client_cdda2wav_supported_distribs,
     NULL,
     N_("'cdda2wav' is used by Gnometoaster to extract Digital Audio from\n"
	"audio compact discs for either encoding or burning.\n"
	"cdda2wav can retrieve audio tracks from CDROM drives (see\n"
	"[cdda2wav-]README  for  a list of drives) that are capable of reading\n"
	"audio data digitally to the host (CDDA). (Source:man cdda2wav)\n")
};

char *dependencies_client_gogo_urls[]=
{
   "http://corpse.acm.ndsu.nodak.edu/archive/gogo/",
     NULL
};

dependencies_client dependencies_client_gogo=
{
   "gogo",
     N_("GoGo"),
   N_("PEN@MarineCat and shigeo"),
   N_("ver. 2.22 (Nov  8 1999)"),
   dependencies_client_gogo_urls,
     NULL,
     NULL,
     N_("Gnometoaster can use this program to encode audio tracks and .wav\n"
	"files into mp3 files.\n"
	"GoGo is an assembly optimized version of the free Lame encoder\n"
	"yielding about 6x encoding speed on my K6-3-400 machine.\n"
	"For many, Lame is considered the best free mp3 encoder available\n"
	"Note that in some countries it may be illegal to use this encoder\n"
	"because of software patent issues. See http://gnometoaster.rulez.org\n"
	"to find more informations about how you can protest against\n"
	"certain legal practices that may eventually make life a lot more\n"
	"difficult for us with intellectual property becoming more and more\n"
	"important an issue.\n")
};

dependencies_distrib *dependencies_client_cdrecord_supported_distribs[]=
{
   &dependencies_distrib_debian,
     NULL
};

char *dependencies_client_cdrecord_urls[]=
{
   "http://www.fokus.gmd.de/research/cc/glone/employees/joerg.schilling/private/cdrecord.html",
     NULL
};

dependencies_client dependencies_client_cdrecord=
{
   "cdrecord",
     N_("Cdrecord"),
   N_("Joerg Schilling (schilling@fokus.gmd.de)"),
   N_("1.9"),
   dependencies_client_cdrecord_urls,
     dependencies_client_cdrecord_supported_distribs,
     NULL,
     N_("Gnometoaster depends on cdrecord for cd recording and CDRW blanking.\n"
	"Note that if cdrecord isn't installed, Gnometoaster may still be able\n"
	"to record discs in disc-at-once mode using cdrdao.\n"
	"Cdrecord is a commmandline cd recording program capable of using\n"
	"most cd recorders in track-at-once and some of them in disc-at-once mode\n"
	"(a rule of thumb would be to check wether your drive supports cuesheet recording).\n"
	"Please go to the cdrecord homepage to get a list of drives currently\n"
	"supported by this program. If your drive isn't supported, you can try\n"
	"using it with cdrdao, yet another backend Gnometoaster is supporting\n")
};

dependencies_distrib *dependencies_client_cdrdao_supported_distribs[]=
{
   &dependencies_distrib_debian,
     NULL
};

char *dependencies_client_cdrdao_urls[]=
{
   "http://cdrdao.sourceforge.net",
     NULL
};

dependencies_client dependencies_client_cdrdao=
{
   "cdrdao",
     N_("CdrDAO"),
   N_("Andreas Mueller <mueller@daneb.ping.de>"),
   N_("1.1.4"),
   dependencies_client_cdrdao_urls,
     dependencies_client_cdrdao_supported_distribs,
     NULL,
     N_("Gnometoaster can use cdrdao to record CDRs in disc-at-once mode.\n"
	"Please make sure your drive is supported by cdrdao before trying\n"
	"to record a disc with this backend. As you don't have cdrdao\n"
	"installed, you can either turn off the \"Disc-at-once\" switch\n"
	"in Gnometoaster's recording panel or uncheck the \"Use alternative\n"
	"DAO client\" option in Gnometoaster's recording options.\n"
	"Gnometoaster will then use cdrecord to write your CDs\n")
};

char *dependencies_client_shorten_urls[]=
{
   "http://www.etree.org",
     NULL
};

dependencies_client dependencies_client_shorten=
{
   "shorten",
     N_("Shorten Lossless Audio Compressor"),
   N_("Tony Robinson and SoftSound Ltd"),
   N_("version 3.2"),
   dependencies_client_shorten_urls,
     NULL,
     NULL,
     N_("Gnometoaster can use shorten to burn .shn files to a CDR on-the-fly."
	"The program can also be used to encode audio into .shn files."
	"Shorten is a fileformat used for lossless audio compression and is therefore "
	"especially useful for musicians (e.g. for exchanging demo tapes)."
	"See http://www.etree.org for details.")
};

dependencies_client dependencies_client_shntool=
{
   "shntool",
     N_("Shorten Lossless Audio Compression Tool"),
   N_("Jason Jordan"),
   N_("version 1.01"),
   dependencies_client_shorten_urls,
     NULL,
     NULL,
     N_("GnomeToaster needs this program if you want to record shorten files "
	"to a CDR. It is used to calculate the exact space required by your "
	"shorten file on the medium."
	"For informations about the Shorten format, visit http://www.etree.org.")
};

char *dependencies_client_ogg123_urls[]=
{
   "http://www.xiph.org/ogg/vorbis/index.html",
     NULL
};

dependencies_client dependencies_client_ogg123=
{
   "ogg123",
     N_("Ogg-Vorbis commandline decoder "),
   N_("Kenneth Arnold"),
   N_("version 1.0beta4"),
   dependencies_client_ogg123_urls,
     NULL,
     NULL,
     N_("GnomeToaster needs this program to decode ogg-vorbis files (.ogg)."
	"Ogg-Vorbis is a totally free implementation of a lossy audio-compression "
	"scheme similar to Fraunhofer's mp3. It yields slightly better quality "
	"at the same bitrate and is free of any patented alghorithms.")
};

char *dependencies_client_oggtst_urls[]=
{
   "http://gnometoaster.rulez.org/archive/oggtst.tgz",
     NULL
};

dependencies_client dependencies_client_oggtst=
{
   "oggtst",
     N_("Ogg Vorbis File Information viewer"),
   N_("James Atwill"),
   N_("Jan 2001"),
   dependencies_client_oggtst_urls,
     NULL,
     NULL,
     N_("GnomeToaster needs this program to determine the playing time of "
	"an ogg-vorbis file. Because it is not currently part of the "
	"main vorbis-tools distribution, a packaged version of the source "
	"can be downloaded from http://gnometoaster.rulez.org")
};

// a list of clients, NULL-terminated
dependencies_client *clients[]=
{
   &dependencies_client_mpg123,
     &dependencies_client_sox,
     &dependencies_client_mkisofs,
     &dependencies_client_cdda2wav,
     &dependencies_client_gogo,
     &dependencies_client_cdrecord,
     &dependencies_client_cdrdao,
     &dependencies_client_shorten,
     &dependencies_client_shntool,
     &dependencies_client_ogg123,
     &dependencies_client_oggtst,
     NULL
};

/* retrieve informations about the given command.
 * returns a dependencies_client struct */
dependencies_client *dependencies_getinfo(const char *command)
{
   int i;
   dependencies_client *current=NULL;

   char *cmd=strdup(command);
   if (strchr(cmd,' '))
     *strchr(cmd,' ')=0;

   for (i=0;clients[i]!=NULL;i++)
     if (!strcmp(cmd,clients[i]->executable))
       current=clients[i];

   free(cmd);
   return current;
};

#define dependencies_showinfo_text(text) \
   if (text) \
       { \
	  line=gtk_label_new(text); \
	  gtk_widget_show(line); \
	  gtk_box_pack_start(GTK_BOX(content),line,0,0,0); \
       }

void dependencies_showinfo(const char *message,
			   dependencies_client *info)
{
   GtkWidget *content=gtk_vbox_new(0,0);
   GtkWidget *line;
   int x;

   dependencies_showinfo_text(message);
   /* do we have a valid dependency entry ? */
   if (info)
     {
	dependencies_showinfo_text(info->product_name);
	dependencies_showinfo_text(info->vendor);
	dependencies_showinfo_text(info->version_number);

	line=gtk_hbox_new(0,0);
	if (info->urls)
	  for (x=0;info->urls[x]!=NULL;x++)
	    {
	       GtkWidget *entry;
#ifdef HAVE_GNOME
	       entry=gnome_href_new(info->urls[x],info->urls[x]);
#else
	       entry=gtk_label_new(info->urls[x]);
#endif
	       gtk_widget_show(entry);
	       gtk_box_pack_start(GTK_BOX(line),entry,0,0,0);
	       if (info->urls[x+1]!=NULL)
		 {
		    entry=gtk_label_new(",");
		    gtk_widget_show(entry);
		    gtk_box_pack_start(GTK_BOX(line),entry,0,0,0);
		 };
	    };
	gtk_widget_show(line);
	gtk_box_pack_start(GTK_BOX(content),line,0,0,0);

	line=gtk_hbox_new(0,0);
	if (info->supported_distribs)
	  for (x=0;info->supported_distribs[x]!=NULL;x++)
	    {
	       GtkWidget *entry=gtk_button_new_with_label(info->supported_distribs[x]->product_name);
	       gtk_widget_show(entry);
	       gtk_box_pack_start(GTK_BOX(line),entry,0,0,0);
	    };
	gtk_widget_show(line);
	gtk_box_pack_start(GTK_BOX(content),line,0,0,0);
	if (info->description)
	  {
	     GtkWidget *scrolling;
	     GtkWidget *descr=gtk_text_new(NULL,NULL);
	     line=gtk_hbox_new(0,0);
	     gtk_text_set_editable(GTK_TEXT(descr),0);
	     gtk_text_set_word_wrap(GTK_TEXT(descr),1);
	     scrolling=gtk_vscrollbar_new(GTK_TEXT(descr)->vadj);
	     gtk_box_pack_start(GTK_BOX(line),descr,1,1,1);
	     gtk_widget_show(descr);
	     gtk_box_pack_start(GTK_BOX(line),scrolling,0,0,0);
	     gtk_widget_show(scrolling);
	     wheel_connect(descr,GTK_TEXT(descr)->vadj,4);
	     gtk_box_pack_start(GTK_BOX(content),line,0,0,0);
	     gtk_widget_show(line);
	     gtk_text_insert(GTK_TEXT(descr),NULL,NULL,NULL,_(info->description),-1);
	  };
     };

   dialog_showuserwidget(_("Dependency database"),
			 content,
			 DIALOG_USERWIDGET_OK,
			 NULL,NULL);
};

void dependencies_showmissingexec(const char *command)
{
   char *message=(char*)malloc(1024);
   char *cmd=strdup(command);
   if (strlen(cmd)>35)
     {
	cmd[35]=0;
	strcat(cmd,"[...]");
     };
   sprintf(message,_("Gnometoaster failed to run the following command\n"
		     "on your system:\n"
		     "%s\n"),
	   cmd);
   free(cmd);

   dependencies_showdep(command,message);
   free (message);
}
