/*
 * rfb_protocol.cpp - class rfbConnection, an implementation of the RFB-protocol, used by VNC servers
 *
 * iTALC
 * Copyright (c) 2004-2005 Tobias Doerffel <tobias@doerffel.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cstring>
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#include <cerrno>

#include <qfileinfo.h>
#include <qmessagebox.h>
#include <qdir.h>
#if QT_VERSION >= 0x030200
#include <qsplashscreen.h>
#endif

#include "rfb_connection.h"
#include "system_environment.h"
#include "qimage_manips.h"
#include "rsa_crypt_lib.h"
#include "italc.h"
#include "paths.h"

#include "rfb_connection.moc"


int endian_test = 1;

const char * RFBP_VERSION_FORMAT = "RFB %03d.%03d\n";
//const int MAX_READ_RETRIES = 1024;


const s_rfb_pixel_format rfbConnection::s_localDisplayFormat =
{
	32, 		// bit per pixel
	32,		// depth
	FALSE,		// big endian
	1,		// true color
	255,		// red max
	255,		// green max
	255,		// blue max
	16,		// red shift
	8,		// green shift
	0,		// blue shift,
	0, 0		// only for padding
} ;



#ifdef RSA_CRYPT_AVAILABLE
RSA * rfbConnection::s_privateKey = NULL;
#endif


void rfbConnection::initializeAuthentication( void )
{
#ifdef RSA_CRYPT_AVAILABLE
	if( italc::ensureConfigPathExists() == FALSE )
	{
		qFatal( "Could not read/write or create directory " + QString( ITALC_CONFIG_PATH ) + "! For running iTALC, make sure you have "
			"write-access to your home-directory and to " + QString( ITALC_CONFIG_PATH ) + " (if already existing)." );
	}
	const QString PRIV_KEY_FILE = QDir::home().absPath() + "/"+ITALC_CONFIG_PATH+"/id_rsa";
	const QString PUB_KEY_FILE = PRIV_KEY_FILE + ".public";

	s_privateKey = readPrivateKey( PRIV_KEY_FILE );
	if( s_privateKey == NULL )
	{
		// generate 1024bit RSA-key and save it
		s_privateKey = rsaGeneratePrivateKey( 1024 );
		writePrivateKey( s_privateKey, PRIV_KEY_FILE );

		// derive public key from private key and save it
		RSA * pub_key = publicKeyFromPrivateKey( s_privateKey );
		writePublicKey( pub_key, PUB_KEY_FILE );
		RSA_free( pub_key );
#if QT_VERSION >= 0x030200
		if( splashScreen != NULL )
		{
			splashScreen->hide();
		}
#endif
		QMessageBox::information( NULL, tr( "New key-pair generated" ), tr( "No authentication-keys were found or your old ones "
											"contained errors, so a new key-pair has been "
											"generated.\nPlease ask your administrator for "
											"distributing your public key. Otherwise you won't "
											"be able to access clients." ), QMessageBox::Ok );
	}
#endif
}




// normal ctor
rfbConnection::rfbConnection( const QString & _host_ip ) :
	QObject(),
	m_connected( FALSE ),
	m_hostIP( _host_ip ),
	m_user( "" ),
	m_gotInvitationResult( FALSE ),
	m_invitationOk( FALSE ),
	m_sock( -1 ),
	m_clientScreen(),
	m_scaledClientScreen(),
	m_destImgSize( 640, 480 ),
	m_bufOutPtr( m_buf ),
	m_buffered( 0 )
{
#ifdef HAVE_LIBZ
	m_zlibStreamActive[0] = m_zlibStreamActive[1] = m_zlibStreamActive[2] = m_zlibStreamActive[3] = FALSE;
#endif
}



/*
// copy ctor for using data frome existing connection
rfbConnection::rfbConnection( const rfbConnection & _rc ) :
	QObject(),
	m_connected( FALSE ),
	m_hostIP( _rc.m_hostIP ),
	m_user( "" ),
	m_gotInvitationResult( FALSE ),
	m_invitationOk( FALSE ),
	m_sock( -1 ),
	m_clientScreen(),
	m_scaledClientScreen(),
	m_destImgSize( 640, 480 ),
	m_bufOutPtr( m_buf ),
	m_buffered( 0 )
{
	m_zlibStreamActive[0] = m_zlibStreamActive[1] = m_zlibStreamActive[2] = m_zlibStreamActive[3] = FALSE;
}
*/



rfbConnection::~rfbConnection()
{
	closeConnection();
}




bool rfbConnection::openConnection( void )
{
	closeConnection ();

        if( systemEnvironment::hostAvailable( m_hostIP ) == FALSE )
	{
		return( FALSE );
	}

	in_addr_t host;
	if( !stringToIPAddr( m_hostIP.ascii(), &host ) )
	{
		printf( "Couldn't convert '%s' to host address\n", m_hostIP.ascii() );
		return( FALSE );
	}

	m_sock = connectToTCPAddr( host, VNC_SERVER_PORT_OFFSET );

	if( m_sock < 0 )
	{
		//printf("Unable to connect to VNC server on client %s\n", m_hostIP.ascii());
		return( FALSE );
	}

	if( fcntl( m_sock, F_SETFL, O_NONBLOCK ) < 0 )
	{
		printf( "Error while setting non-blocking\n" );
		return( FALSE );
	}


	char protocol_version[13];
	int major, minor;
	Q_UINT32 auth_scheme, reason_len;
	char * reason;
	s_rfb_client_init_msg ci;
	
	if( !readFromServer( protocol_version, RFBP_VERSION_MESSAGE_LEN ) )
	{
		return( FALSE );
	}
	
	protocol_version[RFBP_VERSION_MESSAGE_LEN] = 0;

	if( sscanf( protocol_version, RFBP_VERSION_FORMAT, &major, &minor ) != 2 )
	{
		printf( "Not a valid VNC server\n" );
		return( FALSE);
	}

	//printf ("VNC server supports protocol version %d.%d (viewer %d.%d)\n", major, minor, RFBP_MAJOR_VERSION, RFBP_MINOR_VERSION);

	//major = RFBP_MAJOR_VERSION;
	//minor = RFBP_MINOR_VERSION;

	//sprintf (protocol_version, protocol_version, major, minor);

	if( !writeToServer( protocol_version, RFBP_VERSION_MESSAGE_LEN ) )
	{
		return( FALSE );
	}


	if( !readFromServer( (char *)&auth_scheme, sizeof( Q_UINT32 ) ) )
	{
		return( FALSE );
	}

	auth_scheme = Swap32IfLE( auth_scheme );

	switch( auth_scheme )
	{
		case RFB_CONNECTION_FAILED:
			if( !readFromServer( (char *)&reason_len, sizeof( Q_UINT32 ) ) )
			{
				return( FALSE );
			}
			reason_len = Swap32IfLE( reason_len );

			reason = new char[reason_len];

			if( readFromServer( reason, reason_len ) )
			{
				printf( "VNC connection failed: %.*s\n", (int) reason_len, reason );
			}
			delete reason;
			return( FALSE );

		case RFB_NO_AUTHENTICATION:
			//printf ("No authentication needed\n");
			break;
#ifdef RSA_CRYPT_AVAILABLE
		case RFB_VNC_CHALL_RESP_AUTHENTICATION:
		{
			// send user-name to server (server has to encrypt data with the appropriate key)
			s_rfb_authentication ra;
			QString user_name = getenv( "USER" );

			ra.user_name_len = Swap16IfLE( user_name.length()+1 );
			if( !writeToServer( (const char*) &ra, sizeof( ra ) ) )
				return( FALSE );
			if( !writeToServer( user_name.ascii(), user_name.length()+1 ) )
				return( FALSE );

			// challenge-response-authentication
			int ecs = RSA_size( s_privateKey );
			char * encrypted_challenge = new char[ecs];
			if( !readFromServer( encrypted_challenge, ecs ) )
			{
				return( FALSE );
			}

			char * decrypted_response = new char[DEFAULT_CHALLENGE_SIZE];
			
			int bytes = RSA_private_decrypt( ecs, (unsigned char *)encrypted_challenge, (unsigned char *)decrypted_response, s_privateKey, RSA_PKCS1_PADDING );
			// TODO: add error handling if bytes != DEFAULT_CHALLENGE_SIZE

			if( !writeToServer( decrypted_response, DEFAULT_CHALLENGE_SIZE ) )
			{
				return( FALSE );
			}
			delete[] encrypted_challenge;
			delete[] decrypted_response;
			Q_UINT32 auth_result;
			if( !readFromServer( (char *)& auth_result, 4 ) )
			{
				return( FALSE );
			}
			auth_result = Swap32IfLE( auth_result );
			switch( auth_result )
			{
				case RFB_AUTH_OK:
					//printf( "VNC authentication succeeded\n" );
					break;
				case RFB_AUTH_FAILED:
					printf( "VNC authentication failed\n" );
					return( FALSE );
				case RFB_AUTH_TOO_MANY:
					printf( "VNC authentication failed - too many tries\n" );
					return( FALSE );
				default:
					printf( "Unknown VNC authentication result: %d\n", (int) auth_result );
					return( FALSE );
			}
			break;
		}
#endif
		default:
			printf( "Unknown authentication scheme from VNC server: %d\n",(int) auth_scheme );
			return( FALSE);
	}

	ci.shared = TRUE;

	if( !writeToServer( (char *) &ci, sizeof( ci ) ) )
	{
		return( FALSE );
	}

	if( !readFromServer( (char *)&m_si, sizeof( m_si ) ) )
	{
		return( FALSE );
	}

	m_si.framebuffer_width = Swap16IfLE( m_si.framebuffer_width );
	m_si.framebuffer_height = Swap16IfLE( m_si.framebuffer_height );
	m_si.format.red_max = Swap16IfLE( m_si.format.red_max );
	m_si.format.green_max = Swap16IfLE( m_si.format.green_max );
	m_si.format.blue_max = Swap16IfLE( m_si.format.blue_max );
	m_si.name_length = Swap32IfLE( m_si.name_length );

	char desktop_name[m_si.name_length+1];

	if( !readFromServer( desktop_name, m_si.name_length ) )
	{
		return( FALSE );
	}

	//desktop_name[m_si.name_length] = 0;

	//printf ("Desktop name: \"%s\"\n",desktop_name);
	//printf ("Connected to VNC server, using protocol version %d.%d\n", RFBP_MAJOR_VERSION, RFBP_MINOR_VERSION);

	//printf(stderr,"VNC server default format:\n");
	//PrintPixelFormat(&m_si.format);

	s_rfb_set_pixel_format_msg spf;

	spf.type = RFB_SET_PIXEL_FORMAT;
	spf.format = s_localDisplayFormat;
	spf.format.red_max = Swap16IfLE(spf.format.red_max);
	spf.format.green_max = Swap16IfLE(spf.format.green_max);
	spf.format.blue_max = Swap16IfLE(spf.format.blue_max);

	if( !writeToServer( (char *) &spf, sizeof( s_rfb_set_pixel_format_msg ) ) )
	{
		return( FALSE );
	}


	char m_buf[sizeof(s_rfb_set_pixel_format_msg) + MAX_ENCODINGS * sizeof(Q_UINT32)];
	s_rfb_set_encodings_msg * se = (s_rfb_set_encodings_msg *) m_buf;
	se->type = RFB_SET_ENCODINGS;
	se->n_encodings = 0;

	Q_UINT32 * encs = (Q_UINT32 *)(&m_buf[sizeof(s_rfb_set_encodings_msg)]);

	encs[se->n_encodings++] = Swap32IfLE( RFB_ENCODING_TIGHT );
	encs[se->n_encodings++] = Swap32IfLE( RFB_ENCODING_CORRE );
	encs[se->n_encodings++] = Swap32IfLE( RFB_ENCODIG_RRE );
	encs[se->n_encodings++] = Swap32IfLE( RFB_ENCODING_COPYRECT );

#ifdef HAVE_LIBZ
	encs[se->n_encodings++] = Swap32IfLE( RFB_ENCODING_COMPRESS_LEVEL_0+4 );
#ifdef HAVE_LIBJPEG
	encs[se->n_encodings++] = Swap32IfLE( RFB_ENCODING_QUALITY_LEVEL_0+4 );
#endif
#endif
	//encs[se->n_encodings++] = Swap32IfLE(RFB_ENCODING_LAST_RECT);


	unsigned int len = sizeof( s_rfb_set_encodings_msg ) + se->n_encodings * sizeof( Q_UINT32 );

	se->n_encodings = Swap16IfLE( se->n_encodings );

	if( !writeToServer( m_buf, len ) )
	{
		return( FALSE );
	}

	//printf ("Pixel-format and encodings sucessfully setup...\n");

	m_clientScreen = QImage( m_si.framebuffer_width, m_si.framebuffer_height, s_localDisplayFormat.depth );

	sendFramebufferUpdateRequest();
	m_connected = TRUE;


	return( TRUE );

}




void rfbConnection::closeConnection( void )
{
	m_bufOutPtr = m_buf;
	m_buffered = 0;
#ifdef HAVE_LIBZ
	m_zlibStreamActive[0] = m_zlibStreamActive[1] = m_zlibStreamActive[2] = m_zlibStreamActive[3] = FALSE;
#endif
	if( m_sock != -1 )
	{
		close( m_sock );
	}
	m_sock = -1;

	m_connected = FALSE;

	m_user = "";

}




bool rfbConnection::resetConnection( const QString & _new_host_ip )
{
	closeConnection ();

	if( _new_host_ip != "" )
		m_hostIP = _new_host_ip;

	return( openConnection() );
}




bool rfbConnection::dataFromServer( void )
{
	fd_set fds;
	struct timeval tv;
/*	tv.tv_sec = 0;
	tv.tv_usec = 10000;     // timeout = 10ms*/
	tv.tv_sec = 1;		// timeout = 1 s
	tv.tv_usec = 0;
	FD_ZERO( &fds );
	FD_SET( m_sock, &fds );
	if( select( m_sock+1, &fds, NULL, NULL, &tv ) <= 0 )
	{
		return( FALSE );
	}

	return( TRUE );
}




bool rfbConnection::readFromServer( char * _out, unsigned int _n )
{
	if( _n <= m_buffered )
	{
		memcpy( _out, m_bufOutPtr, _n );
		m_bufOutPtr += _n;
		m_buffered -= _n;
		return( TRUE );
	}

	if( m_buffered )
	{
		memcpy( _out, m_bufOutPtr, m_buffered );
		_out += m_buffered;
		_n -= m_buffered;
	}

	m_bufOutPtr = m_buf;
	m_buffered = 0;

	int retries = 0;

	if( _n <= INTERNAL_READ_BUF_SIZE )
	{
		while( m_buffered < _n )
		{
			int i = read( m_sock, m_buf+m_buffered, INTERNAL_READ_BUF_SIZE-m_buffered );
			if( i <= 0 )
			{
				if( i < 0 )
				{
					if( ( errno == EWOULDBLOCK || errno == EAGAIN )/* && ++retries < MAX_READ_RETRIES*/ )
					{
						//QApplication::eventLoop()->processEvents (QEventLoop::AllEvents, MAX_EVENT_PROC_TIME);
						i = 0;
					}
					else
					{
						printf( "readFromServer: read(...) failed\n" );
						closeConnection();
						return( FALSE );
					}
				}
				else
				{
					printf( "VNC server closed connection\n" );
					closeConnection();
					return( FALSE );
				}
			}
			m_buffered += i;
		}

		memcpy( _out, m_bufOutPtr, _n );
		m_bufOutPtr += _n;
		m_buffered -= _n;
	}
	else
	{
		while( _n > 0 )
		{
			int i = read( m_sock, _out, _n );
			if( i <= 0 )
			{
				if( i < 0 )
				{
					if( ( errno == EWOULDBLOCK || errno == EAGAIN )/* && ++retries < MAX_READ_RETRIES*/ )
					{
						//QApplication::eventLoop()->processEvents (QEventLoop::AllEvents, MAX_EVENT_PROC_TIME);
						i = 0;
					}
					else
					{
						printf( "readFromServer: read(...) failed\n" );
						closeConnection();
						return( FALSE );
					}
				}
				else
				{
					printf( "VNC server closed connection\n" );
					closeConnection();
					return( FALSE );
				}
			}
			_out += i;
			_n -= i;
		}
	}

	return( TRUE );
}




bool rfbConnection::writeToServer( const char * _buf, unsigned int _n )
{
	unsigned int i = 0;
	while( i < _n )
	{
		int j = write( m_sock, _buf + i, _n - i );
		if( j <= 0 )
		{
			printf( "error while writing!\n" );
			if( j < 0 )
			{
				if( errno == EWOULDBLOCK || errno == EAGAIN )
				{
					fd_set fds;
					FD_ZERO( &fds );
					FD_SET( m_sock, &fds );
					if( select( m_sock+1, NULL, &fds, NULL, NULL ) <= 0 )
					{
						printf( "writeToServer: select(..) failed\n" );
						closeConnection();
						return( FALSE );
					}
					j = 0;
				}
				else
				{
					printf( "writeToServer: write(..) failed (j < 0)\n" );
					closeConnection();
					return( FALSE );
				}
			}
			else
			{
				printf( "writeToServer: write(..) failed (j == 0)\n" );
				closeConnection();
				return( FALSE );
			}
		}
		i += j;
	}

	return( TRUE );
}




long rfbConnection::readCompactLen( void )
{
	Q_UINT8 b;

	if( !readFromServer( (char *)&b, sizeof( b ) ) )
	{
		return( -1 );
	}

	long len = (int)b & 0x7F;
	if( b & 0x80 )
	{
		if( !readFromServer( (char *)&b, sizeof( b ) ) )
		{
			return( -1 );
		}
		len |= ((int)b & 0x7F) << 7;
		if( b & 0x80 )
		{
			if( !readFromServer( (char *)&b, 1 ) )
			{
				return( -1 );
			}
			len |= ((int)b & 0xFF) << 14;
		}
	}

	return( len );
}




bool rfbConnection::sendFramebufferUpdateRequest( void )
{
	return( sendFramebufferUpdateRequest( 0, 0, m_si.framebuffer_width, m_si.framebuffer_height, FALSE ) );
}




bool rfbConnection::sendIncrementalFramebufferUpdateRequest( void )
{
	return( sendFramebufferUpdateRequest( 0, 0, m_si.framebuffer_width, m_si.framebuffer_height, TRUE ) );
}




bool rfbConnection::sendFramebufferUpdateRequest( Q_UINT16 _x, Q_UINT16 _y, Q_UINT16 _w, Q_UINT16 _h, bool _incremental )
{
	s_rfb_framebuffer_update_request_msg fur;

	fur.type = RFB_FRAMEBUFFER_UPDATE_REQUEST;
	fur.incremental = (_incremental == TRUE)? 1 : 0;
	fur.x = Swap16IfLE( _x );
	fur.y = Swap16IfLE( _y );
	fur.w = Swap16IfLE( _w );
	fur.h = Swap16IfLE( _h );

	return( writeToServer( (char *) &fur, sizeof( fur ) ) );
}




bool rfbConnection::sendPointerEvent( Q_UINT16 _x, Q_UINT16 _y, Q_UINT16 _button_mask )
{
	s_rfb_pointer_event_msg pe;

	pe.type = RFB_POINTER_EVENT;
	pe.button_mask = _button_mask;
	//if (_x < 0) _x = 0;
	//if (_y < 0) _y = 0;
	pe.x = Swap16IfLE( _x );
	pe.y = Swap16IfLE( _y );

	return( writeToServer( (char *) &pe, sizeof( pe ) ) );
}




bool rfbConnection::sendKeyEvent( Q_UINT32 key, bool down )
{
	s_rfb_key_event_msg ke;

	ke.type = RFB_KEY_EVENT;
	ke.down = (down == TRUE)? 1 : 0;
	ke.key = Swap32IfLE( key );

	return( writeToServer( (char *) &ke, sizeof( ke ) ) );
}




bool rfbConnection::sendGetUserRequest( void )
{
	if( !connected() )
	{
		return( FALSE );
	}

	s_rfb_get_user_msg gu;
	gu.cmd = RFB_ITALC_TO_SERVER_REQUEST;
	gu.italc_cmd = ITALC_GET_USER;

	return( writeToServer( (char *) &gu, sizeof( gu ) ) );
}




bool rfbConnection::execCmds( const QString & _cmd )
{
	if( !connected() )
	{
		return( FALSE );
	}

	s_rfb_exec_cmds_msg ec;
	ec.cmd = RFB_ITALC_TO_SERVER_REQUEST;
	ec.italc_cmd = ITALC_EXEC_CMDS;
	ec.cmd_len = Swap16IfLE( _cmd.length()+1 );

	if( !writeToServer( (char *) &ec, sizeof( ec ) ) )
	{
		printf( "Could not send ITALC_EXEC_CMDS-request to client\n" );
		return( FALSE);
	}

	return( writeToServer( _cmd.ascii(), _cmd.length()+1 ) );
}




bool rfbConnection::startDemo( const QString & _master_ip, bool _full_screen )
{
	if( !connected() )
	{
		return( FALSE );
	}

	s_rfb_start_demo_msg sd;
	sd.cmd = RFB_ITALC_TO_SERVER_REQUEST;
	sd.italc_cmd = ITALC_START_DEMO;
	sd.fullscreen = (_full_screen)? !0 : 0;
	sd.demo_master_ip_len = Swap16IfLE( _master_ip.length()+1 );
	if( !writeToServer( (char *) &sd, sizeof( sd ) ) )
	{
		printf( "Could not send ITALC_START_DEMO-request to client\n" );
		return( FALSE );
	}

	return( writeToServer( _master_ip.ascii(), _master_ip.length()+1 ) );
}




bool rfbConnection::stopDemo( void )
{
	if( !connected() )
	{
		return( FALSE );
	}

	s_rfb_stop_demo_msg sd;
	sd.cmd = RFB_ITALC_TO_SERVER_REQUEST;
	sd.italc_cmd = ITALC_STOP_DEMO;

	return( writeToServer( (char *) &sd, sizeof( sd ) ) );
}




bool rfbConnection::lockDisplay( void )
{
	if( !connected() )
	{
		return( FALSE );
	}

	s_rfb_lock_display_msg ld;
	ld.cmd = RFB_ITALC_TO_SERVER_REQUEST;
	ld.italc_cmd = ITALC_LOCK_DISPLAY;

	return( writeToServer( (char *) &ld, sizeof( ld ) ) );
}




bool rfbConnection::unlockDisplay( void )
{
	if( !connected() )
		return( FALSE );

	s_rfb_unlock_display_msg ud;
	ud.cmd = RFB_ITALC_TO_SERVER_REQUEST;
	ud.italc_cmd = ITALC_UNLOCK_DISPLAY;

	return( writeToServer( (char *) &ud, sizeof( ud ) ) );
}




bool rfbConnection::sendMessage( const QString & _msg )
{
	if( !connected() )
	{
		return( FALSE );
	}

	s_rfb_send_msg_msg sm;
	sm.cmd = RFB_ITALC_TO_SERVER_REQUEST;
	sm.italc_cmd = ITALC_SEND_MSG;
	sm.msg_len = Swap16IfLE( _msg.length()+1 );

	if( !writeToServer( (char *) &sm, sizeof( sm ) ) )
	{
		printf( "Could not send ITALC_SEND_MSG-request to client\n" );
		return( FALSE );
	}

	return( writeToServer( _msg.ascii(), _msg.length()+1 ) );
}




bool rfbConnection::postFile( const QString & _fname )
{
/*
	if (!connected() || QFileInfo(_fname).exists() == FALSE || QFileInfo(_fname).isReadable() == FALSE)
		return (FALSE);

	Q_UINT32 fsize = QFileInfo(_fname).size();
	if (fsize == 0)
		return (FALSE);

	QString realFileName = QFileInfo(_fname).fileName();

	s_rfb_post_file_msg pf;
	pf.cmd = RFB_ITALC_TO_SERVER_REQUEST;
	pf.italc_cmd = ITALC_POST_FILE_REQ;
	pf.fname_len = Swap16IfLE(realFileName.length()+1);
	pf.fsize = Swap32IfLE(fsize);

	if (!writeToServer((char *) &pf, sizeof(pf))) {
		printf ("Could not send ITALC_POST_FILE_REQ-request to client\n");
		return (FALSE);
	}

	if (!writeToServer(realFileName.ascii(), realFileName.length()+1)) {
		printf ("Could not send file-name to client\n");
		return (FALSE);
	}

	QFile f (_fname);
	f.open (IO_ReadOnly);
	Q_UINT32 processed = 0;

	while (processed < fsize) {
		const int BUF_SIZE = 1024;
		char m_buffer[BUF_SIZE];
		Q_LONG read = f.readBlock (m_buffer, BUF_SIZE);
		if (read > 0) {
			if (!writeToServer(m_buffer, read)) {
				printf ("Could not send file-content to client (%d of %d bytes already sent)\n", (int) processed, (int) fsize);
				return (FALSE);
			}
			processed += read;
		} else if (read < 0) {
			printf ("An error occured while reading file (%d of %d bytes already sent)\n", (int) processed, (int) fsize);
			return (FALSE);
		}
		usleep(25);	// avoid m_buffer-overflow in client's TCP/IP-m_buffer...
	}

	return (TRUE);
*/
}




bool rfbConnection::getFile( const QString & _nfilter )
{
/*	if (!connected())
		return (FALSE);

	s_rfb_get_file_msg gf;
	gf.cmd = RFB_ITALC_TO_SERVER_REQUEST;
	gf.italc_cmd = ITALC_GET_FILE;
	gf.fmask_len = Swap16IfLE(_nfilter.length()+1);

	if (!writeToServer((char *) &gf, sizeof(gf))) {
		printf ("Could not send ITALC_GET_FILE-request to client\n");
		return (FALSE);
	}

	return (writeToServer(_nfilter.ascii(), _nfilter.length()+1));
*/
}




bool rfbConnection::inviteForSupport( const QString & _user_name )
{
	if( !connected() )
	{
		return( FALSE );
	}

	s_rfb_invite_support_msg is;
	is.cmd = RFB_ITALC_TO_SERVER_REQUEST;
	is.italc_cmd = ITALC_INVITE_SUPPORT;
	is.uname_len = Swap16IfLE( _user_name.length()+1 );

	if( !writeToServer( (char *) &is, sizeof( is ) ) )
	{
		printf( "Could not send ITALC_INVITE_SUPPORT-request to client\n" );
		return( FALSE );
	}

	if( !writeToServer( _user_name.ascii(), _user_name.length()+1 ) )
	{
		printf( "Could not write user-name to client\n" );
		return( FALSE );
	}

	m_gotInvitationResult = FALSE;
	m_invitationOk = FALSE;

	while( m_gotInvitationResult == FALSE && handleServerMessages( FALSE ) == TRUE )
	{
	}

	// return result, if error occured while handling server-messages, m_invitationOk is untouched -> FALSE
	return( m_invitationOk );
}




bool rfbConnection::handleServerMessages( bool _send_screen_update )
{
	u_rfb_server_to_client_msg msg;

	while( dataFromServer() == TRUE )
	{

	if( !readFromServer( (char *) &msg, sizeof( Q_UINT8 ) ) )
	{
		printf( "Reading of message-type failed\n" );
		return( FALSE );
	}

	switch( msg.type )
	{
		case RFB_SET_COLORMAP_ENTRIES:
			printf( "Got signal for setting colormap entries. Ignoring.\n" );
			break;

		case RFB_FRAMEBUFFER_UPDATE:
		{
			if( !readFromServer( ( (char *)&msg.fu ) + 1, sizeof( s_rfb_framebuffer_update_msg )- 1 ) )
			{
				return( FALSE );
			}

			msg.fu.n_rects = Swap16IfLE( msg.fu.n_rects );

			s_rfb_framebuffer_update_rect_header rect;
			for( Q_UINT16 i = 0; i < msg.fu.n_rects; i++ )
			{
				if( !readFromServer( (char *)&rect, sizeof( s_rfb_framebuffer_update_rect_header ) ) )
				{
					return( FALSE );
				}

				rect.r.x = Swap16IfLE( rect.r.x );
				rect.r.y = Swap16IfLE( rect.r.y );
				rect.r.w = Swap16IfLE( rect.r.w );
				rect.r.h = Swap16IfLE( rect.r.h );

				rect.encoding = Swap32IfLE( rect.encoding );
				if( rect.encoding == RFB_ENCODING_LAST_RECT )
				{
					break;
				}

				if( ( rect.r.x + rect.r.w > m_si.framebuffer_width ) ||
				    ( rect.r.y + rect.r.h > m_si.framebuffer_height ) )
				{
					printf( "Rect too large: %dx%d at (%d, %d)\n", rect.r.w, rect.r.h, rect.r.x, rect.r.y );
					return( FALSE );
				}

				if( ( rect.r.h * rect.r.w ) == 0 )
				{
					printf( "Zero size rect - ignoring\n" );
					continue;
				}

				switch( rect.encoding )
				{
					case RFB_ENCODIG_RAW:
					{
						Q_UINT16 bytes_per_line = rect.r.w * (s_localDisplayFormat.bits_per_pixel/8);
						Q_UINT16 lines_to_read = (Q_UINT16) (BUFFER_SIZE / bytes_per_line);
						while( rect.r.h > 0 )
						{
							if( lines_to_read > rect.r.h )
							{
								lines_to_read = rect.r.h;
							}
							if( !readFromServer( m_buffer, bytes_per_line * lines_to_read ) )
							{
								return (FALSE);
							}
							QRgb * src = (QRgb *) m_buffer;
							Q_UINT16 x2 = rect.r.x+rect.r.w;
							for( Q_UINT16 y = 0; y < lines_to_read; y++ )
							{
								QRgb * dest = (QRgb *)m_clientScreen.scanLine( rect.r.y+y );
								for( Q_UINT16 x = rect.r.x; x < x2; ++x )
								{
									dest[x] = *src;
									++src;
								}
							}
							rect.r.h -= lines_to_read;
							rect.r.y += lines_to_read;
						}
						break;
					}
					case RFB_ENCODING_COPYRECT:
					{
						s_rfb_copy_rect cr;
						if( !readFromServer( (char *) &cr, sizeof( s_rfb_copy_rect ) ) )
						{
							return( FALSE );
						}
						cr.src_x = Swap16IfLE( cr.src_x );
						cr.src_y = Swap16IfLE( cr.src_y );

						copyExistingRect( m_clientScreen, cr.src_x, cr.src_y, rect.r.w, rect.r.h, rect.r.x, rect.r.y );
						printf( "copy rect - should not occur!!\n" );
						break;
					}
					case RFB_ENCODIG_RRE:
					{
						if( !handleRRE( rect.r.x, rect.r.y, rect.r.w, rect.r.h ) )
						{
							return( FALSE );
						}
						break;
					}
					case RFB_ENCODING_CORRE:
					{
						if( !handleCoRRE( rect.r.x, rect.r.y, rect.r.w, rect.r.h ) )
						{
							return( FALSE );
						}
						break;
					}
#ifdef HAVE_LIBZ
#ifdef HAVE_LIBJPEG
					case RFB_ENCODING_TIGHT:
					{
						if( !handleTight( rect.r.x, rect.r.y, rect.r.w, rect.r.h ) )
						{
								return( FALSE );
						}
						break;
					}
#endif
#endif
					default:
						printf( "Unknown rect encoding %d\n", (int) rect.encoding );
						return( FALSE);
				}
			}

			if( m_destImgSize.width() > 0 && m_destImgSize.height() > 0 )
			{
				// first create temporary image whose size matches destination-image-size
				QImage temporary_image = QImage( m_destImgSize, s_localDisplayFormat.depth );
				// now scale the image and store the output to temporary_image
				fastScaleImage( m_clientScreen, temporary_image );
				// now copy temporary_image to scaled client-screen
				//qApp->lock();
				m_scaledClientScreen = temporary_image;
				//qApp->unlock();
			}
			break;
		}
		case RFB_BELL:
			// FIXME: bell-action
			break;

		case RFB_SERVER_CUT_TEXT:
		{
			if( !readFromServer( ( (char *) &msg ) + 1, sizeof( s_rfb_server_cut_text_msg ) - 1 ) )
			{
				return( FALSE );
			}
			msg.sct.length = Swap32IfLE( msg.sct.length );
			char server_cut_text[msg.sct.length+1];
			if( !readFromServer( server_cut_text, msg.sct.length ) )
			{
				return( FALSE );
			}
			//server_cut_text[msg.sct.length] = 0;
    			//printf ("Server-Cut-Text: %s\n", server_cut_text);
			break;
    		}

		case RFB_SERVER_TO_ITALC_RESPONSE:
		{
			s_rfb_server_response sr;
			if( !readFromServer( (char *) &sr.italc_cmd, sizeof( sr.italc_cmd ) ) )
			{
				return( FALSE );
			}
			switch( sr.cmd )
			{
				case ITALC_POST_USER:
				{
					if( !readFromServer( ( (char *) &sr ) + 2, sizeof( s_rfb_post_user_msg )-2 ) )
					{
						return( FALSE );
					}
					sr.post_user_msg.user_name_len = Swap16IfLE( sr.post_user_msg.user_name_len );
					char * user_name = new char[sr.post_user_msg.user_name_len];
					if( !readFromServer( user_name, sr.post_user_msg.user_name_len ) )
					{
						delete[] user_name;
						return( FALSE );
					}
					m_user = user_name;
					delete[] user_name;
					break;
				}
				case ITALC_INVITATION_RESULT:
				{
					if( !readFromServer( ( (char *) &sr ) + 2, sizeof( s_rfb_invitation_result_msg ) - 2 ) )
					{
						return( FALSE );
					}
					m_invitationOk = sr.invitation_result_msg.inv_accepted;
					m_gotInvitationResult = TRUE;
					break;
				}
/*				case ITALC_POST_FILE_RESP: {
					if (!readFromServer(((char *) &sr)+2, sizeof(s_rfb_post_file_msg)-2))
						return (FALSE);
					sr.post_file_msg.fname_len = Swap16IfLE(sr.post_file_msg.fname_len);
					sr.post_file_msg.fsize = Swap32IfLE(sr.post_file_msg.fsize);
					char * file_name = new char[sr.post_file_msg.fname_len];
					if (!readFromServer(file_name, sr.post_file_msg.fname_len)) {
						delete[] file_name;
						return (FALSE);
					}
					system ("mkdir -p ~/collected_files/"+m_user);
					int fd = open(QString(getenv("HOME"))+"/collected_files/"+m_user+"/"+QFileInfo(file_name).fileName(), O_CREAT|O_WRONLY, 0666);
					Q_UINT32 total_bytes = sr.post_file_msg.fsize;
					Q_UINT32 bytes_done = 0;
					while (bytes_done < total_bytes) {
						char m_buffer[BUFFER_SIZE];
						Q_UINT32 bytes_todo = BUFFER_SIZE;
						if (total_bytes - bytes_done < bytes_todo)
							bytes_todo = total_bytes - bytes_done;
						if (!readFromServer(m_buffer, bytes_todo)) {
							close (fd);
							delete[] file_name;
							return (FALSE);
						}
						if (fd > 0)
							write(fd, m_buffer, bytes_todo);
						bytes_done += bytes_todo;
					}
					close (fd);
					delete[] file_name;
					break;
				}*/
				default:
					printf( "Unknown server response %d\n", (int) sr.cmd );
					return( FALSE );
			}
			break;
		}

		default:
			printf( "Unknown message type %d from VNC server. Closing connection. Will re-open it later.\n", msg.type );
			closeConnection();
			return( FALSE );
	}

	}	// end while( ... )


	if( _send_screen_update )
	{
		if( !sendIncrementalFramebufferUpdateRequest() )
		{
			return( FALSE );
		}
	}

	return( TRUE );
}









// ========================================================================================================================
// functions for network-IO
// ========================================================================================================================


int rfbConnection::connectToTCPAddr( in_addr_t _host, int _port )
{
	struct sockaddr_in addr;
	int one = 1;

	addr.sin_family = AF_INET;
	addr.sin_port = htons(_port);
	addr.sin_addr.s_addr = _host;
	int m_sock = socket(AF_INET, SOCK_STREAM, 0);
	if( m_sock < 0 )
	{
		printf( "connectToTCPAdress: socket(...) failed\n" );
		return( -1 );
	}

	if( ::connect( m_sock, ( struct sockaddr * )&addr, sizeof( addr ) ) < 0 )
	{
		//printf ("connectToTCPAdress: connect(...) failed\n");
		close( m_sock );
		return( -1 );
	}

	if( setsockopt( m_sock, IPPROTO_TCP, TCP_NODELAY, (char *)&one, sizeof( one ) ) < 0 )
	{
		printf( "connectToTCPAdress: setsockopt(...) failed\n" );
		close( m_sock );
		return( -1 );
	}

	return( m_sock );
}






bool rfbConnection::stringToIPAddr( const QString & _host, in_addr_t * _addr )
{
	if( _host == "" )
	{
		*_addr = 0;	// local
		return( TRUE );
	}

	*_addr = inet_addr( _host.ascii() );

	if( *_addr != (in_addr_t) -1 )
	{
		return( TRUE );
	}

	struct hostent * hp = gethostbyname( _host.ascii() );

	if( hp )
	{
		*_addr = *(in_addr_t *)hp->h_addr;
		return( TRUE );
	}

	return( FALSE );
}




// ========================================================================================================================
// functions for decoding rects
// ========================================================================================================================


bool rfbConnection::handleCoRRE( Q_UINT16 rx, Q_UINT16 ry, Q_UINT16 rw, Q_UINT16 rh )
{
	s_rfb_RRE_header hdr;

	if( !readFromServer( (char *) &hdr, sizeof( s_rfb_RRE_header ) ) )
	{
		return( FALSE );
	}

	hdr.n_subrects = Swap32IfLE( hdr.n_subrects );

	QRgb pix;
	if( !readFromServer( (char *) &pix, sizeof( pix ) ) )
	{
		return( FALSE );
	}

	fillRect( m_clientScreen, rx, ry, rw, rh, pix );

	if( !readFromServer( m_buffer, hdr.n_subrects * ( sizeof( s_rfb_CoRRE_rectangle ) + sizeof( Q_UINT32 ) ) ) )
	{
		return( FALSE );
	}

	Q_UINT8 * ptr = (Q_UINT8 *) m_buffer;

	for( Q_UINT32 i = 0; i < hdr.n_subrects; i++ )
	{
		pix = *(QRgb *) ptr;
		ptr += sizeof( pix );
		Q_UINT8 x = *ptr++;
		Q_UINT8 y = *ptr++;
		Q_UINT8 w = *ptr++;
		Q_UINT8 h = *ptr++;
		fillRect( m_clientScreen, rx+x, ry+y, w, h, pix );
	}

	return( TRUE );
}



bool rfbConnection::handleRRE( Q_UINT16, Q_UINT16, Q_UINT16, Q_UINT16 )
{
	printf ("Fatal: Got RRE-encoded rect. Can't decode.\n");
	return( TRUE );
}


#ifdef HAVE_LIBZ

#define TIGHT_MIN_TO_COMPRESS 12


#define RGB_TO_PIXEL(r,g,b)									\
  (((Q_UINT32)(r) & s_localDisplayFormat.red_max) << s_localDisplayFormat.red_shift |		\
   ((Q_UINT32)(g) & s_localDisplayFormat.green_max) << s_localDisplayFormat.green_shift |	\
   ((Q_UINT32)(b) & s_localDisplayFormat.blue_max) << s_localDisplayFormat.blue_shift)



// type declarations

typedef void (rfbConnection:: *filterPtr) (Q_UINT16, Q_UINT32 *);



// Definitions

bool rfbConnection::handleTight( Q_UINT16 rx, Q_UINT16 ry, Q_UINT16 rw, Q_UINT16 rh )
{
	QRgb fill_color;
	Q_UINT8 comp_ctl;

	if( !readFromServer( (char *) &comp_ctl, 1 ) )
	{
		return( FALSE );
	}

	// Flush zlib streams if we are told by the server to do so.
	for( Q_UINT8 stream_id = 0; stream_id < 4; stream_id++ )
	{
		if( ( comp_ctl & 1 ) && m_zlibStreamActive[stream_id] )
		{
			if( inflateEnd( &m_zlibStream[stream_id] ) != Z_OK && m_zlibStream[stream_id].msg != NULL )
			{
				printf( "inflateEnd: %s\n", m_zlibStream[stream_id].msg );
			}
			m_zlibStreamActive[stream_id] = FALSE;
		}
		comp_ctl >>= 1;
	}

	// Handle solid rectangles.
	if( comp_ctl == RFB_TIGHT_FILL )
	{
		if( !readFromServer( (char*)&fill_color, sizeof( fill_color ) ) )
		{
			return( FALSE );
		}
		fillRect( m_clientScreen, rx, ry, rw, rh, fill_color );
		return( TRUE );
	}

	if( comp_ctl == RFB_TIGHT_JPEG )
	{
#ifdef HAVE_LIBJPEG
		return( decompressJpegRect( rx, ry, rw, rh ) );
#else
		return ( -1 );
#endif
	}


	// Quit on unsupported subencoding value.
	if( comp_ctl > RFB_TIGHT_MAX_SUBENCODING )
	{
		printf( "Tight encoding: bad subencoding value received.\n" );
		return( FALSE );
	}

	// Here primary compression mode handling begins.
	// Data was processed with optional filter + zlib compression.
	filterPtr filter_function;
	Q_UINT8 bits_pixel;

	// First, we should identify a filter to use.
	if( ( comp_ctl & RFB_TIGHT_EXPLICIT_FILTER ) != 0 )
	{
		Q_UINT8 filter_id;
		if( !readFromServer( (char*) &filter_id, 1 ) )
		{
			return( FALSE );
		}

		switch( filter_id )
		{
			case RFB_TIGHT_FILTER_COPY:
				filter_function = &rfbConnection::filterCopy;
				bits_pixel = initFilterCopy( rw, rh );
				break;
			case RFB_TIGHT_FILTER_PALETTE:
				filter_function = &rfbConnection::filterPalette;
				bits_pixel = initFilterPalette( rw, rh );
				break;
			case RFB_TIGHT_FILTER_GRADIENT:
				filter_function = &rfbConnection::filterGradient;
				bits_pixel = initFilterGradient( rw, rh );
				break;
			default:
				printf( "Tight encoding: unknown filter code received.\n" );
				return( FALSE );
		}
	}
	else
	{
		filter_function = &rfbConnection::filterCopy;
		bits_pixel = initFilterCopy( rw, rh );
	}
	if( bits_pixel == 0 )
	{
		printf( "Tight encoding: error receiving palette.\n" );
		return( FALSE );
	}


	// Determine if the data should be decompressed or just copied.
	Q_UINT16 row_size = ( rw * bits_pixel + 7 ) / 8;
	if( rh * row_size < TIGHT_MIN_TO_COMPRESS )
	{
		if( !readFromServer( (char*)m_buffer, rh * row_size ) )
		{
			return( FALSE );
		}

		QRgb * buffer2 = (QRgb *) &m_buffer[TIGHT_MIN_TO_COMPRESS * 4];
		( this->*( filter_function ) )( rh, (Q_UINT32 *)buffer2 );
		copyRect( m_clientScreen, rx, ry, rw, rh, buffer2 );
		return( TRUE );
	}

	// Read the length (1..3 bytes) of compressed data following.
	int compressed_len = (int)readCompactLen();
	if( compressed_len <= 0 )
	{
		printf( "Incorrect data received from the server.\n" );
		return( FALSE );
	}


	// Now let's initialize compression stream if needed. */
	Q_UINT8 stream_id = comp_ctl & 0x03;
	z_streamp zs = &m_zlibStream[stream_id];
	if( !m_zlibStreamActive[stream_id] )
	{
		zs->zalloc = Z_NULL;
		zs->zfree = Z_NULL;
		zs->opaque = Z_NULL;
		int err = inflateInit( zs );
		if( err != Z_OK )
		{
			if( zs->msg != NULL )
			{
				printf( "InflateInit error: %s.\n", zs->msg );
			}
			return( FALSE );
		}
		m_zlibStreamActive[stream_id] = TRUE;
	}


	// Read, decode and draw actual pixel data in a loop. */
	int buffer_size = BUFFER_SIZE * bits_pixel / ( bits_pixel+32 ) & 0xFFFFFFFC;
	if( row_size > buffer_size )
	{
		// Should be impossible when BUFFER_SIZE >= 16384
		printf( "Internal error: incorrect m_buffer size.\n" );
		return( FALSE );
	}
	QRgb * buffer2 = (QRgb *) &m_buffer[buffer_size];


	Q_UINT16 rows_processed = 0;
	int extra_bytes = 0;
	int portion_len;

	while( compressed_len > 0 )
	{
		if( compressed_len > ZLIB_BUFFER_SIZE )	
		{
			portion_len = ZLIB_BUFFER_SIZE;
		}
		else
		{
			portion_len = compressed_len;
		}

		if( !readFromServer( (char*)m_zlibBuffer, portion_len ) )
		{
			return( FALSE );
		}

		compressed_len -= portion_len;

		zs->next_in = (Bytef *)m_zlibBuffer;
		zs->avail_in = portion_len;

		do
		{
			zs->next_out = (Bytef *) &m_buffer[extra_bytes];
			zs->avail_out = buffer_size - extra_bytes;

			int err = inflate(zs, Z_SYNC_FLUSH);
			if( err == Z_BUF_ERROR )   // Input exhausted -- no problem.
			{
				break;
			}
			if( err != Z_OK && err != Z_STREAM_END )
			{
				if( zs->msg != NULL )
				{
					printf( "Inflate error: %s.\n", zs->msg );
				}
				else
				{
					printf( "Inflate error: %d.\n", err );
				}
				return( FALSE );
			}

			Q_UINT16 num_rows = (Q_UINT16)( ( buffer_size - zs->avail_out ) / (int)row_size );

			( this->*( filter_function ) )( num_rows, (Q_UINT32 *)buffer2 );
			extra_bytes = buffer_size - zs->avail_out - num_rows * row_size;
			if( extra_bytes > 0 )
			{
				memcpy( m_buffer, &m_buffer[num_rows * row_size], extra_bytes );
			}

			copyRect( m_clientScreen, rx, ry+rows_processed, rw, num_rows, buffer2 );
			rows_processed += num_rows;
		}
		while( zs->avail_out == 0 );
	}

	if( rows_processed != rh )
	{
		printf( "Incorrect number of scan lines after decompression.\n" );
		return( FALSE );
	}

	return( TRUE );
}



/*----------------------------------------------------------------------------
 *
 * Filter stuff.
 *
 */

Q_UINT8 rfbConnection::initFilterCopy( Q_UINT16 rw, Q_UINT16/* rh*/ )
{
	m_rectWidth = rw;

	return( 32 );
}




void rfbConnection::filterCopy( Q_UINT16 num_rows, Q_UINT32 * dst )
{
	memcpy( dst, m_buffer, num_rows * m_rectWidth * sizeof( Q_UINT32 ) );
}




Q_UINT8 rfbConnection::initFilterGradient( Q_UINT16 rw, Q_UINT16/* rh*/ )
{
	m_rectWidth = rw;
	memset( m_tightPrevRow, 0, rw * 3 * sizeof( Q_UINT16 ) );

	return( sizeof( QRgb ) );
}




void rfbConnection::filterGradient( Q_UINT16 num_rows, Q_UINT32 * dst )
{
	Q_UINT32 * src = (Q_UINT32 *) m_buffer;
	Q_UINT16 * that_row = (Q_UINT16 *) m_tightPrevRow;
	Q_UINT16 this_row[2048*3];
	Q_UINT16 pix[3];
	Q_UINT16 max[3] = { s_localDisplayFormat.red_max, s_localDisplayFormat.green_max, s_localDisplayFormat.blue_max } ;
	Q_UINT8 shift[3] = { s_localDisplayFormat.red_shift, s_localDisplayFormat.green_shift, s_localDisplayFormat.blue_shift } ;
	Q_INT16 est[3];


	for( Q_UINT16 y = 0; y < num_rows; y++ )
	{
		// First pixel in a row
		for( Q_UINT8 c = 0; c < 3; c++ )
		{
			pix[c] = (Q_UINT16)((src[y*m_rectWidth] >> shift[c]) + that_row[c] & max[c]);
			this_row[c] = pix[c];
		}
		dst[y*m_rectWidth] = RGB_TO_PIXEL(pix[0], pix[1], pix[2]);
		// Remaining pixels of a row 
		for( Q_UINT16 x = 1; x < m_rectWidth; x++ )
		{
			for( Q_UINT8 c = 0; c < 3; c++ )
			{
				est[c] = (Q_INT16)that_row[x*3+c] + (Q_INT16)pix[c] - (Q_INT16)that_row[(x-1)*3+c];
				if( est[c] > (Q_INT16)max[c] )
				{
					est[c] = (Q_INT16)max[c];
				}
				else if( est[c] < 0 )
				{
					est[c] = 0;
				}
				pix[c] = (Q_UINT16)((src[y*m_rectWidth+x] >> shift[c]) + est[c] & max[c]);
				this_row[x*3+c] = pix[c];
			}
			dst[y*m_rectWidth+x] = RGB_TO_PIXEL( pix[0], pix[1], pix[2] );
		}
		memcpy( that_row, this_row, m_rectWidth * 3 * sizeof( Q_UINT16 ) );
	}
}




Q_UINT8 rfbConnection::initFilterPalette( Q_UINT16 rw, Q_UINT16/* rh*/ )
{
	Q_UINT8 num_colors;

	m_rectWidth = rw;

	if( !readFromServer( (char*)&num_colors, sizeof( num_colors ) ) )
	{
		return( 0 );
	}

	m_rectColors = (Q_UINT16) num_colors;
	if( ++m_rectColors < 2 )
	{
		return( 0 );
	}

	if( !readFromServer( (char*)&m_tightPalette, m_rectColors * sizeof( Q_UINT32 ) ) )
	{
		return( 0 );
	}

	return( ( m_rectColors == 2 ) ? 1 : 8 );
}




void rfbConnection::filterPalette( Q_UINT16 num_rows, Q_UINT32 * dst )
{
	Q_UINT8 * src = (Q_UINT8 *)m_buffer;
	Q_UINT32 * palette = (Q_UINT32 *) m_tightPalette;

	if( m_rectColors == 2 )
	{
		const Q_UINT16 w = (m_rectWidth + 7) / 8;
		for( Q_UINT16 y = 0; y < num_rows; y++ )
		{
			const Q_UINT32 base = y*m_rectWidth;
			for( Q_UINT16 x = 0; x < m_rectWidth/8; x++ )
			{
				for( Q_INT8 b = 7; b >= 0; b-- )
				{
					dst[base+x*8+7-b] = palette[src[y*w+x] >> b & 1];
				}
			}
			for( Q_INT8 b = 7; b >= 8 - m_rectWidth % 8; b-- )
			{
				dst[base+m_rectWidth+7-b] = palette[src[y*w+m_rectWidth/8] >> b & 1];
			}
		}
	}
	else
	{
		for( Q_UINT16 y = 0; y < num_rows; y++ )
		{
			const Q_UINT32 base = y*m_rectWidth;
			for( Q_UINT16 x = 0; x < m_rectWidth; x++ )
			{
				dst[base+x] = palette[(int)src[base+x]];
			}
		}
	}
}


#ifdef HAVE_LIBJPEG
/*----------------------------------------------------------------------------
 *
 * JPEG decompression.
 *
 */


void jpegInitSource( jpeglib::j_decompress_ptr )
{
}




jpeglib::boolean jpegFillInputBuffer( jpeglib::j_decompress_ptr )
{
	printf( "ERROR: jpegFillInputBuffer(...) called (not implemented, because it should not be needed\n" );
	return( 0 );
}




void jpegSkipInputData( jpeglib::j_decompress_ptr, long )
{
	printf( "ERROR: jpegSkipInputData(...) called (not implemented, because it should not be needed\n" );
}




void jpegTermSource( jpeglib::j_decompress_ptr )
{
}



using jpeglib::jpeg_decompress_struct;

bool rfbConnection::decompressJpegRect( Q_UINT16 x, Q_UINT16 y, Q_UINT16 w, Q_UINT16 h )
{
	int compressed_len = (int) readCompactLen();
	if( compressed_len <= 0 )
	{
		printf( "Incorrect data received from the server.\n" );
		return( FALSE );
	}

	Q_UINT8 compressed_data[compressed_len];

	if( !readFromServer( (char*)compressed_data, compressed_len ) )
	{
		return( FALSE );
	}

	struct jpeglib::jpeg_error_mgr jerr;
	struct jpeglib::jpeg_decompress_struct cinfo;
	cinfo.err = jpeglib::jpeg_std_error( &jerr );
	jpeglib::jpeg_create_decompress( &cinfo );

	//jpegSetSrcManager (&cinfo, compressed_data, compressed_len);
	m_jpegSrcManager.init_source = jpegInitSource;
	m_jpegSrcManager.fill_input_buffer = jpegFillInputBuffer;
	m_jpegSrcManager.skip_input_data = jpegSkipInputData;
	m_jpegSrcManager.resync_to_restart = jpeglib::jpeg_resync_to_restart;
	m_jpegSrcManager.term_source = jpegTermSource;
	m_jpegSrcManager.next_input_byte = (jpeglib::JOCTET *) compressed_data;
	m_jpegSrcManager.bytes_in_buffer = (size_t) compressed_len;

	cinfo.src = &m_jpegSrcManager;


	jpeglib::jpeg_read_header( &cinfo, TRUE );
	cinfo.out_color_space = jpeglib::JCS_RGB;

	jpeglib::jpeg_start_decompress( &cinfo );
	if( cinfo.output_width != w || cinfo.output_height != h || cinfo.output_components != 3 )
	{
		printf( "Tight Encoding: Wrong JPEG data received.\n" );
		jpeglib::jpeg_destroy_decompress( &cinfo );
		return( FALSE );
	}

	jpeglib::JSAMPROW row_pointer[1];
	row_pointer[0] = (jpeglib::JSAMPROW) m_buffer;
	int dy = 0;
	while( cinfo.output_scanline < cinfo.output_height )
	{
		jpeglib::jpeg_read_scanlines( &cinfo, row_pointer, 1 );
		Q_UINT32 * pixel_ptr = (Q_UINT32 *) &m_buffer[BUFFER_SIZE / 2];
		for( Q_UINT16 dx = 0; dx < w; dx++ )
		{
			*pixel_ptr++ = RGB_TO_PIXEL( m_buffer[dx*3], m_buffer[dx*3+1], m_buffer[dx*3+2] );
		}
		copyRect( m_clientScreen, x, y+dy, w, 1, (QRgb *) &m_buffer[BUFFER_SIZE / 2] );
		dy++;
	}

	jpeglib::jpeg_finish_decompress( &cinfo );

	jpeglib::jpeg_destroy_decompress( &cinfo );

	return( TRUE );
}

#endif	/* LIBJPEG */

#endif	/* LIBZ */






// this is an optimized version of Qt's smoothScale... the adventage is, that I could remove all code, dealing with
// alpha-layers. The second thing is, that this code will be optimized for use on CPU with MMX and SSE. The Qt libraries
// are per default compiled for i586 only...
void fastScaleImage( const QImage & src, QImage & dst )
{
	QRgb * xelrow = NULL;
	QRgb * tempxelrow = NULL;
	Q_UINT16 rowswritten = 0;
	const uchar maxval = 255;

	Q_UINT16 cols = src.width();
	Q_UINT16 rows = src.height();
	Q_UINT16 newcols = dst.width();
	Q_UINT16 newrows = dst.height();

	long SCALE;
	long HALFSCALE;

	if( cols > 4096 )
	{
		SCALE = 4096;
		HALFSCALE = 2048;
	}
	else
	{
		Q_UINT16 fac = 4096;
		while( cols * fac > 4096 )
		{
			fac /= 2;
		}
		SCALE = fac * cols;
		HALFSCALE = fac * cols / 2;
	}

	long sxscale = (long)( (double) newcols / (double) cols * SCALE );
	long syscale = (long)( (double) newrows / (double) rows * SCALE );

	if( newrows != rows )	// shortcut Y scaling if possible
		tempxelrow = new QRgb[cols];

	long * rs = new long[cols];
	long * gs = new long[cols];
	long * bs = new long[cols];
	Q_UINT16 rowsread = 0;
	register long fracrowleft = syscale;
	register Q_UINT16 needtoreadrow = 1;

	for( Q_UINT16 col = 0; col < cols; ++col )
	{
		rs[col] = gs[col] = bs[col] = HALFSCALE;
	}

	register long fracrowtofill = SCALE;
	register QRgb * xP;
	register QRgb * nxP;

	for( Q_UINT16 row = 0; row < newrows; ++row )
	{
		// First scale Y from xelrow into tempxelrow.
		if( newrows == rows )
		{
			// shortcut Y scaling if possible 
			tempxelrow = xelrow = (QRgb*)src.scanLine( rowsread++ );
		}
		else
		{
			while( fracrowleft < fracrowtofill )
			{
				if( needtoreadrow && rowsread < rows )
				{
					xelrow = (QRgb*)src.scanLine( rowsread++ );
				}
				xP = xelrow;
				for( Q_UINT16 col = 0; col < cols; ++col, ++xP )
				{
					rs[col] += fracrowleft * qRed( *xP );
					gs[col] += fracrowleft * qGreen( *xP );
					bs[col] += fracrowleft * qBlue( *xP );
				}
				fracrowtofill -= fracrowleft;
				fracrowleft = syscale;
				needtoreadrow = 1;
			}
			// Now fracrowleft is >= fracrowtofill, so we can produce a row.
			if( needtoreadrow && rowsread < rows )
			{
				xelrow = (QRgb*)src.scanLine( rowsread++ );
				needtoreadrow = 0;
			}
			xP = xelrow;
			nxP = tempxelrow;
			for( Q_UINT16 col = 0; col < cols; ++col, ++xP, ++nxP )
			{
				register long r, g, b;
				r = rs[col] + fracrowtofill * qRed( *xP );
				g = gs[col] + fracrowtofill * qGreen( *xP );
				b = bs[col] + fracrowtofill * qBlue( *xP );

				r /= SCALE;
				if( r > maxval )
				{
					r = maxval;
				}

				g /= SCALE;
				if( g > maxval )
				{
					g = maxval;
				}

				b /= SCALE;
				if( b > maxval )
				{
					b = maxval;
				}

				*nxP = qRgb( (int)r, (int)g, (int)b );
				rs[col] = gs[col] = bs[col] = HALFSCALE;
			}
			fracrowleft -= fracrowtofill;
			if( fracrowleft == 0 )
			{
				fracrowleft = syscale;
				needtoreadrow = 1;
			}
			fracrowtofill = SCALE;
		}

		// Now scale X from tempxelrow into dst and write it out.
		if( newcols == cols )
		{
			// shortcut X scaling if possible
			memcpy( dst.scanLine(rowswritten++), tempxelrow, newcols*4 );
		}
		else
		{
			register long r, g, b;
			register long fraccoltofill, fraccolleft = 0;
			register Q_UINT16 needcol;

			nxP = (QRgb*)dst.scanLine( rowswritten++ );
			fraccoltofill = SCALE;
			r = g = b = HALFSCALE;
			needcol = 0;
			xP = tempxelrow;
			for( Q_UINT16 col = 0; col < cols; ++col, ++xP )
			{
				fraccolleft = sxscale;
				while( fraccolleft >= fraccoltofill )
				{
					if( needcol )
					{
						++nxP;
						r = g = b = HALFSCALE;
					}
					r += fraccoltofill * qRed( *xP );
					g += fraccoltofill * qGreen( *xP );
					b += fraccoltofill * qBlue( *xP );

					r /= SCALE;
					if( r > maxval )
					{
						r = maxval;
					}

					g /= SCALE;
					if( g > maxval )
					{
						g = maxval;
					}

					b /= SCALE;
					if( b > maxval )
					{
						b = maxval;
					}

					*nxP = qRgb( (int)r, (int)g, (int)b );
					fraccolleft -= fraccoltofill;
					fraccoltofill = SCALE;
					needcol = 1;
				}
				if( fraccolleft > 0 )
				{
					if( needcol )
					{
						++nxP;
						r = g = b = HALFSCALE;
						needcol = 0;
					}
					r += fraccolleft * qRed( *xP );
					g += fraccolleft * qGreen( *xP );
					b += fraccolleft * qBlue( *xP );
					fraccoltofill -= fraccolleft;
				}
			}
			if( fraccoltofill > 0 )
			{
				--xP;
				r += fraccoltofill * qRed( *xP );
				g += fraccoltofill * qGreen( *xP );
				b += fraccoltofill * qBlue( *xP );
			}
			if( !needcol )
			{
				r /= SCALE;
				if( r > maxval )
				{
					r = maxval;
				}

				g /= SCALE;
				if( g > maxval )
				{
					g = maxval;
				}

				b /= SCALE;
				if( b > maxval )
				{
					b = maxval;
				}

				*nxP = qRgb( (int)r, (int)g, (int)b );
			}
		}
	}

	if( newrows != rows && tempxelrow )// Robust, tempxelrow might be 0 one day
	{
		delete[] tempxelrow;
	}
	if( rs )			// Robust, rs might be 0 one day
	{
		delete[] rs;
	}
	if( gs )			// Robust, gs might be 0 one day
	{
		delete[] gs;
	}
	if( bs )			// Robust, bs might be 0 one day
	{
		delete[] bs;
	}
}
