unit AdminBackup;

// Copyright (C) 2003, 2004 MySQL AB
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

interface

{$Include Compilers.inc}

uses
  gnugettext, Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ExtCtrls, StdCtrls, ComCtrls,
  InstanceSections, AdditionalClasses,
  ImgList, Buttons, Contnrs, Menus, ApplicationDataModule, AuxFuncs,
  myx_public_interface, myx_admin_public_interface, SchemataTreeView,
  AdvancedEdit, PNGImage,
  AdminBackupProgress, AuxAdminBackupRestore,
  MySQLConnection, Options, MyxError, TntExtCtrls, TntComCtrls,
  Math, OptionsEditor, AdminOptionPages,
  TntMenus, TntStdCtrls, TntForms, TntClasses, TntDialogs,
  ActiveX, MSTask;

type
  TBackupThread = class;

  TAdminBackupForm = class(TInstanceSectionForm)
    ServerBackupRestorePnl: TTntPanel;
    BackupRestorePageControl: TTntPageControl;
    BackupContentTabSheet: TTabSheet;
    ItemSelectImageList: TImageList;
    Panel1: TTntPanel;
    RestoreBevel: TTntBevel;
    Label3: TTntLabel;
    Backup2Img: TTntImage;
    SubTreePnl: TTntPanel;
    BackupTreeViewPopupMenu: TTntPopupMenu;
    RemoveBackupNodeMI: TTntMenuItem;
    ProfilesPopupMenu: TTntPopupMenu;
    RefreshProfilesMI: TTntMenuItem;
    SchemataFrame: TSchemataFrame;
    ContentSplitter: TTntSplitter;
    Panel2: TTntPanel;
    TopPnl: TTntPanel;
    SchemataLbl: TTntLabel;
    AdvancedEditFrame: TAdvancedEditFrame;
    SpacerPnl: TTntPanel;
    ProjectListView: TTntListView;
    AddSchemaToBackupBtn: TTntButton;
    RemoveSchemaFromBackup: TTntButton;
    Panel3: TTntPanel;
    BackupTreeView: TTntTreeView;
    BackupTreeViewHeaderControl: THeaderControl;
    Panel4: TTntPanel;
    Label1: TTntLabel;
    Label2: TTntLabel;
    Panel7: TTntPanel;
    N1: TTntMenuItem;
    DeleteProfileMI: TTntMenuItem;
    ClearCompleteContentMI: TTntMenuItem;
    AdvancedOptionsTabSheet: TTabSheet;
    Panel5: TTntPanel;
    Bevel1: TTntBevel;
    Label15: TTntLabel;
    Backup3Img: TTntImage;
    Label16: TTntLabel;
    NewProjectBtn: TTntButton;
    ApplyBtn: TTntButton;
    StartBackupBtn: TTntButton;
    Panel9: TTntPanel;
    GroupBox1: TTntGroupBox;
    Label4: TTntLabel;
    Label5: TTntLabel;
    ProjectNameEd: TTntEdit;
    ScheduleTabSheet: TTabSheet;
    ScheduleGBox: TTntGroupBox;
    ScheduleTargetDirCaptionLbl: TTntLabel;
    ScheduleTargetDirLbl: TTntLabel;
    SchedulePrefixCaptionLbl: TTntLabel;
    SchedulePrefixLbl: TTntLabel;
    ScheduleLbl: TTntLabel;
    ScheduleTargetDirEd: TTntEdit;
    ScheduleBrowseTargetDirBtn: TTntButton;
    ScheduleFilenamePrefixEd: TTntEdit;
    ScheduleCBox: TTntCheckBox;
    Panel6: TTntPanel;
    BackupBevel: TTntBevel;
    Label6: TTntLabel;
    BackupImg: TTntImage;
    Label7: TTntLabel;
    ExecutionTimeGBox: TTntGroupBox;
    SchedulePageControl: TTntPageControl;
    DailyTabSheet: TTabSheet;
    Label26: TTntLabel;
    WeeklyTabSheet: TTabSheet;
    Label21: TTntLabel;
    ScheduleMondayCBox: TTntCheckBox;
    ScheduleTuesdayCBox: TTntCheckBox;
    ScheduleWednesdayCBox: TTntCheckBox;
    ScheduleThursdayCBox: TTntCheckBox;
    ScheduleFridayCBox: TTntCheckBox;
    ScheduleSaturdayCBox: TTntCheckBox;
    ScheduleSundayCBox: TTntCheckBox;
    MonthlyTabSheet: TTabSheet;
    Label22: TTntLabel;
    Label20: TTntLabel;
    Label25: TTntLabel;
    ScheduleDayOfMonthCBox: TTntComboBox;
    TimePnl: TTntPanel;
    Label23: TTntLabel;
    Label24: TTntLabel;
    ScheduleTimeEd: TTntEdit;
    ScheduleTypeCbox: TTntComboBox;
    Panel10: TTntPanel;
    AdvOptionsScrollBox: TTntScrollBox;
    BackExecuteGBox: TTntGroupBox;
    Label17: TTntLabel;
    Label18: TTntLabel;
    Label19: TTntLabel;
    Label14: TTntLabel;
    LockAllTablesRButton: TTntRadioButton;
    SingleTransRButton: TTntRadioButton;
    NormalBackupRadioButton: TTntRadioButton;
    CompleteSchematasCBox: TTntCheckBox;
    OutputFileGBox: TTntGroupBox;
    Label12: TTntLabel;
    Label13: TTntLabel;
    NoCreatesCBox: TTntCheckBox;
    NoExtendedInsertsCBox: TTntCheckBox;
    AddDropTableCBox: TTntCheckBox;
    CompleteInsertsCBox: TTntCheckBox;
    CommentCBox: TTntCheckBox;
    FullPathCBox: TTntCheckBox;
    AnsiQuotesCBox: TTntCheckBox;
    DisableKeysCBox: TTntCheckBox;
    BackupFileTypesLU: TTntComboBox;
    CompatibilityCheckbox: TCheckBox;
    PointInTimeRBtn: TTntRadioButton;
    PointInTimeLbl: TTntLabel;
    procedure BackupTreeViewCompare(Sender: TObject; Node1, Node2: TTreeNode; Data: Integer; var Compare: Integer);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);

    procedure FillSchemaTree(searchStr: WideString = '');
    procedure ServerBackupRestorePnlResize(Sender: TObject);
    procedure BackupTreeViewAdvancedCustomDrawItem(Sender: TCustomTreeView;
      Node: TTreeNode; State: TCustomDrawState; Stage: TCustomDrawStage;
      var PaintImages, DefaultDraw: Boolean);
    procedure BackupTreeViewMouseDown(Sender: TObject;
      Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure BackupTreeViewHeaderControlSectionResize(
      HeaderControl: THeaderControl; Section: THeaderSection);

    procedure AddSchemaToBackupBtnClick(Sender: TObject);
    procedure AddSchemaToBackup(Schema: TMYX_SCHEMA; Selected: Boolean = True);
    procedure BackupTreeViewDragOver(Sender, Source: TObject; X,
      Y: Integer; State: TDragState; var Accept: Boolean);
    procedure BackupTreeViewDragDrop(Sender, Source: TObject; X,
      Y: Integer);
    procedure SchemaTreeViewExpanding(Sender: TObject; Node: TTntTreeNode;
      var AllowExpansion: Boolean);
    procedure BackupTreeViewPopupMenuPopup(Sender: TObject);
    procedure RemoveBackupNodeMIClick(Sender: TObject);

    procedure RemoveTreeNode(Node: TTntTreeNode);
    procedure SchemaTreeViewCollapsing(Sender: TObject; Node: TTntTreeNode;
      var AllowCollapse: Boolean);
    procedure SchemataFrameSchemaTreeViewMouseDown(Sender: TObject;
      Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure StartBackupBtnClick(Sender: TObject);

    procedure NewProjectBtnClick(Sender: TObject);
    procedure BackupTreeViewChanging(Sender: TObject; Node: TTreeNode;
      var AllowChange: Boolean);
    procedure ClearControls(EnableControls: Boolean);
    procedure ProjectNameEdExit(Sender: TObject);
    procedure ScheduleBrowseTargetDirBtnClick(Sender: TObject);
    procedure DoChange(Sender: TObject);
    procedure DiscardBtnClick(Sender: TObject);
    procedure ApplyBtnClick(Sender: TObject);

    procedure RefreshBackupProfiles(SelectProfile: WideString = '');

    procedure ApplyChanges;
    procedure DiscardChanges(AskBeforeDiscard: Boolean);

    procedure LoadProfile(filename: WideString);
    function GetSelectedTablesCount: Integer;

    function GetOptions: Integer;
    procedure AddOption(var options: Integer;
      option: MYX_BACKUP_OPTIONS; Add: Boolean);
    function InOptions(options: Integer; option: MYX_BACKUP_OPTIONS): Boolean;
    procedure RefreshProfilesMIClick(Sender: TObject);
    procedure DeleteProfileMIClick(Sender: TObject);
    procedure ProfilesPopupMenuPopup(Sender: TObject);
    procedure ClearCompleteContentMIClick(Sender: TObject);
    procedure ProjectListViewSelectItem(Sender: TObject; Item: TListItem;
      Selected: Boolean);

    procedure Disconnected(var Message: TMessage); message WM_Disconnected;
    procedure Reconnected(var Message: TMessage); message WM_Reconnected;
    procedure SchemaListChanged(var Message: TMessage); message WM_SchemaListChanged;
    procedure AdvancedEditFrameSearchEdChange(Sender: TObject);

    procedure DisableEnableScheduleControls;
    procedure ScheduleCBoxClick(Sender: TObject);

    procedure ScheduleTypeCboxCloseUp(Sender: TObject);
    procedure SchemataFrameCatalogVSTMouseMove(Sender: TObject;
      Shift: TShiftState; X, Y: Integer);
    procedure SchemataFrameCatalogVSTDblClick(Sender: TObject);
    procedure MenuDrawItem(Sender: TObject; ACanvas: TCanvas; ARect: TRect; State: TOwnerDrawState);
    procedure MenuMeasureItem(Sender: TObject; ACanvas: TCanvas; var Width, Height: Integer);
  private
    MousePos: TPoint;

    BackupPNGImg: TPNGObject;

    BackupProfileDir: WideString;

    BackupThread: TBackupThread;

    OriginalProjectName: WideString;
    FTaskScheduler: ITaskScheduler;
  protected
    function StartScheduler: DWORD;
  public
    CurrentCatalog: TMYX_CATALOG;
    CurrentSchema: TMYX_SCHEMA;

    // User must not expand SchemaTree manually
    LockSchemaTreeExpansion: Boolean;
  end;

  TBackupThread = class(TThread)
    constructor Create(AdminBackupForm: TAdminBackupForm;
      AdminBackupProgressForm: TAdminBackupProgressForm;
      filename: WideString);
    destructor Destroy; override;

    procedure UpdateProgressForm;
    procedure FinishedBackup;
    procedure FreeAdminBackupProgressForm;
    procedure ShowError;
    procedure CleanUp;
  protected
    procedure Execute; override;
  private
    PMySQL: Pointer;

    AdminBackupForm: TAdminBackupForm;
    filename: WideString;
  public
    AdminBackupProgressForm: TAdminBackupProgressForm;

    current_table_name: WideString;
    num_tables: Integer;
    num_tables_processed: Integer;
    num_rows: Integer;
    num_rows_processed: Integer;

    BackupError: MYX_BACKUP_ERROR;
  end;

function BackupProgress(current_table_name: PChar; num_tables: Integer; num_tables_processed: Integer;
  num_rows: Integer; num_rows_processed: Integer; user_data: Pointer): Integer; cdecl;

//----------------------------------------------------------------------------------------------------------------------

implementation

{$R *.dfm}

uses
  {$ifndef COMPILER_9_UP}
    WinError,
  {$endif COMPILER_9_UP}
  Main, VirtualTrees, DateUtils, WinSvc, Password, PNGTools;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.FormCreate(Sender: TObject);

var
  Result: DWORD;
  
begin
  InitForm(self);

  // Initialize scheduler interfaces.
  Result := CoCreateInstance(CLSID_CSchedulingAgent, nil, CLSCTX_INPROC_SERVER, IID_ITaskScheduler, FTaskScheduler);
  if not Succeeded(Result) then
    raise EInOutError.Create(_('Could not initialize task scheduler interface.'));

  DockedPanel := ServerBackupRestorePnl;
  SubTreePanel := SubTreePnl;

  BackupRestorePageControl.ActivePageIndex := 0;

  CurrentCatalog := nil;
  CurrentSchema := nil;

  BackupThread := nil;

  InitControls := True;

  LockSchemaTreeExpansion := True;

  SchemataFrame.MySQLConnection := MySQLConn;
  SchemataFrame.ShowSchemaAssets := False;
  SchemataFrame.ShowAssetsOnSchemaExpansion := True;
  SchemataFrame.FillSchemaTree;

  BackupPNGImg := LoadPNGImageFromResource('backup', BackupImg);
  Backup2Img.Picture.Assign(BackupPNGImg);
  Backup3Img.Picture.Assign(BackupPNGImg);

  if (MYXCommonOptions.XPStyleEnabled) then
  begin
    ContentSplitter.Color := clWhite;
    ExecutionTimeGBox.Color := clWhite;
    AdvOptionsScrollBox.Color := clWhite;
  end;

  ClearControls(False);

  if (not (MySQLConn.Connected)) then
    NewProjectBtn.Enabled := False;

  BackupProfileDir := MYXCommonOptions.UserDataDir;

  RefreshBackupProfiles;

  ScheduleTypeCbox.ItemIndex := 1;
  SchedulePageControl.ActivePageIndex := 1;
  ScheduleDayOfMonthCBox.ItemIndex := 0;

  {if (mysql_full_version_is_later_or_equal_than(
      MySQLConn.MySQL, 4, 1, 8) = 1) then
  begin
    PointInTimeRBtn.Enabled := False;
    PointInTimeLbl.Enabled := False;
  end;}
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.FormDestroy(Sender: TObject);

begin
  BackupPNGImg.Free;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.FormClose(Sender: TObject; var Action: TCloseAction);

begin
  ClearListView(ProjectListView, myx_ndt_pointer);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.FillSchemaTree(searchStr: WideString = '');

begin
  //
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.ServerBackupRestorePnlResize(Sender: TObject);

//var InitSheetWidth, InitSheetHeight: Integer;

begin
  {InitSheetWidth:=563;
  InitSheetHeight:=451;

  //Adjust Page Control
  BackupRestorePageControl.Width:=ServerBackupRestorePnl.Width-591+571;
  BackupRestorePageControl.Height:=ServerBackupRestorePnl.Height-501+443;}

  //Backup Sheet
  BackupTreeViewHeaderControl.Left := BackupTreeView.Left + 1;
  BackupTreeViewHeaderControl.Width := BackupTreeView.Width - 2 - 18;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.BackupTreeViewAdvancedCustomDrawItem(Sender: TCustomTreeView; Node: TTreeNode;
  State: TCustomDrawState; Stage: TCustomDrawStage; var PaintImages, DefaultDraw: Boolean);
  
var
  NodeRect: TRect;
  XPos: Integer;
  theHeaderControl: THeaderControl;
  i: Integer;

begin
  theHeaderControl := BackupTreeViewHeaderControl;

  if (Stage = cdPrePaint) then
  begin
    Sender.Canvas.Pen.Style := psSolid;
    if (cdsFocused in State) then
      Sender.Canvas.Pen.Color := clGray
    else
      Sender.Canvas.Pen.Color := clSilver;

    NodeRect := Node.DisplayRect(False);

    if (cdsSelected in State) then
    begin
      Sender.Canvas.Brush.Color := clWhite;
      Sender.Canvas.Pen.Color := clGray;
      Sender.Canvas.Rectangle(NodeRect.Left + 19 + Node.Level * 19 + 5 - 2, NodeRect.Top,
        Sender.Width - 4 - 18, NodeRect.Bottom);
    end
    else
    begin
      Sender.Canvas.Brush.Color := clWhite;
      Sender.Canvas.FillRect(Rect(NodeRect.Left + 19 + Node.Level * 19 + 5 - 2, NodeRect.Top,
        Sender.Width - 4 - 18, NodeRect.Bottom + 1));
    end;

    Sender.Canvas.Brush.Color := clWhite;

    DefaultDraw := True;
  end
  else
    if (Stage = cdPostPaint) then
    begin
      if (Node.Data <> nil) then
      begin
        NodeRect := Node.DisplayRect(False);

        if (cdsFocused in State) then
        begin
          Sender.Canvas.Pen.Color := clGray;
          Sender.Canvas.Pen.Style := psSolid;
          Sender.Canvas.Brush.Color := clSilver;

          Sender.Canvas.Rectangle(NodeRect.Left + 19 + Node.Level * 19 + 5 - 2, NodeRect.Top,
            Sender.Width - 4 - 18, NodeRect.Bottom);

          Sender.Canvas.Brush.Style := bsClear;

          Sender.Canvas.TextRect(Rect(
            NodeRect.Left + 19 + Node.Level * 19 + 5, NodeRect.Top + 1,
            theHeaderControl.Sections[0].Width + 2, NodeRect.Top + 16),
            NodeRect.Left + 19 + Node.Level * 19 + 5, NodeRect.Top + 1,
            Node.Text);
        end;

        ItemSelectImageList.Draw(Sender.Canvas,
          NodeRect.Left + 19 + Node.Level * 19 + 5, NodeRect.Top + 1,
          Ord(TBackupNode(Node.Data).GetSelectState));

        if (TBackupNode(Node.Data).GetObjType = BOTTable) and
          (TBackupNode(Node.Data).Substrings <> nil) then
        begin
          Sender.Canvas.Brush.Style := bsClear;
          XPos := 0;
          for i := 0 to 3 do
          begin
            XPos := XPos + theHeaderControl.Sections[i].Width;
            Sender.Canvas.TextRect(Rect(XPos + 2, NodeRect.Top + 1,
              XPos + theHeaderControl.Sections[i + 1].Width - 4, NodeRect.Top + 16),
              XPos + 2, NodeRect.Top + 1,
              TBackupNode(Node.Data).Substrings[i]);
          end;
        end;
      end;
    end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.BackupTreeViewMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);

var
  Node: TTntTreeNode;
  NodeRect: TRect;
  Selected: TBackupCheckType;
  
begin
  Node := TTntTreeView(Sender).GetNodeAt(x, y);
  if (Assigned(Node)) then
  begin
    NodeRect := Node.DisplayRect(False);

    if (X > NodeRect.Left + 19 + Node.Level * 19 + 5) and (X < NodeRect.Left + 19 + Node.Level * 19 + 5 + 14) and
      (Node.Data <> nil) then
    begin
      if (TBackupNode(Node.Data).GetSelectState <> BCTAll) then
        Selected := BCTAll
      else
        Selected := BCTNone;

      SetNodeSelectState(Node, Selected);

      DoChange(self);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.BackupTreeViewHeaderControlSectionResize(HeaderControl: THeaderControl; Section: THeaderSection);

begin
  BackupTreeView.Invalidate;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.AddSchemaToBackup(Schema: TMYX_SCHEMA; Selected: Boolean);

var
  CatalogNode, SchemaNode: TTntTreeNode;
  i: Integer;
  pentity_status: PMYX_SCHEMA_ENTITY_STATUS;
  entity_status: TMYX_SCHEMA_ENTITY_STATUS;

  ptable_status: PMYX_TABLE_STATUS;
  table_status: TMYX_TABLE_STATUS;
  pview_status: PMYX_VIEW_STATUS;
  view_status: TMYX_VIEW_STATUS;
  ppsp_status: PMYX_SCHEMA_STORED_PROCEDURE;
  psp_status: TMYX_SCHEMA_STORED_PROCEDURE;
  backup_node: TBackupNode;

begin
  CatalogNode := nil;

  //Check if schema not already in Backup List
  for i := 0 to BackupTreeView.Items.Count - 1 do
    if (BackupTreeView.Items[i].Data <> nil) then
      if (TObject(BackupTreeView.Items[i].Data) is TBackupNode) then
        if (TBackupNode(BackupTreeView.Items[i].Data).GetObjType = BOTSchema) then
          if (CompareText(TBackupNode(BackupTreeView.Items[i].Data).GetObjName,
            Schema.schema_name) = 0) then
          begin
            //If Node has a parent, check catalog name, too
            if (BackupTreeView.Items[i].Parent <> nil) then
            begin
              if (BackupTreeView.Items[i].Parent.Data <> nil) then
                if (TObject(BackupTreeView.Items[i].Parent.Data) is TBackupNode) then
                  if (TBackupNode(BackupTreeView.Items[i].Parent.Data).GetObjType = BOTCatalog) then
                    if (CompareText(TBackupNode(BackupTreeView.Items[i].Parent.Data).GetObjName,
                      Schema.catalog_name) = 0) then
                      Exit;
            end
            else
              Exit;
          end;

  if (BackupTreeView.Items.Count = 0) then
    AddTreeViewChildNode(BackupTreeView, nil, 'BackupRoot', -1, nil);

  for i := 0 to BackupTreeView.Items.Count - 1 do
    if (BackupTreeView.Items[i].Data <> nil) then
      if (TObject(BackupTreeView.Items[i].Data) is TBackupNode) then
        if (TBackupNode(BackupTreeView.Items[i].Data).GetObjType = BOTCatalog) then
          if (CompareText(TBackupNode(BackupTreeView.Items[i].Data).GetObjName,
            Schema.catalog_name) = 0) then
          begin
            CatalogNode := BackupTreeView.Items[i];
            break;
          end;

  if (CatalogNode = nil) then
  begin
    CatalogNode := AddTreeViewChildNode(BackupTreeView, nil, //BackupTreeView.Items[0],
      '      ' + Schema.catalog_name, 2,
      TBackupNode.Create(Schema.catalog_name, BOTCatalog, 0));
  end;

  SchemaNode := AddTreeViewChildNode(BackupTreeView, CatalogNode,
    '      ' + Schema.schema_name, 9,
    TBackupNode.Create(Schema.schema_name, BOTSchema, 0));

  //----------------------------------------------------------
  //Get Tables
  pentity_status := myx_get_schema_entity_status(MySQLConn.MySQL, '',
    Schema.schema_name);
  try
    entity_status := TMYX_SCHEMA_ENTITY_STATUS.Create(pentity_status);

    for i := 0 to entity_status.schema_entities.Count - 1 do
    begin
      if (entity_status.schema_entities[i].entity_type = MYX_ENTITY_TABLE) then
      begin
        ptable_status := entity_status.schema_entities[i].entity;
        table_status := TMYX_TABLE_STATUS.Create(ptable_status);
        try
          backup_node :=
            TBackupNode.Create(table_status.table_name, BOTTable, 0);
          backup_node.Substrings := TTntStringList.Create;
          backup_node.Substrings.Add(table_status.table_type);
          backup_node.Substrings.Add(table_status.rows);
          backup_node.Substrings.Add(table_status.data_length);
          backup_node.Substrings.Add(table_status.update_time);
          AddTreeViewChildNode(BackupTreeView, SchemaNode, '      ' +
            table_status.table_name, 10, backup_node);
        finally
          table_status.Free;
        end;
      end
      else if (entity_status.schema_entities[i].entity_type = MYX_ENTITY_VIEW) then
      begin
        pview_status := entity_status.schema_entities[i].entity;
        view_status := TMYX_VIEW_STATUS.Create(pview_status);
        try
          backup_node := TBackupNode.Create(view_status.view_name, BOTTable, 0);
          backup_node.Substrings := TTntStringList.Create;
          backup_node.Substrings.Add('VIEW');
          backup_node.Substrings.Add('');
          backup_node.Substrings.Add('');
          backup_node.Substrings.Add('');
          AddTreeViewChildNode(BackupTreeView, SchemaNode, '      ' + view_status.view_name, 27, backup_node)
        finally
          view_status.Free;
        end;
      end
      else if (entity_status.schema_entities[i].entity_type = MYX_ENTITY_PROC) then
      begin
        ppsp_status := entity_status.schema_entities[i].entity;
        psp_status := TMYX_SCHEMA_STORED_PROCEDURE.Create(ppsp_status);
        try
          backup_node := TBackupNode.Create(psp_status.name, BOTTable, 0);
          backup_node.Substrings := TTntStringList.Create;
          backup_node.Substrings.Add('PROCEDURE');
          backup_node.Substrings.Add('');
          backup_node.Substrings.Add('');
          backup_node.Substrings.Add('');
          AddTreeViewChildNode(BackupTreeView, SchemaNode, '      ' + psp_status.name, 13, backup_node)
        finally
          psp_status.Free;
        end;
      end
      else if (entity_status.schema_entities[i].entity_type = MYX_ENTITY_FUNC) then
      begin
        ppsp_status := entity_status.schema_entities[i].entity;
        psp_status := TMYX_SCHEMA_STORED_PROCEDURE.Create(ppsp_status);
        try
          backup_node := TBackupNode.Create(psp_status.name, BOTTable, 0);
          backup_node.Substrings := TTntStringList.Create;
          backup_node.Substrings.Add('FUNCTION');
          backup_node.Substrings.Add('');
          backup_node.Substrings.Add('');
          backup_node.Substrings.Add('');
          AddTreeViewChildNode(BackupTreeView, SchemaNode, '      ' + psp_status.name, 13, backup_node)
        finally
          psp_status.Free;
        end;
      end
    end;

    //Select new Schema and all subnodes (tables)
    if (Selected) then
      SetNodeSelectState(SchemaNode, BCTAll);

    entity_status.Free;
  finally
    //myx_free_schema_entity_status(pentity_status);
  end;

  //Expand nodes
  BackupTreeView.Items[0].Expand(False);
  CatalogNode.Expand(False);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.BackupTreeViewDragOver(Sender, Source: TObject; X, Y: Integer; State: TDragState;
  var Accept: Boolean);

begin
  if (Source = SchemataFrame.CatalogVST) then
    Accept := True;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.BackupTreeViewDragDrop(Sender, Source: TObject; X, Y: Integer);

begin
  if (Source = SchemataFrame.CatalogVST) then
  begin
    AddSchemaToBackupBtnClick(self);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.SchemaTreeViewExpanding(Sender: TObject; Node: TTntTreeNode; var AllowExpansion: Boolean);

begin
  //Expansion not allowed in Backup/Restore
  AllowExpansion := not (LockSchemaTreeExpansion);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.BackupTreeViewPopupMenuPopup(Sender: TObject);

begin
  //Tables cannot be removed
  RemoveBackupNodeMI.Enabled := True;
  if (BackupTreeView.Selected <> nil) then
    if (BackupTreeView.Selected.Data <> nil) then
      if (TBackupNode(BackupTreeView.Selected.Data).GetObjType = BOTTable) then
        RemoveBackupNodeMI.Enabled := False;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.RemoveBackupNodeMIClick(Sender: TObject);

begin
  if (BackupTreeView.Selected <> nil) then
    if (BackupTreeView.Selected.Data <> nil) then
    begin
      RemoveTreeNode(BackupTreeView.Selected);

      DoChange(self);

      StartBackupBtn.Enabled := (GetSelectedTablesCount > 0);
    end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.RemoveTreeNode(Node: TTntTreeNode);

begin
  //Remove Child nodes first
  while (Node.Count > 0) do
    RemoveTreeNode(Node.Item[0]);

  TBackupNode(Node.Data).Free;

  TTntTreeView(Node.TreeView).Items.Delete(Node);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.SchemaTreeViewCollapsing(Sender: TObject; Node: TTntTreeNode; var AllowCollapse: Boolean);

begin
  if (Node.Data <> nil) then
    if (TObject(Node.Data) is TMYX_CATALOG) then
      AllowCollapse := False;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.AddSchemaToBackupBtnClick(Sender: TObject);

var
  i, j: Integer;
  Selection: TNodeArray;
  NodeData: ^TObject;

begin
  Selection := SchemataFrame.CatalogVST.GetSortedSelection(False);

  for i := 0 to SchemataFrame.CatalogVST.SelectedCount - 1 do
  begin
    NodeData := SchemataFrame.CatalogVST.GetNodeData(Selection[i]);
    if (NodeData <> nil) then
      if (NodeData^ <> nil) then
      begin
        if (NodeData^ is TMYX_SCHEMA) then
          AddSchemaToBackup(TMYX_SCHEMA(NodeData^));

        if (NodeData^ is TMYX_CATALOG) then
          for j := 0 to TMYX_CATALOG(NodeData^).Schemata.Count - 1 do
            AddSchemaToBackup(TMYX_CATALOG(NodeData^).Schemata[j]);
      end;
  end;

  DoChange(self);

  StartBackupBtn.Enabled := (GetSelectedTablesCount > 0);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.SchemataFrameSchemaTreeViewMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
  
begin
  if (Button = mbLeft) then
  begin
    MousePos := Mouse.CursorPos;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.StartBackupBtnClick(Sender: TObject);

var
  SaveDialog: TTntSaveDialog;
  filename: WideString;
  AdminBackupProgressForm: TAdminBackupProgressForm;

begin
  SaveDialog := TTntSaveDialog.Create(self);
  try
    if (ApplicationDM.Options.AddDateTimeToBackupFiles) then
      SaveDialog.FileName := ProjectNameEd.Text +
        FormatDateTime(' yyyymmdd hhnn', Now) + '.sql'
    else
      SaveDialog.FileName := ProjectNameEd.Text + '.sql';

    SaveDialog.Filter := 'SQL Backup Files|*.sql|All Files|*.*';

    if (not (SaveDialog.Execute)) then
      Exit;

    filename := SaveDialog.Filename;
  finally
    SaveDialog.Free;
  end;

  AdminBackupProgressForm := TAdminBackupProgressForm.Create(Self, True, ProjectNameEd.Text);
  AdminBackupProgressForm.Show;

  BackupThread := TBackupThread.Create(self, AdminBackupProgressForm, filename);
  try
    // Thread will be freed after execution
    BackupThread.FreeOnTerminate := True;

    BackupThread.Resume;
  except
    BackupThread.Free;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.BackupTreeViewChanging(Sender: TObject; Node: TTreeNode; var AllowChange: Boolean);

begin
  if (BackupTreeView.Items.Count > 0) then
    if (Node = BackupTreeView.Items[0]) then
      AllowChange := False;

  if GetSelectedTablesCount < BackupTreeView.Items.Count then
  begin
    CompleteSchematasCBox.Checked := false;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.ClearControls(EnableControls: Boolean);

begin
  InitControls := True;

  DisableEnableControls(BackupContentTabSheet, EnableControls);
  DisableEnableControls(AdvancedOptionsTabSheet, EnableControls);
  DisableEnableControls(ScheduleTabSheet, EnableControls);
  DisableEnableScheduleControls;

  ApplyBtn.Enabled := False;
  //DiscardBtn.Enabled:=False;
  StartBackupBtn.Enabled := False;

  ProjectNameEd.Text := '';
  ScheduleTargetDirEd.Text := '';
  ScheduleFilenamePrefixEd.Text := '';

  NoCreatesCBox.Checked := False;
  NoExtendedInsertsCBox.Checked := False;
  AddDropTableCBox.Checked := True;
  CompleteInsertsCBox.Checked := True;
  CommentCBox.Checked := True;
  FullPathCBox.Checked := False;
  AnsiQuotesCBox.Checked := False;

  SingleTransRButton.Checked := True;

  DisableKeysCBox.Checked := True;
  CompleteSchematasCBox.Checked := False;
  CompatibilityCheckbox.Checked := False;

  BackupFileTypesLU.ItemIndex := 0;

  ClearTreeView(BackupTreeView, myx_ndt_tobject);

  ScheduleCBox.Checked := False;
  ScheduleTypeCbox.ItemIndex := 1;
  SchedulePageControl.ActivePageIndex := 1;

  ScheduleMondayCBox.Checked := False;
  ScheduleTuesdayCBox.Checked := False;
  ScheduleWednesdayCBox.Checked := False;
  ScheduleThursdayCBox.Checked := False;
  ScheduleFridayCBox.Checked := False;
  ScheduleSaturdayCBox.Checked := False;
  ScheduleSundayCBox.Checked := False;

  ScheduleTimeEd.Text := '23:00';

  InitControls := False;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.DisableEnableScheduleControls;

begin
  ScheduleTargetDirCaptionLbl.Enabled := ScheduleCBox.Checked;
  ScheduleTargetDirEd.Enabled := ScheduleCBox.Checked;
  ScheduleBrowseTargetDirBtn.Enabled := ScheduleCBox.Checked;
  ScheduleTargetDirLbl.Enabled := ScheduleCBox.Checked;
  SchedulePrefixCaptionLbl.Enabled := ScheduleCBox.Checked;
  ScheduleFilenamePrefixEd.Enabled := ScheduleCBox.Checked;
  SchedulePrefixLbl.Enabled := ScheduleCBox.Checked;

  DisableEnableControls(ExecutionTimeGBox, ScheduleCBox.Checked);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.NewProjectBtnClick(Sender: TObject);

begin
  DiscardChanges(True);

  OriginalProjectName := '';

  BackupRestorePageControl.ActivePageIndex := 0;

  ProjectListView.Selected := nil;
  ClearControls(True);

  ProjectNameEd.Text := 'New Project';
  ProjectNameEd.SetFocus;
  ProjectNameEd.SelectAll;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.ProjectNameEdExit(Sender: TObject);

begin
  if (ExtractText(ProjectNameEd.Text) <> ProjectNameEd.Text) then
  begin
    ProjectNameEd.SetFocus;
    raise EInOutError.Create(_('You can only use alphanumeric characters for project name.'));
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.ScheduleBrowseTargetDirBtnClick(Sender: TObject);

var
  SaveDialog: TTntSaveDialog;

begin
  SaveDialog := TTntSaveDialog.Create(self);
  SaveDialog.Title := _('Choose Target Directory');
  SaveDialog.InitialDir := '';
  SaveDialog.FileName := _('Browse to the Target Directory');

  if (SaveDialog.Execute) then
  begin
    ScheduleTargetDirEd.Text := ExtractFilePath(SaveDialog.FileName);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.DoChange(Sender: TObject);
var Node: TTntTreeNode;
begin
  if (not (InitControls)) then
  begin
    ApplyBtn.Enabled := True;
    //DiscardBtn.Enabled:=True;
    if Sender = CompleteSchematasCBox then
    begin
      if CompleteSchematasCBox.Checked then
      begin
        Node := BackupTreeView.Items.GetFirstNode;
        repeat
        begin
          if(TObject(Node.Data) is TBackupNode)then
          begin
            TBackupNode(Node.Data).SetSelectState(BCTAll);
          end;
          Node := Node.GetNext;
        end
        until Node = nil;
      end;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.DiscardChanges(AskBeforeDiscard: Boolean);

var
  res: Integer;

begin
  if (not (ApplyBtn.Enabled)) then
  begin
    OriginalProjectName := '';

    Exit;
  end;

  if (AskBeforeDiscard) then
  begin
    res := ShowModalDialog(_('Apply changes?'),
      _('The current backup profile has been changed. ' +
      'Do you want to apply your changes?'),
      myx_mtConfirmation,
      _('Yes') + #13#10 + _('No') + #13#10 + _('Abort'));
    if (res = 1) then
    begin
      ApplyChanges;
      Exit;
    end
    else
      if (res = 3) then
        Abort;
  end;

  if (ProjectListView.Selected = nil) then
    ClearControls(False)
  else
  begin
    ApplyBtn.Enabled := False;
    //DiscardBtn.Enabled:=False;

    //Reload profile
    LoadProfile(UTF8Decode(PChar(ProjectListView.Selected.Data)));
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TAdminBackupForm.GetOptions: Integer;

var
  options: Integer;

begin
  options := 0;

  AddOption(options, MYX_B_NO_CREATES, NoCreatesCBox.Checked);
  AddOption(options, MYX_B_NO_EXTENDED_INSERT, NoExtendedInsertsCBox.Checked);
  AddOption(options, MYX_B_ADD_DROP_TABLE, AddDropTableCBox.Checked);
  AddOption(options, MYX_B_COMPLETE_INSERTS, CompleteInsertsCBox.Checked);
  AddOption(options, MYX_B_COMMENT, CommentCBox.Checked);
  AddOption(options, MYX_B_DONT_WRITE_FULL_PATH, FullPathCBox.Checked);
  AddOption(options, MYX_B_ANSI_QUOTES, AnsiQuotesCBox.Checked);
  AddOption(options, MYX_B_LOCK_ALL_TABLES, LockAllTablesRButton.Checked);
  AddOption(options, MYX_B_SINGLE_TRANSACTION, SingleTransRButton.Checked);
  AddOption(options, MYX_B_DISABLE_KEYS, DisableKeysCBox.Checked);
  AddOption(options, MYX_B_COMPLETE_SCHEMATAS, CompleteSchematasCBox.Checked);
  AddOption(options, MYX_B_COMPATIBILITY_MODE, CompatibilityCheckbox.Checked);
  AddOption(options, MYX_B_POINT_IN_TIME_BACKUP, PointInTimeRBtn.Checked);

  Result := options;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.ApplyChanges;

var
  Profile: TMYX_BACKUP_PROFILE;
  Content: TMYX_BACKUP_CONTENT;
  BackupType: MYX_BACKUP_TYPE;
  filename: WideString;

  S: WideString;
  Index: Word;
  Unknown: IUnknown;
  Task: ITask;
  PersistFile: IPersistFile;
  Trigger: ITaskTrigger;
  TriggerData: TASK_TRIGGER;
  Result: HResult;

  Year,
  Month,
  Day,
  Second,
  MilliSecond: Word;
  DaysFlag: Word;
  DeleteJob: Boolean;
  
begin
  if ScheduleCBox.Checked then
  begin
    if Trim(ScheduleTargetDirEd.Text) = '' then
      raise EInOutError.Create(_('You have to specify a target directory for the backup.'));

    if Trim(ScheduleFilenamePrefixEd.Text) = '' then
      raise EInOutError.Create(_('You have to specify a filename prefix for the backup.'));

    if GetSelectedTablesCount = 0 then
      raise EInOutError.Create(_('You have to specify some content for the backup.'));
  end;

  // Get Backup FileType.
  case BackupFileTypesLU.ItemIndex of
    0:
      BackupType := MYX_BT_SQL_SCRIPT;
  else
    BackupType := MYX_BT_SQL_SCRIPT;
  end;

  Content := GetBackupContent(BackupTreeView);
  try
    Profile := TMYX_BACKUP_PROFILE.create(ProjectNameEd.Text, '', GetOptions, BackupType, Content.get_record_pointer);
    try
      filename := BackupProfileDir + ExtractText(ProjectNameEd.Text) + '.mbp';

      // Detect project name change.
      if (OriginalProjectName <> ProjectNameEd.Text) then
        if (FileExists(BackupProfileDir + OriginalProjectName + '.mbp')) then
        begin
          DeleteFile(filename);
          DeleteFile(ChangeFileExt(filename, '.bak'));
          RenameFile(BackupProfileDir + OriginalProjectName + '.mbp', ChangeFileExt(filename, '.bak'));
        end;

      if (FileExists(filename)) then
      begin
        DeleteFile(ChangeFileExt(filename, '.bak'));
        RenameFile(filename, ChangeFileExt(filename, '.bak'));
      end;

      myx_save_profile(ExtractFileName(filename), ExtractFilePath(filename), Profile.get_record_pointer);
      OriginalProjectName := Profile.profile_name;
    finally
      Profile.Free;
    end;

  finally
    Content.Free;
  end;

  DeleteJob := not ScheduleCBox.Checked;
  
  // If the user scheduled a project.
  if not DeleteJob then
  begin
    // Make sure the service is running...
    StartScheduler;

    // Reuse existing task entry if there is one.
    Result := FTaskScheduler.Activate(PWideChar(OriginalProjectName), IID_ITask, Unknown);
    if Result = S_OK then
      Task := Unknown as ITask
    else
    begin
      Result := FTaskScheduler.NewWorkItem(PWideChar(OriginalProjectName), CLSID_CTask, IID_IScheduledWorkItem, Unknown);
      if Result = S_OK then
        Task := Unknown as ITask
      else
        raise EMyxSystemError.Create(_('Cannot create new backup task.'), Result);
    end;

    if TPasswordDialog.DetermineUserCredentials(Task) then
    begin
      S := Application.ExeName;
      Task.SetApplicationName(PWideChar(S));
      Task.SetFlags(0);

      S := '"-UD' + MYXCommonOptions.UserDataDir + '" ' +
           '"-c' + MySQLConn.user_connection.connection_name + '" ' +
           '"-bp' + ProjectNameEd.Text + '" ' +
           '"-bt' + Trim(ScheduleTargetDirEd.Text) + '" ' +
           '"-bx' + Trim(ScheduleFilenamePrefixEd.Text) + '"';
      Task.SetParameters(PWideChar(S));
      Task.SetWorkingDirectory(PWideChar(MYXCommonOptions.UserDataDir));
      Task.SetPriority(NORMAL_PRIORITY_CLASS);
      Task.SetCreator('MySQL Administrator');
      Task.SetMaxRunTime(INFINITE);

      // Task is ready now. Add a trigger when to run it.
      // Delete existing triggers before adding the new one.
      // TODO: Allow working with more than one trigger.
      Task.GetTriggerCount(Index);
      while Index > 0 do
      begin
        Task.DeleteTrigger(0);
        Dec(Index);
      end;
      Result := Task.CreateTrigger(Index, Trigger);
      if Result <> S_OK then
        raise EMyxSystemError.Create(_('Cannot create new backup task.'), Result);

      FillChar(TriggerData, SizeOf(TriggerData), 0);
      TriggerData.cbTriggerSize := SizeOf(TriggerData);

      // Start day, month and year is today.
      DecodeDate(Now, TriggerData.wBeginYear, TriggerData.wBeginMonth, TriggerData.wBeginDay);
      // Execution time must be splittet as well.
      DecodeDateTime(StrToDateTime(ScheduleTimeEd.Text), Year, Month, Day, TriggerData.wStartHour, TriggerData.wStartMinute,
        Second, Millisecond);

      case ScheduleTypeCbox.ItemIndex of
        0: // Daily.
          begin
            TriggerData.TriggerType := TASK_TIME_TRIGGER_DAILY;
            TriggerData.Type_.Daily.DaysInterval := 1;
          end;
        1: // Weekly at certain days.
          begin
            TriggerData.TriggerType := TASK_TIME_TRIGGER_WEEKLY;
            TriggerData.Type_.Weekly.WeeksInterval := 1;
            DaysFlag := 0;
            if ScheduleMondayCBox.Checked then
              DaysFlag := DaysFlag or TASK_MONDAY;
            if ScheduleTuesdayCBox.Checked then
              DaysFlag := DaysFlag or TASK_TUESDAY;
            if ScheduleWednesdayCBox.Checked then
              DaysFlag := DaysFlag or TASK_WEDNESDAY;
            if ScheduleThursdayCBox.Checked then
              DaysFlag := DaysFlag or TASK_THURSDAY;
            if ScheduleFridayCBox.Checked then
              DaysFlag := DaysFlag or TASK_FRIDAY;
            if ScheduleSaturdayCBox.Checked then
              DaysFlag := DaysFlag or TASK_SATURDAY;
            if ScheduleSundayCBox.Checked then
              DaysFlag := DaysFlag or TASK_SUNDAY;
            TriggerData.Type_.Weekly.rgfDaysOfTheWeek := DaysFlag;
          end;
        2: // Monthly.
          begin
            TriggerData.TriggerType := TASK_TIME_TRIGGER_MONTHLYDATE;

            // Create bitmask for day of month (zero-based) and add all months for execution.
            TriggerData.Type_.MonthlyDate.rgfDays := 1 shl ScheduleDayOfMonthCBox.ItemIndex;
            TriggerData.Type_.MonthlyDate.rgfMonths := TASK_JANUARY or TASK_FEBRUARY or TASK_MARCH or TASK_APRIL or
              TASK_MAY or TASK_JUNE or TASK_JULY or TASK_AUGUST or TASK_SEPTEMBER or TASK_OCTOBER or TASK_NOVEMBER or TASK_DECEMBER;
          end;
      end;

      Trigger.SetTrigger(TriggerData);

      // Now that the task is setup save it to make it active.
      PersistFile := Task as IPersistFile;
      PersistFile.Save(nil, True);
    end
    else
      DeleteJob := True;
  end;

  if DeleteJob then
  begin
    FTaskScheduler.Delete(PWideChar(OriginalProjectName));
    ScheduleCBox.Checked := False;
    DisableEnableScheduleControls;
  end;

  ApplyBtn.Enabled := False;
  RefreshBackupProfiles(filename);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.DiscardBtnClick(Sender: TObject);

begin
  DiscardChanges(False);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.ApplyBtnClick(Sender: TObject);

begin
  ApplyChanges;
end;

//----------------------------------------------------------------------------------------------------------------------

function StrMatchesSearchStr(Str: WideString; searchStr: WideString): Boolean;

begin
  Result := True;

  if (searchStr <> '') then
    Result := myx_match_pattern(Str, searchStr, 0, 1) <> 0;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.RefreshBackupProfiles(SelectProfile: WideString);

var
  CurrentProfileName: WideString;
  SR: TSearchRec;
  filename: PChar;
  i: Integer;

begin
  CurrentProfileName := SelectProfile;

  InitControls := True;
  ProjectListView.Items.BeginUpdate;
  try
    ClearListView(ProjectListView, myx_ndt_pointer);

    if (FindFirst(BackupProfileDir + '*.mbp', faAnyFile and not
      (8 {faVolumeID} or faDirectory), sr) = 0) then
    begin
      repeat
        begin
          if (AdvancedEditFrame.SearchEd.Text <> '') then
            if (not (StrMatchesSearchStr(sr.Name, AdvancedEditFrame.SearchEd.Text))) then
              continue;

          GetMem(filename, Length(UTF8Encode(BackupProfileDir + sr.Name)) + 1);
          StrPCopy(filename, UTF8Encode(BackupProfileDir + sr.Name));

          AddListViewItem(ProjectListView, nil,
            ExtractFileName(Copy(sr.Name, 1, Length(sr.Name) - Length(ExtractFileExt(sr.Name)))),
            -1, filename);
        end;
      until FindNext(sr) <> 0;

      FindClose(sr);
    end;

    for i := 0 to ProjectListView.Items.Count - 1 do
      if (ProjectListView.Items[i].Data <> nil) then
        if (CompareText(CurrentProfileName,
          UTF8Decode(PChar(ProjectListView.Items[i].Data))) = 0) then
        begin
          ProjectListView.Selected := ProjectListView.Items[i];
          break;
        end;
  finally
    ProjectListView.Items.EndUpdate;
    InitControls := False;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TAdminBackupForm.InOptions(options: Integer; option: MYX_BACKUP_OPTIONS): Boolean;

begin
  Result := ((options and ord(option)) = Ord(option));
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.AddOption(var Options: Integer; option: MYX_BACKUP_OPTIONS; Add: Boolean);

begin
  if Add then
    Options := Options + Ord(option);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.LoadProfile(filename: WideString);

  //----------------------------------------------------------------------------

  function ExtractParameter(S: WideString; Start: Integer): WideString;

  // Extracts a parameter from the given string by scanning for the closing
  // quotation mark starting at Start. Search stops also on string end.
  // Note: S must at least contain 1 character.

  var
    Head, Tail: PWideChar;

  begin
    Head := @S[Start];
    // Scan for closing quotation mark.
    Tail := Head + 1;
    while not (Tail^ in [WideChar('"'), WideChar(#0)]) do
      Inc(Tail);
    SetString(Result, Head, Tail - Head);
  end;

  //----------------------------------------------------------------------------

var
  Profile: TMYX_BACKUP_PROFILE;
  PProfile: PMYX_BACKUP_PROFILE;
  error: MYX_ADMIN_LIB_ERROR;

  I: Integer;
  Raw: PWideChar;
  Index: Integer;
  Command: WideString;
  Result: HResult;
  Task: ITask;
  Unknown: IUnknown;
  Trigger: ITaskTrigger;
  TriggerData: TASK_TRIGGER;
  DaysFlag: Word;
  Mask: DWORD;

begin
  if MySQLConn.Connected then
  begin
    PProfile := myx_load_profile(ExtractFileName(filename), ExtractFilePath(filename), @error);
    if (error <> MYX_ADMIN_NO_ERROR) or (PProfile = nil) then
      raise EMyxAdminLibError.Create(_('Error while loading backup profile.'), Ord(error), filename);

    InitControls := True;
    try
      Profile := TMYX_BACKUP_PROFILE.create(PProfile);
    finally
      myx_free_profile(PProfile);
    end;

    try
      //Store original project name
      OriginalProjectName := Profile.profile_name;

      ProjectNameEd.Text := Profile.profile_name;

      BackupFileTypesLU.ItemIndex := Ord(Profile.backup_type);

      NoCreatesCBox.Checked := InOptions(Profile.options, MYX_B_NO_CREATES);
      NoExtendedInsertsCBox.Checked := InOptions(Profile.options, MYX_B_NO_EXTENDED_INSERT);
      AddDropTableCBox.Checked := InOptions(Profile.options, MYX_B_ADD_DROP_TABLE);
      CompleteInsertsCBox.Checked := InOptions(Profile.options, MYX_B_COMPLETE_INSERTS);
      CommentCBox.Checked := InOptions(Profile.options, MYX_B_COMMENT);
      FullPathCBox.Checked := InOptions(Profile.options, MYX_B_DONT_WRITE_FULL_PATH);
      AnsiQuotesCBox.Checked := InOptions(Profile.options, MYX_B_ANSI_QUOTES);
      NormalBackupRadioButton.Checked := True;
      LockAllTablesRButton.Checked := InOptions(Profile.options, MYX_B_LOCK_ALL_TABLES);
      SingleTransRButton.Checked := InOptions(Profile.options, MYX_B_SINGLE_TRANSACTION);
      DisableKeysCBox.Checked := InOptions(Profile.options, MYX_B_DISABLE_KEYS);
      CompleteSchematasCBox.Checked := InOptions(Profile.options, MYX_B_COMPLETE_SCHEMATAS);
      CompatibilityCheckbox.Checked := InOptions(Profile.options, MYX_B_COMPATIBILITY_MODE);
      PointInTimeRBtn.Checked := InOptions(Profile.options, MYX_B_POINT_IN_TIME_BACKUP);

      BuildContentTreeFromBackupContent(MySQLConn.MySQL,
        BackupTreeView,
        Profile.backup_content,
        SchemataFrame,
        CompleteSchematasCBox.Checked);

      // Check if it is scheduled
      Result := FTaskScheduler.Activate(PWideChar(ProjectNameEd.Text), IID_ITask, Unknown);
      if Result = S_OK then
      begin
        ScheduleCBox.Checked := True;

        Task := Unknown as ITask;

        // For now we ignore result values of the getters. If something fails then we will just
        // have strange or no values in the controls. This must be then fixed anyway.
        Task.GetParameters(Raw);
        Command := Raw;
        CoTaskMemFree(Raw);
        
        ScheduleTargetDirEd.Text := '';
        Index := Pos('"-bt', Command);
        if Index > 0 then
          ScheduleTargetDirEd.Text := ExtractParameter(Command, Index + 4);

        ScheduleFilenamePrefixEd.Text := '';
        Index := Pos('"-bx', Command);
        if Index > 0 then
          ScheduleFilenamePrefixEd.Text := ExtractParameter(Command, Index + 4);

        // TODO: Currently only one trigger can be handled here. Enhance this for any trigger count.
        Result := Task.GetTrigger(0, Trigger);
        if Result = S_OK then
        begin
          Trigger.GetTrigger(TriggerData);
          ScheduleTimeEd.Text := Format('%.2d:%.2d', [TriggerData.wStartHour, TriggerData.wStartMinute]);

          case TriggerData.TriggerType of
            TASK_TIME_TRIGGER_DAILY:
              begin
                ScheduleTypeCbox.ItemIndex := 0;
              end;
            TASK_TIME_TRIGGER_WEEKLY:
              begin
                ScheduleTypeCbox.ItemIndex := 1;

                DaysFlag := TriggerData.Type_.Weekly.rgfDaysOfTheWeek;
                ScheduleMondayCBox.Checked := (DaysFlag and TASK_MONDAY) <> 0;
                ScheduleTuesdayCBox.Checked := (DaysFlag and TASK_TUESDAY) <> 0;
                ScheduleWednesdayCBox.Checked := (DaysFlag and TASK_WEDNESDAY) <> 0;
                ScheduleThursdayCBox.Checked := (DaysFlag and TASK_THURSDAY) <> 0;
                ScheduleFridayCBox.Checked := (DaysFlag and TASK_FRIDAY) <> 0;
                ScheduleSaturdayCBox.Checked := (DaysFlag and TASK_SATURDAY) <> 0;
                ScheduleSundayCBox.Checked := (DaysFlag and TASK_SUNDAY) <> 0;
              end;
            TASK_TIME_TRIGGER_MONTHLYDATE:
              begin
                ScheduleTypeCbox.ItemIndex := 2;
                ScheduleDayOfMonthCBox.ItemIndex := 0;
                
                // Scan for the first set bit. We cannot handle more than one day currently.
                Mask := 1;
                for I := 0 to 30 do
                begin
                  if (TriggerData.Type_.MonthlyDate.rgfDays and Mask) <> 0 then
                  begin
                    ScheduleDayOfMonthCBox.ItemIndex := I;
                    Break;
                  end;
                  Mask := Mask shl 1;
                end;
              end;
          end;
        end;
        
        ScheduleTypeCboxCloseUp(self);
      end
      else
        ScheduleCBox.Checked := False;

      DisableEnableScheduleControls;
    finally
      InitControls := False;
      Profile.Free;
    end;

    StartBackupBtn.Enabled := (GetSelectedTablesCount > 0);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TAdminBackupForm.GetSelectedTablesCount: Integer;

var
  i, count: Integer;

begin
  count := 0;

  for i := 0 to BackupTreeView.Items.Count - 1 do
    if (BackupTreeView.Items[i].Data <> nil) then
      if (TObject(BackupTreeView.Items[i].Data) is TBackupNode) then
        if (TBackupNode(BackupTreeView.Items[i].Data).GetObjType = BOTTable) and
          (TBackupNode(BackupTreeView.Items[i].Data).GetSelectState = BCTAll) then
          inc(count);

  Result := count;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.RefreshProfilesMIClick(Sender: TObject);

begin
  RefreshBackupProfiles;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.DeleteProfileMIClick(Sender: TObject);

var
  I: Integer;
  Filename: WideString;
  Profile: PMYX_BACKUP_PROFILE;
  error: MYX_ADMIN_LIB_ERROR;
  S: WideString;

begin
  if (ShowModalDialog(_('Delete Profiles?'), _('Are you sure you want to delete the selected profiles?'),
    myx_mtConfirmation, _('Yes') + #13#10 + _('No')) = 1) then
  begin
    InitControls := True;
    try
      for I := 0 to ProjectListView.Items.Count - 1 do
        if ProjectListView.Items[i].Selected then
          if Assigned(ProjectListView.Items[I].Data) then
          begin
            Filename := UTF8Decode(PChar(ProjectListView.Items[I].Data));
            Profile := myx_load_profile(ExtractFileName(Filename), ExtractFilePath(Filename), @error);

            // TODO: Profile still uses ANSI strings.
            S := Profile.profile_name;
            FTaskScheduler.Delete(PWideChar(S));

            DeleteFile(filename);
          end;

      RefreshBackupProfiles;

      ProjectListView.Selected := nil;
      ClearControls(False);
    finally
      InitControls := False;
    end;
  end;

end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.ProfilesPopupMenuPopup(Sender: TObject);

begin
  DeleteProfileMI.Enabled := ProjectListView.SelCount > 0;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.ClearCompleteContentMIClick(Sender: TObject);

begin
  ClearTreeView(BackupTreeView, myx_ndt_tobject);

  DoChange(self);

  StartBackupBtn.Enabled := (GetSelectedTablesCount > 0);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.ProjectListViewSelectItem(Sender: TObject; Item: TListItem; Selected: Boolean);

var
  filename: WideString;

begin
  if not (InitControls) then
  begin
    if (ProjectListView.Selected <> nil) and (MySQLConn.Connected) then
    begin
      if Assigned(ProjectListView.Selected.Data) then
      begin
        filename := UTF8Decode(PChar(ProjectListView.Selected.Data));

        if ApplyBtn.Enabled then
          DiscardChanges(True);

        LoadProfile(filename);

        DisableEnableControls(BackupContentTabSheet, True);
        DisableEnableControls(AdvancedOptionsTabSheet, True);
        DisableEnableControls(ScheduleTabSheet, True);
        DisableEnableScheduleControls;
      end;
    end
    else
      ClearControls(False);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

constructor TBackupThread.Create(AdminBackupForm: TAdminBackupForm; AdminBackupProgressForm: TAdminBackupProgressForm;
  filename: WideString);

begin
  inherited Create(True);

  self.AdminBackupForm := AdminBackupForm;
  self.AdminBackupProgressForm := AdminBackupProgressForm;
  self.filename := filename;

  PMySQL := myx_mysql_init();
  if (PMySQL = nil) then
    raise EMyxError.Create(_('Error while allocating memory for MySQL Struct in Backup Thread.'));

  if (myx_connect_to_instance(
    AdminBackupForm.MySQLConn.User_Connection.get_record_pointer,
    PMySQL) <> 0) then
    raise EMyxError.Create(_('Backup Thread cannot connect to MySQL'));
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TBackupThread.Destroy;

begin
  CleanUp;

  inherited Destroy;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TBackupThread.CleanUp;

begin
  myx_mysql_close(PMySQL);

  AdminBackupForm.BackupThread := nil;

  Synchronize(FreeAdminBackupProgressForm);
end;

//----------------------------------------------------------------------------------------------------------------------

function BackupProgress(current_table_name: PChar; num_tables: Integer; num_tables_processed: Integer;
  num_rows: Integer; num_rows_processed: Integer; user_data: Pointer): Integer;

var
  PSender: TBackupThread;
  
begin
  PSender := user_data;

  with PSender.AdminBackupProgressForm do
  begin
    PSender.current_table_name := current_table_name;
    PSender.num_tables := num_tables;
    PSender.num_tables_processed := num_tables_processed;
    PSender.num_rows := num_rows;
    PSender.num_rows_processed := num_rows_processed;
  end;

  PSender.Synchronize(PSender.UpdateProgressForm);

  Result := Ord(PSender.AdminBackupProgressForm.Stopping);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TBackupThread.Execute;

var
  backup_content: TMYX_BACKUP_CONTENT;

begin
  with AdminBackupForm do
  begin
    backup_content := GetBackupContent(BackupTreeView);
    try
      BackupError := myx_make_backup(PMySQL, filename, backup_content.get_record_pointer, MYX_BT_SQL_SCRIPT,
        GetOptions, 256, BackupProgress, Self);

      if (BackupError = MYX_BACKUP_NO_ERROR) then
        Synchronize(FinishedBackup)
      else
        Synchronize(ShowError);

    finally
      backup_content.Free;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TBackupThread.UpdateProgressForm;

begin
  with AdminBackupProgressForm do
  begin
    CurrentTableNameLbl.Caption :=
      current_table_name;

    TablesLbl.Caption := IntToStr(num_tables);
    CurrentTableLbl.Caption := IntToStr(num_tables_processed);
    TablesProgressBar.Max := num_tables;
    TablesProgressBar.Position := num_tables_processed - 1;

    TotalRowsLbl.Caption := IntToStr(num_rows);
    CurrentRowLbl.Caption := IntToStr(num_rows_processed);
    RowProgressBar.Max := num_rows;
    RowProgressBar.Position := num_rows_processed - 1;

    Update;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TBackupThread.FinishedBackup;

begin
  //Set Progress Bars to 100%
  num_tables_processed := num_tables + 1;
  num_rows_processed := num_rows + 1;

  UpdateProgressForm;

  ShowModalDialog(_('Backup finished'),
    _('The Backup was finished successfully.') + #13#10#13#10 +
    Format(_('The File %s has been created.'), [filename]),
    myx_mtInformation, _('OK'));
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TBackupThread.FreeAdminBackupProgressForm;

begin
  AdminBackupProgressForm.Free;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TBackupThread.ShowError;

var
  ErrorTxt: WideString;

begin
  if (ord(BackupError) = -1) then
  begin
    ShowModalDialog(_('Backup stopped'),
      _('The Backup has been stopped.'),
      myx_mtConfirmation, _('OK'));
  end
  else
  begin
    ErrorTxt := GetBackupErrorMsg(BackupError);
    case BackupError of
      MYX_BACKUP_SERVER_ERROR:
        ErrorTxt := Format(ErrorTxt, [myx_mysql_errno(AdminBackupForm.MySQLConn.MySQL),
          myx_mysql_error(AdminBackupForm.MySQLConn.MySQL)]);
      MYX_BACKUP_CANT_OPEN_FILE:
        ErrorTxt := Format(ErrorTxt, [filename]);
      MYX_BACKUP_OUTPUTDEVICE_FULL:
        ErrorTxt := Format(ErrorTxt, [filename]);
    end;

    ShowModalDialog(_('Backup error'),
      _('An Error occured while executing the backup.') + #13#10#13#10 +
      ErrorTxt,
      myx_mtConfirmation, _('OK'));
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.Disconnected(var Message: TMessage);

begin
  ClearControls(False);

  NewProjectBtn.Enabled := False;

  ProjectListView.Selected := nil;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.Reconnected(var Message: TMessage);

begin
  NewProjectBtn.Enabled := True;

  SchemataFrame.FillSchemaTree;

  ProjectListView.Selected := nil;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.SchemaListChanged(var Message: TMessage);

begin
  SchemataFrame.ReloadSchemaTree;
  SchemataFrame.FillSchemaTree;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.AdvancedEditFrameSearchEdChange(Sender: TObject);

begin
  AdvancedEditFrame.SearchEdChange(Sender);

  RefreshBackupProfiles;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.ScheduleCBoxClick(Sender: TObject);

var
  OptionsForm: TOptionsForm;

begin
  // Check if service is available
  StartScheduler;
  
  if (ScheduleCBox.Checked) then
  begin
    if (MYXCommonOptions.PasswordStorageType <> MYX_PASSWORD_OBSCURED) then
      if (ShowModalDialog(_('Attention!'),
        _('You have to enable the Password Storage option on the ' +
        'programm''s General Options tabsheet and select Obscured as ' +
        'password storage method to execute scheduled ' +
        'backups. Otherwise the backup task cannot log into ' +
        'the database.'),
        myx_mtError, _('OK') + #13#10 + _('Open Options')) = 2) then
      begin
        OptionsForm := TOptionsForm.Create(self, TAdminOptionPagesForm.Create(self));
        try
          OptionsForm.ShowModal;
        finally
          OptionsForm.Free;
        end;
      end;

    if (Trim(ScheduleTargetDirEd.Text) = '') then
      ScheduleTargetDirEd.Text := 'C:\';

    if (Trim(ScheduleFilenamePrefixEd.Text) = '') then
      ScheduleFilenamePrefixEd.Text := ProjectNameEd.Text;
  end;

  DisableEnableScheduleControls;
  DoChange(self);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.ScheduleTypeCboxCloseUp(Sender: TObject);

begin
  SchedulePageControl.ActivePageIndex := ScheduleTypeCbox.ItemIndex;

  DoChange(self);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.SchemataFrameCatalogVSTMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);

begin
  if (ssLeft in Shift) then
  begin
    //Simulate Treshold before Drag
    if ((abs(MousePos.X - Mouse.CursorPos.X) > 5) or
      (abs(MousePos.Y - Mouse.CursorPos.Y) > 5)) then
      SchemataFrame.CatalogVST.BeginDrag(True);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.SchemataFrameCatalogVSTDblClick(Sender: TObject);

begin
  //Prevent user from doubleclicking
  Abort;
end;

//----------------------------------------------------------------------------------------------------------------------

function TAdminBackupForm.StartScheduler: DWORD;

var
  hsc: SC_HANDLE;
  hSchSvc: SC_HANDLE;
  SvcStatus: TServiceStatus;
  ServiceArgVectors: PChar;

begin
  hsc := OpenSCManager(nil, nil, SC_MANAGER_CONNECT);

  if hsc = 0 then
    Result := GetLastError
  else
  begin
    hSchSvc := OpenService(hsc, 'Schedule', SERVICE_START or SERVICE_QUERY_STATUS);
    CloseServiceHandle(hsc);

    if hSchSvc = 0 then
      Result := GetLastError
    else
    begin
      if not QueryServiceStatus(hSchSvc, SvcStatus) then
      begin
        CloseServiceHandle(hSchSvc);
        Result := GetLastError;
      end
      else
      begin
        if SvcStatus.dwCurrentState = SERVICE_RUNNING then
        begin
          CloseServiceHandle(hSchSvc);
          Result := ERROR_SUCCESS;
        end
        else
        begin
          ServiceArgVectors := nil;
          if not StartService(hSchSvc, 0, ServiceArgVectors) then
          begin
            CloseServiceHandle(hSchSvc);
            Result := GetLastError;
          end
          else
          begin
            CloseServiceHandle(hSchSvc);
            Result := ERROR_SUCCESS;
          end;
        end;
      end;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.BackupTreeViewCompare(Sender: TObject; Node1, Node2: TTreeNode; Data: Integer;
  var Compare: Integer);

begin

end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.MenuMeasureItem(Sender: TObject; ACanvas: TCanvas; var Width, Height: Integer);

var
  Size: TSize;
  Item: TTntMenuItem;

begin
  if Sender is TTntMenuItem then
  begin
    Item := Sender as TTntMenuItem;
    ACanvas.Font := Font;

    if Item.IsLine then
    begin
      Width := 10; // This will actually have no effect, because other entries are much wider.
      Height := 6;
    end
    else
    begin
      GetTextExtentPoint32W(ACanvas.Handle, PWideChar(Item.Caption), Length(Item.Caption), Size);

      // Border around each entry.
      Width := Size.cx + 4;
      Height := Size.cy + 6;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminBackupForm.MenuDrawItem(Sender: TObject; ACanvas: TCanvas; ARect: TRect; State: TOwnerDrawState);

var
  Item: TTntMenuItem;

begin
  if Sender is TTntMenuItem then
  begin
    Item := Sender as TTntMenuItem;
    ACanvas.Font := Font;

    if Item.IsLine then
    begin
      // A menu separator.
      ACanvas.Pen.Color := clBtnShadow;
      ACanvas.MoveTo(ARect.Left + 2, (ARect.Bottom + ARect.Top) div 2);
      ACanvas.LineTo(ARect.Right - 2, (ARect.Bottom + ARect.Top) div 2);
    end
    else
    begin
      // Top level items have an invisible parent, so have to check the parent of the parent.
      if (Item.Parent.Parent = nil) and not (Item.GetParentMenu is TPopupMenu) then
      begin
        if [odHotLight, odSelected] * State <> [] then
          ACanvas.Brush.Color := clHighlight
        else
          ACanvas.Brush.Color := clBtnFace;
      end;
      ACanvas.FillRect(ARect);
      Inc(ARect.Left, 8);
      SetBKMode(ACanvas.Handle, TRANSPARENT);
      Windows.DrawTextW(ACanvas.Handle, PWideChar(Item.Caption), Length(Item.Caption), ARect, DT_LEFT + DT_SINGLELINE +
        DT_HIDEPREFIX + DT_VCENTER);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

var
  NeedToUnitialize: Boolean;

initialization
  // Initialize OLE subsystem for drag'n drop and clipboard operations.
  NeedToUnitialize := Succeeded(OleInitialize(nil));
finalization
  if NeedToUnitialize then
    OleUninitialize;
end.

