# arch-tag: b9095457-7c52-43eb-9eab-0e7f7fbd87dc
# Copyright (C) 2003-2005  David Allouche <david@allouche.net>
#               2005 Canonical Limited
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


import re
import errors

__all__ = [
    'NameParser',
    ]

from deprecation import deprecated_callable

def backend():
    import pybaz as arch
    return arch.backend

def default_archive():
    import pybaz as arch
    return str(arch.default_archive())

def factory():
    import pybaz as arch
    return arch.factory

class ForkNameParser(str):

    """Parse Arch names with with tla.

    All operations run tla, this is generally too expensive for
    practical use. Use NameParser instead, which is implemented
    natively.

    This class is retained for testing purposes.
    """

    def __valid_package_name(self, opt, tolerant=True):
        opts = [ 'valid-package-name' ]
        if self.startswith('-'):
            return False
        if tolerant:
            opts.append('--tolerant')
        if opt:
            opts.append(opt)
        opts.append(self)
        # XXX: should be (0,1) but baz 1.3~200503300337 is buggy
        return 0 == backend().status_cmd(opts, expected=(0,1,2))

    def __parse_package_name(self, opt):
        opts = [ 'parse-package-name' ]
        if self.startswith('-'):
            return None
        if opt:
            opts.append(opt)
        opts.append(self)
        # XXX: should be (0,1) but baz 1.3~200503300337 is buggy
        status, retval = backend().status_one_cmd(opts, expected=(0,1,2))
        if status != 0:
            return None
        return retval

    def is_category(self):
        return self.__valid_package_name('--category', tolerant=False)
    def is_package(self):
        return self.__valid_package_name('--package', tolerant=False)
    def is_version(self):
        return self.__valid_package_name('--vsn', tolerant=False)
    def is_patchlevel(self):
        return self.__valid_package_name('--patch-level', tolerant=False)

    def has_archive(self):
        return self.__valid_package_name('--archive')
    def has_category(self):
        return self.__valid_package_name('--category')
    def has_package(self):
        return self.__valid_package_name('--package')
    def has_version(self):
        return self.__valid_package_name('--vsn')
    def has_patchlevel(self):
        return self.__valid_package_name('--patch-level')

    def get_archive(self):
        return self.__parse_package_name('--arch')
    def get_nonarch(self):
        return self.__parse_package_name('--non-arch')
    def get_category(self):
        return self.__parse_package_name('--category')
    def get_branch(self):
        return self.__parse_package_name('--branch')
    def get_package(self):
        return self.__parse_package_name('--package')
    def get_version(self):
        if not self.has_version(): return None # work around a tla bug
        return self.__parse_package_name('--vsn')
    def get_package_version(self):
        return self.__parse_package_name('--package-version')
    def get_patchlevel(self):
        if not self.has_patchlevel(): return None # work around a tla bug
        return self.__parse_package_name('--patch-level')


class NameParser(str):

    """Parser for names in Arch archive namespace.

    Implements name parsing natively for performance reasons. It
    should behave exactly as tla, any discrepancy is to be considered
    a bug, unless tla is obviously buggy.

    Bare names of archives, category, branch, versions ids, and
    unqualified patchlevel names are not part of the archive
    namespace. They can be validated using static methods.

    :group Specificity level: is_category, is_package, is_version

    :group Presence name components: has_archive, has_category, has_package,
        has_version, has_revision, has_patchlevel

    :group Getting name components: get_archive, get_nonarch, get_category,
        get_branch, get_package, get_version, get_package_version,
        get_patchlevel

    :group Validating name components: is_archive_name, is_category_name,
        is_branch_name, is_version_id, is_patchlevel
    """

    __archive_regex = re.compile('^[a-zA-Z0-9][-a-zA-Z0-9]*(\.[-a-zA-Z0-9]+)*@'
                                 '[-a-zA-Z0-9.]*$')
    __name_regex = re.compile('^[a-zA-Z]([a-zA-Z0-9]|-[a-zA-Z0-9])*$')
    __version_regex = re.compile('^[0-9][A-Za-z0-9+:.~-]*$')
    # tla accepts --version-12, so mimick that:
    __level_regex = re.compile('^base-0$|^(version|patch|versionfix)-[0-9]+$')

    def __init__(self, s):
        """Create a parser object for the given string.

        :param s: string to parse.
        :type s: str
        """
        str.__init__(s)
        parts = self.__parse()
        if parts:
            self.__valid = True
        else:
            parts = None, None, None, None, None, None
            self.__valid = False
        (self.__archive, self.__nonarch, self.__category,
         self.__branch, self.__version, self.__level) = parts

    def __parse(self):
        parts = self.__split()
        if not parts:
            return False
        if not self.__validate(parts):
            return False
        return parts

    def __split(self):
        parts = self.split('/')
        if len(parts) == 1:
            archive, nonarch = None, parts[0]
        elif len(parts) == 2:
            archive, nonarch = parts
        else:
            return False
        parts = nonarch.split('--')
        category, branch, version, level = None, None, None, None
        if len(parts) == 1:
            category = parts[0]
        elif len(parts) == 2:
            if nonarch.endswith('--'):
                category, ignored = parts
            elif self.__name_regex.match(parts[1]):
                category, branch = parts
            elif self.__match_name_with_dash(parts[1]):
                category, branch = parts
            else:
                category, version = parts
        elif len(parts) == 3:
            if nonarch.endswith('--'):
                category, branch, ignored = parts
            elif self.__name_regex.match(parts[1]):
                category, branch, version = parts
            else:
                category, version, level = parts
        elif len(parts) == 4:
            category, branch, version, level = parts
        else:
            return False
        return archive, nonarch, category, branch, version, level

    def __match_name_with_dash(self, string):
        if not string.endswith('-'):
            return False
        if self.__name_regex.match(string[:-1]):
            return True
        else:
            return False

    def __validate(self, args):
        archive, nonarch, category, branch, version, level = args
        if archive is not None and not self.__archive_regex.match(archive):
            return False
        if category.endswith('-'):
            category = category[:-1]
        if not self.__name_regex.match(category):
            return False
        if branch is not None:
            if branch.endswith('-'):
                branch = branch[:-1]
            if not self.__name_regex.match(branch):
                return False
        if version is None and level is None:
            return True
        if not self.__version_regex.match(version):
            return False
        if level is None:
            return True
        if not self.__level_regex.match(level):
            return False
        else:
            return True

    def is_category(self):
        """Is this a category name?

        :rtype: bool
        """
        return bool(self.__category) and not (self.__branch or self.__version)

    def is_package(self):
        """Is this a package name (category or branch name)?

        :rtype: bool
        """
        return bool(self.__category) and not self.__version

    def is_version(self):
        """Is this a version name?

        :rtype: bool
        """
        return bool(self.__version) and not self.__level

    def has_archive(self):
        """Does this include an archive name?

        :rtype: bool
        """
        return bool(self.__archive)

    def has_category(self):
        """Does this include an category name?

        All valid names include a category.

        :rtype: bool
        """
        return bool(self.__category)

    def has_package(self):
        """Does this include an package name?

        All valid names include a package.

        :rtype: bool
        """
        return bool(self.__category)

    def has_version(self):
        """Does this include a version name?

        :rtype: bool
        """
        return bool(self.__version)

    def has_patchlevel(self):
        """Does this include a revision name?

        :rtype: bool
        """
        return bool(self.__level)

    def get_archive(self):
        """Get the archive part of the name

        :return: archive part of the name, or the default archive name, or None
            if the name is invalid.
        :rtype: str, None
        """
        if not self.__valid: return None
        if not self.__archive: return default_archive()
        return self.__archive

    def get_nonarch(self):
        """Get Non-archive part of the name

        :return: the name without the archive component, or None if the name is
            invalid or has no archive component.
        :rtype: str, None
        """
        return self.__nonarch

    def get_category(self):
        """Get the Category name

        :return: part of the name which identifies the category within
            the archive, or None if the name is invalid or has no category
            component.
        :rtype: str, None
        """
        return self.__category

    def get_branch(self):
        """Get the branch part of name

        :return: part of the name which identifies the branch within the
            category, or None if the name is invalid or the empty string if the
            name has no branch component.
        :rtype: str, None
        """
        if not self.__valid: return None
        if not self.__branch: return str()
        return self.__branch

    def get_package(self):
        """Get the package name

        :return: part of the name including the category part and branch part
            (if present) of the name, or None if the name is not valid.
        :rtype: str, None
        """
        if not self.__valid: return None
        if self.__branch is None: return self.__category
        return self.__category + '--' + self.__branch

    def get_version(self):
        """Get the version id part of the name

        :return: part of the name identifying a version in a branch, or None if
            the name is invalid or does not contain a version id.
        :rtype: str, None
        """
        return self.__version

    def get_package_version(self):
        """Get the unqualified version name

        :return: part of the name identifying a version in an archive, or None
            if the name does not contain a version id or is invalid.
        :rtype: str, None
        """
        if not self.__version: return None
        return self.get_package() + '--' + self.__version

    def get_patchlevel(self):
        """Get the patch-level part of the name

        :return: part of the name identifying a patch in a version, or None if
            the name is not a revision or is invalid.
        :rtype: str, None
        """
        return self.__level

    def is_archive_name(klass, s):
        """Is this string a valid archive name?

        :param s: string to validate.
        :type s: str
        :rtype: bool
        """
        return bool(klass.__archive_regex.match(s))
    is_archive_name = classmethod(is_archive_name)

    def is_category_name(klass, s):
        """Is this string a valid category name?

        Currently does the same thing as is_branch_name, but that might
        change in the future when the namespace evolves and it is more
        expressive to have different functions.

        :param s: string to validate.
        :type s: str
        :rtype: bool
        """
        return bool(klass.__name_regex.match(s))
    is_category_name = classmethod(is_category_name)

    def is_branch_name(klass, s):
        """Is this string a valid category name?

        Currently does the same thing as is_category_name, but that might
        change in the future when the namespace evolves and it is more
        expressive to have different functions.

        :param s: string to validate.
        :type s: str
        :rtype: bool
        """
        return bool(klass.__name_regex.match(s))
    is_branch_name = classmethod(is_branch_name)

    def is_version_id(klass, s):
        """Is this string a valid version id?

        :param s: string to validate.
        :type s: str
        :rtype: bool
        """
        return bool(klass.__version_regex.match(s))
    is_version_id = classmethod(is_version_id)

    def is_patchlevel(klass, s):
        """Is this string a valid unqualified patch-level name?

        :param s: string to validate.
        :type s: str
        :rtype: bool
        """
        return bool(klass.__level_regex.match(s))
    is_patchlevel = classmethod(is_patchlevel)


    def object(self):
        """Create the Category, Branch, Version or Revision object

        Create the namespace object corresponding to the name. This
        requires some guessing so, for example, nameless branches will
        not be recognized.

        This function is unsafe (categories and nameless branches are not
        distinguished) and is not really useful. Internally, only namespace
        objects should be used, and external output should be validated in a
        more specific way.
        """
        if not self.__valid:
            return None
        try:
            # handle default archive
            fqname = self.get_archive() + '/' + self.get_nonarch()
            if self.is_category():
                archive = factory().Archive(self.get_archive())
                obj = archive[self.get_category()]
            elif self.is_package():
                archive = factory().Archive(self.get_archive())
                obj = archive[self.get_category()][self.get_branch()]
            elif self.is_version():
                obj = factory().Version(fqname)
            elif self.has_patchlevel():
                obj = factory().Revision(fqname)
            else:
                obj = None
        except errors.NamespaceError:
            # needed to handle category or branch names with trailing dash
            obj = None
        return obj
