/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: reader.cxx,v $
 *
 *  $Revision: 1.6 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/07 22:41:37 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/
#include <stdio.h>

#include <osl/conditn.h>
#include <osl/mutex.hxx>
#include <osl/diagnose.h>

#include <rtl/byteseq.hxx>

#include <uno/threadpool.h>

#include <bridges/remote/context.h>
#include <bridges/remote/unmarshal.hxx>
#include <bridges/remote/marshal.hxx>
#include <bridges/remote/connection.h>
#include <bridges/remote/counter.hxx>

#include <uno/environment.h>

#ifdef major
#undef major
#undef minor
#endif

#include <com/sun/star/corba/giop/MessageHeader_1_2.hpp>
#include <com/sun/star/corba/giop/ReplyHeader_1_2.hpp>
#include <com/sun/star/corba/giop/RequestHeader_1_2.hpp>
#include <com/sun/star/corba/giop/MsgType_1_1.hpp>
#include <com/sun/star/corba/iop/ServiceContext.hpp>
#include <com/sun/star/corba/iop/ServiceIdGroup.hpp>
#include <com/sun/star/corba/iop/ServiceContextList.hpp>
#include <com/sun/star/corba/LogicalThreadID.hpp>

#include "iothreads.hxx"
#include "conversion.h"

using namespace ::rtl;
using namespace bridges_remote;

using namespace ::com::sun::star::corba;
using namespace ::com::sun::star::corba::giop;
using namespace ::com::sun::star::corba::iop;
using namespace ::com::sun::star::uno;


#if OSL_DEBUG_LEVEL > 1
static MyCounter thisCounter( "DEBUG : ReaderThread" );
#endif

namespace bridges_remote
{

void retrieveThreadID ( const ServiceContextList & contextList ,
						ByteSequence *pThreadId,
						sal_Bool bIsLittleEndian)
{
	for( sal_Int32 i = 0 ; i < contextList.getLength() ;  i ++ )
	{
		if( ServiceIdGroup::LogicalThreadId ==
			contextList.getConstArray()[i].context_id )
		{
			Unmarshal unm2( contextList.getConstArray()[i].context_data.getConstArray(),
							contextList.getConstArray()[i].context_data.getLength(),
							bIsLittleEndian );
			LogicalThreadID logId;
			if( ( unm2 >>= logId ) && logId.IDs.getLength() )
			{
				*pThreadId = *( ByteSequence *) &( logId.IDs.getConstArray()[0].threadID);
				break;
			}
		}
	}
}

OReaderThread::OReaderThread( remote_Connection *pConnection,
							  void ( SAL_CALL * doRequest ) ( void *pThreadSpecificData ),
							  uno_Environment *pEnvRemote,
							  OWriterThread * pWriterThread ) :
	m_pConnection( pConnection ),
	m_doRequest( doRequest ),
	m_pEnvRemote( pEnvRemote ),
	m_pWriterThread( pWriterThread ),
	m_nMessageErrorCount( 0 ),
	m_bDestroyMyself( sal_False )
{
	m_pConnection->acquire( m_pConnection );
#if OSL_DEBUG_LEVEL > 1
	thisCounter.acquire();
#endif
}


OReaderThread::~OReaderThread( )
{
	m_pConnection->release( m_pConnection );
#if OSL_DEBUG_LEVEL > 1
	thisCounter.release();
#endif
}

void OReaderThread::sendMessageError( )
{
	Sequence < sal_Int8 > seq( 12 );

	MessageHeader_1_2 hdr;
	hdr.magic_1 = 'G';
	hdr.magic_2 = 'I';
	hdr.magic_3 = 'O';
	hdr.magic_4 = 'P';
	hdr.GIOP_version.major = CURRENT_IIOP_PROTOCOL_MAJOR;
	hdr.GIOP_version.minor = CURRENT_IIOP_PROTOCOL_MINOR;
	hdr.flags = 1;
	hdr.message_size = 0;
	hdr.message_type = MsgType_1_1_MessageError;

	Marshal m( sal_True );
	m <<= hdr;
	m_pWriterThread->enqueue( seq );
}

void OReaderThread::onTerminated()
{
	if( m_bDestroyMyself )
	{
		delete this;
	}
}

void OReaderThread::run()
{
	sal_Int8 pHdr[REMOTE_MARSHALED_MSGHDR_SIZE];

	while( sal_True )
	{
		if( REMOTE_MARSHALED_MSGHDR_SIZE !=
			m_pConnection->read( m_pConnection , pHdr, REMOTE_MARSHALED_MSGHDR_SIZE) )
		{
			remote_Context *pContext =
				( remote_Context * ) m_pEnvRemote->pContext;
			if( ! pContext->m_pBridgeImpl->m_bDisposed )
			{
				m_pEnvRemote->dispose( m_pEnvRemote );
			}

			break;
		}

		sal_Bool bIsLittleEndian = sal_True;
		::com::sun::star::corba::giop::MessageHeader_1_2 hdr;
		{
			Unmarshal unm( pHdr , REMOTE_MARSHALED_MSGHDR_SIZE , sal_True );
			unm >>= hdr;
		}
		if( ! hdr.flags )
		{
			bIsLittleEndian = sal_False;
			// Big endian
			Unmarshal unm( pHdr , REMOTE_MARSHALED_MSGHDR_SIZE , sal_False );
			unm >>= hdr;
		}

		if( hdr.magic_1 != 'G' ||
			hdr.magic_2 != 'I' ||
			hdr.magic_3 != 'O' ||
			hdr.magic_4 != 'P' )
		{
			sendMessageError( );
			continue;
		}

		Sequence < sal_Int8 > seq;

		if( hdr.message_size )
		{
			// ensure, that the length of the sequence is properly aligned.
			// This is to allow range checking during extracting ( the range
			// check may fail because of missing alignment bytes ).
			seq = Sequence<  sal_Int8 > (
				(hdr.message_size & 0xfffffff8) + ((hdr.message_size & 0x7) ? 8 : 0) );
			if( seq.getArray() == 0 )
			{
				// Out of memory ( probably corrupted message header ! )
				sendMessageError( );
				continue;
			}

			// read the correct number of bytes from the stream
			const sal_Int32 nRead = m_pConnection->read( m_pConnection ,
														 seq.getArray() ,
														 hdr.message_size );

			if( nRead != hdr.message_size )
			{
				// Connection closure !
				remote_Context *pContext =
					( remote_Context * ) m_pEnvRemote->pContext;
				osl_setCondition(
					((iiop_BridgeImpl*) pContext->m_pBridgeImpl)->m_cndCloseConnectionReceived );
				m_pEnvRemote->dispose( m_pEnvRemote );
				return;
			}
		}


		switch( hdr.message_type )
		{
		case MsgType_1_1_Request:
		{
			// Start the request
			RequestInfo *pData = new RequestInfo(
				m_pEnvRemote,
				seq,
				bIsLittleEndian );

			if( ! ( pData->m_unmarshal >>= pData->m_requestHeader ) )
			{
				sendMessageError();
				continue;
			}

			// Retrieve the ThreadID !
			ByteSequence seqID;
			retrieveThreadID( pData->m_requestHeader.service_context ,
							  &seqID ,
							  bIsLittleEndian );

			remote_Context *pContext =
				( remote_Context * ) m_pEnvRemote->pContext;
			// non blocking !
			uno_threadpool_putJob(
				((iiop_BridgeImpl*) pContext->m_pBridgeImpl)->m_hThreadPool,
				seqID.getHandle(),
				pData,
				m_doRequest,
				pData->m_requestHeader.response_flags ? 0 : 1 );
			break;
		}
		case MsgType_1_1_Reply:
		{
			ReplyInfo *pData = new ReplyInfo(
				m_pEnvRemote,
				seq,
				bIsLittleEndian );

			if( ! ( pData->m_unmarshal >>= pData->m_replyHeader ) )
			{
				sendMessageError();
				continue;
			}

			// Retrieve the ThreadID !
			ByteSequence seqID;
			retrieveThreadID( pData->m_replyHeader.service_context , &seqID , bIsLittleEndian );

			remote_Context *pContext =
				( remote_Context * ) m_pEnvRemote->pContext;
			// non blocking !
			uno_threadpool_putJob(
				((iiop_BridgeImpl*) pContext->m_pBridgeImpl)->m_hThreadPool,
				seqID.getHandle(),
				pData,
				0,
				sal_False );
			break;
		}
		case MsgType_1_1_CloseConnection:
		{
			// No further data will come over the line !
			remote_Context *pContext =
				( remote_Context * ) m_pEnvRemote->pContext;
			osl_setCondition( ((iiop_BridgeImpl*)pContext->m_pBridgeImpl)->m_cndCloseConnectionReceived );
			if( ! pContext->m_pBridgeImpl->m_bDisposed )
			{
				m_pEnvRemote->dispose( m_pEnvRemote );
			}
			return;
		}
		case MsgType_1_1_MessageError:
		{
			// What to do
			printf( "Message error received, retrying ...\n" );
			m_nMessageErrorCount ++;
			if( m_nMessageErrorCount >= 4 )
			{
				printf( "Giving up\n" );
				remote_Context *pContext =
					( remote_Context * ) m_pEnvRemote->pContext;
				osl_setCondition( ((iiop_BridgeImpl*)pContext->m_pBridgeImpl)->m_cndCloseConnectionReceived );
				m_pEnvRemote->dispose( m_pEnvRemote );
				return;
			}
			break;
		}
		default:
			OSL_ENSURE( 0 , "unhandled iiop-message" );
		}
	}

}
}
