/***************************************************************************
 *   Copyright (C) 2004, 2005 Thomas Nagy                                  *
 *   tnagy2^8@yahoo.fr                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation (see COPYING)            *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 ***************************************************************************/

#ifndef _DDATAITEM_H
#define _DDATAITEM_H

#include <qfont.h>
#include <qpixmap.h>
#include <qcolor.h>
#include <qvaluelist.h>

#include "DItem.h"

class QPixmap;
class QString;
class QColor;
class QTextStream;
class QFont;
class DDataControl;

/**
 * This class holds all data of a reference
 * 
 * @short reference data class
 * @author Thomas Nagy <tnagy2^8@yahoo.fr>
 */
struct DRef
{
	/// id
	int m_ref;
	/// caption
	QString m_text;
};

/**
 * URL handler
 *
 * @short URL handler class
 * @author Thomas Nagy <tnagy2^8@yahoo.fr>
 */
struct URLObject
{
	QString m_url;
	QString m_caption;
};

/**
 * This class handles the data for the mindmap items 
 *
 * @short Mindmap item class
 * @author Thomas Nagy <tnagy2^8@yahoo.fr>
 */
class DDataItem : public DItem
{
	friend class DissertParser;
	friend class FFParser;
	friend class DDeltaCreate;

	public:

        typedef double Coord;
        
	DDataItem(DDataControl*, int);
	~DDataItem();

	/// add an item as a child of the current one
	void addChild( int id );

	/// set the parent of this item
	void setParent( int id );

	/// remove a child
	void removeChild( int id );



	/// number of children (not recursive)
	unsigned int countChildren() const;

	/// counts the amount of leaves in the subtree recursively
	int countFamily() const;


	/// id of the nth child
	int childNum(unsigned int) const;

	/// index of a child (using its id)
	unsigned int childIdx(unsigned int) const;

	/// tell a child to be nth in the family
	void setChildIdx(unsigned int id, unsigned int pos);


	/// id of the parent
	int Parent() const { return m_parent; }

	/// make an orphan ..
	void setNoParent();

	/// print the data
	void printSelf(QTextStream&);

	/// set the brush color
	void setColorScheme(int c);

	/// set a custom color for brush, outline and text
	void setCustomColors(QColor c1, QColor c2, QColor c3);

	/// set the position of items on canvas
	void setXY(Coord, Coord);

	/// set the flags
	void setFlag(int flag, bool add=true);

	/// get the flag !
	bool hasFlag(int flag);

	inline unsigned int countFlags() { return m_flags.count(); }
	
	/// color accessors
	QColor fillColor();
	QColor outlineColor();
	QColor textColor();

	/// real fill color for non-root items - root items have the root color
	QColor realFillColor();

	/*
	 * The font used for this item - it is set to the document font by default (DDataControl)
	 */
	QFont m_defaultFont;

	/// text held by the item
	QString m_text;
	/// summary : one line describing the concept
	QString m_summary;
	/// comment : text that does not appear anywhere
	QString m_comment;
	/// original picture, not resized
	QPixmap m_pix;
	/// thumbnail, set to Settings::pixSize()
	QPixmap m_pixtb;
	/// caption for the picture
	QString m_piccaption;
	QString m_picurl;
	/// name of the file held in the temporary directory
	QString m_picfilename;

	QValueList<URLObject> m_urllist;

	/// the message to display in the views if the summary is empty
	QString dispmsg();

	/// x position of the item on the canvas view
	Coord x() const { return m_x; }
	/// y position of the item on the canvas view
	Coord y() const { return m_y; }

	/**
	 *  Last child idx considered on the mindmap view
	 *  useful for navigating on the map with ctrl+arrows for example
	 */
	int m_lastchild;

	/** 
	 * Load a picture from a file
	 * The file is automatically added to the list of cached documents
	 */
	void loadPix(const QString & file);

	/**
	 * Clear the picture attached
	 */
	void clearPix();

	/**
	 * Return the current color scheme (do not bypass to it directly)
	 */
	int colorScheme() const { return m_colorscheme; }

	/**
	 * Notify all views that the mindmap object has changed
	 */
	void setChanged();

	/**
	 * Change the font size
	 */
	void incrementFontSize(int);

	// META-DATA

	/** 
	 * Add a reference
	 */
	void addRef(int);
	/**
	 * Remove a reference
	 */
	void rmRef(int);
	/**
	 * Indicates if the reference of id i belongs
	 * to the list of references for this item
	 */
	bool isRef(int i);
	/**
	 * Return the reference of index idx
	 */
	DRef* ref_idx(unsigned int idx);
	/**
	 * Return the amount of references for this item
	 */
	unsigned int ref_count();

	private:
	/// list of item children
	QValueList<int> m_children;
	/// list of item references
	QValueList<DRef*> m_references;

	// flags
	QValueList<int> m_flags;

	/// X position of the item on the mindmap view
	Coord m_x;
	/// Y position of the item on the mindmap view
	Coord m_y;

	/// colors - reuse the settings color as much as possible
	int m_colorscheme;

	// these are provided for the "custom" color section
	QColor m_fillcolor;
	QColor m_outlinecolor;
	QColor m_textcolor;

	protected:
	/// data control
	DDataControl *m_manager;

	/// parent item, set initially to NOITEM (no parent)
	int m_parent;

	public:
	static QString protectXML(const QString& str);
	static QString printXMLTag(const QString& tag, const QString& str, int padding=0);
};

#endif // _DDATAITEM_H
