/****************************************************************************
**
** Copyright (C) 1992-$THISYEAR$ Trolltech AS. All rights reserved.
**
** This file is part of the $MODULE$ of the Qt Toolkit.
**
** $LICENSE$
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/
#include "configutils.h"

#include <qdebug.h>
#include <qcoreapplication.h>
#include <qobject.h>
#include <qfile.h>
#include <qdir.h>
#include <qregexp.h>
#include <qlibraryinfo.h>
#include <qprocess.h>
#include <qtextstream.h>
#ifdef Q_WS_WIN
#include <windows.h>
#endif
#ifdef Q_OS_UNIX
#include <stdlib.h>
#endif

QString *qtDir = 0;
QString *qsa_prefix = 0;

int errors = 0;
int warnings = 0;
bool autoShutdown = true;

bool qdoc_warning = false;
bool qmake_warning = false;

static QStringList static_messages;

/** Which program to invoke as qmake. Default is "qmake" */
QString qmake = "qmake";

/** Set program to be used as qmake */
void setQMake( const QString &str ) {
    message ( QString("Using qmake: ") + str);
    qmake = str;
}

void message(const QString &str)
{
    static_messages.append(str);
}

QStringList messages()
{
    return static_messages;
}

static bool execute(const QStringList &arguments)
{
    QProcess bp;

    QStringList args = arguments;
    QString app = args.takeFirst();

    bp.start(app, args);
    if (!bp.waitForFinished(-1))
        return false;

    return bp.exitCode() == 0;
}

static void runQMake(const QString &dir,
                     const QStringList &configs,
                     const QStringList &antiConfigs,
                     const QString &prefix,
                     const QString &target)
{
    QString old_dir = QDir::currentPath();
    QDir::setCurrent(dir);

    // make the top level Makefile
    QStringList args;
    args.append(qmake);
    if (!prefix.isEmpty())
        args.append(QLatin1String("QSA_INSTALL_PREFIX=") + prefix);
    if (!target.isNull()) {
        args.append(QLatin1String("-o"));
        args.append(target);
    }

#if defined Q_WS_MACX
    args << "-spec" << "macx-g++";
#endif

    args.append("-after");
    if (!configs.isEmpty())
        args.append( "CONFIG+=" + configs.join( " " ) );
    if (!antiConfigs.isEmpty())
        args << "CONFIG-=" + antiConfigs.join(" ");
    //Need to pass QMAKE to qmake subprocesses
    args.append( "QMAKE=" + qmake );

    if(!execute(args)) {
        warnings++;
        qmake_warning = true;
    }

    QDir::setCurrent(old_dir);
}


#if QT_VERSION > 0x040100
static void runQMakeRecursive(const QString &dir,
                              const QStringList &configs,
                              const QStringList &anti_configs,
                              const QString &prefix,
                              const QString &target)
{
    QDir d(dir);

    QStringList pro_files = d.entryList(QStringList() << "*.pro", QDir::Files);
    if (pro_files.size())
        runQMake(dir, configs, anti_configs, prefix, target);

    QStringList subdirs = d.entryList(QDir::Dirs);
    for (int i=0; i<subdirs.size(); ++i) {
        if (subdirs.at(i) == "." || subdirs.at(i) == "..")
            continue;
        runQMakeRecursive(d.absoluteFilePath(subdirs.at(i)), configs, anti_configs, prefix, target);
    }
}
#endif

void runQMake(const QStringList &configs, const QStringList &anti_configs, const QString &prefix)
{
#if QT_VERSION > 0x040100
    runQMakeRecursive(QLatin1String("src"), configs, anti_configs,
                      prefix, QLatin1String("Makefile.qsa"));
#endif
    runQMake(QLatin1String("."), configs, anti_configs, prefix, QLatin1String("Makefile.qsa"));
}

bool writeQSConfig(bool buildIde, bool buildEditor, bool buildNewEditor)
{
    QFile file(QLatin1String("src/qsa/qsconfig.h"));
    if(!file.open(QIODevice::WriteOnly)) {
        message(QLatin1String("Failed to open 'src/qsa/qsconfig.h' for writing."));
        return false;
    }
    QTextStream txt(&file);
    txt << "// This file is autogenerated by QSA configure, do not modify it!\n"
        << "#ifndef QS_CONFIG_H\n"
        << "#define QS_CONFIG_H\n"
        << "\n";
    if (!buildIde || !buildEditor)
        txt << "#define QSA_NO_IDE\n";
    if (!buildEditor)
        txt << "#define QSA_NO_EDITOR\n";
    if (buildNewEditor)
        txt << "#define QSA_NEW_EDITOR\n";
    txt << "\n"
        << "#endif\n";
    return true;
}

bool writeExtraConfigs(const QStringList &s)
{
    QFile file(QLatin1String(".qmake.cache"));
    if (!file.open(QIODevice::Text | QIODevice::Append)) {
        message(QLatin1String("Failed write configuration to '.qmake.cache'"));
        return false;
    }

    QTextStream stream(&file);
    stream << "CONFIG += " << s.join(QLatin1String(" ")) << endl;
    return true;
}


const char *contents_qsa_prf =
"# This file is automatically generated, don't touch"
"\n"
"\nCONFIG		+= qt warn_on"
"\n"
"\ncontains(QT_CONFIG, release):contains(QT_CONFIG, debug) {"
"\n    # Qt was configued with both debug and release libs"
"\n    CONFIG += debug_and_release build_all"
"\n}"
"\n"
"\nCONFIG(debug, debug|release) {"
"\n    unix:LIBS += -lqsa_debug"
"\n    else:LIBS += -lqsad"
"\n} else {"
"\n    LIBS += -lqsa"
"\n}"
"\n"
"\n"
"\n!shared {"
"\n	DEFINES += QSA_NO_DLL"
"\n}"
"\n";

bool writeQsaPrfFile(bool include_qt3_support)
{
    QFile file("src/qsa/qsa.prf");
    if (!file.open(QFile::WriteOnly | QFile::Text)) {
        message("Failed to open file '" + file.fileName() + "' for writing");
        return false;
    }

    QTextStream stream(&file);
    stream << contents_qsa_prf;

    if (include_qt3_support)
        stream << "mac:QT += qt3support" << endl;

    if (qsa_prefix && !qsa_prefix->isEmpty()) {
        stream << "INCLUDEPATH += " << QString(*qsa_prefix).replace("\\", "/") << endl;
    }

    stream.flush();

    return true;
}
