# arch-tag: b8a786b6-51c4-42de-80b1-b6130ba39502
# Copyright (C) 2003 David Allouche <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Internal module providing changeset handling.

This module implements some of public interface for the
pybaz_ package. But for convenience reasons the author prefers
to store this code in a file separate from ``__init__.py``.

.. _pybaz: pybaz-module.html

This module is strictly internal and should never be used.
"""

import os
import re
from sets import ImmutableSet

import errors
from pathname import DirName
from _escaping import name_unescape
from _output import classify_changeset_application
from _output import classify_changeset_creation

__all__ = [
    'Changeset',
    'ChangesetApplication',
    'ChangesetCreation',
    'delta',
    'iter_delta',
    ]


def backend():
    import _builtin
    return _builtin.backend

def factory():
    import _builtin
    return _builtin.factory

def _check_working_tree_param(param, name):
    import _builtin
    _builtin._check_working_tree_param(param, name)

def _check_str_param(param, name):
    import _builtin
    _builtin._check_str_param(param, name)


class Changeset(DirName):

    """Arch whole-tree changeset."""

    def __init__(self, name):
        DirName.__init__(self, name)
        self.__index = {}
        self.__index_excl = {}
        self.__metadata = {}
        self.exclude_re = re.compile('^[AE]_')
        self._impl = _Changeset_Baz_1_0

    def get_index(self, name, all=False):
        """Load and parse an index file from the changeset.

        Expectable indexes are:
        mod-dirs mod-files orig-dirs orig-files (more?)
        """
        key = (name, all)
        if name in self.__index: return self.__index[key]
        if all:
            not_exclude = lambda(id_): True
        else:
            not_exclude = lambda(id_): not self.exclude_re.match(id_)
        retval = {}
        fullname = self/(name + '-index')
        if os.path.exists(fullname):
            index_file = open(fullname)
            try:
                for line in index_file:
                    n, id_ = map(lambda(s): s.strip(), line.split())
                    if not_exclude(id_):
                        retval[id_] = os.path.normpath(name_unescape(n))
            finally:
                index_file.close()
        self.__index[key] = retval
        return retval

    def _get_does_nothing(self):
        """Is the changeset a no-op?"""
        # TODO: checking that all indices are empty can yield false negatives
        # rather we should check for the effective absence of changes.
        for index_name in ('mod-files', 'orig-files', 'mod-dirs', 'orig-dirs'):
            index = self.get_index(index_name, True)
            if index: return False
        return True
    does_nothing = property(_get_does_nothing)

#     def get_metadata(self, name):
#         """Load and parse a metadata file from the changeset.
#
#         Expectable metadata tables are:
#         modified-only-dir original-only-dir (more?)
#         """
#         raise NotImplementedError, "not yet implemented"
#         if name in self.__metadata: return self.__metadata[name]
#         retval = []
#         fullname = self/(name + '-metadata')
#         if os.path.exists(fullname):
#             for line in open(fullname):
#                 pass # Not implemented
#         self.__metadata[name] = retval
#         return retval

    def __iter_mod_helper(self, what, all):
        __pychecker__ = 'no-abstract' # for ImmutableSet
        orig_index = self.get_index('orig-' + what, all)
        mod_index = self.get_index('mod-' + what, all)
        orig_ids = ImmutableSet(orig_index.iterkeys())
        mod_ids = ImmutableSet(mod_index.iterkeys())
        for id_ in orig_ids & mod_ids:
            yield (id_, orig_index[id_], mod_index[id_])

    def iter_mod_files(self, all=False):
        """Iterator over (id, orig, mod) tuples for files which are are
        patched, renamed, or have their permissions changed."""
        return self.__iter_mod_helper('files', all)

    def iter_patched_files(self, all=False):
        """Iterate over (id, orig, mod) of patched files."""
        patchdir = self/'patches'
        for f in self.iter_mod_files(all):
            if os.path.isfile(patchdir/f[1]+'.patch'):
                yield f
            else:
                print "file does not exists: %s" % str(patchdir/f[1]+'.patch')
        #return itertools.ifilter(lambda(f): os.path.isfile(patchdir/f[2]),
        #                         self.iter_mod_files())

    def patch_file(self, modname):
        return self/'patches'/modname+'.patch'

    def __iter_renames_helper(self, what):
        for id_, orig, mod in self.__iter_mod_helper(what, all=False):
            if orig != mod:
                yield (id_, orig, mod)

    def iter_renames(self):
        """Iterate over (id, orig, dest) triples representing renames.

        id is the persistant file tag, and the key of the dictionnary.
        orig is the name in the original tree.
        dest is the name in the modified tree.
        """
        from itertools import chain
        return chain(*map(self.__iter_renames_helper, ('files', 'dirs')))

    def __iter_created_helper(self, what, removed=False, all=False):
        __pychecker__ = 'no-abstract' # for ImmutableSet
        orig_index = self.get_index('orig-' + what, all)
        mod_index = self.get_index('mod-' + what, all)
        if removed: orig_index, mod_index = mod_index, orig_index
        orig_ids = ImmutableSet(orig_index.iterkeys())
        mod_ids = ImmutableSet(mod_index.iterkeys())
        for id_ in mod_ids - orig_ids:
            yield (id_, mod_index[id_])

    def iter_created_files(self, all=False):
        """Iterator over tuples (id, dest) for created files.

        :param all: include Arch control files.
        :type all: bool
        """
        return self.__iter_created_helper('files', all=all)

    def created_file(self, name):
        return self/'new-files-archive'/name

    def iter_removed_files(self, all=False):
        """Iterator over tuples (id, orig) for removed files.

        :param all: include Arch control files.
        :type all: bool
        """
        return self.__iter_created_helper('files', removed=True, all=all)

    def removed_file(self, name):
        return self/'removed-files-archive'/name

    def iter_created_dirs(self, all=False):
        """Iterator over tuples (id, dest) for created directories.

        :param all: include Arch control files.
        :type all: bool
        """
        return self.__iter_created_helper('dirs', all=all)

    def iter_removed_dirs(self, all=False):
        """Iterator over tuples (id, orig) for removed directories.

        :param all: include Arch control files.
        :type all: bool
        """
        return self.__iter_created_helper('dirs', removed=True, all=all)

    def iter_apply(self, tree, reverse=False):
        """Apply this changeset to a tree, with incremental output.

        :param tree: the tree to apply changes to.
        :type tree: `WorkingTree`
        :param reverse: invert the meaning of the changeset; adds
            become deletes, etc.
        :type reverse: bool
        :rtype: `ChangesetApplication`
        """
        _check_working_tree_param(tree, 'tree')
        args = self._impl.apply_changeset_args(self, tree, reverse)
        proc = backend().sequence_cmd(args, expected=(0,1))
        return ChangesetApplication(proc)

    def apply(self, tree, reverse=False):
        """Apply this changeset to a tree. Raise on conflict.

        :param tree: the tree to apply changes to.
        :type tree: `WorkingTree`
        :param reverse: invert the meaning of the changeset; adds
            become deletes, etc.
        :type reverse: bool
        :raise errors.ChangesetConflict: a conflict occured while applying the
            changeset.
        """
        _check_working_tree_param(tree, 'tree')
        args = self._impl.apply_changeset_args(self, tree, reverse)
        status = backend().status_cmd(args, expected=(0,1))
        if status == 1:
            raise errors.ChangesetConflict(tree, self)


class _Changeset_Baz_1_0(object):

    def apply_changeset_args(cset, tree, reverse):
        args = ['apply-changeset']
        if reverse:
            args.append('--reverse')
        args.extend((str(cset), str(tree)))
        return args

    apply_changeset_args = staticmethod(apply_changeset_args)


class ChangesetApplication(object):
    """Incremental changeset application process."""

    def __init__(self, proc):
        """For internal use only."""
        self._iter_process = proc

    def __iter__(self):
        """Return an iterator of `MergeOutcome`"""
        return classify_changeset_application(self._iter_process)

    def _get_conflicting(self):
        "Did conflicts occur during changeset application?"
        if not self.finished:
            raise AttributeError("Changeset application is not complete.")
        return self._iter_process.status == 1
    conflicting = property(_get_conflicting)

    def _get_finished(self):
        "Is the changeset application complete?"
        return self._iter_process.finished
    finished = property(_get_finished)


class ChangesetCreation(object):
    """Incremental changeset generation process."""

    def __init__(self, proc, dest):
        """For internal use only."""
        self._iter_process = proc
        self._dest = dest
        self._changeset = None

    def __iter__(self):
        """Return an iterator of `TreeChange`"""
        return classify_changeset_creation(self._iter_process)

    def _get_changeset(self):
        """Generated changeset."""
        if self._changeset is None:
            if not self.finished:
                raise AttributeError("Changeset generation is not complete.")
            self._changeset = Changeset(self._dest)
        return self._changeset
    changeset = property(_get_changeset)

    def _get_finished(self):
        "Is the changeset creation complete?"
        return self._iter_process.finished
    finished = property(_get_finished)


def iter_delta(orig, mod, dest):
    """Compute a whole-tree changeset with incremental output.

    :param orig: old revision or directory.
    :type orig: `Revision`, `ArchSourceTree`
    :param mod: new revision or directory,
    :type mod: `Revision`, `ArchSourceTree`
    :param dest: path of the changeset to create.
    :type dest: str
    :rtype: `ChangesetCreation`
    """
    orig = _tree_or_existing_revision_param(orig, 'orig')
    mod = _tree_or_existing_revision_param(mod, 'mod')
    _check_str_param(dest, 'dest')
    args = ['delta', orig, mod, dest]
    proc = backend().sequence_cmd(args)
    return ChangesetCreation(proc, dest)


def _tree_or_existing_revision_param(param, name):
    if factory().isRevision(param):
        # should check for actual existence, but that is expensive
        if not param.archive.is_registered():
            raise errors.ArchiveNotRegistered(param.archive)
        return param.fullname
    elif factory().isSourceTree(param):
        return param
    else:
        raise TypeError("Parameter \"%s\" must be SourceTree or Revision"
                        " but was: %r" % (name, param))

def delta(orig, mod, dest):
    """Compute a whole-tree changeset.

    Create the output directory ``dest`` (it must not already exist).

    Compare the source trees ``orig`` and ``mod`` (which may be source
    arch source tree or revisions). Create a changeset in ``dest``.

    :param orig: the old revision or directory.
    :type orig: `Revision`, `ArchSourceTree`
    :param mod: the new revision or directory.
    :type mod: `Revision`, `ArchSourceTree`
    :param dest: path of the changeset to create.
    :type dest: str
    :return: changeset from ``orig`` to ``mod``.
    :rtype: `Changeset`
    """
    __pychecker__ = 'unusednames=line'
    iter_ = iter_delta(orig, mod, dest)
    for line in iter_: pass
    return iter_.changeset
