/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.vise;

import java.util.Observer;
import java.util.Observable;
import java.util.ResourceBundle;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

import com.redhat.rhdb.vise.resources.ActionResources;

/*
 * need to set things so that the dialog will take a model, clone it, work on
 * the clone, and set the original values to clone values if and only if the
 * user clicks ok. ???
 */

/**
 * Preferences dialog for Explain. This dialog is modal and
 * associated with a specific ConfigurationModel and Frame (parent).
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 *
 * @see com.redhat.rhdb.vise.ConfigurationModel
 */
public class DatabasesDialog extends JDialog implements Observer, ActionListener {
	private ActionFactory afactory;
    private ResourceBundle res;
	private GenericAction addAction, removeAction, configAction, copyAction;
	
	private int result;
	private Frame owner;
	private JButton cancelButton, okButton;
	private Component dbprefs;
	private JList dblist;
	private DatabaseListModel dblm = null;
	private DatabaseListSelectionListener dblsl;
	private DatabaseEditorDialog ded;

	private ConfigurationModel cm;

	/** User has closed window with intention of accepting changes. */
	public static final int OK_OPTION = 0;

	/** User has closed window with intention of discarding changes. */
	public static final int CANCEL_OPTION = 1;

	/** User has closed window with intention of discarding changes. */
	public static final int CLOSED_OPTION = 2;

	/**
	 * Creates a new <code>DatabasesDialog</code> instance.
	 *
	 * @param cm a <code>ConfigurationModel</code> value
	 * @param owner a <code>Frame</code> value
	 */
	public DatabasesDialog(ConfigurationModel cm, Frame owner)
	{
		this(cm, owner, ExplainResources.getString(ExplainResources.DATABASES_TITLE));
	}

	/**
	 * Creates a new <code>DatabasesDialog</code> instance.
	 *
	 * @param cm a <code>ConfigurationModel</code> value
	 * @param owner a <code>Frame</code> value
	 * @param title a <code>String</code> value
	 */
	public DatabasesDialog(ConfigurationModel cm, Frame owner, String title)
	{
		// we are a modal dialog window, owned by owner
		super(owner, true);
		setTitle(title);

		this.owner = owner;
		this.result = CLOSED_OPTION;
		
		this.cm = cm;
		cm.addObserver(this);

		// Database editor dialog
		ded = new DatabaseEditorDialog(this);
		
		// Listeners
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e)
			{
				close();
			}
		});

		dblsl = new DatabaseListSelectionListener();

		// Actions
		afactory = new ActionFactory();

		addAction    = afactory.getAction(ActionFactory.ACTION_NEW);
		removeAction = afactory.getAction(ActionFactory.ACTION_DELETE);
		configAction = afactory.getAction(ActionFactory.ACTION_EDIT);
		copyAction   = afactory.getAction(ActionFactory.ACTION_COPY);

		addAction.addActionListener(this);
		removeAction.addActionListener(this);
		configAction.addActionListener(this);
		copyAction.addActionListener(this);

		removeAction.setEnabled(false);
		configAction.setEnabled(false);
		copyAction.setEnabled(false);

        res = ResourceBundle.getBundle("com.redhat.rhdb.vise.resources.ActionResources");

		initComponents();

		update();
	}

	/**
	 * Get the ConfigurationModel used.
	 *
	 * @return a <code>ConfigurationModel</code> value
	 */
	public ConfigurationModel getConfigurationModel()
	{
		return cm;
	}

	/**
	 * Set the ConfigurationModel to use.
	 *
	 * @param cm a <code>ConfigurationModel</code> value
	 */
	public void setConfigurationModel(ConfigurationModel cm)
	{
		this.cm = cm;
		cm.addObserver(this);
		
		update();
	}

	/**
	 * Show the dialog.
	 */
	public void show()
	{
		if (dblm.getSize() > 0)
			select(0);

		pack();
		setSize(415, getHeight());
		setLocationRelativeTo(owner);

		super.show();
	}

	/**
	 * Get how/why dialog was closed.
	 *
	 * @return an <code>int</code> value
	 */
	public int getResult()
	{
		return result;
	}

	//
	// Observer stuff
	//

	// inherits doc comment
	public void update(Observable o, Object arg)
	{
		ConfigurationModel m = (ConfigurationModel) o;

		if (m != cm)
			return;

		update();
	}

	//
	// ActionListener
	//
	
	/**
	 * Action listener.
	 *
	 * @param e an <code>ActionEvent</code> value
	 */
	public void actionPerformed(ActionEvent e)
	{
		String command = e.getActionCommand();

		if (command.equals(res.getString(ActionResources.OK_COMMAND)))
		{
			ok();
		}
		else if (command.equals(res.getString(ActionResources.CANCEL_COMMAND)))
		{
			cancel();
		}
		else if (command.equals(addAction.getActionCommand()))
		{
			addEntry();
		}
		else if (command.equals(removeAction.getActionCommand()))
		{
			removeEntry();
		}
		else if (command.equals(copyAction.getActionCommand()))
		{
			copyEntry();
		}
		else if (command.equals(configAction.getActionCommand()))
		{
			editEntry();
		}
		else
		{
			System.err.println("DatabasesDialog: unknown action command" + command);
		}
	}

	//
	// private methods
	//

	private void cancel()
	{
		result = CANCEL_OPTION;
		hide();
	}

	private void close()
	{
		result = CLOSED_OPTION;
		hide();
	}

	private void ok()
	{
		result = OK_OPTION;
		hide();
	}
	
	private void update()
	{
		int idx = dblist.getSelectedIndex();
		dblm = new DatabaseListModel(cm);
		dblist.setModel(dblm);
		if (idx != -1)
			select(idx);
	}

	private void select(int index)
	{
		if (index < -1 || index >= dblm.getSize())
			return;

		dblist.setSelectedIndex(index);
		if (index >= 0)
			dblist.ensureIndexIsVisible(index);
	}
	
	private void addEntry()
	{
		ded.setTitle(ExplainResources.getString(ExplainResources.PREFS_NEWDB_TITLE));
		ded.setConfigurationModel(cm);
		ded.setDatabaseModel(new PostgreSQLDatabaseModel());
		ded.show();

		if (ded.getResult() == DatabaseEditorDialog.OK_OPTION)
		{
			try {
				cm.addDatabaseModel(ded.getDatabaseModel());
				select(dblm.getSize() - 1);
			} catch (ConfigurationException ex) {
				// ignore for now
			}
		}
	}

	private void removeEntry()
	{
		cm.removeDatabaseModel(dblist.getSelectedIndex());
	}
	
	private void editEntry()
	{
		if (dblist.getSelectedIndex() == -1)
			return;

		DatabaseModel sel = cm.getDatabaseModel(dblist.getSelectedIndex());
		DatabaseModel clone = (DatabaseModel) sel.clone();

		ded.setTitle(ExplainResources.getString(ExplainResources.PREFS_EDITDB_TITLE));
		ded.setConfigurationModel(cm);
		ded.setDatabaseModel(clone);
		ded.show();

		if (ded.getResult() == DatabaseEditorDialog.OK_OPTION)
		{
			try {
				cm.setDatabaseModel(dblist.getSelectedIndex(), ded.getDatabaseModel());
			} catch (ConfigurationException ex) {
				// ignore - duplicate names already handled
			}
		}
	}

	private void copyEntry()
	{
		if (dblist.getSelectedIndex() == -1)
			return;

		DatabaseModel sel = cm.getDatabaseModel(dblist.getSelectedIndex());
		DatabaseModel copy = (DatabaseModel) sel.clone();

		boolean ok = false;
		String orig = copy.getName();
		int i = 1;

		while (!ok)
		{
			copy.setName(ExplainResources.getString(ExplainResources.PREFS_COPYDB_NAME,
													new Integer(i),
													orig));
			i++;

			try {
				cm.addDatabaseModel(copy);
				select(dblm.getSize() - 1);
				ok = true;
			} catch (ConfigurationException ex) {
				// ignore
			}
		}
	}
	
	private void initComponents()
	{
		JPanel content;
		JPanel buttonPanel;

		content = new JPanel();
		content.setLayout(new BorderLayout());
		content.setBorder(BorderFactory.createEmptyBorder(0, 0, 0, 0));
		setContentPane(content);

		buttonPanel = new ButtonPanel(this);
		getContentPane().add(buttonPanel, BorderLayout.SOUTH);

		dbprefs = createDatabasePanel();

		getContentPane().add(dbprefs, BorderLayout.CENTER);
	}
	
	private JPanel createDatabasePanel()
	{
		JButton addButton, confButton, remButton, copyButton;
		JPanel panel, buttonpanel;
		JScrollPane scroller;

		GridBagLayout gbl = new GridBagLayout();
		GridBagConstraints gbc = new GridBagConstraints();
		
		panel = new JPanel();
		panel.setBorder(BorderFactory.createEmptyBorder(12, 12, 11, 11));
		panel.setLayout(gbl);

		// the list
		dblist = new JList();
		dblist.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		dblist.addListSelectionListener(dblsl);
		dblist.addMouseListener(new MouseAdapter() {
				public void mouseClicked(MouseEvent e)
				{
					if (e.getClickCount() == 2)
						editEntry();
				}
			});
		scroller = new JScrollPane(dblist);

		gbc.gridx = 0;
		gbc.gridy = 0;
		gbc.gridwidth = 1;
		gbc.gridheight = 4;
		gbc.fill = GridBagConstraints.BOTH;
		gbc.anchor = GridBagConstraints.NORTHWEST;
		gbc.weightx = 1.0;
		gbc.weighty = 1.0;
		gbl.setConstraints(scroller, gbc);
		panel.add(scroller);

		// side buttons
		gbc.anchor = GridBagConstraints.NORTH;
		gbc.fill = GridBagConstraints.NONE;
		gbc.insets = new Insets(0, 11, 5, 0);
		gbc.gridwidth = 1;
		gbc.gridheight = 1;
		gbc.weighty = 0.0;
		gbc.weightx = 0.0;
		gbc.gridx = 1;

		remButton  = removeAction.getButton();
		remButton.setMinimumSize(remButton.getPreferredSize());
		addButton  = addAction.getButton();
		addButton.setPreferredSize(remButton.getPreferredSize());
		addButton.setMinimumSize(remButton.getPreferredSize());
		confButton = configAction.getButton();
		confButton.setPreferredSize(remButton.getPreferredSize());
		confButton.setMinimumSize(remButton.getPreferredSize());
		copyButton = copyAction.getButton();
		copyButton.setPreferredSize(remButton.getPreferredSize());
		copyButton.setMinimumSize(remButton.getPreferredSize());

		gbc.gridy = 0;
		gbl.setConstraints(addButton, gbc);
		panel.add(addButton);
		
		gbc.gridy = 1;
		gbl.setConstraints(confButton, gbc);
		panel.add(confButton);

		gbc.gridy = 2;
		gbl.setConstraints(copyButton, gbc);
		panel.add(copyButton);
		
		gbc.gridy = 3;
		gbc.insets = new Insets(0, 11, 0, 0);
		gbl.setConstraints(remButton, gbc);
		panel.add(remButton);

		return panel;
	}

	//
	// list model for database list
	//

	private class DatabaseListModel extends AbstractListModel {
		private ConfigurationModel cm;
		
		public DatabaseListModel(ConfigurationModel cm)
		{
			this.cm = cm;
		}

		public Object getElementAt(int index)
		{
			return cm.getDatabaseModel(index).getName();
		}

		public int getSize()
		{
			return cm.getDatabaseCount();
		}
	}

	//
	// list selection listener
	//

	private class DatabaseListSelectionListener implements ListSelectionListener {
		public void valueChanged(ListSelectionEvent e)
		{
			if (e.getValueIsAdjusting())
				return;

			if (dblist.isSelectionEmpty())
			{
				configAction.setEnabled(false);
				removeAction.setEnabled(false);
				copyAction.setEnabled(false);
			}
			else
			{
				configAction.setEnabled(true);
				removeAction.setEnabled(true);
				copyAction.setEnabled(true);
			}
		}
	}

}// DatabasesDialog
