/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.vise;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;
import javax.swing.text.html.*;
import java.io.*;
import java.net.*;
import java.util.*;
import java.util.ArrayList;
import java.util.ListIterator;

// need to think about history and how it relates to contents and
// index trees.

/**
 * Help window for Explain.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 */
public class HelpWindow extends javax.swing.JFrame implements ActionListener, TreeSelectionListener {

	private GenericAction backAction, forwardAction, homeAction;
	private ActionFactory afactory;
	private JTree contentsTree, indexTree;
	private JList searchList;
	private JEditorPane viewer;
	private	JScrollPane scrollpane;
	private ArrayList history;
	private int current;
	private Page home;
	private DefaultMutableTreeNode top;
	private DefaultTreeModel contentsModel, indexModel;
	private boolean debugContentsFile = false;	// Turn this one to track table of contents creation
	private int level;

	/** Creates new HelpWindow object */
	public HelpWindow()
	{
		contentsModel = null;

		afactory = new ActionFactory(ExplainResources.getString(ExplainResources.IMAGE_PATH));
		backAction = afactory.getAction(ActionFactory.ACTION_BACK);
		forwardAction = afactory.getAction(ActionFactory.ACTION_FORWARD);
		homeAction = afactory.getAction(ActionFactory.ACTION_HOME);

		backAction.addActionListener(this);
		forwardAction.addActionListener(this);
		homeAction.addActionListener(this);

		initComponents();
		initHelp();
		if (contentsModel != null)
			reset();

		setDefaultCloseOperation(javax.swing.WindowConstants.HIDE_ON_CLOSE);
		setTitle(ExplainResources.getString(ExplainResources.HELP_TITLE));
	}

	/**
	 * Action listener (handles ActionEvent's).
	 *
	 * @param e an <code>ActionEvent</code> value
	 */
	public void actionPerformed(ActionEvent e)
	{
		String command = e.getActionCommand();

		if (command.equals(backAction.getActionCommand()))
		{
			back();
		}
		else if (command.equals(forwardAction.getActionCommand()))
		{
			forward();
		}
		else if (command.equals(homeAction.getActionCommand()))
		{
			load(home);
		}
	}

	/**
	 * Tree selection change listener.
	 *
	 * @param e an <code>TreeSelectionEvent</code> value
	 */
	public void valueChanged(TreeSelectionEvent e)
	{
		DefaultMutableTreeNode node = null;

		if (e.getSource() == contentsTree)
			node = (DefaultMutableTreeNode) contentsTree.getLastSelectedPathComponent();
		else if (e.getSource() == indexTree)
			node = (DefaultMutableTreeNode) indexTree.getLastSelectedPathComponent();

        if (node == null)
			return;

        Object nodeInfo = node.getUserObject();

        if (nodeInfo != null &&
			nodeInfo instanceof Page)
		{
            Page p = (Page) nodeInfo;
			load(p);
        }
	}

	// inherits doc comment
	public void show()
	{
		if (contentsModel == null)
		{
			JOptionPane.showMessageDialog(this,
										  ExplainResources.getString(ExplainResources.MANUAL_NOT_FOUND),
										  ExplainResources.getString(ExplainResources.MANUAL_NOT_FOUND_TITLE),
										  JOptionPane.ERROR_MESSAGE);
			return;
		}

		if (! isVisible())
		{
			// For some reason the JRE forgets the preferred size after
			// we dismiss and re-show the window, so we must reset it here
			scrollpane.getViewport().setPreferredSize(new Dimension(400, 600));
			validate();
			
			pack();
			reset();
		}
		super.show();
	}
	
	/*
	 * going back callback
	 */
	private void back()
	{
		// can we go back?
		if (current != 0)
		{
			// load previous page
			current--;

			update();
		}
	}

	/*
	 * going forward callback
	 */
	private void forward()
	{
		// can we go forward?
		if (current != history.size()-1)
		{
			// load new page
			current++;

			update();
		}
	}

	/*
	 * go to arbitrary page
	 */
	private void load(Page p)
	{
		if (current != -1 &&
			history.get(current).equals(p))
			return;

		// remove [current+1, history.size())
		int idx = current + 1;
		while (idx < history.size())
			history.remove(idx);

		// load page
		history.add(p);
		current++;

		update();
	}

	/*
	 * reset everything to a known state. clear all history, load the
	 * home page.
	 */
	private void reset()
	{
		// collapse all rows, except top one
		// remove all history stuff
		// put us back at home page

		int row = contentsTree.getRowCount() - 1;
		while (row >= 0)
			contentsTree.collapseRow(row--);
		contentsTree.expandRow(0);
		history.clear();
		current = -1;
		if (home != null)
			load(home);
		else
		{
			homeAction.setEnabled(false);
			backAction.setEnabled(false);
			forwardAction.setEnabled(false);
		}

		// Make the node with the title "Visual Explain" visible, expanded and selected
		contentsTree.expandPath(new TreePath(findTitle("Visual Explain", top).getPath()));
		contentsTree.setSelectionPath(new TreePath(findTitle("Visual Explain", top).getPath()));
	}
	
	/*
	 * load currently selected page and update back/forward buttons
	 */
	private void update()
	{
		Page p = (Page) history.get(current);

		// find the associated tree node in contents model and select
		// it int the tree
		DefaultMutableTreeNode node = find(p, (DefaultMutableTreeNode)contentsModel.getRoot());

		if (node != null)
			contentsTree.setSelectionPath(new TreePath(contentsModel.getPathToRoot(node)));

		// load page
		try {
			viewer.setPage(p.getURL());
		} catch (Exception ex) {
			System.err.println("Unable to load page: " + p.getURL() + "\nReason:" + ex);
		}

		// enable/disable back and forward buttons
		if (current == 0)
			backAction.setEnabled(false);
		else
			backAction.setEnabled(true);

		if (current == history.size() - 1)
			forwardAction.setEnabled(false);
		else
			forwardAction.setEnabled(true);
	}

	/*
	 * initialize help stuff
	 */
	private void initHelp()
	{
		// load/create contents & index
		// need to set up search facilities?
		history = new ArrayList();

		top = createContentsTree();
		if (top != null)
		{
			contentsModel = new DefaultTreeModel(top);
			contentsTree.setModel(contentsModel);
		}
	}

	private DefaultMutableTreeNode createContentsTree()
	{
		BufferedReader infile = null;
		try {
			infile = new BufferedReader(new InputStreamReader(new FileInputStream(
										ExplainResources.getString(ExplainResources.DOCS_PATH) + "index.html")));
		} catch (FileNotFoundException fnfe) {
			System.out.println(ExplainResources.getString(ExplainResources.MANUAL_FILE_NOT_FOUND,
									ExplainResources.getString(ExplainResources.DOCS_PATH) + "index.html"));
			return null;
		}

		level = 0;
		DefaultMutableTreeNode root = createChildNodes(infile, null);
		// Avoid null pointer exception if we found nothing
		if (root == null)
			root = new DefaultMutableTreeNode();
		else
			home = (Page)(root.getUserObject());
		
		return root;
	}

	private DefaultMutableTreeNode createChildNodes(BufferedReader infile, DefaultMutableTreeNode parentnode)
	{
		int ichar;
		String line;
		String title;
		String pageref;
		Page p;
		// Initialize this to the parent node in case we found a lost DL tag
		DefaultMutableTreeNode node = parentnode;
		
		try { while ((ichar = infile.read()) != -1)
		{
			char c = (char)ichar;
			
			// This is a simplified parser.  We assume all string values are in one line.
			// Also, we assume a well formed document.
			
			// We start looking for tags...
			if (c != '<')
				continue;
				
			// Found a tag.  Which tag?
			c = (char)infile.read();
			if (c == -1)
				return parentnode;
			if (c == 'a' || c == 'A')
			{
				// confirm it is an A tag
				c = (char)infile.read();
				if (c == -1)
					return parentnode;
				if ((c != ' ') && (c != '\n') && (c != '\r'))
					continue;
					
				// Look for "HREF"
				while ((c != -1) && (c != 'H'))
					c = (char)infile.read();
				if (c == -1)
					return parentnode;

				// Boldly read a line
				line = infile.readLine();
				if (line == null)
					return parentnode;
					
				// Confirm it was an HREF and extract the data
				if (!line.startsWith("REF=\""))
					continue;
				line = line.substring(line.indexOf("\"") + 1);
				pageref = line.substring(0, line.indexOf("\""));
				if (debugContentsFile)
					System.out.println("ref=" + pageref);
				line = line.substring(line.indexOf("\""));
				while (line.indexOf(">") == -1)
					line = infile.readLine();
				line = line.substring(line.indexOf(">") + 1);
				
				StringBuffer titlebuf = new StringBuffer(80);
				while (line.indexOf("<") == -1)
				{
					titlebuf.append(line);
					line = infile.readLine();
				}
				titlebuf.append(line.substring(0, line.indexOf("<")));
				title = titlebuf.toString();
				if (debugContentsFile)
					System.out.println("title=" + title);

				// Ignore links to "Next" and "Copyright"
				if (title.equals("Next") || title.equals("Copyright"))
					continue;

				// We have a complete page reference
				p = new Page(title, pageref);
				node = new DefaultMutableTreeNode(p);
				// If first thing we found, make it a child of this page
				// which we know is index.html and we will always
				// use the title "Table of Contents"
				if (parentnode == null)
					parentnode = new DefaultMutableTreeNode(new Page("Table of Contents", "index.html"));
				// Add this subsession/subchapter to its parent
				parentnode.add(node);
				
				continue;
			}
			else if (c == 'd' || c == 'D')
			{						
				// confirm it is an dl tag
				c = (char)infile.read();
				if (c == -1)
					return parentnode;
				if ((c != 'l') && (c != 'L'))
					continue;
				
				// Go one level down
				level++;
				if (debugContentsFile)
					System.out.println("One DOWN to level " + level);
				// Recurse over children of this node
				DefaultMutableTreeNode retnode = createChildNodes(infile, node);
				// Handle misplaced DL tags
				// We get this because there is no link to index.html
				// on itself, so a DL is found before any chapter title
				if (node == null)
					parentnode = retnode;
				
				continue;
			}
			else if (c == '/')
			{						
				// confirm it is a /dl tag
				c = (char)infile.read();
				if (c == -1)
					return parentnode;
				if ((c != 'd') && (c != 'D'))
					continue;
				c = (char)infile.read();
				if (c == -1)
					return parentnode;
				if ((c != 'l') && (c != 'L'))
					continue;
				
				// We are done with this level
				level--;
				if (debugContentsFile)
					System.out.println("One UP to level " + level);
				return parentnode;
			}
			else
				continue;
		} } catch (IOException ioe)
		{
			// Nothing more we can do but to go up
			ioe.printStackTrace();
			return parentnode;
		}
		return parentnode;
	}

	private DefaultMutableTreeNode findTitle(String title, DefaultMutableTreeNode node)
	{
		if ((title == null) || (node == null))
			return null;

		Page data = (Page)node.getUserObject();
		
		if (data.getTitle().equals("Visual Explain"))
			return node;

		Enumeration e = node.breadthFirstEnumeration();
		while (e.hasMoreElements())
		{
			DefaultMutableTreeNode n = (DefaultMutableTreeNode)e.nextElement();
			data = (Page)n.getUserObject();
			if (data.getTitle().equals(title))
				return n;
		}

		return null;
	}

	private DefaultMutableTreeNode find(Page p, DefaultMutableTreeNode node)
	{
		if (p == null || node == null)
			return null;

		Page data = (Page)node.getUserObject();
		DefaultMutableTreeNode n;
		
		if (data.equals(p))
			return node;

		// FIXME Should use an enumeration instead of doing it by ourselves
		for (int i = 0; i < node.getChildCount(); i++)
		{
			n = find(p, (DefaultMutableTreeNode)node.getChildAt(i));
			if (n != null)
				return n;
		}

		return null;
	}

	/*
	 * This method is called from within the constructor to
	 * initialize the form.
	 */
	private void initComponents()
	{
		JPanel panel1, panel2;
		JButton button;
		JTabbedPane tabbedpane;
		JSplitPane splitpane;
		JTextField textfield;
		JToolBar toolbar;
		JLabel label;

		//
		// toolbar
		//

		toolbar = new JToolBar();
		toolbar.setFloatable(false);

		button = backAction.getToolbarButton();
		toolbar.add(button);

		button = forwardAction.getToolbarButton();
		toolbar.add(button);

		button = homeAction.getToolbarButton();
		toolbar.add(button);

		getContentPane().add(toolbar, java.awt.BorderLayout.NORTH);

		//
		// split pane
		//

		splitpane = new JSplitPane();
		splitpane.setDividerLocation(150);
		splitpane.setOneTouchExpandable(true);

		// contents tab
		
		contentsTree = new JTree();
		//contentsTree.setShowsRootHandles(true);
		contentsTree.addTreeSelectionListener(this);

		scrollpane = new JScrollPane();
		scrollpane.setViewportView(contentsTree);

		splitpane.setLeftComponent(scrollpane);

		// text pane

		viewer = new JEditorPane();
		viewer.setBorder(new javax.swing.border.EmptyBorder(new Insets(2, 2, 2, 2)));
		viewer.setEditable(false);
		viewer.setPreferredSize(new Dimension(400, 600));
		viewer.addHyperlinkListener(new HyperlinkListener()
		{
			public void hyperlinkUpdate(HyperlinkEvent e)
			{
				if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED)
				{
					JEditorPane pane = (JEditorPane) e.getSource();
					if (e instanceof HTMLFrameHyperlinkEvent) {
						HTMLFrameHyperlinkEvent  evt = (HTMLFrameHyperlinkEvent)e;
						HTMLDocument doc = (HTMLDocument)pane.getDocument();
						doc.processHTMLFrameHyperlinkEvent(evt);
					} else {
						load(new Page("", e.getURL()));
					}
				}
			}
		});

		scrollpane = new JScrollPane(viewer);

		splitpane.setRightComponent(scrollpane);

		getContentPane().add(splitpane, BorderLayout.CENTER);

		pack();
	}

	private class Page {
		private String title;
		private String file;
		private URL url;

		public Page(String title, String file)
		{
			this.title = title;
			this.file = file;

			if (debugContentsFile)
				System.out.println("title=" + title + " file=" + file);

			try
			{
				url = new URL("file:" + ExplainResources.getString(ExplainResources.DOCS_PATH) + file);
			}
			catch (MalformedURLException mue)
			{
				mue.printStackTrace();
			}
	
			if (debugContentsFile)
				System.out.println("URL=" + url);
		}

		public Page(String title, URL url)
		{
			this.title = title;
			this.url = url;
			// FIXME Shouldn't we extract the file name from the URL?
			this.file = null;
		}

		public String getTitle()
		{
			return title;
		}

		public URL getURL()
		{
			return url;
		}

		public boolean equals(Page other)
		{
			// Pages are equal if their URLs are equal
			return url.equals(other.getURL());
		}

		public String toString()
		{
			if (!title.equals(""))
				return title;
			else
				return url.toString();
		}
	}
}
