/*
 * Copyright 2002 Murray Cumming
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "bakery_gnomeui/Configuration/Client.h"

namespace Bakery
{

namespace Conf
{

//----------------------------------------------------------------------
// Client_GnomeUI implementation. This is a subclass of
// Bakery::Conf::Client that supports various Gnome::UI widgets.
//----------------------------------------------------------------------

Client_GnomeUI::Client_GnomeUI(const Glib::ustring& configuration_directory)
: Client(configuration_directory)
{
}

Client_GnomeUI::~Client_GnomeUI()
{
}

void Client_GnomeUI::add_implementation(const Glib::ustring& key, Gtk::Widget& widget, bool instant)
{
  // Note: Gnome::UI::Entry is already supported because it is a kind of Gtk::Combo.
  // We don't need to worry about the history support that's added by Gnome.

  if (dynamic_cast<Gnome::UI::ColorPicker*>(&widget))
    add_association(key,static_cast<Gnome::UI::ColorPicker&>(widget),instant);
  else if (dynamic_cast<Gnome::UI::FontPicker*>(&widget))
    add_association(key,static_cast<Gnome::UI::FontPicker&>(widget),instant);
  else if (dynamic_cast<Gnome::UI::FileEntry*>(&widget))
    add_association(key,static_cast<Gnome::UI::FileEntry&>(widget),instant);
  else
    Bakery::Conf::Client::add_implementation(key, widget, instant);

  // TODO: Support for DateEdit, IconEntry and PixmapEntry controls. Issues to be
  // be resolved are:
  // 	* How to express DateEdit values in gconf. It needs to be a string, but should
  //	it be a raw time_t value converted to string or should it be a standard date/time
  //	format expressed in UTC time? If the latter, which standard?
  //	* IconEntry does not have a signal_changed defined, though the underlying GTK+
  //	widget does.
  //	* PixmapEntry does not have a signal_changed defined, nor does it have any
  //	method for setting the value of the widget.
}

//----------------------------------------------------------------------
// Association<T> specializations. These provide widget/key
// association behaviors that are specific to individual widget types.
//----------------------------------------------------------------------

void Association<Gnome::UI::ColorPicker>::connect_widget(Callback widget_changed)
{
  using namespace SigC;
  m_widget.signal_color_set().connect(hide<guint>(hide<guint>(hide<guint>(hide<guint>(widget_changed)))));
}

void Association<Gnome::UI::ColorPicker>::load_widget()
{
  char color_string[8];
  guint8 r, g, b, a; 
  m_widget.get(r,g,b,a);
  sprintf(color_string,"#%02X%02X%02X",r,g,b);

  Gdk::Color widget_color(color_string), conf_color(get_conf_client()->get_string(get_key()));
  
  if (conf_color != widget_color)
  {
    r = conf_color.get_red();
    g = conf_color.get_green();
    b = conf_color.get_blue();
    m_widget.set(r,g,b,0);
  }
}

void Association<Gnome::UI::ColorPicker>::save_widget()
{
  char color_string[8];
  guint8 r, g, b, a;
  m_widget.get(r,g,b,a);
  sprintf(color_string,"#%02X%02X%02X",r,g,b);
  
  Gdk::Color widget_color(color_string), conf_color(get_conf_client()->get_string(get_key()));

  if (conf_color != widget_color)
    get_conf_client()->set(get_key(), Glib::ustring(color_string));
}

//----------------------------------------------------------------------

void Association<Gnome::UI::FontPicker>::connect_widget(Callback widget_changed)
{
  using namespace SigC;
  m_widget.signal_font_set().connect(hide<const Glib::ustring&>(widget_changed));
}

void Association<Gnome::UI::FontPicker>::load_widget()
{
  const Glib::ustring& value = get_conf_client()->get_string(get_key());
  if (value != m_widget.get_font_name() && !value.empty())
    m_widget.set_font_name(value);
}

void Association<Gnome::UI::FontPicker>::save_widget()
{
  const Glib::ustring& value = m_widget.get_font_name();
  if (value != get_conf_client()->get_string(get_key()))
    get_conf_client()->set(get_key(), value);
}

//----------------------------------------------------------------------

void Association<Gnome::UI::FileEntry>::connect_widget(Callback widget_changed)
{
  m_widget.signal_activate().connect(widget_changed);
}

void Association<Gnome::UI::FileEntry>::load_widget()
{
  const Glib::ustring& value = get_conf_client()->get_string(get_key());
  if (value != m_widget.get_full_path(false) && !value.empty())
    m_widget.set_filename(value);
}

void Association<Gnome::UI::FileEntry>::save_widget()
{
  const Glib::ustring& value = m_widget.get_full_path(false);
  if (value != get_conf_client()->get_string(get_key()))
    get_conf_client()->set(get_key(), value);
}

} //namespace Conf

} //namespace Bakery