/*
 * game.hh - header for game.cc for Bombermaze
 * written by Sydney Tang <stang@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more details see the file COPYING.
 */

#ifndef _BOMBER_GAME_H_
#define _BOMBER_GAME_H_

#include "map.hh"
#include "input.hh"
#include "ui.hh"

class GameState;

int game_initialize(UIWidget *widget);
void game_terminate(void);
GameState *game_get_current_state(void);
bool check_if_current_state_allows_theme_change(void);

bool game_set_map(const char *mapfile_path);
void game_start_new(UIWidget *widget, int players);
void game_toggle_pause(UIWidget *widget, void *data);
void game_set_pause(bool pause_setting);
bool game_get_pause(void);

///////////////////////////////////////////////////////////////////////////////

class GameStateMachine
{
 public:

  enum GameStates
  {
    STATE_TITLESCREEN=0,
    STATE_MAINMENU,
    STATE_1_PLAYER,
    STATE_MULTIPLAYER,
    STATE_1_PLAYER_GAME_OVER,
    STATE_MULTIPLAYER_GAME_OVER,
    STATE_HIGHSCORES,
    NUMBER_OF_STATES
  };

  UIWidget *UIContainerWidget;
  UIWidget *UIActiveWidget;

  GameStateMachine();
  ~GameStateMachine();
  
  GameStates get_state(void);
  GameState *get_game_state(void);

  void change_state(GameStates NewState, int InitializationData);
  void start_game(int NumPlayers);
  void start_single_player_game(void);
  void start_multiplayer_game(int NumPlayers);

  void ui_dependent_initialization(UIWidget *widget);

  bool set_map(const char *mapfile_path);
  ParsedMap *get_map(void);

 protected:
  GameState *state[NUMBER_OF_STATES];
  GameStates status;

  ParsedMap ReferenceMap;

  void initialize_current_state(int InitializationData);
  void terminate_current_state(void);
  
}; // class GameStateMachine

GameStateMachine::GameStates game_get_current_state_type(void);

///////////////////////////////////////////////////////////////////////////////

class GameState
{
 public:

  enum GameOverCondition
  {
    NOT_GAME_OVER = -1,
    STALEMATE = -2,
    RESUME_MATCH = -3,
  };

  GameState(GameStateMachine *sm);
  virtual ~GameState();

  virtual void initialize_state(int InitializationData);
  virtual void terminate_state(void);

  virtual void handle_keypress(KeyValue keyval);
  virtual void handle_keyrelease(KeyValue keyval);

 protected:

  GameStateMachine *owner;

  bool handle_new_game_keypress(KeyValue keyval);

}; // class GameState

///////////////////////////////////////////////////////////////////////////////

class TitleScreenState : public GameState
{
 public:
  TitleScreenState(GameStateMachine *sm);
  ~TitleScreenState();

  void initialize_state(int InitializationData);
  void terminate_state(void);

  void handle_keypress(KeyValue keyval);
  void handle_keyrelease(KeyValue keyval);

 protected:
  void draw_title_screen(void);
};

///////////////////////////////////////////////////////////////////////////////

class InGameState : public GameState
{
 public:

  InGameState(GameStateMachine *sm);
  ~InGameState();

  void initialize_state(int n_players);
  void terminate_state(void);

  void handle_keypress(KeyValue keyval);
  void handle_keyrelease(KeyValue keyval);
  void match_status_keypress(KeyValue keyval);
  void match_status_continue(void);

  void toggle_pause(void);
  void set_pause(bool pause_setting);
  bool get_pause(void);

  void set_number_of_players(int NumPlayers);
  int  get_number_of_players(void);

  int timeout_callback(void *data);
  static unsigned get_timeout_interval(void);
  static void set_timeout_interval(unsigned delay);

  static void set_wins_per_match(unsigned wins);

 protected:

  enum InGameStateConstants
  {
    TIMEOUT_INTERVAL_MINIMUM = 1,
    TIMEOUT_INTERVAL_MAXIMUM = 100,
    DEFAULT_TIMEOUT_INTERVAL = 10,

    DEFAULT_WINS_PER_MATCH = 2,

    SUSTAINED_MOVEMENT = 0,
    SUSTAINED_BOMBDROP = 1,
    MAX_ALLOWED_SUSTAINED_ACTIONS = 2
  };

  static unsigned Timeout_Interval;
  static unsigned WinsPerMatch;

  UIWidget *MatchStatusWindow;

  GameMap maze;

  int NumberOfPlayers;
  unsigned NumberOfWins[Player::MAX_NUMBER_OF_PLAYERS];
  unsigned round;
  Player player[Player::MAX_NUMBER_OF_PLAYERS];
  int SustainedPlayerAction
      [MAX_ALLOWED_SUSTAINED_ACTIONS][Player::MAX_NUMBER_OF_LOCAL_PLAYERS];

  bool paused;
  bool TimeoutHandlingInProgress;
  unsigned TimeoutHandlerID;

  void reset_match(void);
  void initialize_round(void);
  void initialize_players(void);
  void terminate_round(void);

  void draw_maze(void);
  static void undraw_maze(void);
  void unref_ui_components(void);

  void initialize_timer(void);
  void disable_timer(void);

  void perform_player_action(PlayerActionKeyBinding *action);

  void handle_game_over_conditions(void);
  int  check_if_game_over(void);

  void update_score_display(void);
  void notify_pause_status(void);
  void display_match_status(int winner);
  void hide_match_status(void);
  void update_ui(void);
};

///////////////////////////////////////////////////////////////////////////////

class GameOverState: public GameState
{
 public:

  GameOverState(GameStateMachine *sm);
  ~GameOverState();

  void initialize_state(int winner_of_match);
  void terminate_state(void);

  void handle_keypress(KeyValue keyval);
  void handle_keyrelease(KeyValue keyval);

 protected:

  UIWidget *GameOverWindow;
  int winner;
  //bool GameOver;

  void draw_game_over_screen(int winner);
  void hide_game_over_screen(void);
};

///////////////////////////////////////////////////////////////////////////////

#endif

