
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * The sermon generates the waves for all the oscillators required by a hammond.
 * Different sources put this at 91, 92, or 95 oscillators. By my reconning we
 * need one for every key on the hammond (61), plus the range of the drawbars,
 * (5 octaves = 62). In short, the lowest drawbar goes 12 notes below the 
 * bottom key, and the highest note goes 3 octaves (36 notes) above, or rather
 *		61 + 12 + 36 = 109 frequencies.
 * We pull back to 91 using "foldback", which is reusing different bus routings
 * on the uppermost and lowermost ranges.
 */

#include <math.h>

#include "bristol.h"
#include "hammond.h"

#define OSC_COUNT 109
#define OSC_LIMIT 91
#define WAVE_SIZE 1024
#define WAVE_COUNT 8
#define BUS_COUNT 9

static int filltriwave(float *, int, double, int);
static int fillWave(float *, int, int, int);

/*
 * These are the midi note offsets for any given key harmonics.
 */
static offsets[9] = {-12, 7, 0, 12, 19, 24, 28, 31, 36};

/*
 * Get a humungous chunk of memory, SAMPLE_COUNT should come from the audiomain
 * structure.
 */
static float *oscillators = NULL;

/*
 * Get a stackload of pointers into this array. We will use a config option to
 * enable foldback.
 */
static float *tonewheel[OSC_COUNT];
static float toneindeces[OSC_COUNT];
static int notelimit = OSC_COUNT; // Will provide for foldback.

/*
 * Get some memory for the base waves, plus some pointers.
 */
static float waves[WAVE_COUNT * WAVE_SIZE * sizeof(float)];
static float *wave[WAVE_COUNT];

thesermon(int samplecount, float *ctab, int sineform)
{
	register int i, count;
	register float index, *source, *dest, freq;

	/*
	 * We now have 8 waves, which are anything from sines, leading edge sines,
	 * tris and ramps. We now need to fill all 109 tables with the number of 
	 * samples required for the next buffer SAMPLE_COUNT samples from these
	 * wave tables. Choice of wave will initially be parameterised, but 
	 * eventually we should be looking to select the wave based on the octave
	 * being stuffed.
	 *
	 * We could consider only filling this table fully when a given key is 
	 * requested. That may have an effect on overall phasing though.
	 */
	for (i = 0; i < OSC_LIMIT; i++)
	{
//		if (sineform > 3)
//			sineform = 3;
		if (sineform < 0)
			source = wave[0];
		else if (i < 12)
			source = wave[sineform + 4 >= 8? 7:sineform + 4];
		else if (i < 18)
			source = wave[sineform + 3 >= 8? 7:sineform + 3];
		else if (i < 24)
			source = wave[sineform + 2 >= 8? 7:sineform + 2];
		else if (i < 30)
			source = wave[sineform + 1 >= 8? 7:sineform + 1];
		else if (i > 84)
			source = wave[sineform + 1 >= 8? 7:sineform + 1];
		else
			source = wave[sineform >= 8? 7:sineform];

		index = toneindeces[i];
		dest = tonewheel[i];
		freq = ctab[i + 24];
		count = samplecount;

		do {
			*dest++ = source[(int) index];
			if ((index += freq) > WAVE_SIZE) index -= WAVE_SIZE;
			*dest++ = source[(int) index];
			if ((index += freq) > WAVE_SIZE) index -= WAVE_SIZE;
			*dest++ = source[(int) index];
			if ((index += freq) > WAVE_SIZE) index -= WAVE_SIZE;
			*dest++ = source[(int) index];
			if ((index += freq) > WAVE_SIZE) index -= WAVE_SIZE;
			*dest++ = source[(int) index];
			if ((index += freq) > WAVE_SIZE) index -= WAVE_SIZE;
			*dest++ = source[(int) index];
			if ((index += freq) > WAVE_SIZE) index -= WAVE_SIZE;
			*dest++ = source[(int) index];
			if ((index += freq) > WAVE_SIZE) index -= WAVE_SIZE;
			*dest++ = source[(int) index];
			if ((index += freq) > WAVE_SIZE) index -= WAVE_SIZE;
		} while ((count -= 8) > 0);

		toneindeces[i] = index;
	}
}

/*
 * This is the bussing algorithm. We should consider bus delays for each bus
 * on key_on, so that each bus is activated at a slightly different time, as
 * per a normal multitap mechanical key.
 */
preach(register float *buf, register float *pbuf, int note, int *gains,
int *percs, register int count, float gn)
{
	register int i, j, index;
	register float gain, *dest = buf, *source;

	gain = 0;

	for (i = 0; i < BUS_COUNT; i++)
	{
		gain = ((float) gains[i]) * gn * 0.125;
		if (percs[i])
			dest = pbuf;
		else
			dest = buf;
		/*
		 * We need to do foldback checks.
		 */
		if (((index = note + offsets[i]) >= 0) && (index < 91))
			source = tonewheel[index];
		else if ((index = note + offsets[i]) < 0)
			source = tonewheel[index + 12];
		else if (index >= (OSC_LIMIT + 12))
			source = tonewheel[index - 24];
		else if (index >= OSC_LIMIT)
			source = tonewheel[index - 12];

		for (j = count; j; j-=16)
		{
			*dest++ += *source++ * gain;
			*dest++ += *source++ * gain;
			*dest++ += *source++ * gain;
			*dest++ += *source++ * gain;
			*dest++ += *source++ * gain;
			*dest++ += *source++ * gain;
			*dest++ += *source++ * gain;
			*dest++ += *source++ * gain;
			*dest++ += *source++ * gain;
			*dest++ += *source++ * gain;
			*dest++ += *source++ * gain;
			*dest++ += *source++ * gain;
			*dest++ += *source++ * gain;
			*dest++ += *source++ * gain;
			*dest++ += *source++ * gain;
			*dest++ += *source++ * gain;
		}
	}
}

initthesermon(int samplecount, int compress)
{
	int i;

	printf("initthesermon(%i)\n", samplecount);
	/*
	 * Build the basic source wavetable.
	 */
	if (oscillators == NULL)
	{
		oscillators
			= (float *) bristolmalloc0(OSC_COUNT * samplecount * sizeof(float));

		for (i = 0; i < OSC_COUNT; i++)
		{
			tonewheel[i] = i * samplecount + oscillators;
			toneindeces[i] = rand() & 0x03ff;
			toneindeces[i] = 0;
		}
	}

	for (i = 0; i < WAVE_COUNT; i++)
	{
		wave[i] = i * WAVE_SIZE + waves;
		fillWave(wave[i], WAVE_SIZE, i, compress);
	}
}

static filltriwave(register float *mem, register int count,
register double reach, int compress)
{
	register int i, k = 0, recalc1 = 1, recalc2 = 1;
	register double j = 0, Count = (double) count, inc = reach, l;

	for (i = 0;i < count; i++)
	{
		if ((j > (count * 3 / 4)) && recalc2)
		{
			recalc2 = 0;
			inc = ((float) (count / 4)) / (((float) count) - i);
		} else if ((j > (count / 4)) && recalc1) {
			recalc1 = 0;
			/* 
			 * J now has the sine on a peak, we need to calculate the number
			 * of steps required to get the sine back to zero crossing by
			 * the time i = count / 2, ie, count /4 steps in total.
			 */
			inc = ((float) (count / 4)) / (((float) (count / 2)) - i);
		}

		j += inc;

		if (compress)
		{
			l = sin(((double) (2 * M_PI * -j)) / Count);
			if (l < 0)
				mem[i] = -16 * sqrt(-l * HAMMOND_WAVE_GAIN);
			else
				mem[i] = 16 * sqrt(l * HAMMOND_WAVE_GAIN);
		} else
			mem[i] = sin(((double) (2 * M_PI * j)) / Count) * HAMMOND_WAVE_GAIN;
	}

/*
	printf("\n%f, %f\n", j, inc);
	for (i = 0; i < count; i+=8)
	{
		printf("\n%i: ", i + k);
		for (k = 0; k < 8; k++)
			printf("%03.1f, ", mem[i + k]);
	}
	printf("\n");
*/
}

static fillWave(register float *mem, register int count, int type, int compress)
{
	register int i;
	register double j, Count;

#ifdef BRISTOL_DBG
	printf("fillWave(%x, %i, %i)\n", mem, count, type);
#endif

	Count = (double) count;

	switch (type) {
		case 0:
			/*
			 * This will be a sine wave. We have count samples, and
			 * 2PI radians in a full sine wave. Thus we take
			 * 		(2PI * i / count) * 2048.
			 */
			filltriwave(mem, count, 1.0, compress);
			return;
		case 1:
		default:
			filltriwave(mem, count, 1.05, compress);
			return;
		case 2:
			filltriwave(mem, count, 1.10, compress);
			return;
		case 3:
			filltriwave(mem, count, 1.30, compress);
			return;
		case 4:
			filltriwave(mem, count, 1.60, compress);
			return;
		case 5:
			filltriwave(mem, count, 2.0, compress);
			return;
		case 6:
			filltriwave(mem, count, 3.0, compress);
			return;
		case 7:
			filltriwave(mem, count, 5.0, compress);
			return;
	}
}

