(* $Id: rpc_server.mli 182 2004-05-25 16:49:11Z gerd $
 * ----------------------------------------------------------------------
 *
 *)

(* Rpc_server module:
 *
 * Like the client, the RPC server module is programmed on top of the
 * Unixqueue event system. It pushes itself on an existing Unixqueue
 * as a new service that accepts RPC calls, forwards them to configurable
 * functions, and sends the replies back.
 * The server module can manage two kinds of RPC functions: synchronous
 * and asynchronous. Synchronous functions compute their result immediately
 * and thus the result can be sent back just after the evaluation of the
 * function has finished. In contrast to this, asynchronous functions only
 * get noticed about the call and need not to know immediately what should
 * be answered. Typically, an asynchronous function initiates a second
 * communication channel and its result depends on what happens on the
 * second channel. The communication on this channel is done in an
 * asynchronous way, too, and can be managed by the same event system that
 * carries out the RPC service. After several input or output events,
 * the result has somehow been computed, and the answer can be sent
 * back to the original caller. To do so, the asynchronous RPC function
 * invokes 'reply' together with the necessary session IDs that identify
 * the answer among all answers.
 *)

open Rtypes
open Xdr
open Rpc

exception Connection_lost
  (* raised by the 'reply' function if the connection to the original caller
   * has been lost in the meantime.
   *)

type t
  (* represents a server for an RPC program *)

type session
  (* identifies a pair of a call and a reply *)

type connection_id
  (* identifies the connection of a session. For connectionless servers,
   * every session gets a new connection_id.
   * You can compare connection_ids to find out whether two sessions
   * belong to the same connection. Use "=" for equality.
   *)

type connector =
    Localhost of int
	(* The service is installed on 'localhost' and listens on the
	 * given port number. A number of 0 means that the port is
	 * chosen by the operating system.
	 * Note: The service is only locally reachable.
	 *)
  | Portmapped
        (* The service is installed on every network interface; the port is
	 * chosen by the operating system; the program is registered with the
	 * portmapper
	 *)
  | Internet of (Unix.inet_addr * int)
        (* The service is installed on the passed interface/port combination.
	 * Use Unix.inet_addr_any to listen on all network interfaces.
	 * Use port 0 to automatically choose the port number.
	 *)
  | Unix of string
	(* The service is installed on a Unix domain socket.
	 * Note: the socket path must not exist when the server is started,
	 * and the socket must be unlinked when the server terminates.
	 *)
  | Descriptor of Unix.file_descr
	(* The service listens on the given file descriptor. *)
  | Dynamic_descriptor of (unit -> Unix.file_descr)
	(* The service listens on the given file descriptor. *)

type binding_sync =
    { sync_name : string;                  (* procedure name *)
      sync_proc : xdr_value -> xdr_value   (* the function that implements the
					    * procedure
					    *)
    }

type binding_async =
    { async_name : string;                 (* procedure name *)
      async_invoke : session -> xdr_value -> unit
	  (* A function that is called when the procedure is called *)
    }

type binding =
    Sync of binding_sync     (* bind a synchonous procedure *)
  | Async of binding_async   (* bind an asynchonous procedure *)


val create :
    ?program_number:uint4 ->  (* Override program number *)
    ?version_number:uint4 ->  (* Override version number *)
    Unixqueue.event_system -> (* the underlying event queue *)
    connector ->           (* the address of the service *)
    protocol ->            (* Tcp: stream-oriented; Udp: datagram-oriented *)
    mode ->                (* Socket: serve multiple connections/datagrams;
			    * BiPipe: serve only a single connection
			    *)
    Rpc_program.t ->       (* The specification of the program *)
    binding list ->        (* The procedures *)
    int ->                 (* maximum number of waiting connections (backlog) *)
      t

  (* Creates a new server that is pushed onto the event queue.
   * The 'connector', 'protocol' and 'mode' values control the network
   * type of the server. Note that not all combinations are valid; the
   * following can be used:
   * - any connector, protocol=Tcp, mode=Socket:
   *     creates a classical Tcp server socket that allows multiple
   *     stream connections at the same time
   * - connector=Descriptor s, protocol=Tcp, mode=BiPipe:
   *     (where s is one half of a socketpair)
   *     creates a stream socket that is the endpoint of a point-to-point
   *     stream connection (bidirectional pipe)
   * - any Internet namespace connector, protocol=Udp, mode=Socket:
   *     creates a Udp server socket that allows serving multiple datagrams
   *
   * Note: If connector = Descriptor _ the file descriptor is not opened by
   * this module and not closed. The other 'connectors' work automatically
   * regarding this point, i.e. descriptors are opened and closed as
   * necessary.
   * connector = Dynamic_descriptor: The open descriptor is closed after use.
   *
   * The 'Rpc_program.t' specifies the procedures that are available and
   * their signatures. The 'binding list' should contain for every procedure
   * name the function that handles calls of the procedures.
   * The remaining integer is the maximum number of waiting connections
   * if a classical Tcp server socket is used; other connection types ignore
   * this number.
   *
   * The optional arguments ?program_number and ?version_number may override
   * the numbers specified in the passed program.
   *
   * NOTES ON SERVERS:
   * - servers that allow multiple connections never terminate by themselves
   *   (but you can force termination by raising an Abort exception, for
   *   example from a procedure)
   * - servers for only one connection (endpoint of a bidirectional pipe)
   *   terminate if they see an EOF on the stream; in this case the stream
   *   is closed by the server
   * - authentication is not yet possible
   * - the 'create' function may block if the connector is Portmapped
   *)

val get_event_system : session -> Unixqueue.event_system
  (* Find out the event system that contains the 'session' *)

val get_connection_id : session -> connection_id
  (* Get the connection_id *)

val get_socket_name : session -> Unix.sockaddr
val get_peer_name : session -> Unix.sockaddr
  (* Return the address of the socket serving the session, and the client
   * socket, resp.
   *)

val get_server : session -> t
  (* Returns the server instance of the session *)

val get_main_socket_name : t -> Unix.sockaddr
  (* Returns the address of the server socket, or the address of the
   * bidirectional pipe.
   * This function fails if the main file descriptor is not a socket.
   *)

type rule =
    [ `Deny
    | `Drop
    | `Reject
    | `Accept
    | `Accept_limit_length of (int * rule)
    ]

val set_session_filter : t -> (Unix.sockaddr -> rule) -> unit
  (* If set, the filter function is invoked every time the beginning of a new
   * RPC call is received, and the result of the filter function determines
   * what to do with the call:
   *
   * `Deny: TCP connections are immediately closed; UDP packets are dropped
   * `Drop: The call is dropped (it does not allocate memory)
   * `Reject: A response is sent back that the call is rejected (unauthorized)
   * `Accept: The call is accepted without limitation (the default if no
   *   filter is installed)
   * `Accept_limit_length(n,r): If the call is longer than n bytes, the rule
   *   r will be applied
   *
   * The parameter of the filter function is the socket address of the
   * client.
   *
   * The intention of filters is to prevent denial of service attacks.
   * A simple but good filter for TCP servers is
   *   set_filter srv (fun _ -> (`Accept_limit_length(n,`Deny))
   * which accepts messages up to n bytes without limit, and denies longer
   * messages. n is the length of the longest sensible message.
   *
   * For UDP servers, there is an implicit limit of 8000 bytes, so it is
   * not necessary to care about this.
   *
   * Another application is to restrict which systems can contact this
   * server, based on the IP address of the client.
   *
   * Note that this is not a protection against distributed denial of service
   * attacks.
   *)


val reply : session -> xdr_value -> unit
  (* Asynchronous procedures can reply their results with this function.
   * NOTES:
   * - As with synchronous procedures, the transfer is not reliable since
   *   the connection may be broken at any time
   * - If it is already known that the connection is down, a Connection_lost
   *   exception is raised.
   * - If you don't want to reply to a certain call, just don't [reply].
   *   Unreplied calls do not allocate memory.
   * - It is possible to reply several times ("batch mode"), but the client
   *   must support it, too. Just call [reply] several times for the same
   *   session.
   *)

val reply_error : session -> Rpc.server_error -> unit
  (* Like [reply], but an error condition is sent back to the caller. *)

val set_exception_handler : t -> (exn -> unit) -> unit
  (* Sets the exception handler for the server.
   * The exception handler gets most exceptions raised by the functions that
   * are bound to procedures. The exception handler does not get Abort
   * exceptions and any exceptions resulting from I/O problems.
   *
   * NOTES ABOUT EXCEPTIONS:
   * - The default exception handler just prints the exception on stderr.
   * - I/O problems usually lead to an 'Abort' of the whole server.
   *)

val set_onclose_action : t -> (connection_id -> unit) -> unit
  (* Every time a connection is closed, the onclose function is called
   * with the closed connection.
   * The default onclose action is to do nothing.
   * Note that this action only applies to closed connections. It will
   * not be executed for closed sockets in general (closed master socket,
   * closed datagram socket).
   *)

val stop_server : t -> unit
  (* Schedules a special event that causes the server to be stopped in the
   * very near future.
   * CHECK: Is it safe to invoke this function from a signal handler?
   *)

(************************* authentication **************************)

type auth_result =
    Auth_positive of (string * string * string)
      (* (username, returned_verifier_flavour, returned_verifier_data) *)
  | Auth_negative of Rpc.server_error

type auth_peeker =
    Unix.file_descr -> string option

class type auth_method =
object
  method name : string
    (* The name of the authentication method *)

  method flavors : string list
    (* Which credential flavors are handled by this method *)

  method peek : auth_peeker option
    (* If available, this function is called for every accepted connection.
     * It may return the user name.
     * Notes:
     * - peeked user names override [authenticate]
     * - [peek] is only called once after the stream connection has been
     *   accepted
     *)

  method authenticate :
           t ->
	   connection_id ->
	   Unix.file_descr ->               (* the serving socket *)
	   Unix.sockaddr ->                 (* address of serving socket *)
	   Unix.sockaddr ->                 (* address of calling socket *)
	   string ->                        (* flavor of credentials *)
	   string ->                        (* data of credentials *)
	   string ->                        (* flavor of verifier *)
	   string ->                        (* data of verifier *)
	   (auth_result -> unit) ->
	     unit
    (* This method is called when a remote call has arrived. Its task is
     * to determine the client user and pass the user name (and the verifier)
     * back. If the user cannot be authenticated, the call must be rejected.
     * When the method has done the authentication, it calls the passed
     * function with the [auth_result]. This function can be called
     * immediately or asynchronously.
     *)
end

val set_auth_methods : t -> auth_method list -> unit
  (* Sets the available authentication methods.
   * By default, the list is set to [ auth_none ].
   * If none of the methods apply, the call is rejected (Auth_too_weak).
   *)

val auth_none : auth_method
  (* The authentication method "AUTH_NONE", i.e. no user name is passed.
   * The function [get_user] will return "".
   *)

val auth_too_weak : auth_method
  (* The method that always rejects. *)

val get_user : session -> string
  (* Returns the user name as returned by the authentication method. See
   * the description of the method for the format of the user name string.
   *)

val get_auth_method : session -> auth_method
  (* Returns the method that was used to authenticate the user. *)

val verbose : bool -> unit
  (* Set whether you want debug messages or not *)
  (* Caution! This function is not protected against concurrent invocations
   * from several threads.
   *)
