/*
 * file-open-dialog.c
 *
 * crafted - a pud editor for the freecraft project.
 * 
 * Copyright (C) 2001-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "crafted.h"
#include "craftedintl.h"

#include "load_pud.h"

#include "file-dialog-utils.h"

#define CLAMP0255(a)  CLAMP(a,0,255)

static void  file_open_dialog_create      (void);


static void  file_open_genbutton_callback (GtkWidget     *widget,
                                           gpointer       data);

static void  file_open_clistrow_callback  (GtkWidget     *widget,
                                           gint           row);

static void  file_open_ok_callback        (GtkWidget     *widget,
                                           gpointer       data);

static void  file_open_type_callback      (GtkWidget     *widget,
                                           gpointer       data);

static GSList * clist_to_slist            (GtkCList      *file_list);


static GtkWidget  *fileload                    = NULL;
static GtkWidget  *open_options                = NULL;

static GtkPreview *open_options_preview        = NULL;
static GtkWidget  *open_options_fixed          = NULL;
static GtkWidget  *open_options_label          = NULL;
static GtkWidget  *open_options_frame          = NULL;
static GtkWidget  *open_options_genbuttonlabel = NULL;

/* Some state for the thumbnailer */
static gchar *preview_fullname = NULL;

extern GSList *display_list; /* from gdisplay.c */

static gint thumbnail_mode = 1;

/*  public functions  */
void file_open_callback(GtkWidget *widget,
                        gpointer   data)
{
  if (!fileload)
    file_open_dialog_create();

  gtk_widget_set_sensitive(GTK_WIDGET(fileload), TRUE);
  if (GTK_WIDGET_VISIBLE(fileload))
    return;

  gtk_file_selection_set_filename(GTK_FILE_SELECTION (fileload),
                                  "." G_DIR_SEPARATOR_S);
  gtk_window_set_title(GTK_WINDOW(fileload), _("Load Pud"));

  file_dialog_show(fileload);
}

gint file_open_with_display(gchar *filename,
                            gchar *raw_filename)
{
#if 0
  GimpImage *gimage;
  GDisplay  *gdisplay;
  gchar     *absolute;
  gint       status;

  if ((gimage = file_open_image(filename,
                                raw_filename,
                                _("Open"),
                                NULL,
                                RUN_INTERACTIVE,
                                &status)) != NULL)
  {
    /* enable & clear all undo steps */
    gimp_image_undo_enable(gimage);

    /* set the image to clean  */
    gimp_image_clean_all(gimage);

    /* display the image */
    gdisplay = gdisplay_new(gimage, 0x0101);

    /* always activate the first display */
    if (g_slist_length(display_list) == 1)
      gimp_context_set_display(gimp_context_get_user(), gdisplay);

    absolute = file_open_absolute_filename(filename);

    document_index_add (absolute);
    menus_last_opened_add(absolute);

    g_free(absolute);
  }
  return status;
#endif
  return 0;
}

/*  private functions  */

/* The readXVThumb function source may be re-used under
 * the XFree86-style license. <adam@gimp.org> */
static guchar *readXVThumb (const gchar  *fnam,
                            gint         *w,
                            gint         *h,
                            gchar       **imginfo /* caller frees if != NULL */)
{
  FILE *fp;
  const gchar *P7_332 = "P7 332";
  gchar P7_buf[7];
  gchar linebuf[200];
  guchar *buf;
  gint twofivefive;
  void *ptr;

  *w = *h = 0;
  *imginfo = NULL;

  fp = fopen(fnam, "rb");
  if (!fp)
    return NULL;

  fread(P7_buf, 6, 1, fp);

  if (strncmp(P7_buf, P7_332, 6)!=0)
  {
    g_warning("Thumbnail doesn't have the 'P7 332' header.");
    fclose(fp);
    return NULL;
  }

  /*newline*/
  fread(P7_buf, 1, 1, fp);

  do
  {
    ptr = fgets(linebuf, 199, fp);
    if ((strncmp(linebuf, "#IMGINFO:", 9) == 0) &&
        (linebuf[9] != '\0') &&
        (linebuf[9] != '\n'))
    {
      if (linebuf[strlen(linebuf)-1] == '\n')
        linebuf[strlen(linebuf)-1] = '\0';

      if (linebuf[9] != '\0')
      {
        if (*imginfo)
          g_free(*imginfo);
        *imginfo = g_strdup(&linebuf[9]);
      }
    }
  } while (ptr && linebuf[0]=='#'); /* keep throwing away comment lines */

  if (!ptr)
  {
    /* g_warning("Thumbnail ended - not an image?"); */
    fclose(fp);
    return NULL;
  }

  sscanf(linebuf, "%d %d %d\n", w, h, &twofivefive);

  if (twofivefive!=255)
  {
    g_warning("Thumbnail is of funky depth.");
    fclose(fp);
    return NULL;
  }

  if ((*w)<1 || (*h)<1 || (*w)>128 || (*h)>128)
  {
    g_warning("Thumbnail size bad.  Corrupted?");
    fclose(fp);
    return NULL;
  }

  buf = g_malloc((*w) * (*h));

  fread(buf, (*w) * (*h), 1, fp);

  fclose(fp);

  return buf;
}

static gboolean file_save_thumbnail(guchar *buf,
                                    const gchar *full_source_filename)
{
  gint               i,j;
  gint               w,h;
  gchar             *pathname;
  gchar             *filename;
  gchar             *xvpathname;
  gchar             *thumbnailname;
  FILE              *fp;
  struct stat        statbuf;

  if (stat(full_source_filename, &statbuf) != 0)
  {
    return FALSE;
  }

  pathname = g_dirname(full_source_filename);
  filename = g_basename(full_source_filename); /* Don't free! */

  xvpathname = g_strconcat(pathname, G_DIR_SEPARATOR_S, ".xvpics", NULL);

  thumbnailname = g_strconcat(xvpathname, G_DIR_SEPARATOR_S,
                              filename, NULL);

  mkdir(xvpathname, 0755);

  fp = fopen(thumbnailname, "wb");
  g_free(pathname);
  g_free(xvpathname);
  g_free(thumbnailname);

  if (fp)
  {
    fprintf(fp, "P7 332\n#IMGINFO:128x128 RGB (%d %s)\n"
                "#END_OF_COMMENTS\n128 128 255\n",
                (int)statbuf.st_size,
                (statbuf.st_size == 1) ? "byte" : "bytes");

    for (i=0; i<128 ; i++)
    {
      /* Do a cheap unidirectional error-spread to look better */
      gint rerr=0, gerr=0, berr=0, a;
      for (j=0; j<128 ; j++)
      {
        gint32 r, g, b;

        r = *(buf++) + rerr;
        g = *(buf++) + gerr;
        b = *(buf++) + berr;

        r = CLAMP0255(r);
        g = CLAMP0255(g);
        b = CLAMP0255(b);

        fputc(((r>>5)<<5) | ((g>>5)<<2) | (b>>6), fp);

        rerr = r - ( (r>>5) * 255 ) / 7;
        gerr = g - ( (g>>5) * 255 ) / 7;
        berr = b - ( (b>>6) * 255 ) / 3;
      }
    }
    fclose(fp);
  } else /* Error writing thumbnail */
  {
    return FALSE;
  }

  return TRUE;
}

static void file_open_dialog_create(void)
{
  fileload = gtk_file_selection_new(_("Load Pud"));
  gtk_window_set_position(GTK_WINDOW(fileload), GTK_WIN_POS_MOUSE);
  gtk_window_set_wmclass(GTK_WINDOW(fileload), "load_pud", "Crafted");

  gtk_container_set_border_width(GTK_CONTAINER(fileload), 2);
  gtk_container_set_border_width(GTK_CONTAINER(GTK_FILE_SELECTION(fileload)->button_area), 2);

/*  dialog_register_fileload (fileload);*/

  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fileload)->cancel_button), "clicked",
                            GTK_SIGNAL_FUNC(file_dialog_hide),
                            GTK_OBJECT(fileload));
  gtk_signal_connect(GTK_OBJECT(fileload), "delete_event",
                     GTK_SIGNAL_FUNC(file_dialog_hide),
                     NULL);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileload)->ok_button), "clicked",
                     GTK_SIGNAL_FUNC(file_open_ok_callback),
                     fileload);
  gtk_quit_add_destroy(1, GTK_OBJECT(fileload));

  /* Catch file-clist clicks so we can update the preview thumbnail */
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileload)->file_list), "select_row",
                     GTK_SIGNAL_FUNC(file_open_clistrow_callback),
                     fileload);

  {
    GtkWidget *frame;
    GtkWidget *vbox;
    GtkWidget *hbox;
    GtkWidget *option_menu;
    GtkWidget *load_menu;
    GtkWidget *open_options_genbutton;

    open_options = gtk_hbox_new(TRUE, 1);

    /* Preview frame */
    open_options_frame = frame = gtk_frame_new("");
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
    gtk_box_pack_end(GTK_BOX(open_options), frame, FALSE, TRUE, 4);

    vbox = gtk_vbox_new(FALSE, 2);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
    gtk_container_add(GTK_CONTAINER(frame), vbox);

    hbox = gtk_hbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    gtk_widget_show(hbox);

    open_options_genbutton = gtk_button_new();
    gtk_signal_connect(GTK_OBJECT(open_options_genbutton), "clicked",
                       GTK_SIGNAL_FUNC(file_open_genbutton_callback),
                       fileload);
    gtk_box_pack_start(GTK_BOX(hbox), open_options_genbutton,
                       TRUE, FALSE, 0);
    gtk_widget_show(open_options_genbutton);

    open_options_fixed = gtk_fixed_new();
    gtk_widget_set_usize(open_options_fixed, 128, 128);
    gtk_container_add(GTK_CONTAINER(GTK_BIN(open_options_genbutton)),
                      open_options_fixed);
    gtk_widget_show(open_options_fixed);

    {
      GtkWidget* abox;
      GtkWidget* sbox;
      GtkWidget* align;

      sbox = gtk_vbox_new(TRUE, 0);
      gtk_container_add(GTK_CONTAINER(open_options_fixed), sbox);
      gtk_widget_show(sbox);

      align = gtk_alignment_new(0.5, 0.5, 0.0, 0.0);
      gtk_widget_set_usize(align, 128, 128);
      gtk_box_pack_start(GTK_BOX(sbox), align, FALSE, TRUE, 0);
      gtk_widget_show(align);

      abox = gtk_hbox_new(FALSE, 0);
      gtk_container_add(GTK_CONTAINER(align), abox);
      gtk_widget_show(abox);

      open_options_preview =
        GTK_PREVIEW(gtk_preview_new(GTK_PREVIEW_COLOR));
      gtk_box_pack_start(GTK_BOX(abox), GTK_WIDGET(open_options_preview),
                         FALSE, TRUE, 0);
      gtk_widget_show(GTK_WIDGET(open_options_preview));

      open_options_genbuttonlabel = gtk_label_new(_("Generate\nPreview"));
      gtk_box_pack_start(GTK_BOX(abox), open_options_genbuttonlabel,
                         FALSE, TRUE, 0);
      gtk_widget_show(open_options_genbuttonlabel);
    }

    open_options_label = gtk_label_new("");
    gtk_box_pack_start(GTK_BOX(vbox), open_options_label, FALSE, FALSE, 0);
    gtk_widget_show(open_options_label);

    gtk_widget_show(vbox);
    gtk_widget_show(frame);

    gtk_box_pack_end(GTK_BOX(GTK_FILE_SELECTION(fileload)->main_vbox),
                     open_options, FALSE, FALSE, 0);
  }

  gtk_frame_set_label(GTK_FRAME(open_options_frame), _("Preview"));
  gtk_label_set_text(GTK_LABEL(open_options_label), _("No Selection."));

  gtk_widget_show(GTK_WIDGET(open_options_genbuttonlabel));
  gtk_widget_hide(GTK_WIDGET(open_options_preview));
  gtk_widget_set_sensitive(GTK_WIDGET(open_options_frame), FALSE);

  gtk_widget_show (open_options);
}

/* don't call with preview_fullname as parameter!  will be clobbered! */
static void set_preview(const gchar *fullfname,
                        guchar      *RGB_source,
                        gint         RGB_w,
                        gint         RGB_h)
{
  guchar      *thumb_rgb;
  guchar      *raw_thumb;
  gint         tnw,tnh, i;
  gchar       *pname;
  gchar       *fname;
  gchar       *tname;
  gchar       *imginfo = NULL;
  struct stat  file_stat;
  struct stat  thumb_stat;
  gboolean     thumb_may_be_outdated = FALSE;
  gboolean     show_generate_label   = TRUE;

  pname = g_dirname(fullfname);
  fname = g_basename(fullfname); /* Don't free this! */
  tname = g_strconcat(pname, G_DIR_SEPARATOR_S,
                      ".xvpics", G_DIR_SEPARATOR_S,
                      fname, NULL);

  g_free(pname);

  /*  If the file is newer than its thumbnail, the thumbnail may
   *  be out of date.
   */
  if ((stat(tname,     &thumb_stat) == 0) &&
      (stat(fullfname, &file_stat ) == 0))
  {
    if ((thumb_stat.st_mtime) < (file_stat.st_mtime))
    {
      thumb_may_be_outdated = TRUE;
    }
  }

  raw_thumb = readXVThumb(tname, &tnw, &tnh, &imginfo);

  g_free(tname);

  gtk_frame_set_label(GTK_FRAME(open_options_frame), fname);

  g_free(preview_fullname);
  preview_fullname = g_strdup(fullfname);
  if (RGB_source)
  {
    gtk_preview_size(open_options_preview, RGB_w, RGB_h);

    for (i = 0; i < RGB_h; i++)
    {
      gtk_preview_draw_row(open_options_preview, &RGB_source[3*i*RGB_w],
                           0, i,
                           RGB_w);
    }
  } else
  {
    if (raw_thumb)
    {
      thumb_rgb = g_malloc(3 * tnw * tnh);

      for (i = 0; i < tnw * tnh; i++)
      {
        thumb_rgb[i*3  ] = ((raw_thumb[i]>>5)*255)/7;
        thumb_rgb[i*3+1] = (((raw_thumb[i]>>2)&7)*255)/7;
        thumb_rgb[i*3+2] = (((raw_thumb[i])&3)*255)/3;
      }

      gtk_preview_size(open_options_preview, tnw, tnh);

      for (i = 0; i < tnh; i++)
      {
        gtk_preview_draw_row(open_options_preview, &thumb_rgb[3*i*tnw],
                             0, i,
                             tnw);
      }

      g_free(thumb_rgb);
    }
  }

  if (raw_thumb || RGB_source)  /* We can show *some* kind of preview. */
  {
    if (raw_thumb) /* Managed to commit thumbnail file to disk */
    {
      gtk_label_set_text(GTK_LABEL(open_options_label),
                         thumb_may_be_outdated ?
                         _("(This thumbnail may be out of date)") :
                         (imginfo ? imginfo : _("(No Information)")));
      if (imginfo)
        g_free(imginfo);
    } else
    {
      switch (thumbnail_mode)
      {
        case 0:
          gtk_label_set_text(GTK_LABEL(open_options_label),
                             _("(Thumbnail saving is disabled)"));
          break;
        case 1:
          gtk_label_set_text(GTK_LABEL(open_options_label),
                             _("(Could not write thumbnail file)"));
          break;
        default:
          gtk_label_set_text(GTK_LABEL(open_options_label),
                             _("(Thumbnail file not written)"));
      }
    }

    gtk_widget_show(GTK_WIDGET(open_options_preview));
    gtk_widget_queue_draw(GTK_WIDGET(open_options_preview));

    show_generate_label = FALSE;

    g_free(raw_thumb);
  } else
  {
    if (imginfo)
      g_free(imginfo);

    gtk_widget_hide(GTK_WIDGET(open_options_preview));
    gtk_label_set_text(GTK_LABEL(open_options_label),
                       _("No preview available"));
  }

  if (show_generate_label)
  {
    gtk_widget_hide(GTK_WIDGET(open_options_preview));
    gtk_widget_show(GTK_WIDGET(open_options_genbuttonlabel));
  } else
  {
    gtk_widget_hide(GTK_WIDGET(open_options_genbuttonlabel));
    gtk_widget_show(GTK_WIDGET(open_options_preview));
  }
}

static void file_open_clistrow_callback(GtkWidget *widget,
                                        gint       row)
{
  gchar *fullfname = NULL;

  fullfname = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileload));

  gtk_widget_set_sensitive(GTK_WIDGET(open_options_frame), TRUE);
  set_preview(fullfname, NULL, 0, 0);
}

static void file_open_genbutton_callback(GtkWidget *widget,
                                         gpointer   data)
{
  guchar *gimage_to_be_thumbed;
  gint       RGBbuf_w;
  gint       RGBbuf_h;

  /* added for multi-file preview generation... */
  GtkFileSelection *fs;
  gchar            *full_filename = NULL;
  gchar            *filedirname;
  struct stat       buf;
  gint              err;

  fs = GTK_FILE_SELECTION (data);

  if (!preview_fullname)
  {
    g_warning("Tried to generate thumbnail for NULL filename.");
    return;
  }

  gtk_widget_set_sensitive(GTK_WIDGET(fileload), FALSE);

  /* new mult-file preview make: */
  {
    GSList *list, *toplist;

    /* Have to read the clist before touching anything else */

    list= clist_to_slist(GTK_CLIST(fs->file_list));
    toplist = list;

    /* Find a real base directory for the multiple selection */

    gtk_file_selection_set_filename(fs, "");
    filedirname= gtk_file_selection_get_filename(fs);
    if (filedirname[strlen(filedirname) - 1] == G_DIR_SEPARATOR)
      filedirname[strlen(filedirname) - 1] = '\0';

    while (list)
    {
      full_filename = g_strconcat(filedirname, G_DIR_SEPARATOR_S,
                                  (gchar *)list->data, NULL);

      err = stat(full_filename, &buf);

      if (! (err == 0 && (buf.st_mode & S_IFDIR)))
      { /* Is not directory. */
        gint dummy;

        g_print("thumbing '%s'\n", full_filename);
        
        gimage_to_be_thumbed = load_pud_for_thumbnail(full_filename);

        if (gimage_to_be_thumbed)
        {
          if (thumbnail_mode)
          {
            file_save_thumbnail(gimage_to_be_thumbed,
                                full_filename);
          }
          set_preview(full_filename, gimage_to_be_thumbed, 128, 128);

          if (gimage_to_be_thumbed)
            g_free(gimage_to_be_thumbed);
        } else
        {
          gtk_label_set_text(GTK_LABEL(open_options_label),
                             _("(could not make preview)"));
        }
      }

      g_free(full_filename);
      list= g_slist_next(list);
    }

    for (list = toplist; list; list = g_slist_next(list))
    {
      if (!(g_slist_next(list)))
      {
        full_filename = g_strconcat(filedirname, G_DIR_SEPARATOR_S,
                                    (gchar *)list->data, NULL);
        gtk_file_selection_set_filename(fs, full_filename);
        g_free(full_filename);
      }

      g_free(list->data);
    }

    g_slist_free(toplist);
    toplist = NULL;
  }

  gtk_widget_set_sensitive(GTK_WIDGET(fileload), TRUE);
}

static void file_open_ok_callback(GtkWidget *widget,
                                  gpointer   data)
{
  GtkFileSelection *fs;
  gchar            *full_filename;
  gchar            *raw_filename;
  gchar            *filedirname;
  struct stat       buf;
  gint              err;
  gint              status;

  fs = GTK_FILE_SELECTION(data);
  full_filename = gtk_file_selection_get_filename(fs);
  raw_filename = gtk_entry_get_text(GTK_ENTRY(fs->selection_entry));

  g_assert(full_filename && raw_filename);

  if (strlen(raw_filename) == 0)
    return;

  err = stat(full_filename, &buf);

  if (err == 0 && (buf.st_mode & S_IFDIR))
  {
    if (full_filename[strlen(full_filename) - 1] != G_DIR_SEPARATOR)
    {
      gchar *s = g_strconcat(full_filename, G_DIR_SEPARATOR_S, NULL);
      gtk_file_selection_set_filename(fs, s);
      g_free(s);
    } else
      gtk_file_selection_set_filename(fs, full_filename);

    return;
  }

  gtk_widget_set_sensitive(GTK_WIDGET(fs), FALSE);

  if (err) /* e.g. http://server/filename.jpg */
    full_filename = raw_filename;

  load_pud(full_filename);

/*  if (status == GIMP_PDB_SUCCESS)
  {*/
    file_dialog_hide(data);
/*  } else if (status != GIMP_PDB_CANCEL)
  {
    g_message(_("Open failed.\n%s"), full_filename);
  }*/

  /*
   * Now deal with multiple selections from the filesel clist
   */
  {
    GSList *list;

    /* Have to read the clist before touching anything else */

    list = clist_to_slist(GTK_CLIST (fs->file_list));

    /* Find a real base directory for the multiple selection */

    raw_filename = g_strdup(raw_filename);
    gtk_file_selection_set_filename(fs, "");
    filedirname = gtk_file_selection_get_filename(fs);

    while (list)
    {
      full_filename = g_strconcat(filedirname, G_DIR_SEPARATOR_S,
                                  (gchar *)list->data, NULL);

      if (strcmp(list->data, raw_filename))
      { /* don't load current selection twice */
        err = stat(full_filename, &buf);
        if (! (err == 0 && (buf.st_mode & S_IFDIR)))
        { /* Is not directory. */
          status = file_open_with_display(full_filename,
                                          (gchar *)list->data);

/*          if (status == GIMP_PDB_SUCCESS)
          {*/
            file_dialog_hide(data);
/*          } else if (status != GIMP_PDB_CANCEL)
          {
            g_message(_("Open failed.\n%s"), full_filename);
          }*/
        }
      }

      g_free(full_filename);
      g_free(list->data);
      list = g_slist_next(list);
    }

    g_slist_free(list);
    list = NULL;
  }

  gtk_file_selection_set_filename(fs, raw_filename);
  g_free(raw_filename);
  gtk_widget_set_sensitive(GTK_WIDGET (fs), TRUE);
}

static GSList *clist_to_slist(GtkCList *file_list)
{
  GSList *list = NULL;
  GList  *row;
  gint    rownum;
  gchar  *temp;

  for (row = file_list->row_list, rownum = 0;
       row;
       row = g_list_next (row), rownum++)
  {
    if (GTK_CLIST_ROW(row)->state == GTK_STATE_SELECTED)
    {
      if (gtk_clist_get_cell_type(file_list, rownum, 0) == GTK_CELL_TEXT)
      {
        gtk_clist_get_text(file_list, rownum, 0, &temp);
        list = g_slist_prepend(list, g_strdup(temp));
      }
    }
  }

  return list;
}


