//  crm114_.c  - Controllable Regex Mutilator,  version v1.0
//  Copyright 2001-2004  William S. Yerazunis, all rights reserved.
//  
//  This software is licensed to the public under the Free Software
//  Foundation's GNU GPL, version 2.  You may obtain a copy of the
//  GPL by visiting the Free Software Foundations web site at
//  www.fsf.org, and a copy is included in this distribution.  
//
//  Other licenses may be negotiated; contact the 
//  author for details.  
//
//  include some standard files
#include "crm114_sysincludes.h"

//  include any local crm114 configuration file
#include "crm114_config.h"

//  include the crm114 data structures file
#include "crm114_structs.h"

//  and include the routine declarations file
#include "crm114.h"

//    the command line argc, argv
extern int prog_argc;
extern char **prog_argv;

//    the auxilliary input buffer (for WINDOW input)
extern char *newinputbuf;

//    the globals used when we need a big buffer  - allocated once, used 
//    wherever needed.  These are sized to the same size as the data window.
extern char *inbuf;
extern char *outbuf;
extern char *tempbuf;
//
//
//     crm_nexpandvar - given a string and it's length, go through it
//     and if there's a variable expansion called for (by the :*:
//     operator) expand the variable.
//
//     the inputs are a buffer with the NULL-safe string in it, the
//     length of this string, and the maximum allocated length of the
//     buffer.  This function returns the new length of the buffer.
//     It will NOT increase the buffer length past maxlen, so
//     expansions beyond that will cause a nonfatal error and be
//     aborted.
//
//     Algorithm: 
//     1) efficiency check- do we need to do any expansions at all.
//     2) Start at buf[0], work up to buf[buflen]-3
//     2a) do \n, \r, \a, \xHH and \oOOO
//     3) are we looking at :*:?  
//     4) no: copy 1 character, increment from and to indexes, go to step 3
//     5) yes: skip from index ahead 3, from there to next : is the varname
//     6) copy var value to tbuf, incrementing tobuf index.
//     7) set from-index to third colon index + 1
//     8) go to 2 (modulo last two chars need copying)
//

long crm_nexpandvar (char *buf, long inlen, long maxlen)
{
  return (crm_zexpandvar (buf, 
			  inlen, 
			  maxlen,
			  NULL,
			  CRM_EVAL_ANSI 
			  | CRM_EVAL_STRINGVAR
			  | CRM_EVAL_REDIRECT
			  ));
}

//
// crm_qexpandvar is the "full evaluate one pass of everything" mode.

long crm_qexpandvar (char *buf, long inlen, long maxlen, long *qex_stat)
{
  return (crm_zexpandvar (buf,
			  inlen,
			  maxlen,
			  qex_stat,
			  CRM_EVAL_ANSI 
			  | CRM_EVAL_STRINGVAR 
			  | CRM_EVAL_REDIRECT
			  | CRM_EVAL_STRINGLEN
			  | CRM_EVAL_MATH ));
}
 
//     crm_zexpandvar - "expanded" expandvar.  Does all the expansions, 
//     but does not repeat the evaluations.  If you want repeats, you
//     must do that yourself (that way, this function will always
//     move down the string at least one character, and thus this this
//     function will always terminate,  Nice, that.  :) 
//
//     the inputs are a buffer with the NULL-safe string in it, the
//     length of this string, and the maximum allocated length of the
//     buffer.  This function returns the new length of the buffer.
//     It will NOT increase the buffer length past maxlen, so
//     expansions beyond that will cause a nonfatal error and be
//     aborted.
//
//     Algorithm: 
//     1) efficiency check- do we need to do any expansions at all.
//     2) Start at buf[0], work up to buf[buflen]-3
//     2a) do \n, \r, \a, \xHH and \oOOO
//     3) are we looking at :<some-operator>:?  
//     4) no: copy 1 character, increment from and to indexes, go to step 3
//     5) yes: skip from index ahead 3, from there to next : is the varname
//     6) copy var value to tbuf, incrementing tobuf index.
//     7) set from-index to third colon index + 1
//     8) go to 2 (modulo last two chars need copying)
//
long crm_zexpandvar (char *buf, 
		     long inlen, 
		     long maxlen, 
		     long *retstat, 
		     long exec_bitmask)
{
  long is, id;
  long vht_index;
  long q;
  //  a temporary work buffer...
  char *tbuf;
  //  and another for variable names...
  char *vname; 
  
  char *cp;
  long vlen;
  
  char opchar;
  
  //    efficiency check - do we even _have_ a :*: in the buffer?
  //
  
  if (inlen == 0)
    return (0);
  
  is= 0; id = 0; 

  if (internal_trace) 
    fprintf (stderr, "qexpandvar on =%s= len: %ld bitmask: %ld \n", 
	     buf, inlen, exec_bitmask);
  
  //  GROT GROT GROT must fix this for 8-bit safe error messages
  if (inlen > maxlen)
    {
      q = fatalerror (
		      "You have blown the gaskets while building a string.  Orig string was: ",
		      buf);
      if (q == 0 )
	return (inlen);
      goto bailout;
    };  
  
  //   First thing- do the ANSI \-Expansions
  //
  if (exec_bitmask & CRM_EVAL_ANSI)
    {
      is = 0;
      id = 0;
      if (internal_trace)
	fprintf (stderr, " Doing backslash expansion \n");
      for (is = 0; is <= inlen ; is++)  
	{
	  if (buf[is] != '\\' )
	    {
	      buf [id] = buf [is];
	      id++;
	    }
	  else
	    {
	      //  we're looking at a '\\'.
	      //  
	      //   Check for a few common things: \n, \a, \xNN, \oNNN
	      is++;
	      //
	      switch (buf[is])
		{
		case '0':
		  {
		    //   it's a NULL.
		    buf[id] = '\0';
		    id++;
		  }
		  break;
		case 'b':
		  {
		    //   it's a backspace
		    buf[id] = '\b';
		    id++;
		  }
		  break;
		case 't':
		  {
		    //   it's a tab
		    buf[id] = '\t';
		    id++;
		  }
		  break;
		case 'n':
		  {
		    //   it's a newline.  stuff in a newline.
		    buf[id] = '\n';
		    id++;
		  }
		  break;
		case 'v':
		  {
		    //   it's a vtab
		    buf[id] = '\v';
		    id++;
		  }
		  break;
		case 'f':
		  {
		    //   it's a form feed.
		    buf[id] = '\f';
		    id++;
		  }
		  break;
		case 'r':
		  {
		    //   it's a carriage return
		    buf[id] = '\r';
		    id++;
		  }
		  break;
		case 'a':
		  {
		    //   it's a BELL.  put that in.
		    buf[id] = '\a';
		    id++;
		  }
		  break;
		case 'x':
		case 'X':
		  {
		    //   it might be a hex char constant.  read it 
		    //    and stuff it.
		    unsigned int value;
		    int conv_count;
		    conv_count = 0;
		    value = '\000';
		    if (is+2 < inlen)      // watch out for end-of-string
		      conv_count = sscanf (&buf[is+1], "%2X", &value);
		    if (conv_count == 1)
		      {
			buf[id] = value;
			id++;
			is++; is++;  // move over the hex digits
		      }
		    else	//   and otherwise, just copy the x
		      {
			buf[id] = buf[is];
			id++;
		      };
		  };
		  break;
		case 'o':
		case 'O':
		  {
		    //   it might be an octal char constant.  read it 
		    //   and stuff it.
		    unsigned int value;
		    int conv_count ;
		    conv_count = 0;
		    value = '\000';
		    if (is+3 < inlen)     // watch out for end-of-string
		      conv_count = sscanf (&buf[is+1], "%3o", &value);
		    if (conv_count == 1)
		      {
			buf[id] = value;
			id++;
			is++; is++; is++;  // move over the octal digits
		      }
		    else	//   and otherwise, just copy the conv. char.
		      {
			buf[id] = buf[is];
			id++;
		      };
		  };
		  break;
		case '>':
		case ')':
		case ']':
		case '/':
		case ';':
		case '{':
		case '}':
		case '#':
		case '\\':
		  {
		    //      >, ), ], ;, {, }, #, and / are themselves after a '\',
		    //    but need the \ escape to pass thru the parser
		    //    without terminating their enclosed args
		    buf[id] = buf[is];
		    id++;
		  };
		  break;
		default:
		  {
		    //       if it's "none of the above" characters, then
		    //       the '\' character _stays_ as a literal
		    buf[id] = '\\';
		    id++;
		    buf[id] = buf[is];
		    id++;
		  };
		  break;
		};
	    };
	};
      //     and update the new inlen
      buf[id] = '\000';    // needed because slimy old GNU REGEX needs it.
      //   and take one off for inlen, because it always gets incremented one 
      //   extra time
      inlen = id - 1;

      if (internal_trace)
	fprintf (stderr, "backslash expansion yields: =%s= len %ld \n", buf, inlen);
    }
  //    END OF ANSI \-EXPANSIONS
      


  //    Do a quick check for :'s - this is just a speedup, as all further 
  //    operators use the : notation.

  //    if no :, then no ":" operators possible.
  cp = memchr (buf, ':', inlen);
  if (cp == NULL)
    {
      if (internal_trace)
	fprintf (stderr, "No further expansions possible");
      return (inlen);
    };
  
  //    OK, we might have a :*: substitution operator, so we actually have
  //    to do some work.
  //    allocate some memory for tbuf and vname;
  tbuf = (char *) malloc (maxlen+1);
  vname = (char *) malloc (maxlen+1);
  
  if (tbuf == NULL || vname == NULL)
    {
      q = fatalerror ("Couldn't allocate memory for Q-variable expansion!",
		      "Try making the window set smaller with the -w option");
      if (q == 0)
	return (inlen);
    };
  
  //
  //    Now, do we have a :*: (singlevar) possible?

  if ( exec_bitmask & CRM_EVAL_STRINGVAR )
    {
      
      is = 0;    //   is is the input position index
      id = 0;    //   id is the destination position index
      if (internal_trace)
	fprintf (stderr, "Doing singlevar eval expansion\n");

      //
      //   First time through the loop, for :*: (variable expansion)
      //
      for (is = 0; is <= inlen && id < maxlen; is++)
	{
	  if (is <= inlen - 5  //  check only if :*:c:" possible
	      && buf[is] == ':' 
	      && buf[is+1] == '*' 
	      && ( buf[is+2] ==':' ))
	    {
	      //   yes, it's probably an expansion of some sort.
	      opchar = buf[is+1];
	      //    copy everything from the colon to the second colon
	      //    ( or the end of the string) into the vname buffer.
	      is = is + 2;
	      vname [0] = buf[is];
	      vlen = 1;
	      is++;
	      while (is < maxlen
		     && is <= inlen
		     && buf [is] != ':')
		{
		  vname[vlen] = buf[is];
		  is++;
		  vlen++;
		};
	      //
	      //    check for the second colon as well...
	      if (buf[is] == ':')
		{
		  vname[vlen] = ':';
		  vlen++;
		}
	      vname [vlen] = '\000';
	      
	      //
	      //      Now we've got the variable name in vname, we can 
	      //      go get it's value and copy _that_ into tbuf as well.
	      if (internal_trace)
		fprintf (stderr, "looking up variable >%s<\n", vname);
	      vht_index = crm_vht_lookup (vht, vname, vlen);
	      
	      if (vht[vht_index] == NULL)
		{
		  //      there was no variable by that name, use the text itself
		  switch (opchar)
		    {
		    case '*':
		      {
			//
			//    simply copy text till the close colon
			//
			for (q = 0; q < vlen && id < maxlen; q++)
			  {
			    tbuf[id] = vname[q];
			    id++;
			  }
		      }
		      break;
		    }
		}
	      else
		{
		  //     There really was a variable value by that name.
		  //     suck it out, and splice it's text value
		  
		  //   check to see if it's one of the self-mutating 
		  //    internal variables, like :_iso: or :_cd:
		  
		  if (strncmp(
			      (char *) &vht[vht_index]->nametxt[vht[vht_index]->nstart],
			      ":_", 2) == 0)
		    {
		      if (strncmp(
				  (char *) &vht[vht_index]->nametxt[vht[vht_index]->nstart],
				  ":_iso:", 6) == 0)
			{
			  //   if this was :_:, update iso's length
			  vht[vht_index]->vlen = tdw->nchars;
			};
		    };
		  
		  switch (opchar)
		    {
		    case '*':
		      {
			for (q = 0; q < vht[vht_index]->vlen && id < maxlen; q++)
			  {
			    tbuf[id] = vht[vht_index]->valtxt
			      [(vht[vht_index]->vstart)+q];
			    id++;
			  }
		      }
		      break;
		    };
		};
	    }
	  //         Now, handle the case where we were NOT looking at
	  //         :*:c: in buf
	  else
	    {
	      tbuf[id] = buf[is];
	      id++;
	    }
	}

      //    and put our results back into buf       
      strncpy (buf, tbuf, id);
      buf[id] = '\000';
      inlen = id - 1;
    }  

  //     END OF :*: EXPANSIONS


  //
  //    Now, do we have a :+: (REDIRECT) operators

  if ( exec_bitmask & CRM_EVAL_REDIRECT )
    {
      
      is = 0;    //   is is the input position index
      id = 0;    //   id is the destination position index
      if (internal_trace)
	fprintf (stderr, "Doing singlevar eval expansion\n");

      //
      //   First time through the loop, for :+: (variable expansion)
      //
      for (is = 0; is <= inlen && id < maxlen; is++)
	{
	  if (is <= inlen - 5  //  check only if :*:c:" possible
	      && buf[is] == ':' 
	      && buf[is+1] == '+' 
	      && ( buf[is+2] ==':' ))
	    {
	      //   yes, it's probably an expansion of some sort.
	      //    copy everything from the colon to the second colon
	      //    ( or the end of the string) into the vname buffer.
	      is = is + 2;
	      vname [0] = buf[is];
	      vlen = 1;
	      is++;
	      while (is < maxlen
		     && is <= inlen
		     && buf [is] != ':')
		{
		  vname[vlen] = buf[is];
		  is++;
		  vlen++;
		};
	      //
	      //    check for the second colon as well...
	      if (buf[is] == ':')
		{
		  vname[vlen] = ':';
		  vlen++;
		}
	      vname [vlen] = '\000';
	      
	      //
	      //      Now we've got the variable name in vname, we can 
	      //      go get it's value and copy _that_ into the vname buffer
	      if (internal_trace)
		fprintf (stderr, "looking up variable >%s<\n", vname);
	      vht_index = crm_vht_lookup (vht, vname, vlen);
	      
	      if (vht[vht_index] == NULL)
		{
		  //   no op - if no such variable, no change...
		}
	      else
		{
		  //     There really was a variable value by that name.
		  //     suck it out, and make that the new vname text
		  
		  //   if this was :_iso:, update iso's length
		  if (strncmp(
			      (char *) &vht[vht_index]->nametxt[vht[vht_index]->nstart],
			      ":_iso:", 6) == 0)
		    {
		      vht[vht_index]->vlen = tdw->nchars;
		    };
		  
		  for (q = 0; q < vht[vht_index]->vlen && id < maxlen; q++)
		    {
		      vname[q] = vht[vht_index]->valtxt
			[(vht[vht_index]->vstart)+q];
		    }
		  //   note that vlen is varname len, but vht[]->vlen is
		  //    the length of the text.  Bad choice of names, eh?
		  vlen = vht[vht_index]->vlen;
		};
	      //      Second time around:
	      //      We have something in vname (either the indirected
	      //      varname, or the original varname), we can 
	      //      go get it's value and copy _that_ into tbuf as well.
	      if (internal_trace)
		fprintf (stderr, "Second lookup variable >%s<\n", vname);
	      vht_index = crm_vht_lookup (vht, vname, vlen);
	      
	      if (vht[vht_index] == NULL)
		{
		  //
		  //    simply copy text including the close colon
		  //
		  for (q = 0; q < vlen && id < maxlen; q++)
		    {
		      tbuf[id] = vname[q];
		      id++;
		    }
		}
	      else
		{
		  //     There really was a variable value by that name.
		  //     suck it out, and splice it's text value
		  
		  //   if this was :_iso:, update iso's length
		  if (strncmp(
			      (char *) &vht[vht_index]->nametxt[vht[vht_index]->nstart],
			      ":_iso:", 6) == 0)
		    {
		      vht[vht_index]->vlen = tdw->nchars;
		    };
		  {
		    for (q = 0; q < vht[vht_index]->vlen && id < maxlen; q++)
		      {
			tbuf[id] = vht[vht_index]->valtxt
			  [(vht[vht_index]->vstart)+q];
			id++;
		      }
		  }
		};
	    }
	  //         Now, handle the case where we were NOT looking at
	  //         :+:c: in buf
	  else
	    {
	      tbuf[id] = buf[is];
	      id++;
	    }
	}
      //    and put our results back into buf       
      strncpy (buf, tbuf, id);
      buf[id] = '\000';
      inlen = id - 1;
    }  
  
  //     END OF :+: EXPANSIONS
  
  
  if (exec_bitmask & CRM_EVAL_STRINGLEN)
    {
      //
      //
      //   Second time through the loop - expand :#: (string lengths)
      //
      if (internal_trace)
	fprintf (stderr, "Doing stringglength expansion\n");

      buf[id] = '\000';
      if (internal_trace)
	fprintf (stderr, " var-expand yields: =%s= len %ld\n", buf, inlen);
      id = 0;
      for (is = 0; is <= inlen && id < maxlen; is++)
	{
	  if (is <= inlen - 5  //  check only if :#:c:" possible
	      && buf[is] == ':' 
	      && ( buf[is+1] == '#' )
	      && buf[is+2] ==':')
	    {
	      //   yes, it's probably an expansion of some sort.
	      opchar = buf[is+1];
	      //    copy everything from the colon to the second colon
	      //    into the vname buffer.
	      is = is + 2;
	      vname [0] = buf[is];
	      vlen = 1;
	      is++;
	      while (is < maxlen
		     && is <= inlen
		     && buf [is] != ':')
		{
		  vname[vlen] = buf[is];
		  is++;
		  vlen++;
		};
	      //
	      //    check for the second colon as well...
	      if (buf[is] == ':')
		{
		  vname[vlen] = ':';
		  vlen++;
		}
	      vname [vlen] = '\000';
	      
	      //
	      //      Now we've got the variable name in vname, we can 
	      //      go get it's value and copy _that_ into tbuf as well.
	      if (internal_trace)
		fprintf (stderr, "looking up variable >%s<\n", vname);
	      vht_index = crm_vht_lookup (vht, vname, vlen);
	      
	      if (vht[vht_index] == NULL)
		{
		  //      there was no variable by that name, use the 
		  //      text itself
		  switch (opchar)
		    {
		    case '#':
		      {
			char lentext[MAX_VARNAME];
			int m, mm;
			//   the vlen-2 is because we need to get
			//    rid of the ':' 
			sprintf (lentext, "%ld", vlen-2);
			mm = strlen (lentext);
			for (m = 0; m < mm && id < maxlen; m++)
			  {
			    tbuf[id] = lentext[m];
			    id++;
			  };
		      }
		      break;
		    }
		}
	      else
		{
		  //     There really was a variable value by that name.
		  //     suck it out, and splice it's text value
		  
		  //   if this was :_iso:, update iso's length
		  if (strncmp(
			      (char *) &vht[vht_index]->nametxt[vht[vht_index]->nstart],
			      ":_iso:", 6) == 0)
		    {
		      vht[vht_index]->vlen = tdw->nchars;
		    };
		  
		  switch (opchar)
		    {
		    case '#':
		      {
			//
			//   Actually, we want the _length_ of the variable
			//
			char lentext[MAX_VARNAME];
			int m, mm;
			sprintf (lentext, "%ld", vht[vht_index]->vlen);
			mm = strlen (lentext);
			for (m = 0; m < mm && id < maxlen; m++)
			  {
			    tbuf[id] = lentext[m];
			    id++;
			  };
		      };
		      break;
		    };
		};
	    }
	  //         Now, handle the case where we were NOT looking at
	  //         :*:c: in buf
	  else
	    {
	      tbuf[id] = buf[is];
	      id++;
	    }
	}
      //    and put our results back into buf       
      strncpy (buf, tbuf, id);
      buf[id] = '\000';
      //    and because id always gets an extra increment...
      inlen = id - 1;
    }
  //       END OF :#: STRING LENGTH EXPANSIONS


  //       Do we have any math expansions?
  if (exec_bitmask & CRM_EVAL_MATH)
    {
  
      //
      //       handle :@:  (math evaluations)
      //
      //
      if (internal_trace)
	fprintf (stderr, "Doing math expansion\n");

      buf[id] = '\000';
      if (internal_trace)
	fprintf (stderr, " length-expand yields: =%s= len %ld\n", buf, inlen);
      id = 0;
      for (is = 0; is <= inlen && id < maxlen; is++)
	{
	  if (is <= inlen - 5  //  check only if :*:c:" possible
	      && buf[is] == ':' 
	      && ( buf[is+1] == '@' )
	      && buf[is+2] ==':')
	    {
	      //   yes, it's probably an expansion of some sort.
	      opchar = buf[is+1];
	      //    copy everything from the colon to the second colon
	      //    into the vname buffer.
	      is = is + 2;
	      vname [0] = buf[is];
	      vlen = 1;
	      is++;
	      while (is < maxlen
		     && is <= inlen
		     && buf [is] != ':')
		{
		  vname[vlen] = buf[is];
		  is++;
		  vlen++;
		};
	      //
	      //    check for the second colon as well...
	      if (buf[is] == ':')
		{
		  vname[vlen] = ':';
		  vlen++;
		}
	      else
		{
		  nonfatalerror ("This math eval didn't end with a ':' which is",
				 " often an error...  check it sometime? ");
		};
	      vname [vlen] = '\000';
	      
	      //
	      //      Now we've got the variable name in vname, we can 
	      //      go get it's value and copy _that_ into tbuf as well.
	      if (internal_trace)
		fprintf (stderr, "looking up variable >%s<\n", vname);
	      vht_index = crm_vht_lookup (vht, vname, vlen);
	      
	      if (vht[vht_index] == NULL)
		{
		  //      there was no variable by that name, use the text itself
		  switch (opchar)
		    {
		    case '@':
		      {
			char mathtext[MAX_VARNAME];
			int m, mm;
			strncpy (mathtext, &vname[1], vlen-2);
			mathtext[vlen-2] = '\000';
			if (internal_trace)
			  fprintf (stderr, "In-Mathtext is -'%s'-\n", mathtext); 
			m = strmath (mathtext, vlen-2, MAX_VARNAME, retstat);
			if (internal_trace)
			  fprintf (stderr, "Out-Mathtext is -'%s'-\n", mathtext);
			if (retstat && *retstat < 0)
			  {
			    q = fatalerror ("Problem during math evaluation of ",
					    mathtext);
			    if (q == 0)
			      return (inlen);
			    goto bailout;
			  }
			mm = strlen (mathtext);
			for (m = 0; m < mm && id < maxlen; m++)
			  {
			    tbuf[id] = mathtext[m];
			    id++;
			  };
		      }
		      break;
		    }
		}
	      else
		{
		  //     There really was a variable value by that name.
		  //     suck it out, and splice it's text value
		  
		  //   if this was :_iso:, update iso's length
		  if (strncmp(
			      (char *) &vht[vht_index]->nametxt[vht[vht_index]->nstart],
			      ":_iso:", 6) == 0)
		    {
		      vht[vht_index]->vlen = tdw->nchars;
		    };
		  
		  switch (opchar)
		    {
		    case '@':
		      {
			char mathtext[MAX_VARNAME];
			int m, mm;
			m = 0;
			for (q = 0; q < vht[vht_index]->vlen && m < maxlen; q++)
			  {
			    mathtext[m] = vht[vht_index]->valtxt
			      [(vht[vht_index]->vstart)+q];
			    m++;
			  }
			mathtext[vlen-1] = '\000';
			m = strmath (mathtext, vlen-2, MAX_VARNAME, retstat );
			if (retstat && *retstat < 0)
			  {
			    q = fatalerror ("Problem during math evaluation of ",
					    mathtext);
			    if (q == 0)
			      return (inlen);
			    goto bailout;
			  }
			mm = strlen (mathtext);
			for (m = 0; m < mm && id < maxlen; m++)
			  {
			    tbuf[id] = mathtext[m];
			    id++;
			  };
		      }
		      break;
		    };
		};
	    }
	  //         Now, handle the case where we were NOT looking at
	  //         :*:c: in buf
	  else
	    {
	      tbuf[id] = buf[is];
	      id++;
	    }
	}
      //    and put our results back into buf       
      strncpy (buf, tbuf, id);
      buf [id] = '\000';
      inlen = id - 1;

      if (internal_trace)
	fprintf (stderr, " math-expand yields: =%s= len %ld\n", buf, inlen);

    }

  //    END OF :@: MATH EXPANSIONS

  //    That's all, folks!  Clean up the temporary buffer.  We null-terminate
  //    it in case we need to do stupid non-8-bit-clean IO on it.
  tbuf[inlen+1] = '\000';
  free (tbuf);
  free (vname);
  if (internal_trace)
    {
      fprintf (stderr, " Returned length from qexpandvar is %ld\n", inlen);
      if (retstat) 
	fprintf (stderr, "retstat was: %ld\n", *retstat);
    };
  return (inlen);
 bailout:
  return (inlen);
}

