/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * action.c: Copyright (C) Eric Prevoteau <www@a2pb.gotdns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: action.c,v 1.4 2003/12/28 08:12:38 uid68112 Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <string.h>
#include <sys/socket.h>
#include <glib.h>

#include "action.h"
#include "display.h"
#include "gts.h"
#include "keyboard.h"

/**************************************************************************************/
/* due to the fact all the following data are used by threads, they must be protected */
/**************************************************************************************/

/* this is the lists of all running connections */
G_LOCK_DEFINE(waiting_action);				/* this mutex protects waiting_action array */
GPtrArray *waiting_action=NULL;				/* array of pointers on WAIT_ACT structure */

/* list of all ReConnectToMe sent */
/* (connection request */
G_LOCK_DEFINE(waiting_revcon);		/* this mutex protects waiting_revcon array */
GPtrArray *waiting_revcon=NULL;

/* list of all commands waiting to be sent to the keyboard */
G_LOCK_DEFINE(sim_input);				/* this mutex protects sim_input array */
GArray *sim_input=NULL;

/**************************/
/* create an action to do */
/*********************************************************************************/
/* nick is copied but action is used as is and should not be freed by the caller */
/*********************************************************************************/
WAIT_REVCON *create_action_to_do(char *nick,GString *action)
{
	WAIT_REVCON *nw;

	nw=malloc(sizeof(WAIT_REVCON));
	if(nw==NULL)
	{
		disp_msg(ERR_MSG,"create_and_add_action_to_do","out of memory",NULL);
		return NULL;
	}

	nw->last_touch=time(NULL);
	nw->remote_nick=g_string_new(nick);
	nw->action_to_do=action;

	return nw;
}


/*********************************************************************/
/* create an action to do and append it to the list of actions to do */
/*********************************************************************************/
/* nick is copied but action is used as is and should not be freed by the caller */
/*********************************************************************************/
void create_and_add_action_to_do(char *nick,GString *action)
{
	WAIT_REVCON *nw;

	nw=create_action_to_do(nick,action);
	if(nw==NULL)
		return;

	G_LOCK(waiting_revcon);
	g_ptr_array_add(waiting_revcon,nw);
	G_UNLOCK(waiting_revcon);
}

/*************************************************************************/
/* remote the given WAIT_REVCON from the waiting_revcon list and free it */
/*************************************************************************/
/* if with_remove==0, the ptr is only freed, else, it is also removed */
/* from waiting_revcon                                                */
/**********************************************************************/
void free_action_to_do(WAIT_REVCON *ptr,int with_remove)
{
	if(ptr==NULL)
		return;

	if(with_remove)
	{
		G_LOCK(waiting_revcon);
		g_ptr_array_remove(waiting_revcon,ptr);
		G_UNLOCK(waiting_revcon);
	}

	if(ptr->remote_nick!=NULL)
	{
		g_string_free(ptr->remote_nick,TRUE);
		ptr->remote_nick=NULL;
	}

	if(ptr->action_to_do!=NULL)
	{
		g_string_free(ptr->action_to_do,TRUE);
		ptr->action_to_do=NULL;
	}

	free(ptr);
}

/******************************************************************************************/
/* find the first wait_revcon for the given nick. The wait_revcon is removed of the array */
/******************************************************************************************/
WAIT_REVCON *get_action_to_do(GString *wanted_nick)
{
	WAIT_REVCON *nw=NULL;
	WAIT_REVCON *cur;
	unsigned int i;

	G_LOCK(waiting_revcon);
	for(i=0;i<waiting_revcon->len;i++)
	{
		cur=(WAIT_REVCON*)(g_ptr_array_index(waiting_revcon,i));
		if(!strcmp(wanted_nick->str, cur->remote_nick->str))
		{
			nw=cur;
			g_ptr_array_remove_index(waiting_revcon,i);
			break;
		}
	}
	G_UNLOCK(waiting_revcon);
	return nw;
}

/******************************************************************************************/
/* find the first wait_revcon for the given nick. The wait_revcon is removed of the array */
/* the revcon is a download task (/LS or /DL).                                            */
/* The problem: we cannot take an existing revcon because it is already managed by another*/
/* thread (if exists). We must look inside gts to find something interesting and create a */
/* wait_revcon for it and return it.                                                      */
/******************************************************************************************/
WAIT_REVCON *get_download_action_to_do(GString *wanted_nick)
{
	WAIT_REVCON *nw=NULL;
	GString *atd;

	atd=gts_retrieve_download(wanted_nick->str);
	if(atd==NULL)
		return NULL;		/* nothing to do */

	indirect_call_process_kbd_com(atd->str,&nw);
	g_string_free(atd,TRUE);
	return nw;
}

/*************************************************************************/
/* add the given wait_revcon at the beginning of the waiting_revcon list */
/*************************************************************************/
void prepend_action_to_do(WAIT_REVCON *wrc)
{
	GPtrArray *nw;
	int i;

	G_LOCK(waiting_revcon);
	if(waiting_revcon->len==0)
	{
		g_ptr_array_add(waiting_revcon,wrc);
	}
	else
	{	/* there is no g_ptr_array_prepend, we must do it manually */
		nw=g_ptr_array_new();
		g_ptr_array_add(nw,wrc);

		for(i=0;i<waiting_revcon->len;i++)
		{
			g_ptr_array_add(nw,g_ptr_array_index(waiting_revcon,i));
		}

		g_ptr_array_free(waiting_revcon,TRUE);
		waiting_revcon=nw;
	}
	G_UNLOCK(waiting_revcon);
}

/***********************************************/
/* destruction of the given WAIT_ACT structure */
/***********************************************/
void free_wait_act(WAIT_ACT *act)
{
	if(act==NULL)
		return;

	if(act->sock_fd)
	{
		shutdown(act->sock_fd,2);
		close(act->sock_fd);
		act->sock_fd=0;
	}

	if(act->remote_nick!=NULL)
	{
		g_string_free(act->remote_nick,TRUE);
		act->remote_nick=NULL;
	}

	if(act->remote_addr!=NULL)
	{
		g_string_free(act->remote_addr,TRUE);
		act->remote_addr=NULL;
	}

	if(act->run_task!=NULL)
	{
		fprintf(stderr,"WARNING, in free_wait_act, run_task still is here.\n");
		act->run_task=NULL;
	}

	if(act->disp_info!=NULL)
	{
		g_string_free(act->disp_info,TRUE);
		act->disp_info=NULL;
	}

	if(act->cap_str!=NULL)
	{
		g_string_chunk_free(act->cap_str);
		act->cap_str=NULL;
	}

	if(act->cap_ptr!=NULL)
	{
		g_ptr_array_free(act->cap_ptr,TRUE);
		act->cap_ptr=NULL;
	}

	free(act);
}

/**************************/
/* create a new sim input */
/*****************************************************************************/
/* input: wait_duration: number of seconds to wait before sending the string */
/*        string: string to enter on the keyboard                            */
/*****************************************************************************/
void add_new_sim_input(time_t wait_duration, const char *string)
{
	int i;
	int fnd=0;

	G_LOCK(sim_input);

	/* check if the given string is not yet inside the sim input list */
	if(strncmp("/XDL",string,4))
	{	/* multiqueueing is allowed for /XDL */
		for(i=0;i<sim_input->len;i++)
		{
			SIM_INPUT *cur;
	
			cur=&(g_array_index(sim_input,SIM_INPUT,i));
	
			if(!strcmp(cur->keyb_string->str,string))
			{
				fnd=1;
				break;
			}
		}
	}
		
	if(!fnd)
	{
		SIM_INPUT nw;
		static unsigned long next_id=0;

		nw.id=next_id++;
		nw.min_start_time=time(NULL)+wait_duration;
		nw.keyb_string=g_string_new(string);
		sim_input=g_array_append_val(sim_input,nw);
	}

	G_UNLOCK(sim_input);
	return;
}

/****************************************************************************/
/* search the given string in the sim input array and return its start time */
/****************************************************************************/
/* output: 1=found, (*start_time) contains the sim_input start time */
/*         0=not found                                              */
/********************************************************************/
int find_sim_input_delay(const char *string,time_t *start_time)
{
	int ret=0;
	int i;

	G_LOCK(sim_input);
	for(i=0;i<sim_input->len;i++)
	{
		SIM_INPUT *cur;

		cur=&(g_array_index(sim_input,SIM_INPUT,i));

		if(!strcmp(cur->keyb_string->str,string))
		{
			*start_time=cur->min_start_time;
			ret=1;
			break;
		}
	}
	G_UNLOCK(sim_input);
	return ret;
}

/**********************************************************************************************/
/* check if the given user already has something in progress in wait_action or in wait_revcon */
/**********************************************************************************************/
/* output: 1=yes, 0=no */
/***********************/
int act_with_user_in_progress(const char *nickname)
{
	return 0;		/* this function needs to be rewritten (if still required) */
}

/********************************************************************************************/
/* scan running transfer list and shutdown connection of all uploads for the given nickname */
/********************************************************************************************/
void terminate_upload_of_user(char *nickname)
{
	int i;
	G_LOCK(waiting_action);
	for(i=0;i<waiting_action->len;i++)
	{
		WAIT_ACT *wa;

		wa=(WAIT_ACT*)(g_ptr_array_index(waiting_action,i));
		if(wa->remote_nick!=NULL)
		{
			/* during connection establishment, remote_nick can be NULL */
			if(!strcmp(nickname,wa->remote_nick->str))
			{
				if((wa->disp_info!=NULL)&&(!strncmp(wa->disp_info->str,"UL/",3)))
				{
					shutdown(wa->sock_fd,2);
				}
			}
		}
	}

	G_UNLOCK(waiting_action);
}

