<?php
/*
 * Copyright (C) 2003-2004 Polytechnique.org
 * http://opensource.polytechnique.org/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


require_once 'HTTP/WebDAV/Server.php';
require_once 'diogenes/diogenes.misc.inc.php';
require_once 'diogenes.webdav.logger.inc.php';

/**
 * Filesystem access using WebDAV
 *
 */
class DiogenesWebDAV extends HTTP_WebDAV_Server
{
  /** The barrel on which we are operating. */
  var $alias;

  /** If the barrel is running on a virtualhost. */
  var $vhost;
  
  /** The database table holding the pages */
  var $table_page;

  /** Handle to the spool */
  var $spool;

  /**
   * The constructor
   */
  function DiogenesWebDAV()
  {
    global $globals;

    // call parent constructor
    $this->HTTP_WebDAV_Server();

    // construct new session each time, there is no practical way 
    // of tracking sessions with WebDAV clients
    $_SESSION['session'] = new $globals->session;
  }


  /** 
   * Return an RCS handle.
   */
  function getRcs()
  {
    global $globals;
    return new $globals->rcs($this,$this->alias,$_SESSION['session']->username);
  }


  /** Report an information. Needed for RCS operations.
   *
   * @param msg
   *
   * @see DiogenesRcs
   */
  function info($msg) {
    // we do nothing
  }


  /**
   * Die with a given error message. Needed for RCS operations.
   *
   * @param msg
   *
   * @see DiogenesRcs
   */
  function kill($msg) {
    $this->http_status("400 Error");
    exit;
  }


  /**
   * Record an action to the log. Needed for RCS operations.
   *
   * @param action
   * @param data
   *
   * @see DiogenesRcs
   */
  function log($action,$data) {
    if (isset($_SESSION['log']) && is_object($_SESSION['log']))
      $_SESSION['log']->log($action,$data);
  }


  /** 
   * Break down a PATH_INFO into site, page id and file for WebDAV
   * We accept a missing trailing slash after the directory name.
   *
   * @param path the path to parse
   */
  function parsePathInfo($path) {
    global $globals;

    if (empty($path) || !preg_match("/^\/([^\/]+)\/webdav(\/((.+)\/)?([^\/]*))?$/",$path,$asplit))
      return false;

    $split['alias'] = $asplit[1];
    $split['dir'] = isset($asplit[4]) ? $asplit[4] : "";
    $split['file'] = isset($asplit[5]) ? $asplit[5] : "";
 
    // check that what we considered as a file is not in fact a directory
    // with a missing trailing slash
    if ( empty($split['dir']) and
        !empty($split['file']) and
        (mysql_num_rows($globals->db->query("select location from {$split['alias']}_page where location='{$split['file']}'"))>0))
    {
      $split['dir'] = $split['file'];
      $split['file'] = "";
    }
    return $split;
  }


  /**
   * Serve a webdav request
   */
  function ServeRequest()
  {
    global $globals;
 
    // break down path into site and location components
    if (!($pathinfo = $this->parsePathInfo($_SERVER['PATH_INFO']))) {
      $this->http_status("404 not found");
      exit;
    }
    $this->alias = $pathinfo['alias'];
    $this->table_page = "{$this->alias}_page";

    // check that the site exits
    $res = $globals->db->query("select alias,vhost from diogenes_site where alias='{$this->alias}'");
    if (!list($this->alias,$this->vhost) = mysql_fetch_row($res)) {
      $this->http_status("404 not found");
      exit;
    }

    // create a spool handle
    $this->spool = new DiogenesSpool($this,$this->alias);

    // let the base class do all the work
    parent::ServeRequest();
  }

  
  /** 
   * Hack to support barrels on virtualhosts.
   *
   * @param path
   * @param for_html
   */    
  function _urlencode($path, $for_html=false) {
    if ($this->vhost) {
      $path = preg_replace("/^(.*)\/site\/{$this->alias}\/(.*)/", "/\$2",$path);
    }
    return parent::_urlencode($path, $for_html);
  }

 
  /**
   * Perform authentication
   *
   * @param  type HTTP Authentication type (Basic, Digest, ...)
   * @param  user Username
   * @param  pass Password
   */
  function check_auth($type, $user, $pass)
  {
    global $globals;

    // WebDAV access is only granted for users who log in
    if (!$_SESSION['session']->doAuthWebDAV($user,$pass))
        return false;

    // we retrieve the user's permissions on the current barrel
    $_SESSION['session']->setBarrelPerms($this->alias);

    return true;
  }


  /**
   * Handle a PROPFIND request.
   *
   * @param options
   * @param files
   */
  function PROPFIND($options, &$files)
  {
    global $globals;
    $pathinfo = $this->parsePathInfo($options["path"]);

    // get the page ID and read permissions for the current path
    $res = $globals->db->query("select PID,perms from {$this->table_page} where location='{$pathinfo['dir']}'");
    if (!list($pid,$perms) = mysql_fetch_row($res))
      return false;

    // check permissions
    if (!$_SESSION['session']->hasPerms($perms))
      return "403 Forbidden";

    // get absolute fs path to requested resource
    $fspath = $this->spool->spoolPath($pid,$pathinfo['file']);

    // sanity check
    if (!file_exists($fspath)) {
      return false;
    }

    // prepare property array
    $files["files"] = array();

    // store information for the requested path itself
    $files["files"][] = $this->fileinfo($fspath, $options["path"]);

    // information for contained resources requested?
    if (!$pathinfo['file'] && !empty($options["depth"]))  { 

      // make sure path ends with '/'
      if (substr($options["path"],-1) != "/") {
        $options["path"] .= "/";
      }

      // if this is the toplevel dir, we list the page directories
      if (!$pathinfo['dir']) {
        $res = $globals->db->query("select PID,location from {$this->table_page} where location!=''");
        while (list($dpid,$dloc) = mysql_fetch_row($res)) {
          $dpath = $this->spool->spoolPath($dpid);
          $duri = $options["path"].$dloc;
          $files["files"][] = $this->fileinfo($dpath, $duri);
        }
	mysql_free_result($res);
      }
      
      // now list the files in the current directory
      $handle = @opendir($fspath);
      if ($handle) {
        // ok, now get all its contents
        while ($filename = readdir($handle)) {
          if ($filename != "." && $filename != "..") {
	    $fpath = $this->spool->spoolPath($pid,$filename);
	    $furi = $options["path"].$filename;
	    if (!is_dir($fpath))
              $files["files"][] = $this->fileinfo ($fpath, $furi);
          }
        }
      }
    }

    // ok, all done
    return true;
  }


  /**
   * Return information about a file or directory.
   *
   * @param fspath path of the filesystem entry
   * @param uri the address at which the item is visible
   */
  function fileinfo($fspath, $uri)
  {
    global $globals;

    $file = array();
    $file["path"]= $uri;

    $file["props"][] = $this->mkprop("displayname", strtoupper($uri));

    $file["props"][] = $this->mkprop("creationdate", filectime($fspath));
    $file["props"][] = $this->mkprop("getlastmodified", filemtime($fspath));

    if (is_dir($fspath)) {
      $file["props"][] = $this->mkprop("getcontentlength", 0);
      $file["props"][] = $this->mkprop("resourcetype", "collection");
      $file["props"][] = $this->mkprop("getcontenttype", "httpd/unix-directory");
    } else {
      $file["props"][] = $this->mkprop("resourcetype", "");
      $file["props"][] = $this->mkprop("getcontentlength", filesize($fspath));
      if (is_readable($fspath)) {
        $file["props"][] = $this->mkprop("getcontenttype", get_mime_type($fspath));
      } else {
        $file["props"][] = $this->mkprop("getcontenttype", "application/x-non-readable");
      }
    }
    return $file;
  }


  /**
   * Handle a GET request.
   *
   * @param options
   */
  function GET(&$options)
  {
    global $globals;
    $pathinfo = $this->parsePathInfo($options["path"]);

    // get the page ID and read permissions for the current path
    $res = $globals->db->query("select PID,perms from {$this->table_page} where location='{$pathinfo['dir']}'");
    if (!list($pid,$perms) = mysql_fetch_row($res))
      return false;

    // check permissions
    if (!$_SESSION['session']->hasPerms($perms))
      return "403 Forbidden";
   
    // create stream
    $fspath = $this->spool->spoolPath($pid,$pathinfo['file']);
    if (file_exists($fspath)) {             
      $options['mimetype'] = get_mime_type($fspath); 
                
      // see rfc2518, section 13.7
      // some clients seem to treat this as a reverse rule
      // requiering a Last-Modified header if the getlastmodified header was set
      $options['mtime'] = filemtime($fspath);
      $options['size'] = filesize($fspath);
            
      // TODO check permissions/result
      $options['stream'] = fopen($fspath, "r");

      return true;
    } else {
      return false;
    }               
  }


  /**
   * Handle a PUT request.
   *
   * @param options
   */
  function PUT(&$options)
  {
    global $globals;
    $pathinfo = $this->parsePathInfo($options["path"]);

    // we do not support multipart
    if (!empty($options["ranges"]))
      return "501";
    
    // get the page ID and write permissions for the current path
    $res = $globals->db->query("select PID,wperms from {$this->table_page} where location='{$pathinfo['dir']}'");
    if (!list($pid,$perms) = mysql_fetch_row($res))
      return false;

    // check permissions
    if (!$_SESSION['session']->hasPerms($perms))
      return "403 Forbidden";

    // create an RCS handle
    $rcs = $this->getRcs();
    $options["new"] = !file_exists($rcs->rcsFile($pid,$pathinfo['file']));

    // read content
    $content = "";    
    while (!feof($options["stream"])) {
      $content .= fread($options["stream"], 4096);
    }

    // if this is a barrel page, strip extraneous tags
    if ($pathinfo['file'] == $globals->htmlfile)
      $content = $rcs->importHtmlString($content);
  
    // perform commit
    if (!$rcs->commit($pid,$pathinfo['file'],$content,"WebDAV PUT of {$pathinfo['file']}"))
      return "400 Error";
    
    // if this is Word master document, do the HTML conversion
    if ($globals->word_import && $pathinfo['file'] == $globals->wordfile) {
      $myfile = $this->spool->spoolPath($pid,$globals->wordfile);
      $rcs->importWordFile($pid, $globals->htmlfile, $myfile);
    }
    
    return $options["new"] ? "201 Created" : "204 No Content";
  }


  /**
   * Handle an MKCOL (directory creation) request.
   *
   * @param options
   */
  function MKCOL($options)
  {
    // we do not allow directory creations
    return "403 Forbidden";
  }


  /**
   * Handle a request to delete a file or directory.
   *
   * @param options
   */
  function delete($options)
  {
    global $globals;
    $pathinfo = $this->parsePathInfo($options["path"]);
    
    // get the page ID and write permissions for the current path
    $res = $globals->db->query("select PID,wperms from {$this->table_page} where location='{$pathinfo['dir']}'");
    if (!list($pid,$perms) = mysql_fetch_row($res))
      return false;

    // check permissions
    if (!$_SESSION['session']->hasPerms($perms))
      return "403 Forbidden";

    // create an RCS handle
    $rcs = $this->getRcs();
    $rcs->del($pid,$pathinfo['file']);
    return "204 No content";
  }


  /**
   * Handle a request to move/rename a file or directory.
   *
   * @param options
   */
  function move($options)
  {
    // we do not allow moving files
    return "403 Forbidden";
  }


  /**
   * Handle a request to copy a file or directory.
   *
   * @param options
   */
  function copy($options)
  {
    // we do not allow copying files
    return "403 Forbidden";
  }


  /**
   * Handle a PROPWATCH request.
   *
   * @param options
   */
  function proppatch(&$options)
  {
    global $prefs, $tab;

    $msg = "";

    $path = $options["path"];

    $dir = dirname($path)."/";
    $base = basename($path);

    foreach($options["props"] as $key => $prop) {
      if($ns == "DAV:") {
        $options["props"][$key][$status] = "403 Forbidden";
      }
    }

    return "";
  }

}

?>
