/*  $Header: /cvsroot/dvipdfmx/src/pdfximage.c,v 1.3 2004/03/26 13:35:37 hirata Exp $
    
    This is dvipdfmx, an eXtended version of dvipdfm by Mark A. Wicks.

    Copyright (C) 2002 by Jin-Hwan Cho and Shunsaku Hirata,
    the dvipdfmx project team <dvipdfmx@project.ktug.or.kr>
    
    Copyright (C) 1998, 1999 by Mark A. Wicks <mwicks@kettering.edu>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
*/

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <ctype.h>
#include <math.h>
#include <string.h>

#include "system.h"
#include "mem.h"
#include "mfileio.h"

#include "numbers.h"
#include "dvi.h"
#include "pdflimits.h"
#include "pdfobj.h"
#include "pdfdoc.h"
#include "pdfdev.h"
#include "pdfparse.h"

#include "epdf.h"
#include "mpost.h"
#include "psimage.h"
#include "pngimage.h"
#include "jpegimage.h"

#include "pdfximage.h"

struct pdf_ximage {
  char *ident;
  char *filename;
  int   subtype;

  char  res_name[16];

  void *info;

  pdf_obj *reference;
  pdf_obj *resource;
};

#define PDF_XIMAGE_ALLOC_SIZE  5

static int verbose = 0;

void
pdf_ximage_set_verbose (void)
{
  verbose++;
}

static pdf_ximage *pdf_ximages = NULL;
static int num_ximages = 0, max_ximages = 0;

static void
pdf_ximage_init_struct (pdf_ximage *ximage)
{
  ximage->ident = NULL;
  ximage->filename = NULL;
  ximage->subtype  = -1;
  memset(ximage->res_name, 0, 16);
  ximage->info = NULL;
  ximage->reference = NULL;
  ximage->resource  = NULL;
}

void
pdf_ximage_init (void)
{
  num_ximages = 0;
  max_ximages = 0;
  if (pdf_ximages)
    ERROR("Enexpected error: pdf_ximage_init() invoked multiple times?");
  pdf_ximages = NULL;
}

void
pdf_ximage_close (void)
{
  int i;

  for (i = 0; i < num_ximages; i++) {
    if (pdf_ximages[i].filename)
      RELEASE(pdf_ximages[i].filename);
    if (pdf_ximages[i].resource)
      pdf_release_obj(pdf_ximages[i].resource);
    pdf_ximages[i].resource = NULL;
    if (pdf_ximages[i].reference)
      pdf_release_obj(pdf_ximages[i].reference);
    pdf_ximages[i].reference = NULL;
    if (pdf_ximages[i].info) {
      if (pdf_ximages[i].subtype == PDF_XOBJECT_TYPE_IMAGE) {
	ximage_info *info;
	info = (ximage_info *) pdf_ximages[i].info;
	RELEASE(info);
      } else {
	xform_info *info;
	info = (xform_info *) pdf_ximages[i].info;
	RELEASE(info);
      }
    }
    pdf_ximages[i].info = NULL;
  }
  if (pdf_ximages)
    RELEASE(pdf_ximages);

  psimage_close(); /* release distiller template */
}

static void
scale_image (transform_info *p, const ximage_info *image_info)
{
  double xscale, yscale;

  xscale = yscale = 0.0;

  if (p->scale  != 0.0) {
    xscale = p->scale * image_info->width;
    yscale = p->scale * image_info->height;
  }
  if (p->xscale != 0.0)
    xscale = p->xscale * image_info->width;
  if (p->yscale != 0.0)
    yscale = p->yscale * image_info->height;

  /* width and height overwrite scale. */
  if (p->width != 0.0)
    xscale = p->width;
  if (p->height != 0.0)
    yscale = p->height;

  /* Keep aspect */
  if (xscale == 0.0 && yscale != 0.0)
    xscale = yscale * image_info->width  / image_info->height;
  else if (xscale != 0.0 && yscale == 0.0)
    yscale = xscale * image_info->height / image_info->width;
  else if (xscale == 0.0 && yscale == 0.0) {
    xscale = image_info->width;  /* 72dpi */
    yscale = image_info->height; /* 72dpi */
  }
    
  /*
   * Raster images are always scaled to width/height, not clipped.
   */
  p->user_bbox = 0;
  p->u_llx = 0.0;
  p->u_lly = 0.0;
  p->u_urx = p->xscale;
  p->u_ury = p->yscale;

  p->xscale = xscale;
  p->yscale = yscale;
}

static void
scale_form (transform_info *p, const xform_info *form_info)
{
  double xscale, yscale;
  double width, height;

  xscale = yscale = 0.0;
  if (p->scale != 0.0) {
    xscale = p->scale;
    yscale = p->scale;
  }
  if (p->xscale != 0.0) {
    xscale = p->xscale;
  }
  if (p->yscale != 0.0) {
    yscale = p->yscale;
  }

  if (!p->user_bbox) {
    p->u_llx = form_info->bbox.llx;
    p->u_lly = form_info->bbox.lly;
    p->u_urx = form_info->bbox.urx;
    p->u_ury = form_info->bbox.ury;
  } else if (p->u_llx != form_info->bbox.llx ||
	     p->u_lly != form_info->bbox.lly ||
	     p->u_urx != form_info->bbox.urx ||
	     p->u_ury != form_info->bbox.ury) {
    /*
     * When the user supplied bounding box information disagreed
     * with the form's one, we clip image and adjust the offset.
     */
    p->clip  = 1;
    p->xoffset = form_info->bbox.llx - p->u_llx;
    p->yoffset = form_info->bbox.lly - p->u_lly;
  }

  width  = p->u_urx - p->u_llx;
  height = p->u_ury - p->u_lly;

  if (p->width != 0.0 && width != 0.0) {
    xscale = p->width  / width;
    if (p->height == 0.0)
      yscale = xscale;
  }
  if (p->height != 0.0 && height != 0.0) {
    yscale = p->height / height;
    if (p->width == 0.0)
      xscale = yscale;
  }

  if (xscale == 0.0 && yscale != 0.0)
    xscale = yscale;
  else if (yscale == 0.0 && xscale != 0.0)
    yscale = xscale;
  else if (xscale == 0.0 && yscale == 0.0)
    xscale = yscale = 1.0;
    
  p->xscale = xscale;
  p->yscale = yscale;
}

#define IMAGE_TYPE_UNKNOWN -1
#define IMAGE_TYPE_PDF  0
#define IMAGE_TYPE_JPEG 1
#define IMAGE_TYPE_PNG  2
#define IMAGE_TYPE_MPS  4
#define IMAGE_TYPE_EPS  5

static int
source_image_type (FILE *fp)
{
  int image_type;

  rewind(fp);
  /*
   * Make sure we check for PS *after* checking for MP since
   * MP is a special case of PS.
   */
  if (check_for_jpeg(fp)) {
    image_type = IMAGE_TYPE_JPEG;
  }
#ifdef HAVE_LIBPNG
  else if (check_for_png(fp)) {
    image_type = IMAGE_TYPE_PNG;
  }
#endif
  else if (check_for_pdf(fp)) {
    image_type = IMAGE_TYPE_PDF;
  } else if (check_for_mp(fp)) {
    image_type = IMAGE_TYPE_MPS;
  } else if (check_for_ps(fp)) {
    image_type = IMAGE_TYPE_EPS;
  } else {
    image_type = IMAGE_TYPE_UNKNOWN;
  }
  rewind(fp);

  return image_type;
}

static int
pdf_ximage_open (pdf_ximage *ximage, const char *filename)
{
  int   result = -1;
  char *fullname;
  FILE *fp;

  ximage->resource  = NULL;
  ximage->reference = NULL;

  fullname = kpse_find_pict(filename);
  if (!fullname) {
    WARN("Error locating image file \"%s\"", filename);
    return -1;
  }
  fp = MFOPEN(fullname, FOPEN_RBIN_MODE);
  if (!fp) {
    WARN("Error opening image file \"%s\"", filename);
    return -1;
  }

  if (verbose) {
    MESG("(IMAGE:");
    MESG("%s", filename);
    if (verbose > 1)
      MESG("[%s]", fullname);
  }

  switch (source_image_type(fp)) {
  case IMAGE_TYPE_JPEG:
    if (verbose)
      MESG("[JPEG]");
    result = jpeg_include_image(ximage, fp);
    ximage->subtype  = PDF_XOBJECT_TYPE_IMAGE;
    break;
#ifdef HAVE_LIBPNG
  case IMAGE_TYPE_PNG:
    if (verbose)
      MESG("[PNG]");
    result = png_include_image(ximage, fp);
    ximage->subtype  = PDF_XOBJECT_TYPE_IMAGE;
    break;
#endif
  case IMAGE_TYPE_PDF:
    if (verbose)
      MESG("[PDF]");
    result = pdf_include_page(ximage, fp);
    ximage->subtype  = PDF_XOBJECT_TYPE_FORM;
    break;
  case IMAGE_TYPE_MPS:
    if (verbose)
      MESG("[MPS]");
    result = mps_include_page(ximage, fp);
    ximage->subtype  = PDF_XOBJECT_TYPE_FORM;
    break;
  case IMAGE_TYPE_EPS:
    if (verbose)
      MESG("[PS]");
    result = ps_include_page(ximage, fullname);
    ximage->subtype  = PDF_XOBJECT_TYPE_FORM;
    break;
  default:
    if (verbose)
      MESG("[UNKNOWN]");
    result = ps_include_page(ximage, fullname);
    ximage->subtype  = PDF_XOBJECT_TYPE_FORM;
  }
  MFCLOSE(fp);
  if (verbose)
    MESG(")");

  return result;
}


int
pdf_ximage_findresource (const char *ident)
{
  int xobj_id;

  for (xobj_id = 0; xobj_id < num_ximages; xobj_id++) {
    if (!strcmp(ident, pdf_ximages[xobj_id].ident))
      break;
  }

  if (xobj_id == num_ximages) {
    pdf_ximage *ximage;

    if (num_ximages >= max_ximages) {
      max_ximages += PDF_XIMAGE_ALLOC_SIZE;
      pdf_ximages = RENEW(pdf_ximages, max_ximages, pdf_ximage);
    }

    ximage = &(pdf_ximages[num_ximages]);

    pdf_ximage_init_struct(ximage);

    if (pdf_ximage_open(ximage, ident) < 0) {
      WARN("pdf: image inclusion failed for (%s).", ident);
      return -1;
    }

    ximage->filename = NEW(strlen(ident)+1, char);
    ximage->ident    = NEW(strlen(ident)+1, char);
    strcpy (ximage->filename, ident);
    strcpy (ximage->ident,    ident);

    switch (ximage->subtype) {
    case PDF_XOBJECT_TYPE_IMAGE:
      sprintf(ximage->res_name, "Im%d", num_ximages);
      break;
    case PDF_XOBJECT_TYPE_FORM:
      sprintf(ximage->res_name, "Fm%d", num_ximages);
      break;
    default:
      ERROR("Unknown XObject subtype: %d", ximage->subtype);
    }

    num_ximages++;
  }

  return xobj_id;
}

void
pdf_ximage_init_form_info (xform_info *info)
{
  info->flags    = 0;
  info->bbox.llx = 0;
  info->bbox.lly = 0;
  info->bbox.urx = 0;
  info->bbox.ury = 0;
  info->matrix.a = 1.0;
  info->matrix.b = 0.0;
  info->matrix.c = 0.0;
  info->matrix.d = 1.0;
  info->matrix.e = 0.0;
  info->matrix.f = 0.0;
}

void
pdf_ximage_init_image_info (ximage_info *info)
{
  info->flags  = 0;
  info->width  = 0;
  info->height = 0;
  info->bits_per_component = 0;
  info->num_components = 0;
  info->min_dpi = 0;
}

void
pdf_ximage_set_image (pdf_ximage *ximage, void *image_info, pdf_obj *resource)
{
  pdf_obj *dict;
  ximage_info *info;

  if (!PDF_OBJ_STREAMTYPE(resource))
    ERROR("Image XObject must be stream type.");

  ximage->subtype = PDF_XOBJECT_TYPE_IMAGE;

  info = NEW(1, ximage_info);
  memcpy(info, image_info, sizeof(ximage_info));
  ximage->info = info;
  ximage->reference = pdf_ref_obj(resource);

  dict = pdf_stream_dict(resource);
  pdf_add_dict(dict, pdf_new_name("Type"),    pdf_new_name("XObject"));
  pdf_add_dict(dict, pdf_new_name("Subtype"), pdf_new_name("Image"));
  pdf_add_dict(dict, pdf_new_name("Width"),   pdf_new_number(info->width));
  pdf_add_dict(dict, pdf_new_name("Height"),  pdf_new_number(info->height));
  pdf_add_dict(dict, pdf_new_name("BitsPerComponent"),
	       pdf_new_number(info->bits_per_component));

  pdf_release_obj(resource); /* Caller don't know we are using reference. */
  ximage->resource  = NULL;
}

void
pdf_ximage_set_form (pdf_ximage *ximage,  void *info, pdf_obj *resource)
{
  ximage->subtype   = PDF_XOBJECT_TYPE_FORM;

  ximage->info = NEW(1, xform_info);
  memcpy(ximage->info, info, sizeof(xform_info));
  ximage->reference = pdf_ref_obj(resource);
  pdf_release_obj(resource); /* Caller don't know we are using reference. */
  ximage->resource  = NULL;
}

/*
 * Compute a transformation matrix
 * transformations are applied in the following
 * order: scaling, rotate, displacement.
 */
static void
make_matrix (pdf_tmatrix *matrix,
	     double ref_x,  double ref_y,
	     double xscale, double yscale,
	     double rotate)
{
  double c, s;

  c = cos(rotate);
  s = sin(rotate);

  matrix->a =  c * xscale;
  matrix->b =  s * xscale;
  matrix->c = -s * yscale;
  matrix->d =  c * yscale;
  matrix->e = ref_x;
  matrix->f = ref_y;
}

pdf_obj *
pdf_ximage_getresource (int xobj_id)
{
  pdf_ximage *ximage;

  if (xobj_id < 0 || xobj_id >= num_ximages)
    ERROR("Invalid XObject ID: %d", xobj_id);

  ximage = &(pdf_ximages[xobj_id]);
  if (!ximage->reference)
    ximage->reference = pdf_ref_obj(ximage->resource);

  return pdf_link_obj(ximage->reference);
}

int
pdf_ximage_get_form_info (int xobj_id, xform_info *form_info)
{
  pdf_ximage *ximage;
  xform_info *info;

  if (xobj_id < 0 || xobj_id >= num_ximages)
    ERROR("Invalid XObject ID: %d", xobj_id);

  ximage = &(pdf_ximages[xobj_id]);

  if (ximage->subtype != PDF_XOBJECT_TYPE_FORM) {
    WARN("XObject not form type.");
    return -1;
  }

  info = ximage->info;
  form_info->bbox.llx = info->bbox.llx;
  form_info->bbox.lly = info->bbox.lly;
  form_info->bbox.urx = info->bbox.urx;
  form_info->bbox.ury = info->bbox.ury;
  form_info->matrix.a = info->matrix.a;
  form_info->matrix.b = info->matrix.b;
  form_info->matrix.c = info->matrix.c;
  form_info->matrix.d = info->matrix.d;
  form_info->matrix.e = info->matrix.e;
  form_info->matrix.f = info->matrix.f;

  return 0;
}

int
pdf_ximage_defineresource (const char *ident,
			   int subtype, void *info, pdf_obj *resource)
{
  pdf_ximage *ximage;

  if (num_ximages >= max_ximages) {
    max_ximages += PDF_XIMAGE_ALLOC_SIZE;
    pdf_ximages = RENEW(pdf_ximages, max_ximages, pdf_ximage);
  }

  ximage = &(pdf_ximages[num_ximages]);

  pdf_ximage_init_struct(ximage);

  if (ident) {
    ximage->ident = NEW(strlen(ident) + 1, char);
    strcpy(ximage->ident, ident);
  }

  switch (subtype) {
  case PDF_XOBJECT_TYPE_IMAGE:
    pdf_ximage_set_image(ximage, info, resource);
    ximage->subtype = PDF_XOBJECT_TYPE_IMAGE;
    break;
  case PDF_XOBJECT_TYPE_FORM:
    pdf_ximage_set_form (ximage, info, resource);
    ximage->subtype = PDF_XOBJECT_TYPE_FORM;
    break;
  default:
    ERROR("Unknown XObject subtype: %d", subtype);
  }
  sprintf(ximage->res_name, "Fm%d", num_ximages);

  return num_ximages++;
}

int
pdf_ximage_put_image (int xobj_id,
		      transform_info *p,
		      double ref_x, double ref_y)
{
  pdf_ximage *ximage;
  pdf_tmatrix matrix;
  int len = 0;

  if (xobj_id < 0 || xobj_id >= num_ximages)
    ERROR("Invalid XObject ID: %d", xobj_id);

  ximage = &(pdf_ximages[xobj_id]);

  pdf_doc_add_to_page_resources("XObject",
				ximage->res_name,
				pdf_link_obj(ximage->reference));

  switch (ximage->subtype) {
  case PDF_XOBJECT_TYPE_IMAGE:
    scale_image(p, ximage->info);
    break;
  case PDF_XOBJECT_TYPE_FORM:
    scale_form (p, ximage->info);
    break;
  }

  make_matrix(&matrix,
	      ref_x, ref_y, p->xscale, p->yscale,
	      (dev_wmode() == 1) ? p->rotate - 0.50 * M_PI : p->rotate);

  work_buffer[len++] = ' ';
  work_buffer[len++] = 'q';
  work_buffer[len++] = ' ';
  len += pdf_sprint_matrix(work_buffer+len, &matrix);
  work_buffer[len++] = ' ';
  work_buffer[len++] = 'c';
  work_buffer[len++] = 'm';

  /* Clip */
  if (p->clip) {
    pdf_rect clip;

    switch (ximage->subtype) {
    case PDF_XOBJECT_TYPE_FORM:
      clip.llx = 0.0;
      clip.lly = 0.0;
      clip.urx = p->u_urx - p->u_llx;
      clip.ury = p->u_ury - p->u_lly;

      work_buffer[len++] = ' ';
      len += pdf_sprint_rect(work_buffer+len, &clip);
      work_buffer[len++] = ' ';
      work_buffer[len++] = 'r';
      work_buffer[len++] = 'e';
      work_buffer[len++] = ' ';
      work_buffer[len++] = 'W';
      work_buffer[len++] = ' ';
      work_buffer[len++] = 'n';
      break;
    case PDF_XOBJECT_TYPE_IMAGE:
      WARN("Clipping doesn't make sense for raster images.");
      break;
    default:
      ERROR("Unknown XObject type: %d", ximage->subtype);
      break;
    }
  }

  if (p->xoffset != 0.0 ||
      p->yoffset - p->depth != 0.0) {
    make_matrix(&matrix,
		p->xoffset, p->yoffset - p->depth, 1.0, 1.0, 0.0);
    work_buffer[len++] = ' ';
    len += pdf_sprint_matrix(work_buffer+len, &matrix);
    work_buffer[len++] = ' ';
    work_buffer[len++] = 'c';
    work_buffer[len++] = 'm';
  }

  len += sprintf(work_buffer+len, " /%s Do Q", ximage->res_name);
  pdf_doc_add_to_page(work_buffer, len);

  if (verbose > 1) {
    switch (ximage->subtype) {
    case PDF_XOBJECT_TYPE_IMAGE:
      {
	ximage_info *info;

	info = ximage->info;
	MESG("[%ldx%ld dpi]",
	     (long) (72 * info->width  / (p->xscale/pdf_dev_scale())),
	     (long) (72 * info->height / (p->yscale/pdf_dev_scale())));
      }
      break;
    case PDF_XOBJECT_TYPE_FORM:
      {
	xform_info *info;

	info = ximage->info;
	/* ... */
      }
      break;
    }
  }

  return 0;
}
