#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "url.h"
#include "edvtypes.h"
#include "edvobj.h"
#include "edvrecbin.h"
#include "edvarchobj.h"
#include "edvfop.h"
#include "edvrecbinfio.h"
#include "edvrecbinfop.h"
#include "edvarchfio.h"
#include "edvarchop.h"
#include "edvconfirm.h"
#include "archiveopts.h"
#include "browser.h"
#include "browserdnd.h"
#include "endeavour.h"
#include "edvop.h"
#include "edvcb.h"
#include "edvutils.h"

#include "edvutilsgtk.h"
#include "config.h"


static void EDVBrowserDNDUpdateStatusBar(
	edv_browser_struct *browser,
	gint gdk_action, guint info,
	gint total_src_objects,
	gint total_objects_processed, gint status
);

static void EDVBrowserDragDataReceivedNexus(
	edv_core_struct *core_ptr, edv_browser_struct *browser,
	GdkDragContext *dc, guint info, GtkSelectionData *selection_data,
	edv_object_struct *obj			/* Can be NULL */
);

gint EDVBrowserLocBarIconCrossingCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

void EDVBrowserDirTreeDNDSetIcon(
	edv_browser_struct *browser, GtkCTreeNode *node
);
void EDVBrowserDirTreeDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
void EDVBrowserDirTreeDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
void EDVBrowserDirTreeDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);

void EDVBrowserContentsDNDSetIcon(
	edv_browser_struct *browser, gint row, gint column
);
void EDVBrowserContentsDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
void EDVBrowserContentsDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
void EDVBrowserContentsDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Updates the File Browser's Status Bar to indicate the result
 *	of the specified operation.
 */
static void EDVBrowserDNDUpdateStatusBar(
	edv_browser_struct *browser,
	gint gdk_action, guint info,
	gint total_src_objects,
	gint total_objects_processed, gint status
)
{
	gchar *buf = NULL;
	edv_statusbar_struct *sb = (browser != NULL) ?
	    browser->status_bar : NULL;
	if(sb == NULL)
	    return;

	/* Begin formatting status message, handle by dnd target type */
	if((info == EDV_DND_TYPE_INFO_TEXT_PLAIN) ||
	   (info == EDV_DND_TYPE_INFO_TEXT_URI_LIST) ||
	   (info == EDV_DND_TYPE_INFO_STRING)
	)
	{
	    switch(gdk_action)
	    {
	      case GDK_ACTION_COPY:
		switch(status)
		{
		  case 0: case -5:
		    buf = g_strdup_printf(
			"Coppied %i object%s",
			total_objects_processed,
			(total_objects_processed == 1) ? "" : "s"
		    );
		    break;
		  case -4:	/* Cancel */
		    buf = g_strdup_printf(
			"Copy operation canceled"
		    );
		    break;
		  default:	/* Error */
		    buf = g_strdup_printf(
			"Unable to copy object%s",
			(total_src_objects == 1) ? "" : "s"
		    );
		    break;
		}
		break;
	      case GDK_ACTION_MOVE:
		switch(status)
		{
		  case 0: case -5:
		    buf = g_strdup_printf(
			"Moved %i object%s",
			total_objects_processed,
			(total_objects_processed == 1) ? "" : "s"
		    );
		    break;
		  case -4:	/* Cancel */
		    buf = g_strdup_printf(
			"Move operation canceled"
		    );
		    break;
		  default:	/* Error */
		    buf = g_strdup_printf(
			"Unable to move object%s",
			(total_src_objects == 1) ? "" : "s"
		    );
		    break;
		}
		break;
	      case GDK_ACTION_LINK:
		switch(status)
		{
		  case 0: case -5:
		    buf = g_strdup_printf(
			"Linked %i object%s",
			total_objects_processed,
			(total_objects_processed == 1) ? "" : "s"
		    );
		    break;
		  case -4:	/* Cancel */
		    buf = g_strdup_printf(
			"Link operation canceled"
		    );
		    break;
		  default:	/* Error */
		    buf = g_strdup_printf(
			"Unable to link object%s",
			(total_src_objects == 1) ? "" : "s"
		    );
		    break;
		}
		break;
	    }
	}
	else if(info == EDV_DND_TYPE_INFO_RECYCLED_OBJECT)
	{
	    switch(gdk_action)
	    {
	      case GDK_ACTION_COPY:
	      case GDK_ACTION_MOVE:
	      case GDK_ACTION_LINK:
		switch(status)
		{
		  case 0: case -5:
		    buf = g_strdup_printf(
			"Recovered %i object%s",
			total_objects_processed,
			(total_objects_processed == 1) ? "" : "s"
		    );
		    break;
		  case -4:  /* Cancel */
		    buf = g_strdup_printf(
			"Recover operation canceled"
		    );
		    break;
		  default:  /* Error */
		    buf = g_strdup_printf(
			"Unable to recover object%s",
			(total_src_objects == 1) ? "" : "s"
		    );
		    break;
		}
		break;
	    }
	}
	else if(info == EDV_DND_TYPE_INFO_ARCHIVE_OBJECT)
	{
	    switch(gdk_action)
	    {
	      case GDK_ACTION_COPY:
	      case GDK_ACTION_MOVE:
	      case GDK_ACTION_LINK:
		switch(status)
		{
		  case 0: case -5:
		    buf = g_strdup_printf(
			"Extracted %i object%s",
			total_objects_processed,
			(total_objects_processed == 1) ? "" : "s"
		    );
		    break;
		  case -4:  /* Cancel */
		    buf = g_strdup_printf(
			"Extract operation canceled"
		    );
		    break;
		  default:  /* Error */
		    buf = g_strdup_printf(
			"Unable to extract object%s",
			(total_src_objects == 1) ? "" : "s"
		    );
		    break;
		}
		break;
	    }
	}

	/* Set status bar message */
	EDVStatusBarMessage(sb, buf, FALSE);

	g_free(buf);
}


/*
 *	File Browser drag data received nextus.
 *
 *	All inputs assumed valid.
 */
static void EDVBrowserDragDataReceivedNexus(
	edv_core_struct *core_ptr, edv_browser_struct *browser,
	GdkDragContext *dc, guint info, GtkSelectionData *selection_data,
	edv_object_struct *obj			/* Can be NULL */
)
{
	gint initial_confirmation_result, nurls;
	gchar *parent_path;
	GList *url_list;
	GtkWidget *toplevel = browser->toplevel;
	const url_struct *url;

#define DO_FREE_LOCALS	{				\
 g_free(parent_path);					\
 parent_path = NULL;					\
							\
 /* Delete URL list */					\
 g_list_foreach(url_list, (GFunc)URLDelete, NULL);	\
 g_list_free(url_list);					\
 url_list = NULL;					\
}

	/* Get copy of parent_path as the object's full path if the
	 * object is not NULL
	 */
	parent_path = (obj != NULL) ? STRDUP(obj->full_path) : NULL;

	/* If parent_path is not obtainable from the disk object then
	 * use the image browser's current location as the parent_path
	 */
	if(parent_path == NULL)
	    parent_path = STRDUP(EDVBrowserCurrentLocation(browser));

	/* Check if parent_path does not lead to a directory, if it does
	 * not lead to a directory then we need to make the parent_path's
	 * parent the directory to drop to
	 */
	if(!ISPATHDIR(parent_path))
	{
	    /* Droped on object does not lead to a directory, so change
	     * the parent_path to reflect its own parent so we know that
	     * it is a directory
	     */
	    gchar *s = g_dirname(parent_path);
	    g_free(parent_path);
	    parent_path = s;
	}

	/* Decode the DDE buffer into a list of URLs */
	url_list = URLDecode(
	    (const guint8 *)selection_data->data,
	    selection_data->length
	);
	nurls = g_list_length(url_list);

	/* External drag source checks go here, for example sources
	 * that have a url with a http or ftp protocol
	 */
	url = (url_list != NULL) ? URL(url_list->data) : NULL;
	if((url != NULL) ? !STRISEMPTY(url->path) : FALSE)
	{
	    const gchar *protocol = url->protocol;
	    if(!g_strcasecmp(protocol, "http") ||
	       !g_strcasecmp(protocol, "ftp") ||
	       !g_strcasecmp(protocol, "https")
	    )
	    {
		EDVInternetDownloadObject(
		    core_ptr, url, parent_path, toplevel
		);

		/* Do not continue with standard dnd handling */
		DO_FREE_LOCALS
		return;
	    }
	}


	/* Make initial user confirmation querying to proceed with this
	 * operation
	 */
	initial_confirmation_result = EDVConfirmDND(
	    core_ptr, (gint)dc->action, info,
	    toplevel, url_list,
	    parent_path
	);
	/* Regular Object? */
	if((initial_confirmation_result == CDIALOG_RESPONSE_YES) &&
	   ((info == EDV_DND_TYPE_INFO_TEXT_PLAIN) ||
	    (info == EDV_DND_TYPE_INFO_TEXT_URI_LIST) ||
	    (info == EDV_DND_TYPE_INFO_STRING)
	   )
	)
	{
	    gint status = -1;
	    gboolean yes_to_all = FALSE;
	    gint objects_processed = 0;
	    gchar *new_path;
	    const gchar *error_mesg;
	    GList *glist;
	    struct stat lstat_buf;

	    EDVBrowserSetBusy(browser, TRUE);
	    browser->processing = TRUE;

	    /* Iterate through URL list */
	    for(glist = url_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		url = URL(glist->data);
		if(url == NULL)
		    continue;

		if(STRISEMPTY(parent_path) || STRISEMPTY(url->path))
		    continue;

		/* Handle by drag action type */
		status = -1;
		new_path = NULL;
		error_mesg = NULL;
		switch((gint)dc->action)
		{
		  case GDK_ACTION_COPY:
		    status = EDVFOPCopy(
			core_ptr, url->path, parent_path,
			&new_path, toplevel,
			TRUE, TRUE,
			&yes_to_all
		    );
		    break;
		  case GDK_ACTION_MOVE:
		    status = EDVFOPMove(
			core_ptr, url->path, parent_path,
			&new_path, toplevel,
			TRUE, TRUE,
			&yes_to_all
		    );
		    break;
		  case GDK_ACTION_LINK:
		    status = EDVFOPLink(
			core_ptr, url->path, parent_path,
			&new_path, toplevel,
			TRUE, TRUE,
			&yes_to_all
		    );
		    break;
		  default:
		    error_mesg = "Unsupported drag operation";
		    break;
		}

		/* Get error message (if any) describing the error that
		 * might have occured in the above operation
		 */
		if(error_mesg == NULL)
		    error_mesg = EDVFOPGetError();
		if(!STRISEMPTY(error_mesg) && (status != -4))
		{
		    EDVPlaySoundError(core_ptr);
		    EDVMessageError(
			"Operation Error",
			error_mesg,
			NULL,
			toplevel
		    );
		}

		/* Check if new_path is valid and that no error
		 * occured
		 *
		 * This implies the operation was successful
		 * and that we need to emit an object added signal
		 */
		if((!STRISEMPTY(new_path) && !status) ?
		    !lstat(new_path, &lstat_buf) : FALSE
		)
		{
		    objects_processed++;

		    browser->processing = FALSE;
		    EDVObjectAddedEmit(
			core_ptr, new_path, &lstat_buf
		    );
		    browser->processing = TRUE;
		}

		/* Delete copy of the new object's path */
		g_free(new_path);
		new_path = NULL;

		/* Skip handling of the rest of the objects on error
		 * (status != 0) and that the error was not a user
		 * response of no (status != -5)
		 */
		if(status && (status != -5))
		    break;
	    }

	    /* Update status bar message */
	    EDVBrowserDNDUpdateStatusBar(
		browser, (gint)dc->action, info,
		nurls, objects_processed, status
	    );

	    /* Unmap progress dialog, it may have been mapped if any
	     * operations occured in the above loop
	     */
	    ProgressDialogBreakQuery(TRUE);
	    ProgressDialogSetTransientFor(NULL);

	    /* Play completed sound on success */
	    if(status == 0)
		EDVPlaySoundCompleted(core_ptr);

	    browser->processing = FALSE;
	    EDVBrowserSetBusy(browser, FALSE);
	}
	/* Recycled Object? */
	else if((initial_confirmation_result == CDIALOG_RESPONSE_YES) &&
		(info == EDV_DND_TYPE_INFO_RECYCLED_OBJECT)
	)
	{
	    gboolean yes_to_all = FALSE;
	    gint status = -1;
	    gint objects_recovered = 0;
	    guint index;
	    gchar *new_path;
	    const gchar *error_mesg;
	    GList *glist;
	    struct stat lstat_buf;

	    EDVBrowserSetBusy(browser, TRUE);
	    browser->processing = TRUE;

	    /* Iterate through URL list */
	    for(glist = url_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		url = URL(glist->data);
		if(url == NULL)
		    continue;

		if(STRISEMPTY(url->path))
		    continue;

		/* Handle by drag action type */
		status = -1;
		new_path = NULL;
		error_mesg = NULL;
		switch((gint)dc->action)
		{
		  case GDK_ACTION_COPY:
		    error_mesg = "Recycled objects may not be coppied";
		    break;
		  case GDK_ACTION_MOVE:
		    index = (guint)ATOI(url->path);
		    status = EDVRecBinFOPRecover(
			core_ptr, index, parent_path,
			&new_path,
			toplevel,
			TRUE, TRUE, &yes_to_all
		    );
		    break;
		  case GDK_ACTION_LINK:
		    error_mesg = "Recycled objects may not be linked";
		    break;
		  default:
		    error_mesg = "Unsupported drag operation";
		    break;
		}

		/* Get error message (if any) describing the error that
		 * might have occured in the above operation
		 */ 
		if(error_mesg == NULL)
		    error_mesg = EDVRecBinFOPGetError();
		if(!STRISEMPTY(error_mesg) && (status != -4))
		{
		    EDVPlaySoundError(core_ptr);
		    EDVMessageError(
			"Operation Error",
			error_mesg,
			NULL,
			toplevel
		    ); 
		}

		/* Check if new_path is valid and that no error
		 * occured
		 *
		 * This implies the operation was successful
		 * and that we need to emit an object added signal  
		 */
		if(((new_path != NULL) && !status) ?
		    !lstat(new_path, &lstat_buf) : FALSE
		)
		{
		    objects_recovered++;

		    browser->processing = FALSE;
		    EDVObjectAddedEmit(
			core_ptr, new_path, &lstat_buf
		    );
		    browser->processing = TRUE;
		}

		/* Delete copy of the new object's path */ 
		g_free(new_path);
		new_path = NULL;

		/* Skip handling of the rest of the objects on error
		 * (status != 0) and that the error was not a user
		 * response of no (status != -5)
		 */
		if(status && (status != -5))
		    break;
	    }

	    /* Update status bar message */
	    EDVBrowserDNDUpdateStatusBar(
		browser, (gint)dc->action, info,
		nurls, objects_recovered, status
	    );

	    /* Unmap progress dialog, it may have been mapped if any
	     * operations occured in the above loop
	     */
	    ProgressDialogBreakQuery(TRUE);
	    ProgressDialogSetTransientFor(NULL);

	    /* Play completed sound on success */
	    if(status == 0)
		EDVPlaySoundCompleted(core_ptr);

	    browser->processing = FALSE;
	    EDVBrowserSetBusy(browser, FALSE);
	}
	/* Archive Object? */
	else if((initial_confirmation_result == CDIALOG_RESPONSE_YES) &&
		(info == EDV_DND_TYPE_INFO_ARCHIVE_OBJECT)
	)
	{
	    gboolean	yes_to_all = FALSE;
	    gboolean	preserve_directories = TRUE,
			preserve_timestamps = TRUE;
	    gint i, status;
	    gint objects_extracted = 0;
	    const gchar *arch_obj = NULL;
	    const gchar *error_mesg;
	    edv_archive_object_struct **aobj = NULL;
	    gint total_aobjs = 0;
	    gchar **new_path = NULL;
	    gint total_new_paths = 0;
	    GList *glist;

	    EDVBrowserSetBusy(browser, TRUE);
	    browser->processing = TRUE;

	    /* Get location of the archive itself (the first object in
	     * the list)
	     */
	    if(nurls >= 1)
	    {
		url = URL(url_list->data);
		arch_obj = (url != NULL) ? url->path : NULL;
	    }

	    /* Query user for extract from archive options */
	    if(!EDVArchExtractOptsGetResponse(
		core_ptr, toplevel, arch_obj,
		&preserve_directories,
		&preserve_timestamps
	    ))
	    {
		/* User canceled */
		browser->processing = FALSE;
		EDVBrowserSetBusy(browser, FALSE);
		DO_FREE_LOCALS
		return;
	    }

	    /* Get stats for all objects in the archive */
	    if(nurls >= 2)
	    {
		gint strc = nurls - 1;
		gchar **strv = (gchar **)g_malloc0(strc * sizeof(gchar *));

		glist = g_list_next(url_list);
		for(i = 0; i < strc; i++)
		{
		    if(glist == NULL)
			break;

		    url = URL(glist->data);
		    strv[i] = (url != NULL) ? url->path : NULL;
		    glist = g_list_next(glist);
		}

		aobj = EDVArchFIOGetListing(
		    core_ptr, arch_obj, &total_aobjs,
		    (const gchar **)strv, strc,
		    NULL
		);

		/* Delete string list (but not each string) */ 
		g_free(strv);
	    }

	    /* Extract object(s) from archive */
	    status = EDVArchOPExtract(
		core_ptr, arch_obj, aobj, total_aobjs,
		parent_path, &new_path, &total_new_paths,
		toplevel, TRUE, TRUE, &yes_to_all,
		preserve_directories,
		preserve_timestamps
	    );

	    /* Get error message (if any) describing the error that
	     * might have occured in the above operation
	     */
	    error_mesg = EDVArchOPGetError();
	    if(!STRISEMPTY(error_mesg))
	    {
		EDVPlaySoundError(core_ptr);
		EDVMessageError(
		    "Operation Error",
		    error_mesg,
		    NULL,
		    toplevel
		); 
	    }

	    /* Report new objects added if operation was a success */
	    if((new_path != NULL) && !status)
	    {
		const gchar *s;
		struct stat lstat_buf;

		browser->processing = FALSE;

		for(i = 0; i < total_new_paths; i++)
		{
		    s = new_path[i];
		    if(STRISEMPTY(s))
			continue;
		    if(lstat(s, &lstat_buf))
			continue;

		    objects_extracted++;

		    EDVObjectAddedEmit(
			core_ptr, s, &lstat_buf
		    );
		}

		browser->processing = TRUE;
	    }

	    /* Delete coppies of extracted object paths */
	    strlistfree(new_path, total_new_paths);
	    new_path = NULL;
	    total_new_paths = 0;

	    /* Update status bar message */
	    EDVBrowserDNDUpdateStatusBar(
		browser, (gint)dc->action, info,
		total_aobjs, objects_extracted, status
	    );

	    /* Unmap progress dialog, it may have been mapped if any
	     * operations occured in the above loop
	     */
	    ProgressDialogBreakQuery(TRUE);
	    ProgressDialogSetTransientFor(NULL);

	    /* Play completed sound on success */
	    if(status == 0)
		EDVPlaySoundCompleted(core_ptr);

	    browser->processing = FALSE;
	    EDVBrowserSetBusy(browser, FALSE);

	    /* Delete archive object stats */
	    for(i = 0; i < total_aobjs; i++)
		EDVArchObjectDelete(aobj[i]);
	    g_free(aobj);
	    aobj = NULL;
	    total_aobjs = 0;
	}

	DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}

/*
 *	File Browser Location Bar icon "enter_notify_event" or
 *	"leave_notify_event" signal callback.
 */
gint EDVBrowserLocBarIconCrossingCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	gint status = FALSE;
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	GtkWidget *w;
	edv_statusbar_struct *sb;
	edv_browser_struct *browser = EDV_BROWSER(data);
	if((widget == NULL) || (crossing == NULL) || (browser == NULL))
	    return(status);

	w = browser->location_icon_pm;
	sb = browser->status_bar;
	if((w == NULL) || (sb == NULL))
	    return(status);

	switch((gint)crossing->type)
	{
	  case GDK_ENTER_NOTIFY:
	    EDVStatusBarMessage(
		sb,
#if defined(PROG_LANGUAGE_SPANISH)
"Arrastre esto crear una conexin a esta ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Traner ceci pour crer un lien  cet emplacement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schleppen sie dies, eine verbindung zu diesem ort zu schaffen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Trascinare questo per creare una maglia a questa posizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Sleep dit om een schakel aan deze plaats te creren"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arraste isto criar um elo a esta localidade"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Slep dette skape et ledd til denne plasseringen"
#else
"Drag this to create a link to this location"
#endif
		, FALSE
	    );
	    gtk_pixmap_get(GTK_PIXMAP(w), &pixmap, &mask);
	    if(pixmap != NULL)
	    {
		gint width = 15, height = 15;
		gdk_window_get_size(pixmap, &width, &height);
		GUIDNDSetDragIcon(
		    pixmap, mask,
		    width / 2, height / 2
		);
	    }
	    status = TRUE;
	    break;

	  case GDK_LEAVE_NOTIFY:
	    EDVStatusBarMessage(sb, NULL, FALSE);
	    status = TRUE;
	    break;
	}

	return(status);
}


/*
 *	Sets the DND icon from the File Browser's Directory Tree node.
 */
void EDVBrowserDirTreeDNDSetIcon(
	edv_browser_struct *browser, GtkCTreeNode *node
)
{
	GtkCTree *ctree;
	edv_object_struct *obj;

	if((browser == NULL) || (node == NULL))
	    return;

	ctree = (GtkCTree *)browser->directory_ctree;
	if(ctree == NULL)
	    return;

	obj = EDV_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
	if(obj != NULL)
	{
	    GdkPixmap *pixmap = NULL;
	    GdkBitmap *mask = NULL;
	    GtkCTreeRow *row_ptr = GTK_CTREE_ROW(node);

	    /* Check if the node's row pointer is valid and obtain the
	     * pixmap and mask pair that will be used as the icon
	     */
	    if(row_ptr != NULL)
	    {
		/* If node is expanded then use the opened pixmap */
		if(row_ptr->expanded)
		{
		    pixmap = row_ptr->pixmap_opened;
		    mask = row_ptr->mask_opened;

		    /* If the opened pixmap was not available then
		     * revert to the closed pixmap
		     */
		    if(pixmap == NULL)
		    {
			pixmap = row_ptr->pixmap_closed;
			mask = row_ptr->mask_closed;
		    }
		}
		else
		{
		    pixmap = row_ptr->pixmap_closed;
		    mask = row_ptr->mask_closed;
		}
	    }

	    /* Got icon pixmap? */
	    if(pixmap != NULL)
	    {
		gint w = 15, h = 15;

		/* Get size of icon pixmap */
		gdk_window_get_size(pixmap, &w, &h);

		/* Set DND icon */
		GUIDNDSetDragIcon(
		    pixmap, mask,
		    w / 2, h / 2
		);
	    }
	}
}

/*
 *	File Browser Directory Tree "drag_data_get" signal callback.
 */
void EDVBrowserDirTreeDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	GList *glist, *url_list = NULL;
	GtkWidget *w;
	GtkCList *clist;
	GtkCTree *ctree;
	GtkCListRow *row_ptr;
	url_struct *url;
	edv_object_struct *obj;
	edv_core_struct *core_ptr;
	edv_browser_struct *browser = EDV_BROWSER(data);
	if((dc == NULL) || (browser == NULL))
	    return;

	w = browser->directory_ctree;
	core_ptr = EDV_CORE(browser->core_ptr);
	if((w == NULL) || (core_ptr == NULL))
	    return;

	ctree = GTK_CTREE(w);
	clist = GTK_CLIST(w);

	EDVBrowserSyncData(browser);

	/* Generate a list of URLs from the selected nodes */
	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    row_ptr = (GtkCListRow *)glist->data;
	    obj = EDV_OBJECT(
		gtk_ctree_node_get_row_data(ctree, (GtkCTreeNode *)row_ptr)
	    );
	    if(obj == NULL)
		continue;

	    url = URLNew();
	    url->path = STRDUP(obj->full_path);
	    url_list = g_list_append(url_list, url);
	}

	/* Encode DDE buffer from the URL list */
	if(url_list != NULL)
	{
	    gint buf_len;
	    guint8 *buf = URLEncode(url_list, &buf_len);
	    if(buf != NULL)
	    {
		/* Send out buffer */
		gtk_selection_data_set(
		    selection_data,
		    GDK_SELECTION_TYPE_STRING,
		    8,			/* Bits Per Character */
		    buf,		/* Data */
		    buf_len		/* Length */
		);
		data_sent = TRUE;
		g_free(buf);
	    }
	}

	/* Delete URL list */
	g_list_foreach(url_list, (GFunc)URLDelete, NULL);
	g_list_free(url_list);

	/* If failed to send out data then respond with error */
	if(!data_sent)
	{
	    const gchar *s = "Error";
	    gtk_selection_data_set(
		selection_data,
		GDK_SELECTION_TYPE_STRING,
		8,			/* Bits Per Character */
		s,			/* Data */
		STRLEN(s)		/* Length */
	    );
	    data_sent = TRUE;
	}
}

/*
 *	File Browser Directory Tree "drag_data_received" signal
 *	callback.
 */
void EDVBrowserDirTreeDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	GtkWidget *w;
	GtkCList *clist;
	GtkCTree *ctree;
	GtkCTreeNode *node;
	edv_object_struct *obj;
	edv_core_struct *core_ptr;
	edv_browser_struct *browser = EDV_BROWSER(data);
	if((dc == NULL) || (browser == NULL))
	    return;

	w = browser->directory_ctree;
	core_ptr = EDV_CORE(browser->core_ptr);
	if((w == NULL) || (core_ptr == NULL))
	    return;

	ctree = GTK_CTREE(w);
	clist = GTK_CLIST(w);

	/* Check if received data is not empty */
	if((selection_data != NULL) ? (selection_data->length <= 0) : TRUE)
	    return;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core_ptr, TRUE, browser->toplevel))
	    return;

	EDVBrowserSyncData(browser);

	/* Find node that this drop occured on, then get the object
	 * referenced by the node
	 */
	node = EDVNodeGetByCoordinates(
	    ctree,
	    x,
	    y - ((clist->flags & GTK_CLIST_SHOW_TITLES) ?
		clist->column_title_area.height +
		clist->column_title_area.y : 0)
	);
	obj = EDV_OBJECT((node != NULL) ?
	    gtk_ctree_node_get_row_data(ctree, node) : NULL
	);

	/* If unable to obtain object or node then that implies we have
	 * nothing to drop on
	 *
	 * Since dropping on the directory tree requires that the drop
	 * occurs on a node, we need to return due to having no node to
	 * drop on
	 */
	if((obj != NULL) ? STRISEMPTY(obj->full_path) : TRUE)
	    return;

	/* Handle received drag data */
	EDVBrowserDragDataReceivedNexus(
	    core_ptr, browser, dc, info, selection_data, obj
	);
}

/*
 *	File Browser Directory Tree "drag_data_delete" signal callback.
 */
void EDVBrowserDirTreeDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	const gchar *path;
	gchar **strv;
	gint i, strc;
	GList *glist;
	GtkWidget *w;
	GtkCList *clist;
	GtkCTree *ctree;
	GtkCListRow *row_ptr;
	edv_object_struct *obj;
	struct stat lstat_buf;
	edv_core_struct *core_ptr;
	edv_browser_struct *browser = EDV_BROWSER(data);
	if((dc == NULL) || (browser == NULL))
	    return;

	w = browser->directory_ctree;
	core_ptr = EDV_CORE(browser->core_ptr);
	if((w == NULL) || (core_ptr == NULL))
	    return;

	ctree = GTK_CTREE(w);
	clist = GTK_CLIST(w);

	/* Generate a list of object paths from the selected nodes */
	strv = NULL;
	strc = 0;
	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    row_ptr = (GtkCListRow *)glist->data;
	    obj = EDV_OBJECT(
		gtk_ctree_node_get_row_data(ctree, (GtkCTreeNode *)row_ptr)
	    );
	    if((obj != NULL) ? !STRISEMPTY(obj->full_path) : FALSE)
	    {
		i = strc;
		strc = i + 1;
		strv = (gchar **)g_realloc(
		    strv, strc * sizeof(gchar *)
		);
		if(strv == NULL)
		{
		    strc = 0;
		    break;
		}
		strv[i] = STRDUP(obj->full_path);
	    }
	}

	/* Emit object delete signal and delete list of paths */
	for(i = 0; i < strc; i++)
	{
	    path = strv[i];
	    if(!STRISEMPTY(path))
	    {
		if(lstat(path, &lstat_buf))
		    EDVObjectRemovedEmit(core_ptr, path);
	    }
	    g_free(strv[i]);
	}
	g_free(strv);
}


/*
 *	Sets the DND icon from the File Browser's Contents List cell
 *	specified by row and column.
 */
void EDVBrowserContentsDNDSetIcon(
	edv_browser_struct *browser, gint row, gint column
)
{
	GtkCList *clist;
	edv_object_struct *obj;

	if(browser == NULL)
	    return;

	clist = (GtkCList *)browser->contents_clist;
	if(clist == NULL)
	    return;

	obj = EDV_OBJECT(gtk_clist_get_row_data(clist, row));
	if(obj != NULL)
	{
	    gint i;
	    gchar *text = NULL;
	    guint8 spacing = 0;
	    GdkPixmap *pixmap = NULL;
	    GdkBitmap *mask = NULL;

	    /* Iterate through each cell of this row, looking for a
	     * useable pixmap
	     */
	    for(i = 0; i < clist->columns; i++)
	    {
		switch(gtk_clist_get_cell_type(clist, row, i))
		{
		  case GTK_CELL_PIXMAP:
		    gtk_clist_get_pixmap(
			clist, row, i,
			&pixmap, &mask
		    );
		    break;
		  case GTK_CELL_PIXTEXT:
		    gtk_clist_get_pixtext(
			clist, row, i,
			&text, &spacing, &pixmap, &mask
		    );
		    break;
		  case GTK_CELL_TEXT:
		  case GTK_CELL_WIDGET:
		  case GTK_CELL_EMPTY:
		    break;
		}
		if(pixmap != NULL)
		    break;
	    }

	    /* Got icon pixmap? */
	    if(pixmap != NULL)
	    {
		gint w = 15, h = 15;

		/* Get size of icon pixmap */
		gdk_window_get_size(pixmap, &w, &h);

		/* Set DND icon */
		GUIDNDSetDragIcon(
		    pixmap, mask,
		    w / 2, h / 2
		);
	    }
	}
}

/*
 *	File Browser Contents List "drag_data_get" signal callback.
 */
void EDVBrowserContentsDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	gint row;
	GList *glist, *url_list = NULL;
	GtkWidget *w;
	GtkCList *clist;
	url_struct *url;
	edv_object_struct *obj;
	edv_core_struct *core_ptr;
	edv_browser_struct *browser = EDV_BROWSER(data);
	if((dc == NULL) || (browser == NULL))
	    return;

	w = browser->contents_clist;
	core_ptr = EDV_CORE(browser->core_ptr);
	if((w == NULL) || (core_ptr == NULL))
	    return;

	clist = GTK_CLIST(w);

	EDVBrowserSyncData(browser);

	/* Generate a list of URLs from the selected rows */
	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    row = (gint)glist->data;
	    obj = EDV_OBJECT(gtk_clist_get_row_data(clist, row));
	    if(obj == NULL)
		continue;

	    url = URLNew();
	    url->path = STRDUP(obj->full_path);
	    url_list = g_list_append(url_list, url);
	}

	/* Encode DDE buffer from the URL list */
	if(url_list != NULL)
	{
	    gint buf_len;
	    guint8 *buf = URLEncode(url_list, &buf_len);
	    if(buf != NULL)
	    {
		/* Send out DND data buffer */
		gtk_selection_data_set(
		    selection_data,
		    GDK_SELECTION_TYPE_STRING,
		    8,			/* Bits Per Character */
		    buf,		/* Data */
		    buf_len		/* Length */
		);
		data_sent = TRUE;
		g_free(buf);
	    }
	}

	/* Delete URL list */
	g_list_foreach(url_list, (GFunc)URLDelete, NULL);
	g_list_free(url_list);

	/* If failed to send out data then respond with error */ 
	if(!data_sent)
	{
	    const gchar *s = "Error";
	    gtk_selection_data_set(
		selection_data,
		GDK_SELECTION_TYPE_STRING,
		8,			/* Bits Per Character */
		s,			/* Data */
		STRLEN(s)		/* Length */
	    );
	    data_sent = TRUE;
	}
}

/*
 *	File Browser Contents List "drag_data_received" signal callback.
 */
void EDVBrowserContentsDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gint row, column;
	GtkWidget *w;
	GtkCList *clist;
	edv_object_struct *obj;
	edv_core_struct *core_ptr;
	edv_browser_struct *browser = EDV_BROWSER(data);
	if((dc == NULL) || (browser == NULL))
	    return;

	w = browser->contents_clist;
	core_ptr = EDV_CORE(browser->core_ptr);
	if((w == NULL) || (core_ptr == NULL))
	    return;

	clist = GTK_CLIST(w);

	/* Check if received data is not empty */
	if((selection_data != NULL) ? (selection_data->length <= 0) : TRUE)
	    return;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core_ptr, TRUE, browser->toplevel))
	    return;

	EDVBrowserSyncData(browser);

	/* Find row that this drop occured on, then get the object
	 * referenced by the row
	 */
	/* Find row and column based on given coordinates */
	if(!gtk_clist_get_selection_info(
	    clist,
	    x,
	    y - ((clist->flags & GTK_CLIST_SHOW_TITLES) ?
		clist->column_title_area.height +
		clist->column_title_area.y : 0),
	    &row, &column
	))
	{
	    row = -1;
	    column = 0;
	}
	/* Get object from selected row */
	if((row >= 0) && (row < clist->rows))
	    obj = EDV_OBJECT(gtk_clist_get_row_data(clist, row));
	else
	    obj = NULL;

	/* Handle received drag data */
	EDVBrowserDragDataReceivedNexus(
	    core_ptr, browser, dc, info, selection_data,
	    obj				/* Can be NULL */
	);
}

/*
 *	File Browser Contents List "drag_data_delete" signal callback.
 */
void EDVBrowserContentsDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	const gchar *path;
	gchar **strv;
	gint i, strc, row;
	edv_core_struct *core_ptr;
	GList *glist;
	GtkWidget *w;
	GtkCList *clist;
	edv_object_struct *obj;
	struct stat lstat_buf;
	edv_browser_struct *browser = EDV_BROWSER(data);
	if((dc == NULL) || (browser == NULL))
	    return;

	w = browser->contents_clist;
	core_ptr = EDV_CORE(browser->core_ptr);
	if((w == NULL) || (core_ptr == NULL))
	    return;

	clist = GTK_CLIST(w);

	/* Generate a list of object paths from the selected rows */
	strv = NULL;
	strc = 0;
	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    row = (gint)glist->data;
	    obj = EDV_OBJECT(gtk_clist_get_row_data(clist, row));
	    if((obj != NULL) ? !STRISEMPTY(obj->full_path) : FALSE)
	    {
		i = strc;
		strc = i + 1;
		strv = (gchar **)g_realloc(
		    strv, strc * sizeof(gchar *)
		);
		if(strv == NULL)
		{
		    strc = 0;
		    break;
		}
		strv[i] = STRDUP(obj->full_path);
	    }
	}

	/* Emit object delete signal and delete list of paths */
	for(i = 0; i < strc; i++)
	{
	    path = strv[i];
	    if(!STRISEMPTY(path))
	    {
		if(lstat(path, &lstat_buf))
		    EDVObjectRemovedEmit(core_ptr, path);
	    }
	    g_free(strv[i]);
	}
	g_free(strv);
}
