/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: world.hh,v 1.33.2.2 2003/10/03 22:21:26 dheck Exp $
 */
#ifndef WORLD_HH_INCLUDED
#define WORLD_HH_INCLUDED

#include "enigma.hh"
#include "px/math.hh"

#include <string>
#include <iosfwd>
#include <vector>

namespace world
{
    using enigma::GridPos;
    using enigma::GridLoc;
    using enigma::GridLayer;
    using enigma::GridLayerBits;
    using enigma::Direction;
    using enigma::DirectionBits;
    using px::V2;

    using std::string;
}

namespace world
{

/*
** ActorInfo
** 
** This class contains the information the physics engine maintains
** about dynamic objects ("actors").
*/
    class ActorInfo {
    public:
        ActorInfo(V2 p, V2 v)
        : pos(p), vel(v), forceacc(),
          charge(0), mass(1), radius(1),
          grabbed(false),
	  last_pos(), oldpos(), force()
        {}

	//
	// Public variables
	//
        V2 pos;			// Absolute position
        V2 vel;			// Velocity
        V2 forceacc;            // Force accumulator
	double charge;		// Electric charge
	double mass;		// Mass
	double radius;		// Radius of the sphere
        bool grabbed;		// Actor not controlled by the physics engine

	//
	// Variables used internally by the physics engine
	// 
        V2 last_pos;            // Position before current tick
	V2 oldpos;		// Backup position for enter/leave notification
        V2 force;		// Force used during tick
        std::vector<V2> contact_normals; // Normal vectors of current contacts
    };

/*
** Impulse is used to transfer force from one Object to another Object
** (currently only Stones can be the destination of Impulses).
*/
    struct Impulse {
        Object    *sender;
        GridPos    dest;
        Direction  dir;

        Impulse(Object *sender_, const GridPos& dest_, Direction dir_)
        :  sender(sender_), dest(dest_), dir(dir_)
        {}
    };

/*
** Things that may happen when an actor hits a stone.
*/
    enum StoneResponse {
        STONE_PASS,             // Actor passes stone
        STONE_REBOUND           // Actor bounces off the stone
    };

/*
** StoneContact
** 
** All the information that is necessary to handle collisions between
** stones and actors.
*/
    struct StoneContact
    {
        // Constructor.
        StoneContact (Actor *a, 
                      GridPos stonepos,
                      const V2 & contact_point, 
                      const V2 & surface_normal);
        StoneContact();

        // Variables.
        Actor           *actor;
        GridPos         stonepos;
        StoneResponse   response;

        V2      contact_point;  // Where do the shapes meet? (world coordinates)
        V2      normal;         // The surface normal at the contact point
        bool    is_collision;   // Actor moves towards the stone, not away
        bool    fake_collision;
        bool    new_collision;  // True if actor did not touch the stone before
        bool    is_contact;	// if false, contact_point is actually closest feature
        std::string sound;
    };

    // Which of the four faces of the stone does the actor touch?
    Direction     contact_face (const StoneContact &sc);
    DirectionBits contact_faces (const StoneContact &sc);
}

namespace world
{
    class ForceField {
    public:
        virtual ~ForceField() {}
        virtual V2 get_force(Actor *a, V2 x, V2 v, double time)=0;
        virtual void tick(double /*dtime*/) {}
    };

    class ConstantForce : public ForceField {
    public:
        ConstantForce(V2 f) : force(f) {}
        V2 get_force(Actor */*a*/,V2 /*x*/, V2 /*v*/, double /*time*/) {
            return force;
        }
    private:
        V2 force;
    };
}

//----------------------------------------------------------------------
// Global variables
//----------------------------------------------------------------------
namespace world
{
    /* The global timer for all objects that need to be notified at
       regular intervals. */
    extern enigma::Timer g_timer;
}

namespace world
{
    void Init();

    // Create a new, empty world.
    void Create (int w, int h);

    void Reset ();

    /*! Initialize the world after loading it.  Call this after
      loading the world to force laser beams to be recalculated
      etc. */
    bool InitWorld();
    void BroadcastMessage(const std::string& msg, const enigma::Value& value, GridLayerBits grids);

    void Tick(double dtime);

    // Destroy all objects etc.
    void Shutdown();

    // Named objects:
    void    NameObject (Object *obj, const string &name);
    void    UnnameObject (Object *obj);
    void    TransferObjectName(Object *source, Object *target);
    Object *GetNamedObject (const string &name);
    Object *GetObject (const GridLoc &l);

    bool world::IsLevelBorder(GridPos p);

//----------------------------------------
// Force fields.
//----------------------------------------

    void AddForceField (ForceField *ff);
    void RemoveForceField (ForceField *ff);
    void SetMouseForce (V2 f);

//----------------------------------------
// Rubber bands
//----------------------------------------

    /* Add a rubber band that connects an actor with either a stone or
       another actor.  `strength' is the force constant, and `length'
       is the natural length of the elastic: if it is shorter than
       `length' it will exert no force on the actor(s). */
    void AddRubberBand (Actor *a, Stone *st, double strength, double length);
    void AddRubberBand (Actor *a, Actor *a2, double strength, double length);

    /* Remove the rubber band between `a' and `st'.  If `st' is 0, all
       rubber bands connecting `a' to a stone will be cut. */
    void KillRubberBand (Actor *a, Stone *st);

    /* Remove the rubber band between `a' and `a2'.  If `a2' is 0, all
       rubber bands connecting `a' to other actors will be cut. */
    void KillRubberBand (Actor *a, Actor *a2);

    /* Remove all rubber bands attached to stone ST. */
    void KillRubberBands (Stone *st);

    /* fills given vector with basic info about rubbers attached to given stone */
    struct Rubber_Band_Info {
	Actor *act;
	double length;
	double strength;
    };
    typedef std::vector<Rubber_Band_Info> RBI_vector;
    void GiveRubberBands (Stone *st, RBI_vector &rubbers);

    /* Returns true if there already is a rubber band connecting `a'
       and `st'. */
    bool HasRubberBand (Actor *a, Stone *st);


//----------------------------------------
// Signals
//----------------------------------------

    void    AddSignal (GridLoc src, GridLoc dst, const string &msg);
    void    EmitSignals (Object *src, int value = 0);
    Object *FindSignalDestination(Object *src);

//----------------------------------------
// Actors
//----------------------------------------

    void   AddActor(double x, double y, Actor* a);
    Actor *YieldActor(Actor *a);
    void   WarpActor(Actor *a, double newx, double newy, bool fast);
    void   RespawnActor(Actor *a); // like WarpActor but goto respawnpos
    void   GrabActor(Actor *a);
    void   ReleaseActor(Actor *a);

    /* Find all actors at most RANGE units away from CENTER.  Returns
       false if none were found. */
    bool   GetActorsInRange (px::V2 center, double range,
                             std::vector<Actor*> &actors);

//----------------------------------------
// Stones
//----------------------------------------

    void   SetStone(GridPos p, Stone* st);
    void   ReplaceStone(GridPos p, Stone *st);
    Stone *GetStone(GridPos p);
    Stone *YieldStone(GridPos p);
    void   KillStone(GridPos p);

    void SwapStones (GridPos p, GridPos newp);
    void MoveStone(GridPos oldPos, GridPos newPos);

    // Puzzle stone scrambling

    void SetScrambleIntensity(int intensity);
    void AddScramble(GridPos p, Direction d);

//----------------------------------------
// Items
//----------------------------------------
    void  SetItem (GridPos p, Item* it);
    void  SetItemAsStone (GridPos p, Item* it); // special case (e.g. it-brake)
    Item *GetItem (GridPos p);
    Item *YieldItem (GridPos p);
    void  KillItem (GridPos p);

// --------------------------------------
//      Item <-> Stone Conversion
// --------------------------------------

    Stone *ConvertToStone(Item *it);
    Item *ConvertToItem(Stone *st);

//----------------------------------------
// Floors
//----------------------------------------
    void   SetFloor (GridPos p, Floor* st);
    Floor *GetFloor (GridPos p);
    void   KillFloor (GridPos p);

//----------------------------------------
// Impulses
//----------------------------------------
    void addDelayedImpulse(const Impulse& impulse, double delay, const Stone *estimated_receiver);

}
#endif
