/********************************************************************
*
*  File:  glutgraph.c
*
*  Contents:  OpenGL/GLUT display for Surface Evolver
*             For OpenGL Version 1.1
*             (modified from oglgraph.c, which used aux)
*/

/* This runs in a separate thread from the main program, so it avoids
   calling kb_error, since that would longjump to the wrong place.
   Except for WARNINGS, which are OK since kb_error just returns.
   */

/* Some timings in various OpenGL drawing modes:
   (done with cat.fe and  gtime.cmd)(list, arrays exclude setup time)
   (done in 'u' mode for 10 drawings; time for one reported here)
   (flat shading for no arrays; doesn't matter for others)

facets   edges    no arrays  display list  arrays   indexed arrays   strips  
  6144    9132      0.2224       0.1500    0.1021      0.0691        0.0420
 24576   37056      0.8552       0.5740    0.3846      0.2544        0.1292
 98304  147840      3.4059                 1.4932      1.0154        0.4888
 98304       0      1.6513       0.6630    0.5208      0.5348        0.2223
393216       0      7.8273       3.2527    2.1140      2.1827        0.8720

To get a sense of the set-up times, here are recalc times,
flat shading except for strips.
facets   edges    no arrays  display list  arrays   indexed arrays   strips  
  6144    9132      0.32         1.2319    0.3796      0.5558        0.9143
 24576   37056      0.9164      11.1961    1.0064      1.6804        2.9993
 98304  147840      3.6503                 4.2781      6.9260       12.5481

On transforms, using srol.8.fe
Transforms   facets  edges    arrays   indexed arrays   strips
    24       196608  297216   3.3058        2.2122      1.2307
A little slower than one would expect, maybe due to normal flipping??
Or memory caching?  But at least set-up is fast.
*/


#undef DOUBLE 


/* so can use glutPostWindowRedisplay() */
#define GLUT_API_VERSION 4

#if defined(WIN32) || defined(_WIN32)

#undef FIXED
#include <process.h>   
#include <glut.h>

#else

#define __cdecl

#if defined(MAC_OS_X)
#include <glut.h>
#else
#include <GL/glut.h>

#endif

#endif

#include "include.h"

#ifdef _DEBUG
#define GL_ERROR_CHECK \
{ int err = glGetError(); \
  if ( err ) \
    fprintf(stderr,\
     "OpenGL error %08X window %d at line %d\n",err,glutGetWindow(),\
         __LINE__);\
}
#else
#define GL_ERROR_CHECK
#endif

/* Multiple graphing windows stuff */
int graph_thread_running;  /* whether graph thread is running */
int draw_pid; /* draw process id, for signalling from main */
int main_pid;  /* to see if same as main thread */
#ifdef PTHREADS
sigset_t newset;   /* mask SIGKICK from main thread */
#define SIGKICK SIGALRM
#endif
int dup_window;  /* window for new window to duplicate */
#define GET_DATA (gthread_data + glutGetWindow())
#define MAXGRAPHWINDOWS 10
#define WINTITLESIZE 120
struct graph_thread_data {
   int in_use;
   int new_title_flag;
   char wintitle[WINTITLESIZE];
   int arrays_timestamp; /* last arrays declared in this thread */
   long win_id;  /* GLUT graphics window id */
#ifdef WIN32
   HWND draw_hwnd;  /* handle to graphics window */
#endif
   int olddim;
   double scrx[4],scry[4];  /* screen corners */
   double aspect;  /* aspect ratio */
   REAL window_aspect; /* set by user */
   double xscale,yscale;  /* scaling factors */
   GLsizei xsize,ysize; /* window size */
   int oldx,oldy,newx,newy;  /* for tracking motion */
   int basex,basey;  /* translation to do on object */
   REAL *view[MAXCOORD+1];  /* this window's view matrix */
   REAL viewspace[MAXCOORD+1][MAXCOORD+1];
   vertex_id focus_vertex_id;    /* rotate and zoom focus */
   REAL *to_focus[MAXCOORD+1];
   REAL to_focus_space[MAXCOORD+1][MAXCOORD+1];
   REAL *from_focus[MAXCOORD+1];
   REAL from_focus_space[MAXCOORD+1][MAXCOORD+1];
   int projmode;    /* kind of projection to do */
   float projmat[16];  /* for saving projection matrix */
   int stereomode;
   int facetshow_flag; /* whether to show facets */
   REAL linewidth;
   REAL edge_bias; /* amount edges drawn in front */
   int mouse_mode;
   int mouse_left_state; /* state of left mouse button */
   int idle_flag; /* Mac OS X kludge for eliminating excessive
                         glutPostRedisplays, set by idle_func and cleared by 
                         draw_screen */
   int resize_flag; /* whether redraw event caused by resize.
              To get around Mac OS X problem of not combining events. */
                /* not used; didn't work as expected */

   int aspect_flag; /* whether pending reshape due to aspect fixing */

 } gthread_data[MAXGRAPHWINDOWS];

/* end multiple graphing windows stuff */

struct vercol { float c[4]; float n[3]; float x[3]; int inx; };
static  int mainmenu, submenu;  /* menu identifiers */
static char opengl_version[20]; /* from glGetString */
int close_flag = 0; /* whether close_show has been done */
void Ogl_close ARGS((void));
void Ogl_close_show ARGS((void));
void idle_func ARGS((void));
#ifdef PTHREADS
void * draw_thread ARGS((void *));
#else
void __cdecl draw_thread ARGS((void *));
#endif
static int glutInit_called; /* so don't call again if close and reopen */
static REAL gleps = 1e-5; /* tolerance for identifying vertices */
static REAL imagescale = 1.0;  /* scaling factor for image */
static float rgba[16][4]; 
void declare_arrays ARGS((void));
void draw_one_image ARGS((void));
void enlarge_edge_array ARGS((void));
void enlarge_facet_array ARGS((void));
void pick_func ARGS((int,int));
element_id name_to_id ARGS((GLuint));
void myMenuInit ARGS((void));
void my_glLoadName ARGS(( element_id ));
void e_glColor ARGS(( int ));
void f_glColor ARGS(( int ));
void e_glVertex3dv ARGS(( REAL * ));
void f_glVertex3dv ARGS(( REAL * ));
int hashfunc ARGS (( struct vercol *));
int vercolcomp ARGS(( struct vercol *, struct vercol *));
int eecomp ARGS(( int *, int *));
static int initz_flag = 0;  /* z buffer */

/* projmode values */
#define P_ORTHO 0
#define P_PERSP 1

/* stereomode values */
#define NO_STEREO 0
#define CROSS_STEREO 1

#define NOLIST     0
#define NORMALLIST 1
#define RESETLIST  2

static int dlistflag = NOLIST;  /* whether to use display list */
static int arraysflag=0; /* whether to use OpenGL 1.1 arrays */
static struct vercol *fullarray;
static float *colorarray;
static int edgestart,edgecount,facetstart,facetcount;
static int vertexcount;
static struct vercol *edgearray,*facetarray;
static int edgemax,facetmax; /* allocated */
static long arrays_timestamp; /* for remembering surface version */
static int interleaved_flag = 1; /* whether to do arrays as interleaved */
static int indexing_flag = 1; /* whether to use indexed arrays (smaller,but random access) */
static int *indexarray;
static int strips_flag; /* whether to do GL strips */
static int strip_color_flag; /* whether to color facets according to strip number */
static void make_strips(void);
static void make_indexlists(void);
struct stripstruct { int mode;  /* GL_TRIANGLE_STRIP or GL_LINE_STRIP */
                     int start; /* starting offset in indexarray */
                     int count; /* number of vertices in strip */
                   };
static struct stripstruct *striparray;
static int stripcount;  /* size of striparray */
static int estripcount; /* number of edge strips */
static int fstripcount; /* number of facet strips */
static int *stripdata;
static int doing_lazy;  /* whether glutgraph should do transforms itself */
static int q_flag;  /* whether to print drawing stats */
 
/*****************************************************************************
*
* function: my_glLoadName()
*
* purpose: translate element id into GLuint name.
*          Needful if element_id is longer than GLuint.
*          Format: high 2 bits set for type of element.
*                  then machine id bits, for MPI
*                  then element number.
*/
#define NAMEOFFSETBITS (8*sizeof(GLuint) - MACHINE_ID_BITS - 2)
#define NAMEOFFSETMASK  (((GLuint)1 << NAMEOFFSETBITS) - 1 )
#define NAMETYPESHIFT  (8*sizeof(GLuint)-2)
#define NAMETYPE_MASK  (3 << NAMETYPESHIFT)
#define NAMEMACHMASK (((1 << MACHINE_ID_BITS) - 1) << NAMEOFFSETBITS)

void my_glLoadName(id)
element_id id;
{ GLuint name;
  name = id_type(id) << NAMETYPESHIFT;
#ifdef MPI_EVOLVER
  name |= id_machine(id) << NAMEOFFSETBITS;
#endif
  if ( valid_id(id) )
    name |= id & OFFSETMASK & NAMEOFFSETMASK;
  else 
    name |= NAMEOFFSETMASK;
  glLoadName(name);
}

/***************************************************************************
*
* function: name_to_id()
*
* purpose: unravel picked name to element id.
*/
element_id name_to_id(name)
GLuint name;
{ element_id id;
  id = ((name & NAMETYPE_MASK) >> NAMETYPESHIFT) << TYPESHIFT;
  if ( (name & NAMEOFFSETMASK) != NAMEOFFSETMASK )
     id |=   VALIDMASK | (name & NAMEOFFSETMASK);
#ifdef MPI_EVOLVER
  id |= ((name & NAMEMACHMASK) >> NAMEOFFSETBITS) << MACH_ID_SHIFT;
#endif
  return id;
}

/********************************************************************
*
* function: enlarge_edge_array()
*
* purpose: Expand the space for the edge list
*/

void enlarge_edge_array()
{ int more = edgemax + 10;

  edgearray = (struct vercol*)realloc((char*)edgearray,
                (edgemax+more)*sizeof(struct vercol));
  edgemax += more;
}

/********************************************************************
*
* function: enlarge_facet_array()
*
* purpose: Expand the space for the facet list
*/

void enlarge_facet_array()
{ int more = 3*web.skel[FACET].count + 10;

  facetarray = (struct vercol*)realloc((char*)facetarray,
                     (facetmax+more)*sizeof(struct vercol));
  facetmax += more;
}

/***********************************************************************
*
* function: kb_glNormal3fv() etc.
*
* purpose: Either save current normal in kb_norm[] for later list entry,
*          or pass it on to gl.
*/

static float kb_norm[3] = { 0.0, 0.0, 1.0 }; /* state of normal vector */

void kb_glNormal3fv(float *v) /* save for edges and facets */
{ if ( !arraysflag ) glNormal3fv(v);
  else { kb_norm[0] = v[0]; kb_norm[1] = v[1]; kb_norm[2] = v[2]; }
}
void kb_glNormal3dv(REAL *v) /* save for edges and facets */
{ kb_norm[0] = (float)v[0]; kb_norm[1] = (float)v[1]; kb_norm[2] = (float)v[2]; 
  if ( !arraysflag ) glNormal3fv(kb_norm);
}
void kb_glAntiNormal3dv(REAL *v) /* save for edges and facets */
{ kb_norm[0] = -(float)v[0]; kb_norm[1] = -(float)v[1]; kb_norm[2] = -(float)v[2]; 
  if ( !arraysflag ) glNormal3fv(kb_norm);
}

/***********************************************************************
*
* function: e_glColor()
*
* purpose: Either save current edge color in er,eg,eb for later list entry,
*          or pass it on to gl.
*/
static float er,eg,eb,ea; /* current edge color */
void e_glColor(c) 
int c;
{ 
  if ( edge_rgb_color_attr > 0 )
  {
    if ( !arraysflag ) glColor4ubv((const GLubyte*)&c);
    else 
    { er=(float)(((c>>24)&0xFF)/255.); eg=(float)(((c>>16)&0xFF)/255.); 
      eb=(float)(((c>>8)&0xFF)/255.); ea= (float)(((c)&0xFF)/255.); 
    }
  }
  else 
  { if ( !arraysflag ) glColor4fv(rgba[c]);
    else 
    { er = rgba[c][0]; eg = rgba[c][1]; eb = rgba[c][2]; ea = rgba[c][3]; }
  }
}

/*********************************************************************
*
* function: e_glVertex3dv()
*
* purpose: Either save current edge data in edge list, or pass on to gl.
*/
void e_glVertex3dv(x)
REAL *x;
{ if ( !arraysflag ) 
  { glVertex3d((float)x[0],(float)x[1],(float)x[2]); return; }
  if ( !edgearray ) edgemax = 0;
  if ( edgecount > edgemax-5 ) enlarge_edge_array();
  edgearray[edgecount].c[0] = er;
  edgearray[edgecount].c[1] = eg;
  edgearray[edgecount].c[2] = eb;
  edgearray[edgecount].c[3] = ea;
  edgearray[edgecount].x[0] = (fabs(x[0]) < 30000) ? (float)x[0] : 0.0;
  edgearray[edgecount].x[1] = (fabs(x[1]) < 30000) ? (float)x[1] : 0.0;
  edgearray[edgecount].x[2] = (fabs(x[2]) < 30000) ? (float)x[2] : 0.0;
  edgearray[edgecount].n[0] = kb_norm[0];
  edgearray[edgecount].n[1] = kb_norm[1];
  edgearray[edgecount].n[2] = kb_norm[2];
  edgecount++;
}
/***********************************************************************
*
* function: f_glColor()
*
* purpose: Either save current facet color in fr,fg,fb for later list entry,
*          or pass it on to gl.
*/
static float fr,fg,fb,fa; /* current facet color */
void f_glColor(c) 
int c;
{ 
  if ( facet_rgb_color_attr > 0 )
  {
    if ( !arraysflag ) glColor4ubv((const GLubyte*)&c);
    else 
    { fr=(float)(((c>>24)&0xFF)/255.); fg=(float)(((c>>16)&0xFF)/255.);
      fb=(float)(((c>>8)&0xFF)/255.); fa=(float)(((c)&0xFF)/255.); 
    }
  }
  else 
  { if ( !arraysflag ) glColor4fv(rgba[c]);
    else 
    { fr = rgba[c][0]; fg = rgba[c][1]; fb = rgba[c][2]; fa = rgba[c][3]; }
  }
 
}
/*********************************************************************
*
* function: f_glVertex3dv()
*
* purpose: Either save current facet data in facet list, or pass on to gl.
*/
void f_glVertex3dv(x)
REAL *x;
{ if ( !arraysflag )
  { glVertex3f((float)x[0],(float)x[1],(float)x[2]); return; }
  if ( !facetarray ) facetmax = 0;
  if ( facetcount > facetmax-5 ) enlarge_facet_array();
  facetarray[facetcount].c[0] = fr;
  facetarray[facetcount].c[1] = fg;
  facetarray[facetcount].c[2] = fb;
  facetarray[facetcount].c[3] = fa;
  facetarray[facetcount].x[0] = (fabs(x[0]) < 30000) ? (float)x[0] : 0;
  facetarray[facetcount].x[1] = (fabs(x[1]) < 30000) ? (float)x[1] : 0;
  facetarray[facetcount].x[2] = (fabs(x[2]) < 30000) ? (float)x[2] : 0;
  facetarray[facetcount].n[0] = kb_norm[0];
  facetarray[facetcount].n[1] = kb_norm[1];
  facetarray[facetcount].n[2] = kb_norm[2];
  facetcount++;
}

/* gl matrices have vector on left! */
typedef double Matrix[4][4];
Matrix vt3 =  /* gl transform matrix */
  { {0.,0.,1.,0.},
    {1.,0.,0.,0.},
    {0.,-1.,0.,0.},
    {0.,0.,0.,1.} };
Matrix vt3p =  /* gl transform matrix, perspective version */
  { {0.,0.,1.,0.},
    {1.,0.,0.,0.},
    {0.,1.,0.,0.},
    {0.,0.,0.,1.} };

Matrix vt2 =  /* gl transform matrix */
  { {2.,0.,0.,0.},            /* can't figure why I need the 2's */
    {0.,-2.,0.,0.},
    {0.,0.,2.,0.},
    {0.,0.,0.,1.} };

Matrix flip =  /* gl transform matrix */
  { {1.,0.,0.,0.},
    {0.,1.,0.,0.},
    {0.,0.,-1.,0.},
    {0.,0.,0.,1.} };

Matrix flip2D =  /* gl transform matrix */
  { {1.,0.,0.,0.},
    {0.,-1.,0.,0.},
    {0.,0.,1.,0.},
    {0.,0.,0.,1.} };

static long prev_timestamp; /* for remembering surface version */

#define MM_ROTATE       1
#define MM_TRANSLATE   2
#define MM_SCALE       3
#define MM_SPIN        4
static int newarraysflag = 0; /* to cause rebuild of arrays */
static int dindex = 1;  /* display list object index */

void draw_screen ARGS((void));


void Ogl_init ARGS((void))
{ 
}

void Ogl_finish ARGS((void))
{                
}

void graph_new_surface()
{ int i,j,k;
  struct graph_thread_data *td;
  int titlespot;
 
  ENTER_GRAPH_MUTEX; 

  /* to account for global deallocation at start of new surface */
  /* but we've given that up due to synchronization problems */
  /*
  fullarray = NULL;
  colorarray = NULL;
  edgearray = NULL;
  facetarray = NULL;
  indexarray = NULL;
  striparray = NULL;
  stripdata = NULL;
  */
  if ( view )
  for ( k = 0, td = gthread_data ; k < MAXGRAPHWINDOWS ; k++,td++ )
  { for ( i = 0 ; i < HOMDIM ; i++ )
    { td->view[i] = td->viewspace[i];
      td->to_focus[i] = td->to_focus_space[i];
      td->from_focus[i] = td->from_focus_space[i];
      for ( j = 0 ; j < HOMDIM ; j++ )
       td->view[i][j] = view[i][j];  /* initialize to global view */
    }
    matcopy(td->to_focus,identmat,HOMDIM,HOMDIM);
    matcopy(td->from_focus,identmat,HOMDIM,HOMDIM);

    titlespot = (strlen(datafilename) > 60) ? (strlen(datafilename)-60):0;
#ifdef MPI_EVOLVER
    if ( k == 1 )
      sprintf(td->wintitle,"(task %d)  %-*s ",this_machine,WINTITLESIZE-10,
          datafilename+titlespot);
    else  sprintf(td->wintitle,"  %s (task %d) - Camera %d",
         datafilename+titlespot,this_machine,k);
#else
    if ( k == 1 )
      sprintf(td->wintitle,"  %-*s",WINTITLESIZE-10,datafilename+titlespot);
    else  sprintf(td->wintitle,"  %s - Camera %d",datafilename+titlespot,k);
#endif
    td->new_title_flag = 1;
  }
  LEAVE_GRAPH_MUTEX; 
}

void init_Oglz ARGS((void));
 
/************************************************************************
*
* function: pick_func()
*
* purpose: To handle mouse picking of element.  Invoked by right click.
*/

#define PICKBUFLENGTH 500
GLuint pickbuf[PICKBUFLENGTH];
int pick_flag;

void pick_func(x,y)
int x,y;
{ struct graph_thread_data *td = GET_DATA;
  GLint hits, viewport[4];
  int i,n;
  unsigned int enearz = 0xFFFFFFFF;
  unsigned int fnearz = 0xFFFFFFFF;
  unsigned int vnearz = 0xFFFFFFFF;
  facet_id f_id = NULLID;
  edge_id e_id = NULLID;
  vertex_id v_id = NULLID;
  int count;


  glSelectBuffer(PICKBUFLENGTH,pickbuf);
  glGetIntegerv(GL_VIEWPORT,viewport);
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glRenderMode(GL_SELECT);  /* see what picked when drawn */
  glLoadIdentity();
  gluPickMatrix(x,y,4,4,viewport);
   
  glMultMatrixf(td->projmat);
  if ( SDIM >= 3 ) glMultMatrixd(flip[0]);    /* don't know why, but need */
  else glMultMatrixd(flip2D[0]);

  pick_flag = 1;
  if ( arraysflag )       /* have to turn arrays off during picking */
  { arraysflag = 0; graph_timestamp = ++global_timestamp; 
    draw_screen();
    arraysflag = 1; graph_timestamp = ++global_timestamp;
  }
  else
    draw_screen();
  pick_flag = 0;

  hits = glRenderMode(GL_RENDER); /* back to ordinary drawing */
  if ( hits < 0 ) 
  { hits = PICKBUFLENGTH/4;  /* buffer overflow */
    kb_error(2173,
      "Pick buffer overflow; selected element may not be foreground element.\n",
        WARNING);
  }
  
  for ( i=0, n=0 ; (i < hits) && (n < PICKBUFLENGTH-4) ; i++, n += count + 3 )
  { element_id id = name_to_id(pickbuf[n+3]);
    count = pickbuf[n];
    switch ( id_type(id) )
    { case FACET: 
        if ( pickbuf[n+1] < fnearz ) 
        { f_id = id; fnearz = pickbuf[n+1]; }
        break;
      case EDGE:
        if ( pickbuf[n+1] < enearz )
          if ( valid_id(id) || !valid_id(e_id) )
          { e_id = id; enearz = pickbuf[n+1]; }
        break;
      case VERTEX: 
        if ( pickbuf[n+1] < vnearz )
          if ( valid_id(id) )
          { v_id = id; vnearz = pickbuf[n+1]; }
        break;
       
    }
  }
  outstring("\n");  /* to get to next line after prompt */
  if ( valid_id(v_id) )
    { pickvnum = loc_ordinal(v_id) + 1;
      #ifdef MPI_EVOLVER
      sprintf(msg,"Picked vertex %d@%d\n",pickvnum,id_machine(v_id));
      #else
      sprintf(msg,"Picked vertex %d\n",pickvnum);
      #endif
      outstring(msg); 
    }

  if ( valid_id(e_id) ) 
  { 
#ifdef OLDPICKVERTEX
    /* check for vertex in common to picked edges */
    for ( i=0, n=0 ; (i < hits) && (n < PICKBUFLENGTH-4) ; i++, n += count+3 )
    { element_id id,ee_id;
      int ii,nn,ccount;

      count = pickbuf[n];
      id = name_to_id(pickbuf[n+3]);
      if ( (id_type(id) == EDGE) && valid_id(id) )
      { vertex_id v1 = get_edge_headv(id);
        vertex_id v2 = get_edge_tailv(id);
        for ( ii = i+1, nn = n+count+3 ; (ii < hits) && (n < PICKBUFLENGTH-4) ;
           ii++, nn += ccount + 3 )
        { ccount = pickbuf[nn];
          ee_id = name_to_id(pickbuf[nn+3]);
          if ( (id_type(ee_id) == EDGE) && valid_id(ee_id)
               && !equal_element(id,ee_id) )
          { if ( v1 == get_edge_tailv(ee_id) )
            { v_id = v1; break; }
            if ( v1 == get_edge_headv(ee_id) )
            { v_id = v1; break; }
            if ( v2 == get_edge_tailv(ee_id) )
            { v_id = v2; break; }
            if ( v2 == get_edge_headv(ee_id) )
            { v_id = v2; break; }
          }
        }
      }
    }
    if ( valid_id(v_id) )
    { pickvnum = loc_ordinal(v_id) + 1;
      #ifdef MPI_EVOLVER
      sprintf(msg,"Picked vertex %d@%d\n",pickvnum,id_machine(v_id));
      #else
      sprintf(msg,"Picked vertex %d\n",pickvnum);
      #endif
      outstring(msg); 
    }
#endif
    pickenum = loc_ordinal(e_id) + 1;
    #ifdef MPI_EVOLVER
    sprintf(msg,"Picked edge %d@%d\n",pickenum,id_machine(e_id));
    #else
    sprintf(msg,"Picked edge %d\n",pickenum);
    #endif
    outstring(msg);
  }
  else 
  if ( e_id == NULLEDGE )
     outstring("Picked facet subdivision edge.\n");

  if ( valid_id(f_id) ) 
  { pickfnum = loc_ordinal(f_id) + 1;
    #ifdef MPI_EVOLVER
    sprintf(msg,"Picked facet %d@%d\n",pickfnum,id_machine(f_id));
    #else
    sprintf(msg,"Picked facet %d\n",pickfnum);
    #endif
    outstring(msg);
  }
  outstring(current_prompt);

  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
}

/***********************************************************
*
* function: mouse_func()
*
* purpose: Called on mouse button events, records position.
*/

void mouse_func ARGS((int,int,int,int));

void mouse_func(button,state,x,y)
int button, state, x, y;
{ struct graph_thread_data *td = GET_DATA;
  switch ( button )
  { case GLUT_LEFT_BUTTON:
      switch ( state )
      { case GLUT_DOWN:  /* start tracking */
           td->oldx = x; td->oldy = y;
           td->mouse_left_state = GLUT_DOWN;
           glutIdleFunc(idle_func);
           break;
        case GLUT_UP:  /* stop tracking */
           td->basex = td->newx;
           td->basey = td->newy;
           td->mouse_left_state = GLUT_UP;
           glutPostRedisplay();
           glutIdleFunc(idle_func);
           break;
      }
      break;

    case GLUT_RIGHT_BUTTON:
      switch ( state )
      { case GLUT_DOWN: 
          pick_func(x,y);
#if MAC_OS_X
          glutPostRedisplay();  /* get image back */  
#endif
          break;
      }
  }
}

/*******************************************************************
*
* function: mouse_loc_func()
*
* purpose: Called as mouse moves with left button down, this
*          moves surface according to current mouse_mode.
*/
void mouse_loc_func ARGS((int,int));

void mouse_loc_func(x,y)
int x,y;
{ struct graph_thread_data *td = GET_DATA;
  int i,j;

  td->newx = x;
  td->newy = y; 
  if ( td->mouse_left_state == GLUT_DOWN )
  { switch ( td->mouse_mode )
    { case MM_ROTATE:    
       mat_mult(td->to_focus,td->view,td->view,HOMDIM,HOMDIM,HOMDIM);
       fix_ctm(td->view,(REAL)( td->newx - td->oldx),
                       -(REAL)(td->newy - td->oldy));
       mat_mult(td->from_focus,td->view,td->view,HOMDIM,HOMDIM,HOMDIM);
       break;
      case MM_SCALE:
        mat_mult(td->to_focus,td->view,td->view,HOMDIM,HOMDIM,HOMDIM);
        for(i = 0 ; i < HOMDIM-1; i++ )
            for ( j = 0 ; j < HOMDIM ; j++ )
                td->view[i][j] *= 1.0 +0.002*(td->newx-td->oldx);
        mat_mult(td->from_focus,td->view,td->view,HOMDIM,HOMDIM,HOMDIM);
        break;
      case MM_TRANSLATE:
        if ( SDIM == 2 )
        { td->view[0][2] += (td->newx-td->oldx)*td->xscale;
          td->view[1][2] -= (td->newy-td->oldy)*td->yscale;
          td->to_focus[0][2] -= (td->newx-td->oldx)*td->xscale;
          td->to_focus[1][2] += (td->newy-td->oldy)*td->yscale;
          td->from_focus[0][2] += (td->newx-td->oldx)*td->xscale;
          td->from_focus[1][2] -= (td->newy-td->oldy)*td->yscale;
        } else
        {
          td->view[1][HOMDIM-1] += (td->newx-td->oldx)*td->xscale;
          td->view[2][HOMDIM-1] -= (td->newy-td->oldy)*td->yscale;
          td->to_focus[1][HOMDIM-1] -= (td->newx-td->oldx)*td->xscale;
          td->to_focus[2][HOMDIM-1] += (td->newy-td->oldy)*td->yscale;
          td->from_focus[1][HOMDIM-1] += (td->newx-td->oldx)*td->xscale;
          td->from_focus[2][HOMDIM-1] -= (td->newy-td->oldy)*td->yscale;
        };
        break;
      case MM_SPIN: /* about z axis */
        { MAT2D(rot,MAXCOORD+1,MAXCOORD+1);
          REAL dth;
          REAL dang; /* fourth dimension */
          for ( i = 0 ; i < HOMDIM ; i++ )
          { for ( j = 0 ; j < HOMDIM ; j++ )
              rot[i][j] = 0.0;
            rot[i][i] = 1.0;
          }
          dth = (td->newx - td->oldx)/300.0*M_PI;
          dang = (td->newy - td->oldy)/300.0*M_PI;
          if ( SDIM == 2 )
          { rot[0][0] = rot[1][1] = cos(dth);
            rot[0][1] = -(rot[1][0] = sin(dth));
             } else
             { rot[1][1] = rot[2][2] = cos(dth);
                rot[1][2] = -(rot[2][1] = sin(dth));
          } 
          mat_mult(td->to_focus,td->view,td->view,HOMDIM,HOMDIM,HOMDIM);
          mat_mult(rot,td->view,td->view,HOMDIM,HOMDIM,HOMDIM);
          mat_mult(td->from_focus,td->view,td->view,HOMDIM,HOMDIM,HOMDIM);
          }
        break;
      }
      if ( td->idle_flag )
         glutPostRedisplay();
    }

  td->oldx = td->newx; td->oldy = td->newy;
}

/************************************************************************
*
* function: reshape_func()
*
* purpose: handle window resize messages.
*/
void reshape_func ARGS(( int, int ));

void reshape_func ( x, y )
int x,y;
{ struct graph_thread_data *td = GET_DATA;

  if ( window_aspect_ratio != 0.0 )
  { /* munge x,y */
    REAL area = x*y;
    
    /* see if moving just one side of frame */
    if ( abs(td->xsize - x) < abs(td->ysize - y) )
    { /* moving y */
      /* if ( td->ysize == y ) return; */ /* not changing */
      x = y/fabs(window_aspect_ratio);
    }
    else //if ( td->ysize == y )
    { /* moving x */
      y = x*fabs(window_aspect_ratio);
    }
   // else
   // { /* must be moving both */
    //  x = (int)(sqrt((area/window_aspect_ratio)));
    //  y = (int)(sqrt((area*window_aspect_ratio)));
   // }
    td->window_aspect = window_aspect_ratio;
    td->aspect_flag = 0;
    glutReshapeWindow(x,y);
  }
  
  td->xsize = x; td->ysize = y;
  td->aspect = (double)y/x;
  glViewport(0,0,x,y);
  if ( td->aspect > 1 ) 
  { td->xscale = 2.8/td->aspect/x; td->yscale = 2.8/y; 
    imagescale = td->yscale*100;
  }
  else 
  { td->xscale = 2.8/x; td->yscale = 2.8*td->aspect/y; 
    imagescale = td->xscale*100;
  }
  if ( td == gthread_data+1 )
  { /* first window corresponds to printing graphics state */
    if ( x < y ) 
    { minclipx = -1.5; maxclipx = 1.5;
      minclipy = -1.5*y/x; maxclipy = 1.5*y/x;
    }
    else
    { minclipx = -1.5*x/y; maxclipx = 1.5*x/y;
      minclipy = -1.5; maxclipy = 1.5;
    }
  } 
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  
  if ( (td->projmode == P_PERSP) || td->stereomode )
  {     
     gluPerspective((float)(y/1600.*180/3.14159),1/td->aspect,1.0,20.0);
     if ( SDIM == 2 ) glMultMatrixd(vt2[0]); 
     else glMultMatrixd(vt3p[0]);  /* rotate axes */
  }
  else
  {
    if ( td->aspect >= 1.0 )
    { glOrtho(td->scrx[0],td->scrx[2],td->aspect*td->scry[0],
             td->aspect*td->scry[2],-20.0,20.0);
      if ( td == gthread_data+1 )
      { minclipx = td->scrx[0]; maxclipx = td->scrx[2]; 
        minclipy = td->aspect*td->scry[2]; maxclipy = td->aspect*td->scry[0];
      }
    }
    else
    { glOrtho(td->scrx[0]/td->aspect,td->scrx[2]/td->aspect,td->scry[0],
          td->scry[2],-20.0,20.0);
      if ( td == gthread_data+1 )
      { minclipx = td->scrx[0]/td->aspect; maxclipx = td->scrx[2]/td->aspect; 
        minclipy = td->scry[2]; maxclipy = td->scry[0];
      }
    }
 
    if ( SDIM == 2 ) glMultMatrixd(vt2[0]); /* upside down */
    else glMultMatrixd(vt3[0]);  /* upside down and rotate axes */
   }

  glGetFloatv(GL_PROJECTION_MATRIX,td->projmat); /* save */

  GL_ERROR_CHECK

#ifdef resizequick
  td->resize_flag = 1; /* So Mac OS X won't try too much redrawing */
  glutIdleFunc(idle_func);
#endif
}


/***********************************************************************
*
* Function: specialkey_func()
*
* purpose: handle special keystrokes in graphics window.
*/

void specialkey_func ARGS((int,int,int));

void specialkey_func(key,x,y)
int key,x,y;
{ struct graph_thread_data *td = GET_DATA;
  switch ( key )
  { 
    case GLUT_KEY_RIGHT: /* right arrow */
      td->view[SDIM>2?1:0][HOMDIM-1] += .25; break;
    case GLUT_KEY_LEFT:  /* left arrow */ 
      td->view[SDIM>2?1:0][HOMDIM-1] -= .25; break;
    case GLUT_KEY_UP:    /* up arrow */  
      td->view[SDIM>2?2:1][HOMDIM-1] += .25; break;
    case GLUT_KEY_DOWN:  /* down arrow */
      td->view[SDIM>2?2:1][HOMDIM-1] -= .25; break;
  }
  glutPostRedisplay();  /* generate redraw message */
}

/***********************************************************************
*
* Function: key_func()
*
* purpose: handle ASCII keystrokes in graphics window.
*/

void key_func ARGS((unsigned char,int,int));

void key_func(key,x,y)
unsigned char key;
int x,y;
{ struct graph_thread_data *td = GET_DATA;
  int i,j;

  switch ( key )
  { 
    case 'G': /* new graphics window */
        dup_window = glutGetWindow();
        draw_thread(NULL);  /* actually, just a new window */
      break;

    case 'r':  td->mouse_mode = MM_ROTATE; break;
    case 't':  td->mouse_mode = MM_TRANSLATE; break;
    case 'z':  td->mouse_mode = MM_SCALE; break;
    case 'c':  td->mouse_mode = MM_SPIN; break;

    case 'o': box_flag = !box_flag;
              graph_timestamp = ++global_timestamp; 
              break;

    case 'b':
        td->edge_bias -= 0.001; 
        sprintf(msg,"\nEdge front bias now %f\n",td->edge_bias); outstring(msg);
        outstring(current_prompt);
        break;

    case 'B':
        td->edge_bias += 0.001; 
        sprintf(msg,"\nEdge front bias now %f\n",td->edge_bias); outstring(msg);
        outstring(current_prompt);
        break;

    case 'R':
        resize(); 
        for ( i = 0 ; i < HOMDIM ; i++ )
          for ( j = 0 ; j < HOMDIM ; j++ )
            td->view[i][j] = view[i][j];
        matcopy(td->to_focus,identmat,HOMDIM,HOMDIM);
        matcopy(td->from_focus,identmat,HOMDIM,HOMDIM);
        break;
 
    case '-':
        if ( td->linewidth > 0.6 )
        { td->linewidth -= 0.5;  
          glLineWidth(td->linewidth);
        } 
        break;
  
    case '+':
        if ( td->linewidth < 9.9 ) 
        { td->linewidth += 0.5; 
          glLineWidth(td->linewidth);
        }
        break;

    case 'e':
        edgeshow_flag = !edgeshow_flag; 
        graph_timestamp = ++global_timestamp; 
        newarraysflag = 1;
        break;

    case 'f':
        td->facetshow_flag = !td->facetshow_flag; 
        graph_timestamp = ++global_timestamp; 
        newarraysflag = 1;
        break;

    case 'g':  /* Gourard toggle */
        normflag = !normflag; graph_timestamp = ++global_timestamp; 
        newarraysflag = 1;
        break;

    case 'm': /* move to middle */
         { do_gfile(0,NULL); /* get bounding box */
           if ( SDIM == 2 )
            { td->view[0][HOMDIM-1] -= (bbox_maxx+bbox_minx)/2;
              td->view[1][HOMDIM-1] -= (bbox_maxy+bbox_miny)/2;
            } else
            { td->view[1][HOMDIM-1] -= (bbox_maxx+bbox_minx)/2;
              td->view[2][HOMDIM-1] -= (bbox_maxy+bbox_miny)/2;
            }
            break;
         }

    case 'i': /* toggle interleaved elements in opengl arrays */ 
        interleaved_flag = !interleaved_flag;
        outstring(interleaved_flag?"Interleaving ON.\n":"Interleaving OFF.\n"); 
        outstring(current_prompt);
        newarraysflag = 1;
        break;

    case 'I': /* indexed arrays */
        indexing_flag = !indexing_flag;
        outstring(indexing_flag?"Array indexing ON.\n":"Array indexing OFF.\n"); 
        outstring(current_prompt);
        newarraysflag = 1;
        break;
   
    case 'S': /* toggle sending strips in arrays */
        strips_flag = !strips_flag;
        outstring(strips_flag?"Element strips ON.\n":"Element strips OFF.\n"); 
        outstring(current_prompt);
        normflag = 1; /* gourard shading */
        indexing_flag = 1;
        newarraysflag = 1;
        break;
 
    case 'Y': /* colored strips */
      { int *fcolors;
        int i;
        fcolors = (int*)temp_calloc(web.skel[FACET].maxcount,sizeof(int));
        for ( i = 0 ; i < web.skel[FACET].maxcount ; i++ )
           fcolors[i] = get_facet_color(i);
        if ( !strips_flag ) key_func(0,0,'S');  /* make sure strips on */
        strip_color_flag = 1;
        newarraysflag = 1;
        draw_screen();  /* get facets colored */
        newarraysflag = 1;
        draw_screen();  /* draw colored facets */
        strip_color_flag = 0;
        for ( i = 0 ; i < web.skel[FACET].maxcount ; i++ )
           set_facet_color(i,fcolors[i]);
        temp_free((char*)fcolors);
      }
      break;

    case 'F': /* use last pick to set rotation center */
        if ( pickvnum > 0 )
        { int i,m;
          REAL *x,xx[MAXCOORD];
          REAL focus[MAXCOORD];

          td->focus_vertex_id = get_ordinal_id(VERTEX,pickvnum-1);
          x = get_coord(td->focus_vertex_id);  
          for ( m = 0 ; m < SDIM ; m++ ) xx[m] = x[m];
          if ( torus_display_mode == TORUS_CLIPPED_MODE )
          { 
            for ( m = 0 ; m < SDIM ; m++ )
            { int wrap = (int)floor(SDIM_dot(web.inverse_periods[m],x));
              for ( i = 0 ; i < SDIM ; i++ )
                xx[i] -= wrap*web.torus_period[m][i];
            }
          }

          matvec_mul(td->view,xx,focus,HOMDIM-1,HOMDIM-1);
          for ( i = 0 ; i < SDIM ; i++ ) 
          { td->to_focus[i][HOMDIM-1] = -focus[i] - td->view[i][HOMDIM-1];
            td->from_focus[i][HOMDIM-1] = focus[i] + td->view[i][HOMDIM-1];
          }
        }
        break;

    case 'D': /* toggle display list */
        dlistflag = dlistflag ? NOLIST:RESETLIST; 
        if ( dlistflag )
        { outstring("\nOpenGL display list now ON.\n");
          if ( arraysflag )
          { glDisableClientState(GL_COLOR_ARRAY);
            glDisableClientState(GL_NORMAL_ARRAY);
            glDisableClientState(GL_VERTEX_ARRAY);
            outstring("\nOpenGL arrays now OFF.\n");
          }
          arraysflag = 0;
        }
        else outstring("\nOpenGL display list now OFF.\n");
        outstring(current_prompt);
        graph_timestamp = ++global_timestamp; /* force recalculate arrays */
        break;
  
    case 'a': /* toggle vertex arrays */
        if ( strcmp(opengl_version,"1.1") < 0 )
        { sprintf(errmsg,
           "Vertex arrays require OpenGL version at least 1.1. This is %s.\n",
                   opengl_version);
          kb_error(2174,errmsg,WARNING);
         return;
        }
        arraysflag = !arraysflag; 
        if ( arraysflag )
        { /* Workaround really bizarre line-drawing bug */
          if ( td->linewidth == 1.0 ) { td->linewidth = 0.5; glLineWidth(0.5);}
          outstring("\nOpenGL vertex arrays now ON.\n");
          glInterleavedArrays(GL_C4F_N3F_V3F,0,(void*)fullarray);
          newarraysflag = 1;
          dlistflag = 0;
        }
        else
        {
          glDisableClientState(GL_COLOR_ARRAY);
          glDisableClientState(GL_NORMAL_ARRAY);
          glDisableClientState(GL_VERTEX_ARRAY);
          outstring("\nOpenGL vertex_arrays now OFF.\n");
        }
        outstring(current_prompt);
        break;

   case 's': /* toggle stereo */
      td->stereomode = (td->stereomode==NO_STEREO) ? CROSS_STEREO:NO_STEREO; 
      reshape_func(td->xsize,td->ysize);
      break;
 
   case 'p': /* perspective mode */
      if ( td->stereomode ) 
         outstring("\nOpenGL projection now CROSS-EYED STEREO.\n"); 
      else if ( td->projmode==P_ORTHO ) 
      { td->projmode=P_PERSP; 
        outstring("\nOpenGL projection now PERSPECTIVE.\n");
      }
      else 
      { td->projmode=P_ORTHO; 
        outstring("\nOpenGL projection now ORTHOGONAL.\n");
      }
      outstring(current_prompt);
      reshape_func(td->xsize,td->ysize);
      break;
 
   case 'M': /* menu mode on right mouse button */
      glutSetMenu(mainmenu);
      glutAttachMenu(GLUT_RIGHT_BUTTON);
      break;


   case 'P': /* pick mode on right mouse button */
      glutSetMenu(mainmenu);
      glutDetachMenu(GLUT_RIGHT_BUTTON);
      break;

   case 'Q': /* toggle drawing stats printing */
      q_flag = !q_flag;
      outstring(q_flag ?
       "\nPrinting drawing stats ON\n":"\nPrinting drawing stats OFF\n");
      outstring(current_prompt);
      break;

   #ifdef MPI_EVOLVER
   case 'y': /* MPI version only */
      mpi_show_corona_flag = ! mpi_show_corona_flag;
      newarraysflag = 1;
      outstring(mpi_show_corona_flag ?
       "\nShowing MPI corona ON\n":"\nShowing MPI corona OFF\n");
      outstring(current_prompt);
      break;
   #endif

   case 'x': 
      Ogl_close();
      return;

   case 'h': case '?':
      outstring("\nGraphics window help:\n");
      outstring("Left mouse: move   Right mouse: pick\n");
      outstring("Graphics window keys:\n");
      outstring(td->mouse_mode==MM_ROTATE?
        "r  Rotate mode for left mouse button, now ACTIVE\n":
        "r  Rotate mode for left mouse button\n");
      outstring(td->mouse_mode==MM_TRANSLATE?
        "t  Translate mode for left mouse button, now ACTIVE\n":
        "t  Translate mode for left mouse button\n");
      outstring(td->mouse_mode==MM_SCALE?
        "z  Zoom mode for left mouse button, now ACTIVE\n":
        "z  Zoom mode for left mouse button\n");
      outstring(td->mouse_mode==MM_SPIN?
        "c  Clockwise/counterclockwise mode, now ACTIVE\n":
        "c  Clockwise/counterclockwise mode\n");
      outstring("W  Widen edges\n");
      outstring("w  Narrow edges\n");
      outstring("B  Increase edge front bias by 0.001\n");
      outstring("b  Decrease edge front bias by 0.001\n");
      outstring(normflag? 
        "g  Gourard shading (smooth shading) toggle, now ON\n":
        "g  Gourard shading (smooth shading) toggle, now OFF\n" );
      outstring("R  Reset view\n");
      outstring("m  Center image\n");
      outstring(edgeshow_flag?"e  Toggle showing all edges, now ON\n":
        "e  Toggle showing all edges, now OFF\n");
      outstring(td->facetshow_flag?"f  Toggle showing facets, now ON\n":
        "f  Toggle showing facets, now OFF\n");
      outstring(td->projmode==P_PERSP?
        "p  Toggle orthogonal/perspective projection, now perspective\n":
        "p  Toggle orthogonal/perspective projection, now orthogonal\n");
      outstring("s  Toggle cross-eyed stereo\n");
      outstring(normflag? "g  Gourard shading (smooth shading) toggle, now ON\n":
             "g  Gourard shading (smooth shading) toggle, now OFF\n" );
      outstring("F  Set rotate/zoom focus to last picked vertex\n");
      outstring("arrow keys  translate image\n");
      outstring("G  Another graphics window\n");
      outstring("H  Guru-level help items\n");
      outstring("x  Close graphics\n");
      outstring(current_prompt);
      break;

   case 'H': /* guru level help */
     outstring("\nFollowing for fiddling with OpenGL drawing modes:\n");
     outstring(dlistflag?"D  Toggle using display list, now ON\n":
       "D  Toggle using display list, now OFF\n");
     outstring(arraysflag?"a  Toggle using vertex and color arrays, now ON\n"
        :"a  Toggle using vertex and color arrays, now OFF\n");
     outstring(indexing_flag ? "I  Indexed vertex arrays, now ON\n"
                           : "I  Indexed vertex arrays, now OFF\n");
     outstring(interleaved_flag ? "i  Interleaved vertex arrays, now ON\n"
                           : "i  Interleaved vertex arrays, now OFF\n");
     outstring(strips_flag?
    "S  Use element strips, now ON (indexed elements automatically turned on)\n"
    :"S  Use element strips, now OFF\n");
     outstring("Y  One-time coloring of strips generated in S mode.\n");
     outstring("       Caution: assumes display facets same as real, with no gaps.\n");
     outstring("u  Toggle window update every graphics command, now ");
     outstring("Q  Toggle printing some statistics during drawing\n");
     #ifdef MPI_EVOLVER
     outstring("y  Toggle corona display\n");
     #endif
 
     outstring(current_prompt);
     break;
  }
  glutPostRedisplay();  /* generate redraw message */
} /* end of key_func() */


 void mainmenu_func ARGS((int));
 void submenu_func ARGS((int));

 void mainmenu_func(choice)
 int choice;
 { 
   key_func(choice,0,0); 
 }
 
 void submenu_func(choice)
 int choice;
 { 
   key_func(choice,0,0);
 }
 
 void myMenuInit()
 { mainmenu = glutCreateMenu(mainmenu_func);
   glutSetMenu(mainmenu);
   glutAddMenuEntry("Left button modes:",' ');
   glutAddMenuEntry("Rotate mode (r)",'r');
   glutAddMenuEntry("Translate mode (t)",'t');
   glutAddMenuEntry("Spin mode (c)",'c');
   glutAddMenuEntry("Scale mode (z)",'z');
   glutAddMenuEntry("Right button modes:",' ');
   glutAddMenuEntry("  Pick mode (P)",'P');
   glutAddMenuEntry("  Menu mode (M)",'M');
   glutAddMenuEntry("Edges front (B)",'B');
   glutAddMenuEntry("Edges back (b)",'b');
   glutAddMenuEntry("Edges thicker (+)",'+');
   glutAddMenuEntry("Edges thinner (-)",'-');
   glutAddMenuEntry("Center object (m)",'m');
   glutAddMenuEntry("Toggle edges (e)",'e');
   glutAddMenuEntry("Toggle faces (f)",'f');
   glutAddMenuEntry("Focus on picked vertex (F)",'F');
   glutAddMenuEntry("Bounding box (o)",'o');
   glutAddMenuEntry("Reset graphics (R)",'R');
   glutAddMenuEntry("",' ');  /* skip funny entry before submenu */
   submenu = glutCreateMenu(submenu_func);
   glutSetMenu(submenu);
   glutAddMenuEntry("Another graphics window (G)",'G');
   glutAddMenuEntry("Toggle Gourard shading (g)",'g');
   glutAddMenuEntry("Toggle stereo view (s)",'s');
   glutAddMenuEntry("Toggle orthogonal/perspective view (p)",'p');
   glutAddMenuEntry("Toggle arrays (a)",'a');
   glutAddMenuEntry("Toggle interleaved arrays (i)",'i');
   glutAddMenuEntry("Toggle indexed arrays (I)",'I');
   glutAddMenuEntry("Toggle strips (S)",'S');
   glutAddMenuEntry("Toggle strip coloring (Y)",'Y');
   glutAddMenuEntry("Toggle display lists (D)",'D');
   glutAddMenuEntry("Print drawing stats (Q)",'Q');
   #ifdef MPI_EVOLVER
   glutAddMenuEntry("Toggle corona display (y)",'y');
   #endif
   glutSetMenu(mainmenu);
   glutAddSubMenu("Advanced",submenu);
   glutAddMenuEntry("Close graphics (x)",'x');
   glutAttachMenu(GLUT_MIDDLE_BUTTON);
 }
 

 /* lighting info for surface */
 static GLfloat mat_specular[] = {.5f,.5f,.5f,1.0f};
 static GLfloat mat_shininess[] = {10.0f};
 static GLfloat mat_diffuse[] = {1.0f,1.0f,1.0f,1.0f}; 
 static GLfloat mat_white[] = {1.0f,1.0f,1.0f,1.0f};
 static GLfloat mat_emission[] = {0.3f,0.3f,0.3f,1.0f};
#define INTENSITY1  0.5f
 static GLfloat light0_position[] = {1.0f,0.0f,1.0f,0.0f};  /* front */
 static GLfloat light0_diffuse[] = {INTENSITY1,INTENSITY1,INTENSITY1,1.0f};
 static GLfloat light0_ambient[] = {.3f,.3f,.3f,1.0f};

#define INTENSITY2  0.5f
 static GLfloat light1_position[] = {0.0f,0.0f,1.0f,0.0f};  /* above */
 static GLfloat light1_diffuse[] = {INTENSITY2,INTENSITY2,INTENSITY2,1.0f};
 static GLfloat light_none[] = {0.f,0.f,0.f,.0f};

#ifdef WIN32
 /*******************************************************************
 *
 * function: handle_func()
 *
 * purpose: callback for EnumThreadWindows()
 */

static handle_count;

BOOL __stdcall handle_func(HWND hwnd, LPARAM lParam)
{ int i;
  for ( i = 1 ; i < MAXGRAPHWINDOWS ; i++ )
  { if ( gthread_data[i].draw_hwnd == hwnd ) return TRUE;
    if ( !gthread_data[i].draw_hwnd )
    { gthread_data[i].draw_hwnd = hwnd;
      break;
    }    
  }
  return TRUE;
}
#endif

#ifndef WIN32

 /*****************************************************************
 *
 * function: glut_catcher()
 *
 * purpose: catch signals meant to wake up thread.
 */

 void glut_catcher(x)
 int x;
 {
   signal(SIGKICK,glut_catcher);
 }

 #endif

 /*****************************************************************
 *
 * function: draw_thread()
 *
 * purpose: Create OpenGL display thread and window.
 */
#ifdef PTHREADS
void * draw_thread(arglist)
void *arglist;
#else
void __cdecl draw_thread(arglist)
void *arglist;
#endif
{ int i,j;
  int argc = 1; /* to keep glutInit happy */
  char *argv = "Evolver";
  struct graph_thread_data *td;
  char wintitle[WINTITLESIZE];
  static int win_id;
  int glut_id;
  int xpixels,ypixels;

#ifdef WIN32
  HANDLE foregroundhwnd = GetForegroundWindow();
  draw_thread_id = GetCurrentThreadId();
#else
  draw_thread_id = pthread_self();  /* get thread id */
  draw_pid = getpid();
 
#endif

  for ( i = 0 ; i < 16 ; i++ )
    for ( j = 0 ; j < 4 ; j++ )
      rgba[i][j] = (float)rgb_colors[i][j];

  if ( !glutInit_called )
    glutInit(&argc,&argv);
  glutInit_called = 1;
  if ( !graph_thread_running ) win_id = 0;
  win_id++;
  if ( win_id >= MAXGRAPHWINDOWS )
  { kb_error(2556,"No more graphics windows available.\n",WARNING);
  }
  glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH );
  glutInitWindowPosition((20*win_id)%400,(20*win_id)%400);
  if ( window_aspect_ratio )
  { xpixels = (int)(400/sqrt(fabs(window_aspect_ratio)));
    ypixels = (int)(400*sqrt(fabs(window_aspect_ratio)));
  }
  else 
  { xpixels = 400;
    ypixels = 400;
  }
  glutInitWindowSize(xpixels,ypixels); 
#ifdef MAC_OS_X
  { char title[1000];
    sprintf(title,"   %s (CTRL-click for right mouse button)",datafilename);
    glutCreateWindow(title);
  }
#else
  glutCreateWindow(datafilename);
#endif

  glutMouseFunc(mouse_func); 
  glutMotionFunc(mouse_loc_func); 
  glutKeyboardFunc(key_func);
  glutSpecialFunc(specialkey_func);
  glutReshapeFunc(reshape_func);
  glutIdleFunc(idle_func); 
  myMenuInit();
  glutShowWindow();  /* start on top */

  glut_id = glutGetWindow(); /* window identifier */
  if ( glut_id >= 10 )
     kb_error(2596,"glut window id too high.\n",RECOVERABLE);
  else td = gthread_data + glut_id;
  td->in_use = 1;
  td->win_id = glut_id;
  td->aspect = 1;
  td->xscale=2.8/xpixels;
  td->yscale=2.8/ypixels;
  background_color = LIGHTBLUE;
  td->xsize = xpixels; 
  td->ysize = ypixels;
  td->projmode = P_ORTHO;    /* kind of projection to do */
  td->stereomode = NO_STEREO;
  td->facetshow_flag = 1; /* whether to show facets */
  td->linewidth = 1.0;
  td->edge_bias = 0.005; /* amount edges drawn in front */
  td->mouse_left_state = GLUT_UP; /* state of left mouse button */
  td->mouse_mode = MM_ROTATE;
  if (glut_id > 1 ) 
  { if ( strlen(datafilename) > 60 )
      sprintf(wintitle,"  %s - Camera %d",datafilename+strlen(datafilename)-60,
          glut_id);
    else  sprintf(wintitle,"  %s - Camera %d",datafilename,glut_id);
#ifdef MPI_EVOLVER
    sprintf(wintitle+strlen(wintitle)," (task %d)",this_machine);
#endif
    glutSetWindowTitle(wintitle);
  }

  ENTER_GRAPH_MUTEX; /* due to view[][] */
  if ( dup_window )
  { struct graph_thread_data *tdd = gthread_data+dup_window;
    for ( i = 0 ; i < HOMDIM ; i++ )
    { td->view[i] = td->viewspace[i];
      td->to_focus[i] = td->to_focus_space[i];
      td->from_focus[i] = td->from_focus_space[i];
      for ( j = 0 ; j < HOMDIM ; j++ )
      { td->view[i][j] = tdd->view[i][j]; 
        td->to_focus[i][j] = tdd->to_focus[i][j];
        td->from_focus[i][j] = tdd->from_focus[i][j];
      }
    }
    dup_window = 0;
  }
  else
  { for ( i = 0 ; i < HOMDIM ; i++ )
    { td->view[i] = td->viewspace[i];
      td->to_focus[i] = td->to_focus_space[i];
      td->from_focus[i] = td->from_focus_space[i];
      for ( j = 0 ; j < HOMDIM ; j++ )
        td->view[i][j] = view[i][j];  /* initialize to global view */
    }
    matcopy(td->to_focus,identmat,HOMDIM,HOMDIM);
    matcopy(td->from_focus,identmat,HOMDIM,HOMDIM);
  }
  LEAVE_GRAPH_MUTEX;

  glDepthFunc(GL_LEQUAL);   
  glEnable(GL_DEPTH_TEST);
  glEnable(GL_NORMALIZE); 
  glEnable(GL_COLOR_MATERIAL);
  glColorMaterial(GL_FRONT_AND_BACK,GL_AMBIENT_AND_DIFFUSE);
  /*glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular); */
  /*glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, mat_shininess); */
  glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, mat_white);
  /*glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, mat_emission);  */
  glEnable(GL_LIGHTING);
  glLightfv(GL_LIGHT0, GL_POSITION, light0_position);  
  glLightfv(GL_LIGHT0, GL_DIFFUSE, light0_diffuse);  
  glLightfv(GL_LIGHT0, GL_AMBIENT, light0_ambient);  
  glLightfv(GL_LIGHT1, GL_POSITION, light1_position);
  glLightfv(GL_LIGHT1, GL_DIFFUSE, light1_diffuse);
  glLightfv(GL_LIGHT0, GL_SPECULAR, light_none); 
  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE,GL_TRUE);
  
  /* screen corners */
  td->scrx[0] = td->scrx[1] = -1.4;
  td->scrx[2] = td->scrx[3] =  1.4;
  td->scry[0] = td->scry[3] =  1.4;
  td->scry[1] = td->scry[2] = -1.4;
  
  go_display_flag = 1; /* default, since fast graphics */

  /* version check */
  strncpy(opengl_version,(char*)glGetString(GL_VERSION),sizeof(opengl_version));
  if ( strcmp(opengl_version,"1.1") >= 0 ) arraysflag = 1;

  glutDisplayFunc(draw_screen);

#ifdef WIN32
  /* get window handle */
  handle_count = 1;
  EnumThreadWindows(draw_thread_id,handle_func,0);

  /* try to get on top */
 
  SetForegroundWindow(gthread_data[glut_id].draw_hwnd);
  SetForegroundWindow(foregroundhwnd);  /* get console back on top */

#endif

  GL_ERROR_CHECK
  if ( !graph_thread_running ) 
  {
  
#ifndef WIN32
   signal(SIGKICK,glut_catcher);
#endif

    graph_thread_running = 1;
    glutMainLoop();
  } 
#ifdef PTHREADS
  return NULL;
#endif
}
 
/************************************************************************
*
* function: init_Oglz()
*
* purpose: start up graphics thread 
*/

void init_Oglz()
{ int n;

  if ( !initz_flag )
  { initz_flag = 1;
    if ( !to_focus )
    { to_focus = dmatrix(0,MAXCOORD,0,MAXCOORD);
      from_focus = dmatrix(0,MAXCOORD,0,MAXCOORD);
      for ( n = 0 ; n <= MAXCOORD ; n++ )
      to_focus[n][n] = from_focus[n][n] = 1.0;
    }

#ifdef WIN32
     _beginthread(draw_thread,0,0);
#elif defined(PTHREADS)
   { pthread_t th;
    /*
    {
      sigemptyset(&newset);
      sigaddset(&newset,SIGKICK);
      pthread_sigmask(SIG_BLOCK,&newset,NULL);
    }
    */
    main_pid = getpid();
#ifdef MAC_OS_X
    /* GLUT doesn't work as secondary thread on 10.0.2, so main draws */
    pthread_create(&th,NULL,(void*(*)(void*))mac_exec_commands,NULL);
    draw_thread(NULL);
#else
    pthread_create(&th,NULL,draw_thread,NULL);
#endif
   }
#else
    kb_error(2447,"Internal error: Evolver compiled with GLUT graphics without threading.\n",RECOVERABLE);
#endif

  }
}

/****************************************************************
*
*  function: Oglz_start()
*
*  purpose: To be called at the start of each Evolver-initiated redraw.
*/
void Oglz_start ARGS((void));
void Oglz_start()
{
  if ( initz_flag == 0 ) init_Oglz();
  if ( !to_focus )
  { int n;
    to_focus = dmatrix(0,MAXCOORD,0,MAXCOORD);
    from_focus = dmatrix(0,MAXCOORD,0,MAXCOORD);
    for ( n = 0 ; n <= MAXCOORD ; n++ )
      to_focus[n][n] = from_focus[n][n] = 1.0;
  }

}    

  
/******************************************************************
*
* function: Oglz_edge()
*
* purpose:  graph one edge.
*/
void Oglz_edge ARGS((struct graphdata *,edge_id));

void Oglz_edge(g,e_id)
struct graphdata *g;
edge_id e_id;
{  
  int k;
  int e_color;

  e_color = g[0].ecolor;
  if ( e_color == CLEAR ) return;
  if ( (e_color < 0) || (e_color >= IRIS_COLOR_MAX) )
    e_color = DEFAULT_EDGE_COLOR;

  if ( pick_flag )
    my_glLoadName(e_id); /* for picking */

  /* display */ 
  e_glColor(e_color);
  glBegin(GL_LINES);
  for ( k = 0 ; k < 2 ; k++ )
     e_glVertex3dv(g[k].x);
  glEnd();

  /* pickable vertices */
  if ( pick_flag )
  { for ( k = 0 ; k < 2 ; k++ )
    { my_glLoadName(g[k].v_id); /* for picking */
      glBegin(GL_POINTS);
        glVertex3dv(g[k].x);
      glEnd();
    }
  }

}

/******************************************************************
*
* function: Oglz_facet()
*
* purpose:  graph one facet.
*/


void Oglz_facet ARGS((struct graphdata *,facet_id));

void Oglz_facet(g,f_id)
struct graphdata *g;
facet_id f_id;
{  
  int i,k;
  REAL len;
  facetedge_id fe;
  struct graph_thread_data *td = GET_DATA;
  float norm[3];
  float backnorm[3];


  /* need normal for lighting */
  for ( i = 0 ; i < 3 ; i++ )
  { int ii = (i+1)%3;
    int iii = (i+2)%3;
    norm[i] = (float)((g[1].x[ii]-g[0].x[ii])*(g[2].x[iii]-g[0].x[iii])
             -  (g[1].x[iii]-g[0].x[iii])*(g[2].x[ii]-g[0].x[ii]));
  }
  len = sqrt(dotf(norm,norm,3));
  if ( len <= 0.0 ) return;
  for ( i = 0 ; i < 3 ; i++ ) norm[i]= (float)(norm[i]/len);

  if ( web.hide_flag && (g[0].color != UNSHOWN) 
           && td->facetshow_flag )
  { if ( pick_flag ) my_glLoadName(f_id); /* for picking */
    if ( g[0].color != CLEAR )
    { if ( color_flag ) f_glColor(g->color);
      else f_glColor(INDEX_TO_RGBA(WHITE));
      kb_glNormal3fv(norm);
      glBegin(GL_TRIANGLES);
      for ( k = 0 ; k < 3 ; k++ )
      { if ( normflag ) kb_glNormal3dv(g[k].norm);
         f_glVertex3dv(g[k].x);
      }
      glEnd();
    }
    if ( (g->color != g->backcolor) && (g->backcolor != CLEAR) )
    { REAL x[MAXCOORD];
      f_glColor(g->backcolor);
      for ( i = 0 ; i < 3 ; i++ ) backnorm[i] = -norm[i];
      kb_glNormal3fv(backnorm);
      glBegin(GL_TRIANGLES);
      for ( k = 2 ; k >= 0 ; k-- )
      { for ( i = 0 ; i < 3 ; i++ )
        x[i] = g[k].x[i] + thickness*backnorm[i];
        if ( normflag ) kb_glAntiNormal3dv(g[k].norm);
        f_glVertex3dv(x);
      }
      glEnd();
    }
  }

  fe = valid_id(f_id) ? get_facet_fe(f_id) : NULLID;          
  for ( k = 0 ; k < 3 ; k++, fe = valid_id(fe)?get_next_edge(fe):NULLID )
  { if ( g[k].ecolor == CLEAR ) continue;
    if ( !edgeshow_flag || (g[0].color == UNSHOWN) )
    { if ( (g[k].etype & EBITS) == INVISIBLE_EDGE ) continue;      
    }
    if ( pick_flag ) my_glLoadName(g[k].id); /* for picking */
    e_glColor(g[k].ecolor);
    kb_glNormal3fv(norm);
    glBegin(GL_LINES);
      if ( normflag ) kb_glNormal3dv(g[k].norm);
      e_glVertex3dv(g[k].x);
      if ( normflag ) kb_glNormal3dv(g[(k+1)%3].norm);
      e_glVertex3dv(g[(k+1)%3].x);
    glEnd();

#ifdef WIN32XX
/* kludge due to OpenGL not applying FRONT_AND_BACK to lines */
	if ( g[k].ecolor != BLACK )
	{ double x[3];
      for ( i = 0 ; i < 3 ; i++ ) backnorm[i] = -norm[i];
      kb_glNormal3fv(backnorm);
      glBegin(GL_LINES);
        for ( i = 0 ; i < 3 ; i++ )
          x[i] = g[k].x[i] + thickness*backnorm[i];
        e_glVertex3dv(x);
        for ( i = 0 ; i < 3 ; i++ )
          x[i] = g[(k+1)%3].x[i] + thickness*backnorm[i];
        e_glVertex3dv(x);
      glEnd();
	}
#endif
  }

  /* pickable vertices */
  if ( pick_flag )
  { for ( k = 0 ; k < 3 ; k++ )
    { my_glLoadName(g[k].v_id); /* for picking */
      glBegin(GL_POINTS);
        glVertex3dv(g[k].x);
      glEnd();
    }
  }

} /* end Oglz_facet() */


/**********************************************************************
*
* function: Oglz_end()
*
* purpose: to be called at end of presenting data.
*/

void Oglz_end ARGS((void))
{
  prev_timestamp = graph_timestamp;
}

void Ogl_close ARGS((void))
{ int i;
  struct graph_thread_data *td = GET_DATA;

  glutDestroyWindow(td->win_id);
  memset((char*)td,0,sizeof(struct graph_thread_data));

  for ( i = 0 ; i < MAXGRAPHWINDOWS ; i++ )
  { if ( gthread_data[i].in_use != 0 ) break;
  }
  if ( i == MAXGRAPHWINDOWS )
  { /* all graph windows closed */
    glDeleteLists(dindex,1);
    go_display_flag = 0;
    init_flag = 0;
    initz_flag = 0;
    arrays_timestamp = 0;
    graph_thread_running = 0;
  
#ifdef WIN32
  ExitThread(0);
#elif defined(PTHREADS)
  pthread_exit(NULL);
#endif

  }
}

/***********************************************************************
*
* function: Ogl_close_show()
*
* purpose: to be called by main thread when wanting to close graphics.
*          sets close_flag to be read by threads' idle_func().
*/
void Ogl_close_show()
{ close_flag = 1;
}

/**********************************************************************
*
* function: vercolcomp()
*
* purpose: comparison function for sorting vertex data in case of indexed arrays.
*
*/

int vercolcomp(a,b)
struct vercol *a,*b;
{ 
  int i;
  for ( i = 0 ; i < 10 ; i++ )
  { REAL diff = ((float*)a)[i] - ((float*)b)[i];
    if ( diff < -gleps ) return -1;
    if ( diff > gleps ) return 1;
  }
  return 0;
}

/***********************************************************************
*
* function: eecomp()
*
* purpose: comparison function for sorting indexed edges 
*/

int eecomp(a,b)
int *a,*b;
{ if ( *a < *b ) return -1;
  if ( *a > *b ) return 1;
  if ( a[1] > b[1] ) return 1;
  if ( a[1] < b[1] ) return -1;
  return 0;
}

/******************************************************************************
*
* function: declare_arrays()
*
* purpose: Declare element arrays to current OpenGL context.  Each thread
*          should check before drawing if arrays have changed since last 
*          declaration.
*/
void declare_arrays()
{

  GL_ERROR_CHECK

  if ( arraysflag )
  {
    /* declare arrays to OpenGL */
    if ( interleaved_flag )
    {
     glInterleavedArrays(GL_C4F_N3F_V3F,sizeof(struct vercol),(void*)fullarray);
    }
    else
    { 
      glColorPointer(4,GL_FLOAT,0,colorarray);     
      glNormalPointer(GL_FLOAT,sizeof(struct vercol),fullarray->n);
      glVertexPointer(3,GL_FLOAT,sizeof(struct vercol),fullarray->x);
    }
  }
  GL_ERROR_CHECK
}

/**************************************************************************
*
* function: draw_one_image()
*
* purpose: draw one image of surface. separated out so can be
*          called twice in stereo mode.
*/
void draw_one_image()
{ struct graph_thread_data *td = GET_DATA;

  if ( dlistflag )
    glCallList(dindex);    
  else if ( arraysflag )
  { int i,j,m;

  GL_ERROR_CHECK

    for ( m = 0 ; (m < transform_count) || (m < 1) ; m++ )
    { float tmat[4][4];  /* transform matrix in proper form */

      if ( transforms_flag && doing_lazy && view_transform_det 
                                  && (view_transform_det[m] == -1.0) )
      { /* have to flip normals */
        for ( i = 0 ; i < edgecount+facetcount ; i++ )
         for ( j = 0 ; j < 3 ; j++ )
          fullarray[i].n[j] *= -1.0;
      }

      if ( transforms_flag && doing_lazy && view_transforms )
      { int hi = (SDIM <= 3) ? SDIM : 3;
        for ( i = 0 ; i < hi; i++ )
        { for ( j = 0 ; j < hi; j++ )
            tmat[i][j] = (float)view_transforms[m][j][i];
          for ( ; j < 3 ; j++ ) tmat[i][j] = 0.0;
          tmat[i][3] = (float)view_transforms[m][SDIM][i];
        }
        for ( ; i < 3 ; i++ )
        { for ( j = 0 ; j < 4 ; j++ ) tmat[i][j] = 0.0;
          tmat[i][i] = 1.0;
        }
        for ( j = 0 ; j < hi ; j++ )
          tmat[3][j] = (float)view_transforms[m][j][SDIM];
        for ( ; j < 3 ; j++ )
          tmat[3][j] = 0.0;
        tmat[3][3] = (float)view_transforms[m][SDIM][SDIM];
      
        glMatrixMode(GL_MODELVIEW);
        glPushMatrix();
        glMultMatrixf((float*)tmat); 
      }

  GL_ERROR_CHECK

      if ( strips_flag )
      { int i;
        /* do facets first, then edges in front */
        for ( i = estripcount ; i < stripcount ; i++ )
          glDrawElements(striparray[i].mode,striparray[i].count,
                        GL_UNSIGNED_INT,stripdata+striparray[i].start);
        glMatrixMode(GL_PROJECTION);
        glTranslated(td->edge_bias*imagescale,0.0,0.0);
        for ( i = 0 ; i < estripcount ; i++ )
          glDrawElements(striparray[i].mode,striparray[i].count,
                        GL_UNSIGNED_INT,stripdata+striparray[i].start);
        glTranslated(-td->edge_bias*imagescale,0.0,0.0);
      }     
      else if ( indexing_flag )
      {
        glDrawElements(GL_TRIANGLES,facetcount,GL_UNSIGNED_INT,
           indexarray+facetstart);
        glMatrixMode(GL_PROJECTION);
        glTranslated(td->edge_bias*imagescale,0.0,0.0);
glDisable(GL_LIGHTING);
        glDrawElements(GL_LINES,edgecount,GL_UNSIGNED_INT,indexarray+edgestart);
glEnable(GL_LIGHTING);
        glTranslated(-td->edge_bias*imagescale,0.0,0.0);
      }
      else
      {
        glDrawArrays(GL_TRIANGLES,facetstart,facetcount);
        glMatrixMode(GL_PROJECTION);
        glTranslated(td->edge_bias*imagescale,0.0,0.0);
        glDrawArrays(GL_LINES,edgestart,edgecount);
        glTranslated(-td->edge_bias*imagescale,0.0,0.0);
      }
      if ( transforms_flag && doing_lazy && view_transforms )
      { glMatrixMode(GL_MODELVIEW); glPopMatrix(); }

      if ( transforms_flag &&  doing_lazy && view_transform_det 
                 && (view_transform_det[m] == -1.0) )
      { /* have to flip normals back */
        for ( i = 0 ; i < edgecount+facetcount ; i++ )
         for ( j = 0 ; j < 3 ; j++ )
          fullarray[i].n[j] *= -1.0;
      }

      if ( !doing_lazy  || !transforms_flag || !view_transforms ) break;
    } /* end transform loop */
  } /* end arrays_flag */
  else /* not using display list or arrays */
  {
    ENTER_GRAPH_MUTEX
    graphgen();
    LEAVE_GRAPH_MUTEX
  }
  GL_ERROR_CHECK
} /* end draw_one_image() */

/**********************************************************************
*
* function: draw_screen()
* 
* purpose: Handle redraw messages from operating system.
*/

void draw_screen()
{ struct graph_thread_data *td = GET_DATA;
 
  Matrix viewf;
  int i,j;
 
  if ( td->aspect_flag )
    reshape_func(td->xsize,td->ysize);

#ifdef quickresize
  if ( td->resize_flag && !td->idle_flag ) /* Mac OS X resize recombine */
  { glutIdleFunc(idle_func);
    return; 
  }
  td->resize_flag = 0;
#endif

  td->idle_flag = 0;

  if ( td->new_title_flag )
    glutSetWindowTitle(td->wintitle);
  td->new_title_flag = 0;

  /* Set up longjmp to return here in case of error  */
  if ( setjmp(graphjumpbuf) )
  { return; }

  GL_ERROR_CHECK

  /* build arrays if needed */
  if ( arraysflag && (
      ((graph_timestamp != arrays_timestamp) && go_display_flag )
         || newarraysflag || (dlistflag == RESETLIST))
     )
  { /* if long time since last build, block and wait */
    static REAL last_mutex_time;
    REAL now = get_internal_variable(V_CLOCK);
    int timeout = (now-last_mutex_time > .5) ? LONG_TIMEOUT : IMMEDIATE_TIMEOUT;

    if ( TRY_GRAPH_MUTEX(timeout) )
    { int oldflag;
  
      last_mutex_time = now;
      if ( fullarray )
      { free((char*)fullarray);     
        fullarray = NULL;
      }
      edgecount = 0; 
      edgemax = (web.representation==SIMPLEX) ? SDIM*web.skel[FACET].count + 100 :
           4*web.skel[EDGE].count+10;   /* 2 vertices per edge, each edge twice */
      edgearray = (struct vercol *)calloc(edgemax,sizeof(struct vercol));
      facetcount = 0; facetmax = 3*web.skel[FACET].count+10; /* 3 vertices per facet */
      facetarray = (struct vercol *)calloc(facetmax,sizeof(struct vercol));
      if ( !edgearray || !facetarray )
      { erroutstring("Cannot allocate memory for graphics.\n");
        goto bailout;
      }
  
      graph_start = Oglz_start;
      graph_facet = Oglz_facet;
      graph_edge  = Oglz_edge;
      graph_end = Oglz_end;
      init_graphics = Ogl_init;
      finish_graphics = Ogl_finish;
      close_graphics = Ogl_close_show;
  
      /*glDisableClientState(GL_COLOR_ARRAY); */
      /*glDisableClientState(GL_NORMAL_ARRAY); */
      /*glDisableClientState(GL_VERTEX_ARRAY); */
      
      oldflag = markedgedrawflag;
      markedgedrawflag = 1;
      if ( !transform_colors_flag && (SDIM <= 3)) 
         lazy_transforms_flag = 1;   /* for graphgen use */
      doing_lazy = lazy_transforms_flag;  /* for glutgraph use */
      arrays_timestamp = graph_timestamp = ++global_timestamp; /* prevent stale data */ 
      graphgen();   /* fill in arrays */
      END_TRY_GRAPH_MUTEX
  
      lazy_transforms_flag = 0;
      markedgedrawflag = oldflag;
      if ( q_flag )
      { sprintf(msg,"\n%d edges, %d facets\n",edgecount/2,facetcount/3);
        outstring(msg);
      }
  
      /* doing this here since facet_alpha_flag set in graphgen */
      if ( facet_alpha_flag ) glEnable(GL_BLEND);
      else glDisable(GL_BLEND);
      glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
  
      /* unify lists */
  
      fullarray = (struct vercol *)realloc((char*)edgearray,
                 (edgecount+facetcount)*sizeof(struct vercol));       
      declare_arrays();
  
      memcpy((char*)(fullarray+edgecount),(char*)facetarray,
           facetcount*sizeof(struct vercol));
      free((char*)facetarray);  facetarray = NULL; edgearray = NULL;
      edgestart = 0; facetstart = edgecount;      
     
      /* Workaround really bizarre line-drawing bug */
      if ( td->linewidth == 1.0 ) 
      { td->linewidth = (float)1.1; 
        glLineWidth(td->linewidth); 
      }
  
    GL_ERROR_CHECK
  
      if ( indexing_flag ) 
      { 
        make_indexlists();
        if ( q_flag )
        { sprintf(msg,"After indexing: %d unique vertices, %d unique edges\n",
              vertexcount,edgecount/2);
          outstring(msg);
        }
        if ( strips_flag ) make_strips();
      }
  
      if ( !interleaved_flag )
      { /* kludge for broken nVidia Detonater 2.08 driver */
        if ( colorarray ) free((char*)colorarray);
        colorarray = (float*)calloc(edgecount+facetcount,4*sizeof(float));
        for ( i = 0 ; i < edgecount+facetcount ; i++ )
          for ( j = 0 ; j < 4 ; j++ )
            colorarray[4*i+j] = fullarray[i].c[j];
        declare_arrays();
      }
  
    GL_ERROR_CHECK
  
  
      newarraysflag = 0;
      if ( dlistflag )
      {
        declare_arrays();
        glNewList(dindex,GL_COMPILE);
        if ( indexing_flag )
        {
          glDrawElements(GL_TRIANGLES,facetcount,GL_UNSIGNED_INT,indexarray+facetstart);
          glMatrixMode(GL_PROJECTION);
          glTranslated(td->edge_bias*imagescale,0.0,0.0);   /* edges in front */
          glDrawElements(GL_LINES,edgecount,GL_UNSIGNED_INT,indexarray+edgestart);
          glTranslated(-td->edge_bias*imagescale,0.0,0.0);
          glMatrixMode(GL_MODELVIEW);
        }
        else
        {
          glDrawArrays(GL_TRIANGLES,facetstart,facetcount); 
          glMatrixMode(GL_PROJECTION);
          glTranslated(td->edge_bias*imagescale,0.0,0.0);    /* edges in front */
          glDrawArrays(GL_LINES,edgestart,edgecount);
          glTranslated(-td->edge_bias*imagescale,0.0,0.0);
          glMatrixMode(GL_MODELVIEW);
       }
        glEndList();           
        dlistflag = NORMALLIST;
      }
      if ( q_flag ) outstring(current_prompt);  
     }
     else 
       glutPostRedisplay();
  }
  else
  if ( ((dlistflag != NORMALLIST) || 
        ((graph_timestamp != prev_timestamp) && go_display_flag ))
     )
  { 
    /* if long time since last build, block and wait */
    static REAL last_mutex_time;
    REAL now = get_internal_variable(V_CLOCK);
    int timeout = (now-last_mutex_time > .5) ? LONG_TIMEOUT : IMMEDIATE_TIMEOUT;
    if ( TRY_GRAPH_MUTEX(timeout) )
    { /* regenerate display list */
      graph_start = Oglz_start;
      graph_facet = Oglz_facet;
      graph_edge  = Oglz_edge;
      graph_end = Oglz_end;
    
      init_graphics = Ogl_init;
      finish_graphics = Ogl_finish;
      close_graphics = Ogl_close_show;
      last_mutex_time = now;
 
      if ( dlistflag != NOLIST )
      { glNewList(dindex,GL_COMPILE);
        facetcount = edgecount = 0;
        graphgen();
        glEndList();
        dlistflag = NORMALLIST;
      }
      END_TRY_GRAPH_MUTEX
    }
   else 
     glutPostRedisplay();
  }

  if ( SDIM != td->olddim )
  { reshape_func(td->xsize,td->ysize);  /* in case dimension changes */
    td->olddim = SDIM;
  }
  glEnable(GL_LIGHT0);
  /*glEnable(GL_LIGHT1); */

  GL_ERROR_CHECK

  /* clear screen and zbuffer */
  glClearColor(rgba[background_color][0], 
          rgba[background_color][1],
          rgba[background_color][2],    
          rgba[background_color][3]
          );
         
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

  if ( backcull_flag ) { glEnable(GL_CULL_FACE); }
  else { glDisable(GL_CULL_FACE); }

  glDepthFunc(GL_LEQUAL); /* so later things get drawn */

  /* make sure global view matrix is same as first window */
  if ( td->win_id == 1 )
  { 
    if ( view == NULL ) { return; } /* rapid permload?? */
    for ( i = 0 ; i < HOMDIM ; i++ )
      for ( j = 0 ; j < HOMDIM ; j++ )
        view[i][j] = td->view[i][j];
  }

  glMatrixMode(GL_MODELVIEW);
  for ( i = 0 ; i < 4 ; i++ )
    for ( j = 0 ; j < 4 ; j++ )
      viewf[i][j] = td->view[(j<3)?j:(SDIM)][(i<3)?i:(SDIM)];
  if ( SDIM == 2 ) 
  { for ( i = 0 ; i < 3 ; i++ ) viewf[i][2] = viewf[2][i] = 0.0;
    viewf[2][2] = 1.0;
  }
  /* transpose, picking first 3 coordinates */
  if ( (td->projmode == P_PERSP) || td->stereomode ) 
  {   if ( SDIM == 2 ) viewf[3][2] -= 16;
      else viewf[3][0] -= 16.0;
  } 
  GL_ERROR_CHECK


  glLoadMatrixd(viewf[0]); 
  
  /* for picking */
  if ( !arraysflag ) { glInitNames();  glPushName(0);}

  if ( arraysflag )
    if ( td->arrays_timestamp < arrays_timestamp )
    { declare_arrays();
      td->arrays_timestamp = arrays_timestamp;
    }

  /* Now the actual drawing */
  if ( td->stereomode )
  { int w = (SDIM==2) ? 0 : 1;
    /* Stereo mode always perspective */
    glMatrixMode(GL_MODELVIEW);
    viewf[3][w] -= 1.5; glLoadMatrixd(viewf[0]); draw_one_image();
    glMatrixMode(GL_MODELVIEW);
    viewf[3][w] += 3.0; glLoadMatrixd(viewf[0]); draw_one_image();
  }
  else draw_one_image();
 

  glFlush();
  glutSwapBuffers();  

bailout:
  glutIdleFunc(idle_func);

} /* end draw_screen() */

/************************************************************************
*
* function:  idle_func()
* 
* purpose: Mac OS X kludge to prevent excessive redisplays on 
*          mouse movement.
*/
void idle_func()
{ struct graph_thread_data *td = GET_DATA;

  if ( close_flag ) 
     Ogl_close();

  td->idle_flag = 1;
#ifdef quickresize
  if ( td->resize_flag ) 
     glutPostRedisplay();
  td->resize_flag = 0;
#endif
  if ( draw_pid != main_pid ) /* can use signals */
    glutIdleFunc(NULL); 
  /* else relying on idle_func to pick up redraw messages */
  else
  {
#if (defined(SUN) || defined(LINUX)) && !defined(__STRICT_ANSI__)
    /* let the thread could sleep .1 sec; need link with -lrt */
    struct timespec delay;
    delay.tv_sec = 0; delay.tv_nsec = 100000000;
    nanosleep(&delay,NULL); 
#endif

#ifdef WIN32
    /* sleep until some event */
    { HANDLE pHandles[MAXIMUM_WAIT_OBJECTS];
      MsgWaitForMultipleObjects(0,pHandles,FALSE,100,QS_ALLINPUT);
    }
#endif
  }
}

/***********************************************************************
*
* function: display()
*
* purpose: called by Evolver to display on screen.
*/
void display()
{ struct graph_thread_data *td;
  int i,j;

  close_flag = 0;
  Oglz_start();
  
  /* set first screen view to anything modified by main thread */
  for ( i = 0 ; i < HOMDIM ; i++ )
  { gthread_data[1].view[i] = gthread_data[1].viewspace[i];
    for ( j = 0 ; j < HOMDIM ; j++ )
      gthread_data[1].view[i][j] = view[i][j];
  }

  for ( i = 0, td = gthread_data ; i < MAXGRAPHWINDOWS ; i++, td++ )
    if ( td->win_id )
    { if ( window_aspect_ratio != td->window_aspect )
        td->aspect_flag = 1;
 
#ifdef MAC_OS_X
      /* glutPostRedisplay() generates ugly NSEvent leak messages */
      glutSetWindow(td->win_id);
      draw_screen(); 
         
      /* glutPostWindowRedisplay(td->win_id); */  /* generate redraw message */    
#else
/*   WARNING: glutSetWindow() in non-drawing thread causes problems. */
/*      glutSetWindow(td->win_id); */
 /*     glutPostRedisplay(); */  /* generate redraw message */ 
      glutPostWindowRedisplay(td->win_id);  /* generate redraw message */
#ifdef WIN32
      InvalidateRect(td->draw_hwnd,NULL,FALSE);  /* give draw thread a kick */
#else
     if ( draw_pid != main_pid )
     { 
       kill(draw_pid,SIGKICK);  /* unix version of kick */
     } 
#endif
#endif
    }

}



/****************************************************************************
* 
* function: make_strips()
*
* purpose: Converts indexed arrays of edges and triangles to strips
*          for more efficient plotting.
* Input: global array indexarray
*        global variables edgecount,facetcount
* Output:  striparray
*/

int ecomp(a,b)
char *a,*b;
{ int ai = indexarray[edgestart+*(int*)a]; 
  int bi = indexarray[edgestart+*(int*)b]; 
  if ( ai < bi ) return -1;
  if ( bi < ai ) return 1;
  return 0;
}

struct festruct { int v[3];  /* head and tail and opposite vertices */
                  int f;     /* facet on left */
                };

int fecomp ( a,b)
struct festruct *a,*b;
{ if ( a->v[0] < b->v[0] ) return -1;
  if ( a->v[0] > b->v[0] ) return  1;
  if ( a->v[1] < b->v[1] ) return -1;
  if ( a->v[1] > b->v[1] ) return  1;
  return 0;
}


void make_strips()
{ int *edgeinx; /* oriented edge numbers, sorted by first vertex */
  int *evlist;  /* vertex indexed list of offsets into edgeinx */
  int v;
  int i,k,kk;
  int *estripno; /* number of current strip */
  int *fstripno; /* number of current strip */
  int stripnum;
  int dataspot;
  struct festruct *felist;
  int striplength[3];  /* for testing 3 ways from each starting facet */
  int *trialstrip;  /* for unerasing markings */
  int bestlength;
  int *bestverts;
  int *bestfacets;
  int way;
  int bestway;

  /* strip arrays */
  if ( stripdata ) free((char*)stripdata);
  if ( striparray ) free((char*)striparray);
  stripdata = (int*)calloc(edgecount+facetcount+5,sizeof(int));
  striparray = (struct stripstruct *)calloc(edgecount/2+facetcount/3 + 10,
                                         sizeof(struct stripstruct));
  dataspot = 0; stripnum = 0;

  /* edges */
  /* make list indexed by vertex */
  /* in einx, entry is 2*edgeindex+orientationbit */
  edgeinx = (int *)temp_calloc(edgecount,sizeof(int));
  for ( i = 0 ; i < edgecount ; i++ )
  { edgeinx[i] = i;  } /* using bit for orientation */  
  qsort((void*)edgeinx,edgecount,sizeof(int),FCAST ecomp);
  /* find where individual vertex segments start */
  evlist = (int *)temp_calloc(edgecount+10,sizeof(int));
  for ( i = 0, v = 0 ; i < edgecount ; i++ )
  { while ( indexarray[edgestart+edgeinx[i]] > v ) evlist[++v] = i;
  }
  evlist[++v] = i;  /* last sentinel */

  /* now make strips.  start with some edge and just keep going. */
  estripno = (int*)temp_calloc(edgecount+5,sizeof(int));
  for ( i = 0 ; i < edgecount/2 ; i++ )
  { int nexte,headv;
    if ( estripno[i] ) continue;
    /* new strip */
    striparray[stripnum].mode = GL_LINE_STRIP;
    striparray[stripnum].start = dataspot;
    nexte = 2*i;
    for (;;)
    { 
      estripno[nexte>>1] = stripnum+1;
      headv = indexarray[edgestart+(nexte^1)];
      stripdata[dataspot++] = headv;
      for ( k = evlist[headv] ; k < evlist[headv+1] ; k++ )
      { if ( estripno[edgeinx[k]>>1] == 0 )
        { nexte = edgeinx[k]; 
          break;
        }
      }
      if ( k == evlist[headv+1] ) break; /* end of strip */
    }
    /* flip list around */
    for ( k = striparray[stripnum].start, kk = dataspot-1 ; k < kk ; k++,kk-- )
      { int temp = stripdata[k]; stripdata[k] = stripdata[kk]; 
        stripdata[kk] = temp;
      }
    /* now backwards */
    nexte = 2*i+1;
    for (;;)
    {
      estripno[nexte>>1] = stripnum+1; 
      headv = indexarray[edgestart+(nexte^1)];
      stripdata[dataspot++] = headv;
      for ( k = evlist[headv] ; k < evlist[headv+1] ; k++ )
      { if ( estripno[edgeinx[k]>>1] == 0 )
        { nexte = edgeinx[k]; 
          break;
        }
      }
      if ( k == evlist[headv+1] ) break; /* end of strip */
    }
    striparray[stripnum].count = dataspot - striparray[stripnum].start;

    stripnum++;
  }

  estripcount = stripnum;
  temp_free((char*)evlist);
  temp_free((char*)estripno); 

  /* facets */

  /* make list of edges with left-hand facets */
  felist = (struct festruct *)temp_calloc(facetcount,sizeof(struct festruct));
  for ( i = 0, k = 0 ; i < facetcount ; i += 3, k++ )
  { felist[i].v[0] = indexarray[facetstart+i];
    felist[i].v[1] = indexarray[facetstart+i+1];
    felist[i].v[2] = indexarray[facetstart+i+2];
    felist[i].f = k;
    felist[i+1].v[0] = indexarray[facetstart+i+1];
    felist[i+1].v[1] = indexarray[facetstart+i+2];
    felist[i+1].v[2] = indexarray[facetstart+i];
    felist[i+1].f = k;
    felist[i+2].v[0] = indexarray[facetstart+i+2];
    felist[i+2].v[1] = indexarray[facetstart+i];
    felist[i+2].v[2] = indexarray[facetstart+i+1];
    felist[i+2].f = k;
  }
  qsort((void*)felist,facetcount,sizeof(struct festruct),FCAST fecomp);

  fstripno = (int *)temp_calloc(facetcount+5,sizeof(int));
  trialstrip = (int *)temp_calloc(facetcount+5,sizeof(int));
  bestverts = (int *)temp_calloc(facetcount+5,sizeof(int));
  bestfacets = (int *)temp_calloc(facetcount+5,sizeof(int));

  /* now make strips.  start with some facet and just keep going. */
  for ( i = 0 ; i < facetcount/3 ; i++ )
  { int nextf,va,vb,vc,whichway;
    int firstcount,secondcount; /* for checking orientation at start */
    if ( fstripno[i] ) continue;
    /* new strip */
    striparray[stripnum].mode = GL_TRIANGLE_STRIP;
    striparray[stripnum].start = dataspot;
    bestlength = 0;
    for ( way = 0 ; way < 3 ; way++)
    { int m = 0;  /* trialstrip index */
      dataspot = striparray[stripnum].start;
      nextf = 3*i;
      stripdata[dataspot++] = va = indexarray[facetstart+nextf+((1+way)%3)];
      stripdata[dataspot++] = vb = indexarray[facetstart+nextf+way];
      whichway = 1;
      for (;;)
      { struct festruct *fe,key;
        /* find in felist */
        if ( whichway ) { key.v[0] = va; key.v[1] = vb; }
        else { key.v[1] = va; key.v[0] = vb; }
        fe = (struct festruct *)bsearch(&key,(void*)felist,facetcount,
            sizeof(struct festruct), FCAST fecomp);
        if ( fe==NULL ) break;  /* done; maybe hit edge of surface */
  
        /*see if facet done yet */
        if ( fstripno[fe->f] != 0 ) break;  /* done in this direction */

        /*add opposite vertex */
        vc = fe->v[2];
        stripdata[dataspot++] = vc;                                 
        fstripno[fe->f] = stripnum+1;
        trialstrip[m++] = fe->f;
  
        /* ready for next time around */
        va = vb;
        vb = vc;
        whichway = !whichway;
      }
      firstcount = dataspot - striparray[stripnum].start;
      /* flip list around */
      for ( k = striparray[stripnum].start, kk = dataspot-1 ; k < kk ; k++,kk-- )
      { int temp = stripdata[k]; stripdata[k] = stripdata[kk]; 
        stripdata[kk] = temp;
      }
      /* now backwards */
      va = indexarray[facetstart+nextf+way];
      vb = indexarray[facetstart+nextf+((way+1)%3)];
      whichway = 1;
      for (;;)
      { struct festruct *fe,key;
        /*find in felist */
        if ( whichway ) { key.v[0] = va; key.v[1] = vb; }
        else { key.v[1] = va; key.v[0] = vb; }
        fe = bsearch(&key,(void*)felist,facetcount,sizeof(struct festruct),
             FCAST fecomp);
        if ( fe==NULL ) break;  /* done; maybe hit edge of surface */

        /*see if facet done yet */
        if ( fstripno[fe->f] != 0 ) break;  /* done in this direction */

        /*add opposite vertex */
        vc = fe->v[2];
        stripdata[dataspot++] = vc;
        fstripno[fe->f] = stripnum+1;
        trialstrip[m++] = fe->f;
        /* ready for next time around */
        va = vb;
        vb = vc;
        whichway = !whichway;
      }
      striplength[way] = dataspot - striparray[stripnum].start;
      secondcount = striplength[way] - firstcount;

      /* check orientation at start */
      if ( firstcount & 1 )
      { if ( secondcount & 1 ) 
        { striplength[way]--;  /* omit last, if necessary */
          if ( i == trialstrip[m-1] ) i--;  /* so loop doesn't skip omitted facet */
        }
        /* flip order */
        for ( k = striparray[stripnum].start, 
            kk = striparray[stripnum].start+striplength[way]-1 ; k < kk ; k++,kk-- )
        { int temp = stripdata[k]; stripdata[k] = stripdata[kk]; 
          stripdata[kk] = temp;
        }
      }

      if ( striplength[way] > bestlength )
      { bestlength = striplength[way];
        bestway = way;
        memcpy(bestverts,stripdata+striparray[stripnum].start,bestlength*sizeof(int));
        memcpy(bestfacets,trialstrip,(bestlength-2)*sizeof(int));
      }
      for ( k = 0 ; k < m ; k++ )  /* unmark */
        fstripno[trialstrip[k]] = 0;
    }  /* end ways */

    memcpy(stripdata+striparray[stripnum].start,bestverts,bestlength*sizeof(int));
    for ( k = 0 ; k < bestlength-2 ; k++ )  /* remark */
    { fstripno[bestfacets[k]] = stripnum+1;
      if ( strip_color_flag && (bestfacets[k] < web.skel[FACET].maxcount) ) 
        set_facet_color(bestfacets[k],(stripnum % 14) + 1);
    }

    striparray[stripnum].count = bestlength;
    dataspot = striparray[stripnum].start + bestlength;

    stripnum++;
  }

  temp_free((char*)fstripno); 
  temp_free((char*)trialstrip); 
  temp_free((char*)bestverts); 
  temp_free((char*)bestfacets); 

  stripcount = stripnum;
  fstripcount = stripcount - estripcount;
  /* cut down arrays to needed size */
  stripdata = (int*)realloc((char*)stripdata,dataspot*sizeof(int));
  striparray = (struct stripstruct *)realloc((char*)striparray,
                     stripcount*sizeof(struct stripstruct));
  if ( q_flag )
  { sprintf(msg,"After stripping: %d edgestrips, %d facetstrips\n",
              estripcount,fstripcount);
    outstring(msg);
  }
}

/***************************************************************************
*
* function: hashfunc()
*
* purpose: Compute hash value for vertex.
*/
int hashsize; /* size of hashtable */
int hashfunc(a)
struct vercol *a;
{
  int h;
  int scale = 100000;
  double eps = 3.e-6;  /* to prevent coincidences */
  h = 15187*(int)(scale*a->x[0]+eps);
  h += 4021*(int)(scale*a->x[1]+eps);
  h += 2437*(int)(scale*a->x[2]+eps);
  h += 7043*(int)(scale*a->n[0]+eps);
  h += 5119*(int)(scale*a->n[1]+eps);
  h += 8597*(int)(scale*a->n[2]+eps);
  h += 1741*(int)(scale*a->c[0]+eps);
  h += 4937*(int)(scale*a->c[1]+eps);
  h += 1223*(int)(scale*a->c[2]+eps);
  h = h % hashsize;
  if ( h < 0 ) h += hashsize;
  return h;
}
/***************************************************************************
*
* function: make_indexlists()
*
* purpose: Uniquify vertex and edge lists, and create index array for
*          OpenGL.  Also needed for strips.
*/

void make_indexlists()
{ int i,j;
  int rawcount = edgecount+facetcount;  /* number of unsorted vertices */
  struct vercol **hashlist;
  float mat[4][4];

  /* get reasonable epsilon for identifying vertices */
  glGetFloatv(GL_MODELVIEW_MATRIX,mat[0]);
  gleps = 1e-5/sqrt(mat[0][0]*mat[0][0]+mat[0][1]*mat[0][1]
                       +mat[0][2]*mat[0][2]);
  
  /* Uniquify using hash table */
  /* qsort here is a time hog */
  if ( indexarray ) free((char*)indexarray);
  indexarray = (int*)calloc(rawcount,sizeof(int));
  if ( !fullarray ) return;
  hashsize = 2*rawcount + 10;
  hashlist = (struct vercol**)calloc(hashsize,sizeof(struct vercol *));
  hashlist[hashfunc(fullarray)] = fullarray;
  indexarray[0] = 0;
  for ( i = 1, j = 1 ; i < rawcount ; i++ )
  { int h = hashfunc(fullarray+i);
    while ( hashlist[h] && vercolcomp(hashlist[h],fullarray+i) ) 
    { h++; if ( h == hashsize ) h = 0; }
    if ( hashlist[h] == NULL ) /* new one */
    { 
      fullarray[j] = fullarray[i];
      hashlist[h] = fullarray+j;
      j++;
    }
    indexarray[i] = hashlist[h]-fullarray;
  } 
  free((char*)hashlist);
  vertexcount = j;


   /* Uniquify edges */
   for ( i = edgestart ; i < edgestart+edgecount ; i += 2 )
   { if ( indexarray[i] > indexarray[i+1] )
     { int temp = indexarray[i];
       indexarray[i] = indexarray[i+1];
       indexarray[i+1] = temp;
     }
   }
   /* qsort here relatively minor in time */
   qsort((void*)(indexarray+edgestart),edgecount/2,2*sizeof(int), FCAST eecomp);
   for ( i = 2, j = 0 ; i < edgecount ; i += 2 )
   { if ( eecomp(indexarray+edgestart+i,indexarray+edgestart+j) != 0 )
     { j += 2;
       if ( i > j )
       { indexarray[edgestart+j] = indexarray[edgestart+i];
         indexarray[edgestart+j+1] = indexarray[edgestart+i+1];
       }
     }
   }
   if ( edgecount ) edgecount = j+2;
} /* end make_indexlists() */
