/******************************************************************** 
   Copyright (C) 2000 Bassoukos Tassos <abas@aix.meng.auth.gr>
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later
   version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*********************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib.h>
#include <gtk/gtk.h>
#include <gnome.h>

#include "guiprefs.h"
#include "protocol.h"
#include "users.h"
#include "server.h"
#include "main.h"
#include "guiutils.h"
#include "chat.h"
#include "userwidget.h"
#include "privs.h"
#include "privchat.h"
#include "hldat.h"
#include "pixmap.h"

#define PRVCHAT "PrivChat"

typedef struct {
  Connection *c;
  int counter;
  GList *chats;
} PrivChatData;

typedef struct {
  Connection *c;
  PrivChatData *pcd;
  guint32 chatwindowid;
  ChatWindowWidgets cww;
  NotebookPage *nb;
  char *subject;
} PrivChat;

static PrivChat *privchat_find_or_create(Connection *c,guint32 chatwindow);

/* ============================ */

static int find_chat_func(PrivChat *pc,gpointer data){
  return pc->chatwindowid - GPOINTER_TO_INT(data);
}

static PrivChat *get_chat(Connection *c, gint32 cwid){
  PrivChatData *pcd=hooks_get_data(c,PRVCHAT);
  GList *l;
  if(pcd==NULL) return NULL;
  l=g_list_find_custom(pcd->chats,GINT_TO_POINTER(cwid),(GCompareFunc)find_chat_func);
  return l==NULL?NULL:l->data;
}

static void privchat_destroy(PrivChat *pc){
  pc->pcd->chats=g_list_remove(pc->pcd->chats,pc);
  userwidget_destroy(pc->cww.userwidget);
  gutils_nbpage_destroy(pc->nb);
  free(pc->subject);
  free(pc);
}

static void privchat_destroy_all(Connection *c,gpointer data){
  PrivChatData *pcd=hooks_get_data(c,PRVCHAT);
  if(pcd==NULL) return;
  hooks_set_data(c,PRVCHAT,NULL);
  while(pcd->chats!=NULL)
    privchat_destroy(pcd->chats->data);
  free(pcd);
}

static PrivChatData *privchatdata_get_or_create(Connection *c){
  PrivChatData *pcd=hooks_get_data(c,PRVCHAT);
  if(pcd==NULL){
    pcd=malloc(sizeof(PrivChatData));
    pcd->c=c;
    pcd->counter=0;
    pcd->chats=NULL;
    hooks_create(c,PRVCHAT,pcd,NULL,privchat_destroy_all);
  }
  return pcd;
}

void privchat_handle_other_leave(Connection *c,HLTransaction *t,gpointer data){
  HLObject *cwid=transaction_find_object(t,HLO_CHATWINDOW);
  HLObject *who=transaction_find_object(t,HLO_SOCKET);
  PrivChat *pc;
  
  if(cwid!=NULL && who!=NULL && (pc=get_chat(c,cwid->data.number))!=NULL)
    userwidget_remove(pc->cww.userwidget,who->data.number);
  transaction_destroy(t);
}
void privchat_handle_other_join(Connection *c,HLTransaction *t,gpointer data){
  HLObject *cwid=transaction_find_object(t,HLO_CHATWINDOW);
  HLObject *who=transaction_find_object(t,HLO_SOCKET);
  PrivChat *pc;
  
  if(cwid!=NULL && who!=NULL && (pc=get_chat(c,cwid->data.number))!=NULL)
    userwidget_add(pc->cww.userwidget,who->data.number);
  transaction_destroy(t);
}
void privchat_handle_subject_change(Connection *c,HLTransaction *t,gpointer data){
  HLObject *cwid=transaction_find_object(t,HLO_CHATWINDOW);
  HLObject *subject=transaction_find_object(t,HLO_SUBJECT);
  PrivChat *pc;

  if(cwid!=NULL && subject!=NULL && (pc=get_chat(c,cwid->data.number))!=NULL){
    gtk_entry_set_text(GTK_ENTRY(pc->cww.subject),subject->data.string);
    free(pc->subject);
    pc->subject=strdup(subject->data.string);
  }
  transaction_destroy(t);
}
void privchat_handle_chat(Connection *c,HLTransaction *t,gpointer data){
  HLObject *cwid=transaction_find_object(t,HLO_CHATWINDOW);
  PrivChat *pc=NULL;

  if(cwid!=NULL && (pc=get_chat(c,cwid->data.number))!=NULL){
    play_sound(HL_SOUND_DOORBELL);
    chat_pane_handle_chat(&pc->cww,t);
  } else {
    transaction_destroy(t);
  }
}

static gboolean privchat_handle_deferred_privchat(gpointer data){
  HLTransaction *t=data;
  void (*func)(Connection *c,HLTransaction *t,gpointer data)=(gpointer)t->task;

  /* too bored to fix this kludge */
  t->task=NULL;
  func(t->data,t,NULL);
  return FALSE;
}

void privchat_handle_privchat(Connection *c,HLTransaction *t,gpointer data){
  transaction_read_objects(c,t);
  t->data=c;
  t->task=data;
  gtk_idle_add(privchat_handle_deferred_privchat,t);
}

static void privchat_send(Connection *c,gpointer data){
  HLTransaction *t=(HLTransaction *)data;
  server_set_default_reply(c,t);
  transaction_send(t,c);
}
static void privchat_change_subject(GtkEntry *e,PrivChat *pc){
  char *m;
  HLTransaction *t=transaction_new(HLCT_CHANGESUBJECT,NULL,TRUE);

  m=gtk_editable_get_chars(GTK_EDITABLE(pc->cww.subject),0,-1);
  gtk_entry_set_text(GTK_ENTRY(pc->cww.subject),"");
  transaction_add_object(t,create_string(HLO_SUBJECT,m));
  transaction_add_object(t,create_number(HLO_CHATWINDOW,pc->chatwindowid));
  free(m);
  server_transaction_start(pc->c,privchat_send,t);
}
static void privchat_say(ChatWindowWidgets *cww, gpointer data,int as_emote){
  PrivChat *pc=data;
  chat_pane_send_chat(pc->c,cww,create_number(HLO_CHATWINDOW,pc->chatwindowid),as_emote);
}
static void privchat_leave(GtkWidget *w,PrivChat *pc){
  HLTransaction *t=transaction_new(HLCT_LEAVEPCHAT,NULL,FALSE);
  
  transaction_add_object(t,create_number(HLO_CHATWINDOW,pc->chatwindowid));
  server_transaction_start(pc->c,privchat_send,t);
  privchat_destroy(pc);
}

static GnomeUIInfo privchat_toolbar[] = {
  GNOMEUIINFO_ITEM_STOCK(N_("Leave"),N_("Leave this chat"),
			 privchat_leave,
			 HL_STOCK_PIXMAP_CLOSE_PAGE),
  GNOMEUIINFO_END
};

static PrivChat *privchat_find_or_create(Connection *c,guint32 chatwindow){
  PrivChatData *pcd=privchatdata_get_or_create(c);
  PrivChat *pc;
  char buf[128];

  if((pc=get_chat(c,chatwindow))!=NULL)
    return pc;
  
  pc=malloc(sizeof(PrivChat));
  pcd->counter++;
  pc->pcd=pcd;
  pc->c=c;
  pc->chatwindowid=chatwindow;
  sprintf(buf,_("Chat #%d"),pcd->counter);
  pc->nb=gutils_nbpage_new("PrivChat",privchat_toolbar,c,buf,"privchat",pc);
  pc->subject=strdup(buf);

  pc->cww.sendmsg=privchat_say;
  pc->cww.user_data=pc;
  chat_pane_create(pc->nb,&pc->cww,TRUE);

  gtk_signal_connect(GTK_OBJECT(pc->cww.subject),"activate",
		     GTK_SIGNAL_FUNC(privchat_change_subject),(gpointer)pc);
  pcd->chats=g_list_prepend(pcd->chats,pc);
  return pc;
}

/* ======================================= */

static gboolean privchat_do_accept_privchat(gpointer data){
  HLTransaction *t=data;
  Connection *c=t->data;
  HLObject *cwid=transaction_find_object(t,HLO_CHATWINDOW);
  HLObject *subject=transaction_find_object(t,HLO_SUBJECT);
  int count;
  HLObject **users=transaction_extract_objects(t,HLO_USERLISTENTRY,&count);
  PrivChat *pc;

  if(users!=NULL && cwid!=NULL){
    int i;
    pc=privchat_find_or_create(c,cwid->data.number);
    for(i=0;i<count;i++){
      userwidget_add(pc->cww.userwidget,users[i]->data.userlistentry->u.socket);
    }
    if(subject!=NULL){
      gtk_entry_set_text(GTK_ENTRY(pc->cww.subject),subject->data.string);
    }
  }
  if(users!=NULL)
    objects_destroy(users);
  transaction_destroy(t);
  return FALSE;
}

static void privchat_accept_privchat(Connection *c,HLTransaction *t,
				     HLTransaction *r,gpointer data){
  HLObject *cwid=transaction_find_object(t,HLO_CHATWINDOW);
  if(r->w.error!=0 || cwid==NULL){
    server_reply_default(c,t,r,strdup(_("Could not join chat channel:")));
    return;
  }
  transaction_read_objects(c,r);
  transaction_add_object(r,create_number(HLO_CHATWINDOW,cwid->data.number));
  r->data=c;
  transaction_destroy(t);
  gtk_idle_add(privchat_do_accept_privchat,r);
}

static void privchat_send_accept(Connection *c,gpointer data){
  server_transaction_reply_set(c,data,privchat_accept_privchat,NULL);
  transaction_send(data,c);
}

static void privchat_invite_accept(GtkButton *but,gpointer data){
  HLTransaction *t=data;
  Connection *c=t->data;
  HLObject *cwid=transaction_find_object(t,HLO_CHATWINDOW);

  if(cwid!=NULL){
    HLTransaction *r=transaction_new(HLCT_REQJOINPCHAT,0,TRUE);
    transaction_add_object(r,create_number(HLO_CHATWINDOW,cwid->data.number));
    server_transaction_start(c,privchat_send_accept,r);
  }
}

static void privchat_invite_reject(GtkButton *but,gpointer data){
  HLTransaction *t=data;
  Connection *c=t->data;
  HLObject *cwid=transaction_find_object(t,HLO_CHATWINDOW);

  if(cwid!=NULL){
    HLTransaction *r=transaction_new(HLCT_REJECTPCHAT,0,TRUE);
    transaction_add_object(r,create_number(HLO_CHATWINDOW,cwid->data.number));
    server_transaction_start(c,privchat_send,r);
  }
}

static gboolean privchat_dialog_invite(gpointer data){
  HLTransaction *t=data;
  char buf[256];
  GtkWidget *top;
  UserlistEntry *ue;
  HLObject *sock=transaction_find_object(t,HLO_SOCKET);
  Connection *c=t->data;
  char *uname;
  int socket;
  
  socket=sock->data.number;
  ue=get_user_by_socket(c,socket);
  uname=(ue!=NULL && ue->name!=NULL)?ue->name:NULL;
  if(uname!=NULL)
    sprintf(buf,_("Invitation from %s"),uname);
  else
    sprintf(buf,_("Invitation from socket %d ???"),socket);
  top=gnome_dialog_new(buf,_("Accept"),_("Decline"),NULL);
  gnome_dialog_button_connect(GNOME_DIALOG(top),0,
			      GTK_SIGNAL_FUNC(privchat_invite_accept),
			      (gpointer)t);
  gnome_dialog_button_connect(GNOME_DIALOG(top),1,
			      GTK_SIGNAL_FUNC(privchat_invite_reject),
			      (gpointer)t);
  if(uname!=NULL)
    sprintf(buf,"%s has invited tou a chat channel",uname);
  else
    sprintf(buf,"Socked %d has invited tou a chat channel",socket);
  gtk_box_pack_end(GTK_BOX(GNOME_DIALOG(top)->vbox),gtk_label_new(buf),FALSE,FALSE,0);
  gtk_signal_connect_object(GTK_OBJECT(top),"destroy",
			    GTK_SIGNAL_FUNC(transaction_destroy),
			    (GtkObject *)t);
  gnome_dialog_button_connect_object(GNOME_DIALOG(top),0,
				     GTK_SIGNAL_FUNC(gtk_widget_destroy),
				     GTK_OBJECT(top));
  gnome_dialog_button_connect_object(GNOME_DIALOG(top),1,
				     GTK_SIGNAL_FUNC(gtk_widget_destroy),
				     GTK_OBJECT(top));
  gtk_widget_show_all(top);
  return FALSE;
}

void privchat_handle_invite(Connection *c,HLTransaction *t,gpointer data){
  HLObject *cwid=transaction_find_object(t,HLO_CHATWINDOW);
  HLObject *who=transaction_find_object(t,HLO_SOCKET);
  
  t->data=c;
  if(cwid!=NULL && who!=NULL){
    if(privchat_auto_join){
      privchat_invite_accept(NULL,t);
    } else {
      gtk_idle_add(privchat_dialog_invite,t);
      return;
    }
  }
  transaction_destroy(t);
}

/* ======================================= */

static gboolean privchat_do_start_privchat(gpointer data){
  HLTransaction *r=data;
  HLObject *sock=transaction_find_object(r,HLO_SOCKET);
  HLObject *cw=transaction_find_object(r,HLO_CHATWINDOW);

  if(sock!=NULL && cw!=NULL){
    privchat_find_or_create(r->data,cw->data.number);
    privchat_handle_other_join(r->data,r,NULL);
  } else {
    transaction_destroy(r);
  }
  return FALSE;
}

static void privchat_handle_start_privchat(Connection *c,HLTransaction *t,
					   HLTransaction *r,gpointer data){
  if(r->w.error!=0){
    server_reply_default(c,t,r,strdup(_("Could not invite to chat channel:")));
    return;
  }
  transaction_destroy(t);
  transaction_read_objects(c,r);
  r->data=c;
  gtk_idle_add(privchat_do_start_privchat,r);
}

static void privchat_send_start_privchat(Connection *c,gpointer data){
  HLTransaction *t=(HLTransaction *)data;
  server_transaction_reply_set(c,data,privchat_handle_start_privchat,NULL);
  transaction_send(t,c);
}

void privchat_start_privchat_with_user(Connection *c,int socket){
  PrivChatData *pcd=privchatdata_get_or_create(c);
  HLTransaction *t=transaction_new(HLCT_CREATEPCHAT,NULL,TRUE);

  transaction_add_object(t,create_number(HLO_SOCKET,socket));
  server_transaction_start(pcd->c,privchat_send_start_privchat,t);
}

/* ======================================= */

static gboolean privchat_invite_ok(gpointer data){
  HLTransaction *r=data;
  UserWidget *uw=r->data;
  HLObject *socket=transaction_find_object(r,HLO_SOCKET);
  HLObject *cwid=transaction_find_object(r,HLO_CHATWINDOW);
  PrivChat *pc;

  pc=privchat_find_or_create(uw->c,cwid->data.number);
  userwidget_add(pc->cww.userwidget,socket->data.number);
  transaction_destroy(r);
  return FALSE;
}
static void privchat_sent_invite(Connection *c,HLTransaction *t,
				 HLTransaction *r,gpointer data){
  if(r->w.error!=0){
    server_reply_default(c,t,r,strdup(_("Could not invite to chat channel:")));
    return;
  }
  transaction_read_objects(c,r);
  r->data=t->data;
  transaction_destroy(t);
  gtk_idle_add(privchat_invite_ok,r);
}
static void privchat_send_invite(Connection *c,gpointer data){
  server_transaction_reply_set(c,data,privchat_sent_invite,NULL);
  transaction_send(data,c);
}
static void privchat_invite_new(GtkWidget *w,gpointer cwid){
  HLTransaction *r;
  UserWidget *uw=gtk_object_get_user_data(GTK_OBJECT(w));

  r=transaction_new(HLCT_CREATEPCHAT,uw,FALSE);
  transaction_add_object(r,create_number(HLO_SOCKET,userwidget_get_seluser(uw)));
  server_transaction_start(uw->c,privchat_send_invite,r);
}
static void privchat_invite_exists(GtkWidget *w,gpointer cwid){
  HLTransaction *r;
  UserWidget *uw=gtk_object_get_user_data(GTK_OBJECT(w));

  r=transaction_new(HLCT_ADDTOPCHAT,0,FALSE);
  transaction_add_object(r,create_number(HLO_CHATWINDOW,GPOINTER_TO_INT(cwid)));
  transaction_add_object(r,create_number(HLO_SOCKET,userwidget_get_seluser(uw)));
  server_transaction_start(uw->c,privchat_send,r);
}

void privchat_make_invite_menu(UserWidget *uw, GtkMenu *menu){
  GtkWidget *mi,*m;
  GList *gl;
  gboolean need_separator=TRUE;
  PrivChatData *pcd=privchatdata_get_or_create(uw->c);

  m=gtk_menu_new();
  mi=gtk_menu_item_new_with_label(_("New Chat"));
  gtk_object_set_user_data(GTK_OBJECT(mi),uw);
  gtk_signal_connect(GTK_OBJECT(mi),"activate",GTK_SIGNAL_FUNC(privchat_invite_new),NULL);
  gtk_menu_append(GTK_MENU(m),mi);
  for(gl=pcd->chats;gl!=NULL;gl=gl->next){
    if(need_separator==TRUE){
      need_separator=FALSE;
      gtk_menu_append(GTK_MENU(m),gtk_menu_item_new());
    }
    mi=gtk_menu_item_new_with_label(((PrivChat *)gl->data)->subject);
    gtk_object_set_user_data(GTK_OBJECT(mi),uw);
    gtk_signal_connect(GTK_OBJECT(mi),"activate",GTK_SIGNAL_FUNC(privchat_invite_exists),
		       GINT_TO_POINTER(((PrivChat *)gl->data)->chatwindowid));
    gtk_menu_append(GTK_MENU(m),mi);
  }
  gtk_widget_show_all(m);
  mi=gtk_object_get_data(GTK_OBJECT(menu),"privchat_id");
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi),m);
}

void privchat_attach_to_menu(Connection *c, GtkMenu *menu, GtkWidget *mi){
  gtk_object_set_data(GTK_OBJECT(menu),"privchat_id",mi);
}
