//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: MeshReader.cpp,v 1.10 2004/04/22 15:30:41 delpinux Exp $

#include <MeshReader.hpp>

#include <SurfaceMeshOfTriangles.hpp>
#include <SurfaceMeshOfQuadrangles.hpp>

#include <MeshOfHexahedra.hpp>
#include <MeshOfTetrahedra.hpp>

#include <set>

MeshReader::Keyword MeshReader::__nextKeyword()
{
  MeshReader::Keyword kw;

  std::string aKeyword;
  if (!(__is >> aKeyword)) {
    if (__is.eof()) {
      kw.second = EndOfFile;
      return kw;
    }
  }

  KeywordList::iterator i = __keywordList.find(aKeyword.c_str());

  if (i != __keywordList.end()) {
    kw.first = (*i).first;
    kw.second = (*i).second;
    return kw;
  }
  fferr(0) << "\nThe keyword " << aKeyword << " is unknown\n";
  std::exit(1);
  kw.first = aKeyword;
  kw.second = Unknown;
  return kw;
}


void MeshReader::__removeComments(std::istream& is)
{
  char c=' ';
  // remove comments!
  while(is.get(c)) {
    if (c=='#') {
      while (c!='\n') {
	if (! is.get(c)) {
	  break;
	}
      }
    }
    __is << c;
  }
}

void MeshReader::__checkVertex(const size_t& n)
{
  if ((n == 0) or (n > (*__vertices).numberOfVertices())) {
    throw MeshReader::VertexError();
  }
}

size_t MeshReader::__getVertexNumber()
{
  size_t n = this->__getInteger();
  this->__checkVertex(n);

  return n;
}

size_t MeshReader::__getInteger()
{
  size_t i;
  if(!(__is >> i)) {
    fferr(0) << "\nExpected integer not found\n";
    std::exit(1);
    return 0;
  }
  return i;
}

real_t MeshReader::__getReal()
{
  real_t r;
  if(!(__is >> r)) {
    fferr(0) << "\nExpected integer not found\n";
    std::exit(1);
    return 0;
  }
  return r;
}

void MeshReader::
__writeReferences(const std::set<size_t>& references,
		  std::string objectName)
{
  ffout(3) << "  " << objectName << " references: ";
  for (std::set<size_t>::const_iterator i = references.begin();
       i != references.end(); ++i) {
    if (i != references.begin()) ffout(3) << ',';
    ffout(3) << *i;
  }
  ffout(3) << '\n';
}

void MeshReader::__readVertices()
{
  std::set<size_t> references;
  VerticesSet& V = *__vertices;
  for (size_t i=0; i<V.numberOfVertices(); ++i) {
    const double x   = __getReal();
    const double y   = __getReal();
    const double z   = __getReal();
    const size_t ref = __getInteger();
    references.insert(ref);
    V[i] = Vertex(x,y,z, ref);
  }

  this->__writeReferences(references, "Vertices");
}

void MeshReader::__readElements()
{
  std::string elementType;
  Keyword kw = __nextKeyword();
  while (kw.second != EndOfFile) {
    switch (kw.second) {
    case Vertices: {
      fferr(0) << "error: while reading \"" << __fileName
	       << "\": vertices list declared twice!\n";
      std::exit(1);
      break;
    }
    case Hexahedra: {
      size_t numberOfHexahedra = __getInteger();

      __hexahedra = new Vector<Hexahedron>(numberOfHexahedra);

      ffout(3) << "- Number of hexahedra: " << numberOfHexahedra << '\n';
      this->__readHexahedra();

      break;
    }
    case Tetrahedra: {
      size_t numberOfTetrahedra = __getInteger();

      __tetrahedra = new Vector<Tetrahedron>(numberOfTetrahedra);

      ffout(3) << "- Number of tetrahedra: " << numberOfTetrahedra << '\n';
      this->__readTetrahedra();

      break;
    }
    case Triangles: {
      size_t numberOfTriangles = __getInteger();

      __triangles = new Vector<Triangle>(numberOfTriangles);

      ffout(3) << "- Number of triangles: " << numberOfTriangles << '\n';
      this->__readTriangles();

      break;
    }
    case Quadrilaterals: {
      size_t numberOfQuadrilaterals = __getInteger();

      __quadrilaterals = new Vector<Quadrangle>(numberOfQuadrilaterals);

      ffout(3) << "- Number of quadrilaterals: "
	       << numberOfQuadrilaterals << '\n';
      this->__readQuadrilaterals();

      break;
    }
    case Corners: {
      size_t numberOfCorners = __getInteger();
      for (size_t j=0; j<numberOfCorners; ++j) {
	__getInteger();
      }

      fferr(3) << "  warning: while reading \"" << __fileName
	       << "\": skipping Corners\n";
      break;
    }
    case Edges: {
      size_t numberOfEdges = __getInteger();
      for (size_t j=0; j<numberOfEdges; ++j) {
	__getInteger();
	__getInteger();
	__getInteger();
      }

      fferr(3) << "  warning: while reading \"" << __fileName
	       << "\": skipping Edges\n";
      break;
    }
    case Ridges: {
      size_t numberOfRidges = __getInteger();
      for (size_t j=0; j<numberOfRidges; ++j) {
	__getInteger();
      }

      fferr(3) << "  warning: while reading \"" << __fileName
	       << "\": skipping Ridges\n";
      break;
    }
    default: {
      fferr(0) << "error: while reading \"" << __fileName
	       << "\": unknown keyword: " << kw.first << "\n";
      std::cerr << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
      std::exit(1);
    }
    }
    kw = __nextKeyword();
  }
}


void MeshReader::__readHexahedra()
{
  VerticesSet& V = *__vertices;
  Vector<Hexahedron>& H = *__hexahedra;
  std::set<size_t> references;

  size_t i=0;
  try {
    for (; i<H.size(); ++i) {
      const size_t a   = __getInteger() - 1;
      const size_t b   = __getInteger() - 1;
      const size_t c   = __getInteger() - 1;
      const size_t d   = __getInteger() - 1;
      const size_t e   = __getInteger() - 1;
      const size_t f   = __getInteger() - 1;
      const size_t g   = __getInteger() - 1;
      const size_t h   = __getInteger() - 1;

      const size_t ref   = __getInteger();
      references.insert(ref);

      H[i] = Hexahedron(V[a],V[b],V[c],V[d],V[e],V[f],V[g],V[h], ref);
    }
  }
  catch (MeshReader::VertexError e) {
    fferr(0) << "error: reading file \"" << __fileName << "\"\n";
    fferr(0) << "       invalid vertex number reading hexahedron " << i << "\n";
    std::exit(1);
  }

  this->__writeReferences(references, "Hexahedra");  
}

void MeshReader::__readQuadrilaterals()
{
  VerticesSet& V = *__vertices;
  Vector<Quadrangle>& Q = *__quadrilaterals;

  std::set<size_t> references;
  size_t i;
  try {
    for (i=0; i < Q.size(); ++i) {
      const size_t a   = __getInteger() - 1;
      const size_t b   = __getInteger() - 1;
      const size_t c   = __getInteger() - 1;
      const size_t d   = __getInteger() - 1;

      const size_t ref   = __getInteger();
      references.insert(ref);

      Q[i] = Quadrangle(V[a], V[b], V[c], V[d], ref);
    }
  }
  catch (MeshReader::VertexError e) {
    fferr(0) << "error: reading file \"" << __fileName << "\"\n";
    fferr(0) << "       invalid vertex number reading quadrilateral " << i << "\n";
    std::exit(1);
  }

  this->__writeReferences(references, "Quadrilaterals");
}


void MeshReader::__readTetrahedra()
{
  VerticesSet& V = *__vertices;
  Vector<Tetrahedron>& T = *__tetrahedra;

  std::set<size_t> references;
  size_t i;
  try {
    for (i=0; i<T.size(); ++i) {
      const size_t a   = __getInteger() - 1;
      const size_t b   = __getInteger() - 1;
      const size_t c   = __getInteger() - 1;
      const size_t d   = __getInteger() - 1;

      const size_t ref   = __getInteger();
      references.insert(ref);

      T[i] = Tetrahedron(V[a],V[b],V[c],V[d], ref);
    }
  }
  catch (MeshReader::VertexError e) {
    fferr(0) << "error: reading file \"" << __fileName << "\"\n";
    fferr(0) << "       invalid vertex number reading tetrahedron " << i << "\n";
    std::exit(1);
  }

  this->__writeReferences(references, "Tetrahedra");
}

void MeshReader::__readTriangles()
{
  VerticesSet& V = *__vertices;
  Vector<Triangle>& T = *__triangles;

  std::set<size_t> references;
  size_t i;
  try {
    for (i=0; i<T.size(); ++i) {
      const size_t a = __getVertexNumber() - 1;
      const size_t b = __getVertexNumber() - 1;
      const size_t c = __getVertexNumber() - 1;

      const size_t ref   = __getInteger();
      references.insert(ref);
    
      T[i] = Triangle(V[a],V[b],V[c], ref);
    }
  }
  catch (MeshReader::VertexError e) {
    fferr(0) << "error: reading file \"" << __fileName << "\"\n";
    fferr(0) << "       invalid vertex number reading triangle " << i << "\n";
    std::exit(1);

  }

  this->__writeReferences(references, "Triangles");
}

MeshReader::MeshReader(std::istream& is,
		       std::string s)
  : MeshGenerator(),
    __fileName(s)
{

  __keywordList["MeshVersionFormatted"]=MeshVersionFormatted;
  __keywordList["Dimension"]=MeshDimension;
  __keywordList["Vertices"]=Vertices;
  __keywordList["Triangles"]=Triangles;
  __keywordList["Quadrilaterals"]=Quadrilaterals;
  __keywordList["Tetrahedra"]=Tetrahedra;
  __keywordList["Hexahedra"]=Hexahedra;
  __keywordList["Normals"]=Normals;
  __keywordList["Tangents"]=Tangents;
  __keywordList["Corners"]=Corners;
  __keywordList["Edges"]=Edges;
  __keywordList["Ridges"]=Ridges;
  __keywordList["End"]=EndOfFile;

  size_t integer;

  ffout(3) << "\n";
  ffout(3) << "Reading file " << s << '\n';
  this->__removeComments(is);
  ffout(3) << "- commentaries removed\n";
  MeshReader::Keyword kw;

  kw = this->__nextKeyword();
  if (kw.second != MeshVersionFormatted) {
    fferr(0) << "\nExpecting MeshVersionFormatted\n";
    std::exit(1);
  }
  integer = __getInteger();

  ffout(3) << "- MeshVersionFormatted: " << integer << '\n';

  kw = this->__nextKeyword();
  if (kw.second != MeshDimension) {
    fferr(0) << "Expecting Dimension!\n";
    std::exit(1);
  }
  integer = __getInteger();  

  ffout(3) << "- Dimension: " << integer << '\n';

  if (integer != 3) {
    fferr(0) << "Only dimension 3 meshes can be read by ff3d!\n";
    std::exit(1);
  }

  kw = this->__nextKeyword();
  if (kw.second != Vertices) {
    fferr(0) << "Expecting Vertices list!\n";
    std::exit(1);
  }
  integer = __getInteger();  

  ffout(3) << "- Number of vertices: " << integer << '\n';

  __vertices = new VerticesSet(integer);

  this->__readVertices();

  this->__readElements();

  if (__hexahedra != 0) {
    MeshOfHexahedra* m
      = new MeshOfHexahedra(__vertices,
			    __hexahedra,
			    new SurfaceMeshOfQuadrangles(__vertices,
							 __quadrilaterals));
    __mesh = m;
    ffout(3) << "- Number of Hexahedra: " << (*__hexahedra).size() << '\n';
    return;
  }


  if (__tetrahedra != 0) {
    MeshOfTetrahedra* m
      = new MeshOfTetrahedra(__vertices,
			     __tetrahedra,
			     new SurfaceMeshOfTriangles(__vertices,
							__triangles));
    __mesh = m;
    ffout(3) << "- Number of Tetrahedra: " << (*__tetrahedra).size() << '\n';
    return;
  }

  if (__triangles != 0) {
    SurfaceMeshOfTriangles* m
      = new SurfaceMeshOfTriangles(__vertices,
				   __triangles);
    __mesh = m;
    ffout(3) << "- Number of Triangles: " << (*__triangles).size() << '\n';
    return;
  }

  if (__quadrilaterals != 0) {
    SurfaceMeshOfQuadrangles* m
      = new SurfaceMeshOfQuadrangles(__vertices,
				     __quadrilaterals);
    __mesh = m;
    ffout(3) << "- Number of Quadrilaterals: " << (*__quadrilaterals).size() << '\n';
    return;
  }
}
