//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: SecondOrderOperator.hpp,v 1.3 2003/09/20 21:31:09 delpinux Exp $

#ifndef SECOND_ORDER_OPERATOR_HPP
#define SECOND_ORDER_OPERATOR_HPP

#include <TinyMatrix.hpp>
#include <PDEOperator.hpp>

/**
 * @file   SecondOrderOperator.hpp
 * @author Stphane Del Pino
 * @date   Mon Jun  9 23:27:51 2003
 * 
 * @brief  second order partial differencial operators
 * 
 * The class SecondOrderOperator describes second order partial
 * differencial operators: \f$ \nabla\cdot A \nabla \f$ where \f$ A
 * \f$ is a \f$ 3\times 3 \f$ matrix of functions.
 *
 * \par example: if \f$ A = I \f$, the operator is the Laplacian.
 */

class SecondOrderOperator
  : public PDEOperator
{
public:
  typedef TinyMatrix<3,3, ReferenceCounting<UserFunction> > Matrix;

private:
  /*!
    Coeficients of \f$ A \f$: \f$ a_{ij} \f$.
  */
  ReferenceCounting<SecondOrderOperator::Matrix> __A; 

public:
  const UserFunction& coefficient(const size_t i)
  {
    assert (i<9);
    
    return (*(*__A)(i/3,i%3));
  }

  const bool isSet(const int&i, const int&j) const
  {
    return ((*__A)(i,j) != 0);
  }

  std::string TypeName() const
  {
    return std::string("SecondOrderOperator");
  }

  //! Read-only access to operator coefs.
  const UserFunction& A(const int&i, const int&j) const
  {
    return *(*__A)(i,j);
  }

  //! Read-only access to operator coefs.
  const SecondOrderOperator::Matrix& A() const
  {
    return *__A;
  }

  //! Returns a pointer on a SecondOrderOperator that was "multiplied" by u
  ReferenceCounting<PDEOperator> operator * (ConstReferenceCounting<UserFunction> u) const
 {
    ReferenceCounting<SecondOrderOperator::Matrix> A2
      = new SecondOrderOperator::Matrix;

    for (size_t i=0; i<3; ++i)
      for (size_t j=0; j<3; ++j) {
	if ((*__A)(i,j) != 0)
	  (*A2)(i,j) = new MulUserFunction((*__A)(i,j), u);
      }

    return (new SecondOrderOperator(A2));
  }

  //! Returns a pointer on the opposed SecondOrderOperator operator.
  ReferenceCounting<PDEOperator> operator - () const
 {
    ReferenceCounting<SecondOrderOperator::Matrix> A2
      = new SecondOrderOperator::Matrix;

    for (size_t i=0; i<3; ++i)
      for (size_t j=0; j<3; ++j) {
	if ((*__A)(i,j) != 0)
	  (*A2)(i,j) = new UnaryMinusUserFunction((*__A)(i,j));
      }

    return (new SecondOrderOperator(A2));
  }

  //! Constructs the operator using the \a Mesh and a given function matrix \a initA.
  SecondOrderOperator(ReferenceCounting<SecondOrderOperator::Matrix> initA)
    : PDEOperator(PDEOperator::secondorderop,
		  9),
      __A(initA)
  {
    ;
  }

  //! Copy constructor.
  SecondOrderOperator(const SecondOrderOperator& D)
    : PDEOperator(D),
      __A(D.__A)
  {
    ;
  }

  //! Destrutor.
  ~SecondOrderOperator()
  {
    ;
  }
};

#endif // SECOND_ORDER_OPERATOR_HPP
