// -*- C++ -*-

/*

  Heap Layers: An Extensible Memory Allocation Infrastructure
  
  Copyright (C) 2000-2003 by Emery Berger
  http://www.cs.umass.edu/~emery
  emery@cs.umass.edu
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#ifndef _MANAGESUPERBLOCKS_H_
#define _MANAGESUPERBLOCKS_H_

#include "superblockheap.h"

namespace HL {

  template <class SuperHeap, int SuperblockSize>
  class ManageSuperblocksBase : public SuperHeap {
  public:
    virtual void * malloc (size_t) = 0;
    virtual void free (void *) = 0;
  };

  template <class SuperHeap, int SuperblockSize, int ObjectSize>
  class ManageSuperblocks : public ManageSuperblocksBase<SuperHeap, SuperblockSize> {

  public:

    ManageSuperblocks (void)
      : sbList (NULL)
    {}

    inline void * malloc (size_t sz) {
      void * ptr = NULL;
      while (ptr == NULL) {
	if (sbList == NULL) {
				//printf ("Obtaining a superblock of size %d.\n", ObjectSize);
	  sbHeapType * sb
	    = new (SuperHeap::malloc(SuperblockSize)) sbHeapType;
				// Put it on the freelist.
	  sb->setNext (sbList);
	  if (sbList != NULL)
	    sbList->setPrev (sb);
	  sb->setPrev (NULL);
	  sbList = sb;
	}
	ptr = sbList->malloc(sz);
	if (ptr == NULL) {
				// This superblock is now completely in use.
				// Remove it.
	  sbHeapType * next = (sbHeapType *) sbList->getNext();
	  if (next != NULL)
	    next->setPrev (NULL);
	  sbList->setNext (NULL);
	  sbList->setPrev (NULL);
	  sbList = next;
	}
	//printf ("[%d] sbList = %x, ptr = %x\n", i, sbList, ptr);
      }
      return ptr;
    }

    inline void free (void * ptr) {
      sbHeapType * sb = (sbHeapType *) sbHeapType::getHeap (ptr);
      sb->free (ptr);
      if ((sb->getNext() == NULL) && (sb->getPrev() == NULL) && (sb != sbList)) {
	// The heap is not on the freelist.
	// Add it.
	sb->setNext (sbList);
	sb->setPrev (NULL);
	if (sbList != NULL) {
	  assert (sbList->getPrev() == NULL);
	  sbList->setPrev (sb);
	}
	sbList = sb;
      }
      if (sb->isFree()) {
	// The superblock is completely free.
	// Remove it.
	sbHeapType * prev, * next;
	prev = (sbHeapType *) sb->getPrev();
	next = (sbHeapType *) sb->getNext();
	if (prev != NULL) {
	  prev->setNext (next);
	}
	if (next != NULL) {
	  next->setPrev (prev);
	}
	sb->setPrev (NULL);
	sb->setNext (NULL);
	if (sbList == sb) {
	  if (next != NULL) {
	    sbList = next;
	  } else {
	    sbList = prev;
	  }
	}
	// Now free it.
	assert (sb != sbList);
	//printf ("Freeing a superblock of size %d.\n", ObjectSize);
	SuperHeap::free (sb);
      }
    }

  private:

    typedef SuperblockHeap<SuperblockSize, ObjectSize> sbHeapType;
    sbHeapType * sbList;

  };

};

#endif
