/***************************************************************************
                          LatexCode.cpp  -  description
                             -------------------
    begin                : Mit Jul 24 2002
    copyright            : (C) 2002 by Andr Simon
    email                : andre.simon1@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "latexgenerator.h"

namespace highlight {

LatexGenerator::LatexGenerator(const string &colourTheme,
                               bool replQuotes)
    : CodeGenerator(colourTheme),
      replaceQuotes(replQuotes)
{
  styleTagOpen.push_back( "\\hlstd{");
  styleTagOpen.push_back( "\\hlstr{");
  styleTagOpen.push_back( "\\hlnum{");
  styleTagOpen.push_back( "\\hlslc{");
  styleTagOpen.push_back( "\\hlcom{");
  styleTagOpen.push_back( "\\hlesc{");
  styleTagOpen.push_back( "\\hldir{");
  styleTagOpen.push_back( "\\hldstr{");
  styleTagOpen.push_back( "\\hlline{");
  styleTagOpen.push_back( "\\hlsym{");

  for (int i=0;i<NUMBER_BUILTIN_STYLES; i++){
   styleTagClose.push_back( "}");
  }

  newLineTag = "\\\\\n";
  spacer = "\\ ";

  maskWs=true;
  maskWsBegin = "\\hlstd{";
  maskWsEnd = "}";

  excludeWs=true;
}

LatexGenerator::LatexGenerator()
{}
LatexGenerator::~LatexGenerator()
{}

string LatexGenerator::formatStyleAttributes(const string & elemName,
                                             const ElementStyle &elem)
{
  ostringstream s;
  s  << "\\newcommand{\\hl"
     << elemName
     << "}[1]{\\textcolor[rgb]{"
     << elem.getColour().getLatexRedValue() << ","
     << elem.getColour().getLatexGreenValue() << ","
     << elem.getColour().getLatexBlueValue()
     << "}{";

  if (elem.isBold())
    s << "\\bf{";
  if (elem.isItalic())
    s << "\\it{";

  s  <<"#1";

  if (elem.isBold())
    s << "}";
  if (elem.isItalic())
    s << "}";

  s  <<"}}\n";
  return s.str();
}

void LatexGenerator::printBody()
{
  *out << "\\noindent\n"
       << "\\ttfamily\n";

  processRootState();

  *out << "\\mbox{}\\\\"
       << "\n\\normalfont\n";
}

string LatexGenerator::getHeader(const string & title)
{
  ostringstream os;
  os << "\\documentclass{article}\n"
     << "\\usepackage{color}\n"
     << "\\usepackage{alltt}\n";

  os << formatStyleAttributes("std", docStyle.getDefaultStyle());
  os << formatStyleAttributes("num", docStyle.getNumberStyle());
  os << formatStyleAttributes("esc", docStyle.getEscapeCharStyle());
  os << formatStyleAttributes("str", docStyle.getStringStyle());
  os << formatStyleAttributes("dstr", docStyle.getDirectiveStringStyle());
  os << formatStyleAttributes("slc", docStyle.getSingleLineCommentStyle());
  os << formatStyleAttributes("com", docStyle.getCommentStyle());
  os << formatStyleAttributes("dir", docStyle.getDirectiveStyle());
  os << formatStyleAttributes("sym", docStyle.getSymbolStyle());
  os << formatStyleAttributes("line", docStyle.getLineStyle());

  KeywordStyles styles = docStyle.getKeywordStyles();
  for (KSIterator it=styles.begin(); it!=styles.end(); it++){
    os << formatStyleAttributes(it->first, *(it->second));
  }

  os << "\n\\title{" << title << "}\n"
     << "\\oddsidemargin -3mm\n\\textwidth 165,2truemm\n"
     << "\\topmargin 0truept\n\\headheight 0truept\n"
     << "\\headsep 0truept\n\\textheight 230truemm\n\n"
     << "\\begin{document}\n"
     << "\\definecolor{bgcolor}{rgb}{";
  os << docStyle.getBgColour().getLatexRedValue() << ","
     << docStyle.getBgColour().getLatexGreenValue() << ","
     << docStyle.getBgColour().getLatexBlueValue()
     << "}\n\\pagecolor{bgcolor}\n";
  return os.str();
}

string LatexGenerator::getFooter()
{
  ostringstream os;
  os << "\\end {document}\n"
     << "(* LaTeX generated by highlight "
     << HIGHLIGHT_VERSION
     << ", "
     << HIGHLIGHT_URL
     << " *)\n";
  return os.str();
}

string LatexGenerator::getNewLine(){
  return (line.length() || showLineNumbers)?newLineTag:"\\mbox{}"+newLineTag;
}

string LatexGenerator::maskCharacter(unsigned char c)
{
  switch (c)
    {
    case '<' :
      return "$<$";
      break;
    case '>' :
      return "$>$";
      break;
    case '{':
    case '}':
    case '&':
    case '$':
    case '#':
    case '%':
      {
       string m;
       m  ="\\";
       m += c;
       return m;
      }
      break;
    case '\"':
      return (fragmentOutput && replaceQuotes)?"\\dq{}":"\"";
      break;
    case '_':
      return "\\textunderscore ";
      break;
    case '^':
      return "\\textasciicircum ";
      break;
    case '\\':
      return "$\\backslash$";
      break;
    case '~':
      return "$\\sim$";
      break;
    case '|':
      return "\\textbar ";
      break;
    // avoid latex compilation failure if [ or * follows a line break (\\)
    case '*':
    case '[':
    case ']':
    // avoid "merging" of consecutive '-' chars when included in bold font ( \bf )
    case '-':
      {
       string m;
       m= "{";
       m+= c;
       m+= "}";
       return m;
      }
      break;
    /*case ' ':
      return spacer;
      break;*/
    case AUML_LC:
      return "\\\"a";
      break;
    case OUML_LC:
      return "\\\"o";
      break;
    case UUML_LC:
      return "\\\"u";
      break;
    case AUML_UC:
      return "\\\"A";
      break;
    case OUML_UC:
      return "\\\"O";
      break;
    case UUML_UC:
      return "\\\"U";
      break;
    case AACUTE_LC:
      return "\\'a";
      break;
    case EACUTE_LC:
      return "\\'e";
      break;
    case OACUTE_LC:
      return "\\'o";
      break;
    case UACUTE_LC:
      return "\\'u";
      break;
    case AGRAVE_LC:
      return "\\`a";
      break;
    case EGRAVE_LC:
      return "\\`e";
      break;
    case OGRAVE_LC:
      return "\\`o";
      break;
    case UGRAVE_LC:
      return "\\`u";
      break;
    case AACUTE_UC:
      return "\\'A";
      break;
    case EACUTE_UC:
      return "\\'E";
      break;
    case OACUTE_UC:
      return "\\'O";
      break;
    case UACUTE_UC:
      return "\\'U";
      break;
    case AGRAVE_UC:
      return "\\`A";
      break;
    case EGRAVE_UC:
      return "\\`E";
      break;
    case UGRAVE_UC:
      return "\\`O";
      break;
    case OGRAVE_UC:
      return "\\`U";
      break;
    case SZLIG:
      return "\\ss ";
      break;
     #ifndef _WIN32
 // skip  first byte of multibyte chracters
    case 195:
      return string("");
      break;
     #endif

    default :
      {
      string m;
      return m+=c;
     }
    }
}

string LatexGenerator::getMatchingOpenTag(unsigned int styleID){
  return "\\hl"+langInfo.getKeywordClasses()[styleID]+"{";
 }

string LatexGenerator::getMatchingCloseTag(unsigned int styleID){
  return "}";
}

}
