/* GIMLET: GNOME Input Method Language Enabing Tool
 *
 * Copyright 2003 Sun Microsystems Inc.
 *
 * This is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <gtk/gtkmain.h>
#include <gtk/gtkdrawingarea.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtksignal.h>
#include <libart_lgpl/libart.h>
#include <panel-applet.h>
#include <gnome.h>

#include <signal.h>
#include <wait.h>

#include "gnome-im-switcher.h"
#include "widgets.h"
#include "iiim-interface.h"
#include "quick-access-menu.h"
#include "preference.h"

static void update_and_scale_icon (GimletWindow *gimlet);
static void gimlet_register_signal_child_handler();
static void gimlet_signal_child_handler(int sig, siginfo_t * info, void * ucontext);

void gimlet_register_signal_child_handler()
{
        struct sigaction act;
                                                                                                                   
        act.sa_handler = NULL;
        act.sa_sigaction = gimlet_signal_child_handler;
        sigfillset(&act.sa_mask);
        act.sa_flags = SA_SIGINFO;
                                                                                                                   
        sigaction(SIGCHLD, &act, NULL);
}
                                                                                                                   
void gimlet_signal_child_handler(int sig, siginfo_t * info, void * ucontext)
{
        pid_t   pid;
        int     status;
                                                                                                                   
        while ((pid = waitpid(info->si_pid, &status, WNOHANG|WUNTRACED)) > 0) {
                printf("pid %d: die\n", pid);
        }
}

static void
help_verb_cb (BonoboUIComponent *uic,
	      GimletWindow *gimlet,
	      const gchar *verbname)
{
  int pid;
  char *command_str;
/*
  GtkWidget *dialog = gtk_message_dialog_new (NULL,
				   0, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
				   "help is not implemented yet",
				   NULL);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
*/
  gimlet_register_signal_child_handler();

  pid = fork();
  if (pid < 0) {
    return;
  } else if (pid == 0) {
    /* in child process */
    command_str = "mozilla /usr/share/doc/IIIM/index.html";
    execl("/bin/sh", "sh", "-c", command_str, (char *)0);
                                                                                                                  
    /* Exec failed. */
    fprintf (stderr, "Error: Could not exec %s!\n", command_str);
    _exit(1);
  }
  return;
}

static void
edit_preferences_destroyed_callback (GtkWidget   *new_profile_dialog,
				     GimletWindow *gimlet)
{
  gimlet->edit_preferences_dialog = NULL;
}

static void
show_preferences_dialog (BonoboUIComponent *uic,
			 GimletWindow *gimlet,
			 const gchar       *verbname)
{
  GtkWindow *transient_parent;
  GtkWindow *old_transient_parent;

  transient_parent = GTK_WINDOW (gimlet->applet);
  if (gimlet->edit_preferences_dialog == NULL)
    {
      old_transient_parent = NULL;
      /* passing in transient_parent here purely for the
       * glade error dialog
       */
      gimlet->edit_preferences_dialog =
	gimlet_preference_dialog_new (transient_parent, gimlet);
      if (gimlet->edit_preferences_dialog == NULL)
        return; /* glade file missing */
      
      g_signal_connect (G_OBJECT (gimlet->edit_preferences_dialog),
                        "destroy",
                        G_CALLBACK (edit_preferences_destroyed_callback),
                        gimlet);
    }
  else
    {
      old_transient_parent = gtk_window_get_transient_for (GTK_WINDOW (gimlet->edit_preferences_dialog));
    }
  if (old_transient_parent != transient_parent)
    {
      gtk_window_set_transient_for (GTK_WINDOW (gimlet->edit_preferences_dialog),
                                    transient_parent);
      gtk_widget_hide (gimlet->edit_preferences_dialog); /* re-show the window on its new parent */
    }
  
  gtk_widget_show_all (gimlet->edit_preferences_dialog);
  gtk_window_present (GTK_WINDOW (gimlet->edit_preferences_dialog));
  return;
}

static void
about_verb_cb (BonoboUIComponent *uic,
	       GimletWindow *gimlet,
	       const gchar *verbname)
{
  gchar **authors;
  const gchar *documenters[] = {
    NULL
  };
  const char *translator_credits = _("translator_credits");
  char *file;

  authors = g_new (gchar*, 2);
  authors[0] = g_strdup(_("Hidetoshi Tajima <hidetoshi.tajima@sun.com>"));
  authors[1] = NULL;

  if (gimlet->about_dialog != NULL) {
    gtk_window_present (GTK_WINDOW (gimlet->about_dialog));
    return;
  }

  file = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_PIXMAP,
				    _("gnome-settings-im.png"),
				    TRUE, NULL);
  if (file)
    {
      GdkPixbuf *pixbuf = NULL;
      pixbuf = gdk_pixbuf_new_from_file (file, NULL);
      gtk_window_set_icon (GTK_WINDOW (gimlet->about_dialog), pixbuf);
      g_object_unref (pixbuf);
      g_free (file);
    }

  gimlet->about_dialog = gnome_about_new (_("Gnome Input Method Switcher"),
					  VERSION,
					  "Copyright \xc2\xa9 2002-2004 Sun Microsystems, Inc.",
					  _("Switch Input Languages on the panel."),
					  (const gchar**)authors,
					  documenters,
					  strcmp (translator_credits, "translator_credits") != 0 ? translator_credits : NULL,
					  NULL);
  gtk_signal_connect (GTK_OBJECT(gimlet->about_dialog),
		      "destroy",
                      GTK_SIGNAL_FUNC(gtk_widget_destroyed),
		      &gimlet->about_dialog);

  g_strfreev (authors);
  gtk_widget_show (gimlet->about_dialog);
}

static const BonoboUIVerb menu_verbs [] = {
  BONOBO_UI_UNSAFE_VERB ("Props",	show_preferences_dialog),
  BONOBO_UI_UNSAFE_VERB ("Help",        help_verb_cb),
  BONOBO_UI_UNSAFE_VERB ("About",	about_verb_cb),
  BONOBO_UI_VERB_END
};

static gboolean 
key_press_cb (GtkWidget *widget, GdkEventKey *event,
	      GimletWindow *gimlet)
{
  switch (event->keyval)
    {
    case GDK_KP_Enter:
    case GDK_ISO_Enter:
    case GDK_3270_Enter:
    case GDK_Return:
    case GDK_space:
    case GDK_KP_Space:
      gimlet_quick_access_menu_show (gimlet);
      return TRUE;
      break;
    default:
      break;
    }
  return FALSE;
}

static gboolean
do_not_eat_button_press (GtkWidget      *widget,
                         GdkEventButton *event)
{
  if (event->button != 1)
    g_signal_stop_emission_by_name (widget, "button_press_event");

  return FALSE;
}

static void
show_dialog (GtkWidget *button,
	     GimletWindow *gimlet)
{
  gimlet_quick_access_menu_show (gimlet);
  return;
}

static void
unfix_size (GimletWindow *gimlet)
{
    gimlet->fixed_width = -1;
    gimlet->fixed_height = -1;
#if 0
    gtk_widget_queue_resize (gimlet->push_button);
#endif
}

/* Don't request smaller size then the last one we did, this avoids
   jumping when proportional fonts are used.  We must take care to 
   call "unfix_size" whenever options are changed or such where
   we'd want to forget the fixed size */
static void
size_request (GtkWidget *label, GtkRequisition *req, gpointer data)
{
    GimletWindow *gimlet = data;

    if (req->width > gimlet->fixed_width)
	gimlet->fixed_width = req->width;
    if (req->height > gimlet->fixed_height)
	gimlet->fixed_height = req->height;
    req->width = gimlet->fixed_width;
    req->height = gimlet->fixed_height;
}

static gboolean 
handle_button_release (GimletWindow *gimlet,
		       GdkEventButton *event)
{
  if (event->button != 1)
    return FALSE; 

  gimlet_quick_access_menu_show (gimlet);
  return TRUE;
}

static void
icon_size_allocate (GtkWidget      *widget,
		    GtkAllocation *allocation,
		    GimletWindow     *gimlet)
{
  if (widget->allocation.width  != gimlet->prev_allocation.width ||
      widget->allocation.height != gimlet->prev_allocation.height)
    update_and_scale_icon (gimlet);

  gimlet->prev_allocation = *allocation;
  return;
}

static GdkPixbuf *init_base_pixbuf (const char *name);
static gboolean expose_cb (GtkWidget *widget, GdkEventExpose *event,
			   GimletWindow *gimlet);

static void
place_widgets (GimletWindow *gimlet)
{
  GtkWidget *hbox;
  GtkWidget *label;
  GtkWidget *alignment;

  gimlet->prev_allocation.x      = -1;
  gimlet->prev_allocation.y      = -1;
  gimlet->prev_allocation.width  = -1;
  gimlet->prev_allocation.height = -1;

  hbox = gtk_hbox_new (FALSE, 3);
  gtk_container_add (GTK_CONTAINER (gimlet->applet), hbox);
  gtk_widget_show (hbox);

  gimlet->applet_icon_area = gtk_image_new ();
  g_signal_connect (G_OBJECT(gimlet->applet_icon_area), "expose-event",
		    G_CALLBACK(expose_cb), gimlet);
  gimlet->c_pixbuf = g_new0 (ConstrainedPixbuf, 1);
  gimlet->pixbuf = init_base_pixbuf (gimlet->base_iconname);

  gimlet_change_icon (gimlet, " ");

  gtk_box_pack_start_defaults (GTK_BOX (hbox), gimlet->applet_icon_area);

  g_signal_connect (gimlet->applet_icon_area, "size-allocate",
		    G_CALLBACK (icon_size_allocate), gimlet);

  gimlet_draw_icon (gimlet);

  /* translators: 
   * GIMLET is: "G"NOME "I"nput "M"ethod "L"anguage "E"nabling "T"ool
   */
  label = gtk_label_new (_("GIMLET"));

  g_signal_connect (label, "size_request",
		    G_CALLBACK (size_request),
		    gimlet);
  g_signal_connect_swapped (label, "style_set",
			    G_CALLBACK (unfix_size),
			    gimlet);
  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_CENTER);
  gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);

  gimlet->status_label = label;

  gtk_widget_show (label);

#if 0
  gimlet->push_button = gtk_button_new ();
  gtk_container_set_resize_mode (GTK_CONTAINER (gimlet->push_button),
				 GTK_RESIZE_IMMEDIATE);
  gtk_button_set_relief (GTK_BUTTON (gimlet->push_button), GTK_RELIEF_NONE);
  alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
  gtk_container_add (GTK_CONTAINER (alignment), label);
  gtk_container_set_resize_mode (GTK_CONTAINER (alignment), GTK_RESIZE_IMMEDIATE);
  gtk_container_add (GTK_CONTAINER (gimlet->push_button), alignment);
  gtk_widget_show (alignment);

  gtk_box_pack_start_defaults (GTK_BOX (hbox), gimlet->push_button);

  if (gimlet->showtext)
    gtk_widget_show (gimlet->push_button);

  g_signal_connect (gimlet->push_button, "button_press_event",
		    G_CALLBACK (do_not_eat_button_press), NULL);
  g_signal_connect (gimlet->push_button, "clicked",
		    G_CALLBACK (show_dialog), gimlet);
#endif

  g_signal_connect_swapped (gimlet->applet, "button_press_event",
			    G_CALLBACK (do_not_eat_button_press), NULL);
  g_signal_connect_swapped (gimlet->applet, "button_release_event",
			    G_CALLBACK (handle_button_release), gimlet);

  gtk_container_set_border_width (GTK_CONTAINER (gimlet->applet), 0);
#if 0
  gtk_container_set_border_width (GTK_CONTAINER (gimlet->push_button), 0);
#endif

  gimlet_change_status_font (gimlet);
  gimlet_change_status_color (gimlet);
}

static void
draw_scaled_icon (GimletWindow *gimlet)
{
  GdkRectangle image_bound;
  int w, h, x, y;
  ConstrainedPixbuf *c_pixbuf = gimlet->c_pixbuf;
  GtkWidget *widget = gimlet->applet_icon_area;

  x = y = 0;
  w = gdk_pixbuf_get_width (c_pixbuf->scaled);
  h = gdk_pixbuf_get_height (c_pixbuf->scaled);
  
  image_bound.x = x;
  image_bound.y = y;      
  image_bound.width = w;
  image_bound.height = h;

  if (c_pixbuf->scaled && gimlet->showicon)
    gdk_draw_pixbuf (widget->window, NULL, c_pixbuf->scaled,
		     image_bound.x, image_bound.y,
		     image_bound.x, image_bound.y,
		     image_bound.width, image_bound.height,
		     GDK_RGB_DITHER_NORMAL,
		     0, 0);
  if (c_pixbuf->layout && gimlet->showtext)
    {
      GdkColor *foreground = NULL;
      GdkColor *background = NULL;

      if (!gimlet->use_theme_colors)
	{
	  foreground = &gimlet->foreground;
	  background = (gimlet->showicon ? NULL : &gimlet->background);
	}
      gdk_draw_layout (widget->window, widget->style->base_gc [widget->state],
		       c_pixbuf->border_left_scaled + c_pixbuf->center_x + x + 1,
		       c_pixbuf->border_top + c_pixbuf->center_y + y + 1, 
		       c_pixbuf->layout);
      gdk_draw_layout_with_colors (widget->window, widget->style->text_gc [widget->state],
				   c_pixbuf->border_left_scaled + c_pixbuf->center_x + x,
				   c_pixbuf->border_top + c_pixbuf->center_y + y, 
				   c_pixbuf->layout,
				   foreground,
				   background);
    }
}

static void
update_and_scale_icon (GimletWindow *gimlet)
{
  GtkWidget *widget = gimlet->applet_icon_area;
  GdkGC *gc;
  int        width  = -1;
  int        height = -1;
  int        pixbuf_width = -1;
  int        pixbuf_height = -1;
  PanelAppletOrient orient;
  GdkPixbuf *scaled;

  gtk_widget_show (widget);

  if (!GTK_WIDGET_REALIZED (widget) ||
      widget->allocation.width <= 0 ||
      widget->allocation.height <= 0)
    return;

  pixbuf_width  = gdk_pixbuf_get_width  (gimlet->pixbuf);
  pixbuf_height = gdk_pixbuf_get_height (gimlet->pixbuf);

  g_assert (pixbuf_width != -1 && pixbuf_height != -1);

  orient = panel_applet_get_orient (PANEL_APPLET (gimlet->applet));
  
  if (orient == PANEL_APPLET_ORIENT_UP || orient == PANEL_APPLET_ORIENT_DOWN)
    {
      height = widget->allocation.height;
      width  = pixbuf_width * ((gdouble) height / pixbuf_height);
      widget->requisition.width = width;
    }
  else
    {
      width = widget->allocation.width;
      height = pixbuf_height * ((gdouble) width / pixbuf_width);
      widget->requisition.height = height;
    }

  g_assert (width != -1 && height != -1);

  if (width == 0 || height == 0)
    return;

  gtk_widget_queue_resize (widget);

  if (constrained_pixbuf_init (gimlet->c_pixbuf, gimlet->pixbuf))
    {
      constrained_pixbuf_compute_layout (GTK_WIDGET (gimlet->applet_icon_area),
					 gimlet->c_pixbuf,
					 height, gimlet->textwidth);

      scaled = gdk_pixbuf_scale_simple (gimlet->pixbuf,
					gimlet->c_pixbuf->resulting_width,
					height,
					GDK_INTERP_BILINEAR);
      if (scaled == NULL)
	return;

      g_object_unref (gimlet->c_pixbuf->scaled);
      gimlet->c_pixbuf->scaled = scaled;

      widget->requisition.width = gimlet->c_pixbuf->resulting_width;

      draw_scaled_icon (gimlet);
    }
}

static gboolean
expose_cb (GtkWidget *widget, GdkEventExpose *event, GimletWindow *gimlet)
{
  draw_scaled_icon (gimlet);
}

/* public */
void
gimlet_init (GimletWindow *gimlet)
{
  GtkTooltips *tooltips;

  if (gimlet->status_label)
    return;

  panel_applet_set_flags (gimlet->applet, PANEL_APPLET_EXPAND_MINOR);

  g_signal_connect (G_OBJECT(gimlet->applet), "key_press_event",
		    G_CALLBACK(key_press_cb), gimlet);
                     
  tooltips = gtk_tooltips_new();

  gtk_tooltips_set_tip (tooltips, GTK_WIDGET(gimlet->applet),
			_("GIMLET: GNOME Input Method Switcher Applet"), NULL);

  panel_applet_setup_menu_from_file (gimlet->applet,
				     NULL,
				     "GNOME_InputMethodSwitcherApplet.xml",
				     NULL,
				     menu_verbs,
				     gimlet);
#ifdef GNOME_PANEL_APPLET_2_4
  if (panel_applet_get_locked_down (gimlet->applet))
    {
      BonoboUIComponent *popup_component;

      popup_component = panel_applet_get_popup_component (gimlet->applet);

      bonobo_ui_component_set_prop (popup_component,
				    "/commands/Props",
				    "hidden", "1",
				    NULL);
    }
#endif
  gimlet->tooltips = tooltips;

  place_widgets (gimlet);

  gimlet_window_show_or_hide (gimlet);

  return;
}

void
gimlet_window_show_or_hide (GimletWindow *gimlet)
{
  if (gimlet->im_enabled && gimlet->status_enabled &&
      gimlet->status_placement== ON_DESKTOP_PANEL)
    gtk_widget_show (GTK_WIDGET (gimlet->applet));
  else
    gtk_widget_hide (GTK_WIDGET (gimlet->applet));
}

void
gimlet_update (GimletWindow *gimlet)
{
  unfix_size (gimlet);
}

static GdkPixbuf *
init_base_pixbuf (const char *name)
{
  GdkPixbuf *pixbuf;
  GError    *error = NULL;
  char      *path = NULL;

  if (g_path_is_absolute (name))
    path = g_strdup (name);
  else
    {
      path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_PIXMAP,
					name, FALSE, NULL);
      if (!path)
	path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_PIXMAP,
					  name, FALSE, NULL);
      if (!path)
	return NULL;
    }

  pixbuf = gdk_pixbuf_new_from_file (path, &error);

  if (error)
    {
      g_warning ("Cannot load '%s': %s", path, error->message);
      g_error_free (error);
    }
  g_free (path);
  return pixbuf;
}

void
gimlet_change_base_icon (GimletWindow *gimlet, const char *filename)
{
  GdkPixbuf *new_pixbuf = NULL;

  new_pixbuf = init_base_pixbuf (filename);
  if (new_pixbuf)
    {
      g_free (gimlet->base_iconname);
      gimlet->base_iconname = g_strdup (filename);
      g_object_unref (gimlet->pixbuf);
      gimlet->pixbuf = new_pixbuf;
    }
  return;
}

void
gimlet_change_icon (GimletWindow *gimlet, const char *text)
{
  if (gimlet->c_pixbuf == NULL)
    return;
  if (text && text[0])
    {
      if (gimlet->c_pixbuf->text &&
	  !strcmp (gimlet->c_pixbuf->text, text))
	return;

      gimlet->c_pixbuf->font =
	pango_font_description_get_family (gimlet->status_font);

      gimlet->c_pixbuf->text = g_strdup (text);
    }
  update_and_scale_icon (gimlet);
  return;
}

void
gimlet_draw_icon (GimletWindow *gimlet)
{
  update_and_scale_icon (gimlet);
  gimlet_update (gimlet);
  return;
}

void
gimlet_change_status_font (GimletWindow *gimlet)
{
  if (gimlet && gimlet->status_label && 
      GTK_WIDGET_REALIZED (gimlet->status_label))
    gtk_widget_modify_font (GTK_WIDGET (gimlet->status_label),
			    gimlet->status_font);

  if (gimlet->c_pixbuf)
    {
      gimlet->c_pixbuf->font =
	pango_font_description_get_family (gimlet->status_font);
      update_and_scale_icon (gimlet);
    }
}

void
gimlet_change_status_color (GimletWindow *gimlet)
{
  GdkColor fg;

  if (gimlet && gimlet->status_label && 
      GTK_WIDGET_REALIZED (gimlet->status_label))
    {
      if (gimlet->use_theme_colors)
	fg = gimlet->status_label->style->text[GTK_STATE_NORMAL];
      else
	fg = gimlet->foreground;
      gtk_widget_modify_fg (GTK_WIDGET (gimlet->status_label),
			    GTK_STATE_NORMAL, &fg);
    }

  if (gimlet->c_pixbuf)
    update_and_scale_icon (gimlet);
}
