/*  Inti-GL: Integrated Foundation Classes
 *  Copyright (C) 2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/gdk/gl/shapes.h
//! @brief A C++ wrapper interface for the Open-GL shape functions.

#ifndef INTI_GDK_GL_SHAPES_H
#define INTI_GDK_GL_SHAPES_H

#ifndef __GDK_GL_SHAPES_H__
#include <gdk/gdkglshapes.h>
#endif

namespace Inti {

namespace Gdk {

namespace GL {

//! @name Shape Methods
//! @{

void draw_cube(bool solid, double   size);
//!< Renders a cube.
//!< @param solid Set <EM>true</EM> if the cube should be solid.
//!< @param size The length of cube sides.
//!<
//!< <BR>The cube is centered at the modeling coordinates origin with sides of length size.

void draw_sphere(bool solid, double radius, int slices, int stacks);
//!< Renders a sphere centered at the modeling coordinates origin of the specified radius.
//!< @param solid Set <EM>true</EM> if the sphere should be solid.
//!< @param radius The radius of the sphere.
//!< @param slices The number of subdivisions around the Z axis (similar to lines of longitude).
//!< @param stacks The number of subdivisions along the Z axis (similar to lines of latitude).
//!<
//!< <BR>The sphere is subdivided around the Z axis into slices and along the Z axis into stacks.

void draw_cone(bool solid, double base, double height, int slices, int stacks);
//!< Renders a cone oriented along the Z axis.
//!< @param solid Set <EM>true</EM> if the cone should be solid.
//!< @param base The radius of the base of the cone.
//!< @param height The height of the cone.
//!< @param slices The number of subdivisions around the Z axis.
//!< @param stacks The number of subdivisions along the Z axis.
//!<
//!< <BR>The base of the cone is placed at Z = 0, and the top at Z = height. The cone is
//!< subdivided around the Z axis into slices, and along the Z axis into stacks.

void draw_torus(bool solid, double inner_radius, double outer_radius, int nsides, int rings);
//!< Renders a torus (doughnut) centered at the modeling coordinates origin
//!< whose axis is aligned with the Z axis.
//!< @param solid Set <EM>true</EM> if the torus should be solid.
//!< @param inner_radius The inner radius of the torus.
//!< @param outer_radius The outer radius of the torus.
//!< @param nsides The number of sides for each radial section.
//!< @param rings The number of radial divisions for the torus.

void draw_tetrahedron(bool solid);
//!< Renders a tetrahedron centered at the modeling coordinates origin
//!< with a radius of the square root of 3.

void draw_octahedron(bool solid);
//!< Renders a octahedron centered at the modeling coordinates origin with a radius of 1.0.
//!< @param solid Set <EM>true</EM> if the octahedron should be solid.

void draw_dodecahedron(bool solid);
//!< Renders a dodecahedron centered at the modeling coordinates origin
//!< with a radius of the square root of 3.
//!< @param solid Set <EM>true</EM> if the dodecahedron should be solid.

void draw_icosahedron(bool solid);
//!< Renders a icosahedron.
//!< @param solid Set <EM>true</EM> if the icosahedron should be solid.
//!<
//!< <BR>The icosahedron is centered at the modeling coordinates origin and has a radius of 1.0.

void draw_teapot(bool solid, double scale);
//!< Renders a teapot.
//!< @param solid Set <EM>true</EM> if the teapot should be solid.
//!< @param scale The relative size of the teapot.
//!<
//!< <BR>Both surface normals and texture coordinates for the teapot are generated.
//!< The teapot is generated with OpenGL evaluators.

//! @}

} // namespace GL

} // namespace Gdk

} // namespace Inti

#endif // INTI_GDK_GL_SHAPES_H

