/*  Inti-GL: Integrated Foundation Classes
 *  Copyright (C) 2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/gtk/gl/drawingarea.h
//! @brief An Open-GL capable drawing area.

#ifndef INTI_GTK_GL_DRAWING_AREA_H
#define INTI_GTK_GL_DRAWING_AREA_H

#ifndef INTI_GTK_DRAWING_AREA_H
#include <inti/gtk/drawingarea.h>
#endif

#ifndef INTI_GTK_GL_WIDGET_H
#include <inti/gtk/gl/widget.h>
#endif

namespace Inti {

namespace Gdk {

namespace GL {
class Config;
class Context;
}

} // namespace Gdk

namespace Gtk {

namespace GL {

//! @class DrawingArea drawingarea.h inti/gdk/gl/drawingarea.h
//! @brief An Open-GL capable drawing area widget.
//!
//! DrawingArea is a convenience class that sets up an Open-GL capable drawing area.

class DrawingArea : public Gtk::DrawingArea, public Gtk::GL::Widget
{
	DrawingArea(const DrawingArea&);
	DrawingArea& operator=(const DrawingArea&);

public:
//! @name Constructors
//! @{

	DrawingArea();
	//!< Construct a new DrawingArea that can render Open-GL graphics. To make
	//!< the drawing area Open-GL capable you must call set_gl_capability().

	DrawingArea(const Gdk::GL::Config& glconfig, bool direct = true);
	//!< Construct a new Open-GL capable DrawingArea.
	//!< @param glconfig A Gdk::GL::Config.
	//!< @param direct Whether rendering is to be done with a direct connection to the graphics system.
	//!<
	//!< <BR>This constructor prepares the drawing area for its use with OpenGL. No sharing of
	//!< display lists and texture objects takes place when you call this constructor, and
	//!< the render type is set to Gdk::GL::RGBA_TYPE.

	DrawingArea(const Gdk::GL::Config& glconfig, int width, int height, bool direct = true);
	//!< Construct a new Open-GL capable DrawingArea with the specified width and height.
	//!< @param glconfig A Gdk::GL::Config.
	//!< @param width The width to request.
	//!< @param height The height to request.
	//!< @param direct Whether rendering is to be done with a direct connection to the graphics system.
	//!<
	//!< <BR>This constructor prepares the drawing area for its use with OpenGL. No sharing of
	//!< display lists and texture objects takes place when you call this constructor, and
	//!< the render type is set to Gdk::GL::RGBA_TYPE.

	DrawingArea(const Gdk::GL::Config& glconfig, const Gdk::GL::Context& share_list, bool direct = true);
	//!< Construct a new Open-GL capable DrawingArea.
	//!< @param glconfig A Gdk::GL::Config.
	//!< @param share_list The Gdk::GL::Context with which to share display lists and texture objects.
	//!< @param direct Whether rendering is to be done with a direct connection to the graphics system.
	//!<
	//!< <BR>This method prepares the widget for its use with OpenGL. The render type
	//!< is set to Gdk::GL::RGBA_TYPE.

	DrawingArea(const Gdk::GL::Config& glconfig, const Gdk::GL::Context& share_list, int width, int height, bool direct = true);
	//!< Construct a new Open-GL capable DrawingArea.
	//!< @param glconfig A Gdk::GL::Config.
	//!< @param share_list The Gdk::GL::Context with which to share display lists and texture objects.
	//!< @param width The width to request.
	//!< @param height The height to request.
	//!< @param direct Whether rendering is to be done with a direct connection to the graphics system.
	//!<
	//!< <BR>This method prepares the widget for its use with OpenGL. The render type
	//!< is set to Gdk::GL::RGBA_TYPE.

	virtual ~DrawingArea();
	//!< Destructor.

//! @}
//! @name Accessors
//! @{

	bool is_double_buffered() const;
	//!< Determines whether the drawable supports the double-buffered visual.
	//!< @return <EM>true</EM> if the double-buffered visual is supported, <EM>false</EM> otherwise.
	//!<
	//!< <BR>This is a convenience method that retrieves a pointer to the drawing area's
	//!< drawable, and then calls Gdk::GL::Drawable::is_double_buffered().

//! @}
//! @name Methods
//! @{

	void swap_buffers();
	//!< Exchange the front and back buffers.
	//!< This is a convenience method that retrieves a pointer to the drawing area's
	//!< drawable, and then calls Gdk::GL::Drawable::swap_buffers().

	bool gl_begin();
	//!< Delimits the begining of the OpenGL execution.
	//!< @returns <EM>true</EM> if it is successful, <EM>false</EM> otherwise.
	//!<
	//!< <BR>This is a convenience method that retrieves a pointer to the drawing area's
	//!< drawable, and then calls Gdk::GL::Drawable::gl_begin().

	void gl_end();
	//!< Delimits the end of the OpenGL execution.
	//!< This is a convenience method that retrieves a pointer to the drawing area's
	//!< drawable, and then calls Gdk::GL::Drawable::gl_end().

//! @}
};

} // namespace GL

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_GL_DRAWING_AREA_H

