/****************************************************************************
 *                            XObjWidgetBase.cc
 * 
 * Author: Matthew Ballance
 * Desc:   Implements the base object for X widgets
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "XObjWidgetBase.h"
#include "CallbackMgr.h"

/********************************************************************
 * XObjWidgetBase()
 ********************************************************************/
XObjWidgetBase::XObjWidgetBase(
        Tcl_Interp        *interp,
        Uint32             objc,
        Tcl_Obj           *const objv[]) :
    ObjWidgetBase(interp, objc, objv),
    d_RedrawFlags(0)
{
    char *instName;

    d_bindCnt = 0;

    /**** Check to ensure that initialization of the base class 
     **** succeeded...
     ****/
    if (!ok) {
        return;
    }
    ok = 0;

    instName = Tcl_GetString(objv[1]);

    d_mainWindow = Tk_CreateWindowFromPath(interp,
            Tk_MainWindow(interp), instName, (char *)NULL);

    if (!d_mainWindow) {
        Tcl_AppendResult(interp, "cannot create window ", instName, 0);
        return;
    }

    d_display = Tk_Display(d_mainWindow);
    d_WinPixmap.setWindow(d_mainWindow);

    d_PageMoveNumerator   = 8;
    d_PageMoveDenominator = 10;
    d_PageScrollUnitMult  = 5;
    d_YscrollOvershoot    = 0;

    Tk_CreateEventHandler(d_mainWindow,
            ExposureMask|StructureNotifyMask|FocusChangeMask|
                ButtonPressMask|ButtonReleaseMask|Button1MotionMask,
            &XObjWidgetBase::EventProc, (ClientData)this);

    for (Uint32 i=0; i<3; i++) {
        d_buttonTimes[i] = 0;
    }

    d_BPCbList     = CallbackMgr::GetCbList(CBTYPE_X_BUTTON_PRESS, instName);
    d_BRCbList     = CallbackMgr::GetCbList(CBTYPE_X_BUTTON_RELEASE, instName);
    d_MotionCbList = CallbackMgr::GetCbList(CBTYPE_X_MOTION, instName);
    d_BDPCbList    = CallbackMgr::GetCbList(CBTYPE_X_BUTTON_DOUBLE_PRESS,
            instName);

    for (Uint32 i=0; i<16; i++) {
        cbArgs[i] = Tcl_NewObj();
        Tcl_IncrRefCount(cbArgs[i]);
    }

    ok = 1;
}

/********************************************************************
 * ~XObjWidgetBase()
 ********************************************************************/
XObjWidgetBase::~XObjWidgetBase()
{
    Tcl_CancelIdleCall(&XObjWidgetBase::IdleProc, this);

    Tk_DeleteEventHandler(d_mainWindow,
            ExposureMask|StructureNotifyMask|FocusChangeMask|
                ButtonPressMask|ButtonReleaseMask|Button1MotionMask,
            &XObjWidgetBase::EventProc, (ClientData)this);
}

/********************************************************************
 * base_SetupRedraw()
 ********************************************************************/
void XObjWidgetBase::base_SetupRedraw(Uint32 redraw_flags)
{
    if (d_mainWindow && Tk_IsMapped(d_mainWindow) &&
            !(d_RedrawFlags & RedrawPending)) {
        Tcl_DoWhenIdle(&XObjWidgetBase::IdleProc, this);
        d_RedrawFlags |= RedrawPending;
    }

    d_RedrawFlags |= redraw_flags;    
}

/********************************************************************
 * SetupRedraw()
 ********************************************************************/
void XObjWidgetBase::SetupRedraw(Uint32 redraw_flags)
{
    base_SetupRedraw(redraw_flags);
}

/********************************************************************
 * winValid()
 ********************************************************************/
int XObjWidgetBase::winValid()
{
    return (d_mainWindow && Tk_IsMapped(d_mainWindow));
}

/********************************************************************
 * UpdateScrollbar()
 ********************************************************************/
void XObjWidgetBase::UpdateScrollbar(
        char        *cmd,
        Uint32       start_pos,
        Uint32       visible_height,
        Uint32       total_height)
{
    double d_start = 0.0;
    double d_end   = 1.0;
    double d_max   = total_height;
    char   str[64];

    if (!cmd) {
        return;
    }

    if (d_max > 0.0) {
        d_start = ((double)start_pos)/d_max;
    }

    if ((start_pos+visible_height) < total_height) {
        if (d_max > 0.0) {
            d_end = ((double)(start_pos+visible_height))/d_max;
        }
    }

    sprintf(str, " %f %f ", d_start, d_end);
    if (Tcl_VarEval(d_interp, cmd, str, 0) != TCL_OK) {
        Tcl_BackgroundError(d_interp);
    }
}

/********************************************************************
 * ProcessViewCmd()
 *
 * Expects cmd of: <inst> <xview|yview> <cmd>
 ********************************************************************/
int XObjWidgetBase::ProcessViewCmd(
        Uint32        objc,
        Tcl_Obj      *const objv[],
        Uint32        display_total,
        Uint32        display_visible,
        Uint32       &display_start)
{
    Int32    type, count, tmpStart, tmpStart2;
    double   fraction;

#if 0
    fprintf(stderr, "ProcessViewCmd: objc=%d objv=(", objc);
    for (int i=0; i<objc; i++) {
        fprintf(stderr, "0x%08x ", objv[i]);
    }
    fprintf(stderr, ")\n");
    fprintf(stderr, "\t");
    for (int i=0; i<objc; i++) {
        fprintf(stderr, "%s ", Tcl_GetString(objv[i]));
    }
    fprintf(stderr, "\n");
#endif
    
    type = Tk_GetScrollInfoObj(d_interp, objc, objv, &fraction, &count);

    switch (type) {
        default:
        case TK_SCROLL_ERROR:
            return TCL_ERROR;

        case TK_SCROLL_MOVETO:
            if (fraction >= 0.0 && fraction <= 1.0) {
                tmpStart = (Uint32)(fraction*display_total);
            } else if (fraction <= 0.0) {
                tmpStart = 0;
            } else {
                tmpStart = display_total+50 - display_visible;
            }
            break;

        case TK_SCROLL_PAGES:
            if (count > 0) {
                tmpStart = (display_visible * d_PageMoveNumerator)/ 
                    d_PageMoveDenominator;
                tmpStart += display_start;
            } else {
                tmpStart = (display_visible * d_PageMoveNumerator)/
                    d_PageMoveDenominator;
                tmpStart -= display_start;

                if (tmpStart < 0) {
                    tmpStart = 0;
                }
            }
            break;

        case TK_SCROLL_UNITS:
            if ((display_start + (count*d_PageScrollUnitMult)) >= 0) {
                tmpStart = display_start + (count*d_PageScrollUnitMult);

                if ((tmpStart >= display_total) || (tmpStart < 0)) {
                    tmpStart = display_start;
                }
            }
            break;
    }

    if ((tmpStart+display_visible) < (display_total+d_YscrollOvershoot)) {
        display_start = tmpStart;
    } else {
        if ((display_total+d_YscrollOvershoot) >= display_visible) {
            display_start = (display_total+d_YscrollOvershoot)-display_visible;
        } else {
            display_start = 0;
        }
    }

    return TCL_OK;
}

/********************************************************************
 * Center()
 ********************************************************************/
void XObjWidgetBase::Center(
        Uint32        displayStart,
        Uint32        displaySize,
        Uint32       &pixmapStart,
        Uint32        pixmapSize)
{
    Uint32    half = (pixmapSize-displaySize)/2;

    if (half <= displayStart) {
        pixmapStart = (!displayStart)?0:(displayStart-half);
    } else {
        pixmapStart = 0;
    }
}

/********************************************************************
 * Bind()
 ********************************************************************/
int XObjWidgetBase::Bind(
        const char       *bindTag,
        const char       *argPattern,
        Tcl_ObjCmdProc   *cmdProc,
        ClientData        clientData)
{
    char cmd[1024];
    int  ret = TCL_OK;

    sprintf(cmd, "%s%d", getInstName(), d_bindCnt);
    if (!Tcl_CreateObjCommand(d_interp, cmd, cmdProc, clientData, 0)) {
        fprintf(stderr, "ERROR: Couldn't create command: %s\n", 
                Tcl_GetStringResult(d_interp));
    }

    sprintf(cmd, "bind %s %s \"%s%d %s\"",
            getInstName(), bindTag, 
            getInstName(), d_bindCnt++,
            (argPattern)?argPattern:"");

    ret = Tcl_EvalEx(d_interp, cmd, -1, TCL_EVAL_GLOBAL);

    if (ret != TCL_OK) {
        fprintf(stderr, "Bind failed - \"%s\"\n",
                Tcl_GetStringResult(d_interp));
    }

    return ret;
}

/********************************************************************
 * InstCmd()
 ********************************************************************/
int XObjWidgetBase::InstCmd(
        ClientData         clientData,
        Tcl_Interp        *interp,
        int                objc,
        Tcl_Obj           *const objv[])
{
    XObjWidgetBase *obj = (XObjWidgetBase *)clientData;

    return obj->InstCmd(interp, objc, objv);
}

/********************************************************************
 * IdleProc()
 ********************************************************************/
void XObjWidgetBase::IdleProc(ClientData clientData)
{
    XObjWidgetBase *obj = (XObjWidgetBase *)clientData;
    obj->IdleProc();
    obj->d_RedrawFlags &= ~(RedrawPending|RedrawForce);
}

/********************************************************************
 * IdleProc()
 ********************************************************************/
void XObjWidgetBase::IdleProc()
{
    fprintf(stderr, "ERROR :: XObjWidgetBase base IdleProc called\n");
}

/********************************************************************
 * base_InstCmd(
 ********************************************************************/
int XObjWidgetBase::base_InstCmd(
        Tcl_Interp        *interp,
        Uint32             objc,
        Tcl_Obj           *const objv[])
{
    Tcl_AppendResult(interp, "default InstCmd of XObjWidgetBase called", 0);
    return TCL_ERROR;
}

/********************************************************************
 * ProcessEvents()
 ********************************************************************/
void XObjWidgetBase::ProcessEvents(XEvent *eventPtr)
{
    Uint32 button;
    Uint32 delta;

    switch (eventPtr->type) {

        case ButtonPress:
            button = eventPtr->xbutton.button;

            Tcl_SetIntObj(cbArgs[0], eventPtr->xbutton.button);
            Tcl_SetIntObj(cbArgs[1], eventPtr->xbutton.x);
            Tcl_SetIntObj(cbArgs[2], eventPtr->xbutton.y);
            Tcl_SetIntObj(cbArgs[3], eventPtr->xbutton.state);

            delta = (eventPtr->xbutton.time - d_buttonTimes[button]);

            /*
            fprintf(stderr, "time=%u ; buttonTimes=%u ; delta=%u\n",
                    eventPtr->xbutton.time, d_buttonTimes[button], delta);
             */


            /**** See if this is a first button-press ****/
            if (!d_buttonTimes[button] || delta > 700) {
                d_buttonTimes[button] = eventPtr->xbutton.time;
            }

            d_buttonTimes[button] = eventPtr->xbutton.time;
            d_BPCbList->Invoke(3, cbArgs);

            if ( delta > 100 && delta < 300 ) {
/*                fprintf(stderr, "Double-press\n"); */
                d_BDPCbList->Invoke(3, cbArgs);
                d_buttonTimes[button] = 0;
            }
            break;

        case ButtonRelease:
            Tcl_SetIntObj(cbArgs[0], eventPtr->xbutton.button);
            Tcl_SetIntObj(cbArgs[1], eventPtr->xbutton.x);
            Tcl_SetIntObj(cbArgs[2], eventPtr->xbutton.y);

            d_BRCbList->Invoke(3, cbArgs);
            break;

        case MotionNotify:
            Tcl_SetIntObj(cbArgs[0], eventPtr->xmotion.x);
            Tcl_SetIntObj(cbArgs[1], eventPtr->xmotion.y);

            d_MotionCbList->Invoke(2, cbArgs);
            break;

        default:
            break;
    }
}

/********************************************************************
 * Destroy()
 ********************************************************************/
void XObjWidgetBase::Destroy(char *ptr)
{
    XObjWidgetBase *widget = (XObjWidgetBase *)ptr;

    delete widget;
}

/********************************************************************
 * EventProc
 ********************************************************************/
void XObjWidgetBase::EventProc(ClientData clientData, XEvent *eventPtr)
{
    XObjWidgetBase *obj = (XObjWidgetBase *)clientData;

    switch (eventPtr->type) {
        case DestroyNotify:
            Tcl_EventuallyFree(clientData, &XObjWidgetBase::Destroy);
            break;

        default:
            obj->ProcessEvents(eventPtr);
            obj->EventProc(eventPtr);

            if (obj->IsFlagSet(RedrawForce) && 
                    !obj->IsFlagSet(RedrawPending)) {
                obj->SetupRedraw();
            }
            break;
    }
}

/********************************************************************
 * EventProc
 ********************************************************************/
void XObjWidgetBase::EventProc(XEvent *eventPtr)
{
    ;
}

/********************************************************************
 * PixmapPt &operator ()
 ********************************************************************/

static const char *prv_XCbTypes[] = {
    CBTYPE_X_BUTTON_PRESS,
    CBTYPE_X_BUTTON_RELEASE,
    CBTYPE_X_BUTTON_DOUBLE_PRESS,
    CBTYPE_X_MOTION,
    ""
};

/********************************************************************
 * XWidgetInit()
 ********************************************************************/
int XObjWidgetBase::XWidgetInit(
        Tcl_Interp        *interp,
        const char        *widgetName,
        const char        *widgetVersion,
        const char        *widgetCmd,
        Tcl_ObjCmdProc    *cmdProc)
{
    int ret = TCL_OK;

    ret = ObjWidgetBase::WidgetInit(interp, widgetName, widgetVersion,
            widgetCmd, cmdProc);

    if (ret != TCL_OK) {
        return ret;
    }

    if (Tk_InitStubs(interp, "8.1", 0) == NULL) {
        Tcl_AppendResult(interp, "tk stubs init failed...", 0);
        return TCL_ERROR;
    }

    return ret;
}

/********************************************************************
 *
 ********************************************************************/
extern "C" int XObjWidgetBase_Init(Tcl_Interp *interp)
{
    Uint32 idx = 0;

    while (prv_XCbTypes[idx][0]) {
        CallbackMgr_AddCbType(prv_XCbTypes[idx]);
        idx++;
    }

    return TCL_OK;
}


