#*****************************************************************************
#*                           wave_cmds.tcl
#*
#* Author: Matthew Ballance
#* Desc:   Implements the user-level commands for wave-cmd manipulation
#*
#* <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
#*
#*    This source code is free software; you can redistribute it
#*    and/or modify it in source code form under the terms of the GNU
#*    General Public License as published by the Free Software
#*    Foundation; either version 2 of the License, or (at your option)
#*    any later version.
#*
#*    This program is distributed in the hope that it will be useful,
#*    but WITHOUT ANY WARRANTY; without even the implied warranty of
#*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*    GNU General Public License for more details.
#*
#*    You should have received a copy of the GNU General Public License
#*    along with this program; if not, write to the Free Software
#*    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
#*
#* </Copyright>
#*
#*****************************************************************************

#********************************************************************
#* prv_lookup_wave
#********************************************************************
proc prv_lookup_wave {opts_v} {
    upvar $opts_v opts
    global wave_data
    global signal_windows
    global ivi_global

    set wnd_name ""
    set wnd_path ""

    #**** Did the user specify a specific window to use???
    if {$opts(-win) == ""} {
        set wnd_path [lindex $wave_data(win_list) 0]
    } elseif {$opts(-win) == "new_auto"} {
        #**** Do nothing - wnd_name/wnd_path already null
    } else {
        set wnd_name $opts(-win)
        set wnd_path [lindex [array get signal_windows $wnd_name] 1]
    }

    #**** Pick a window-name... 
    if {$wnd_path == ""} {
        for {set i 0} {$i < 256} {incr i} {
            set wnd_path ".signalWindow$i"
            global $wnd_path

            if {![winfo exists $wnd_path]} {
                if {$wnd_name == ""} {
                    set wnd_name "Window$i"
                }
                break
            }
        }
    }

    #**** Now, see if the window must be created...
    if {![winfo exists $wnd_path]} {
        set wnd_path [SigWindow::SigWindow $wnd_path -title $wnd_name ]
        $wnd_path show_page wave

        lappend wave_data(win_list) $wnd_path

        if {$wnd_name != "Window0"} {
            $wnd_path config -win_name $wnd_name
        }

        wm title $wnd_path $wnd_name

	set signal_windows($wnd_name) $wnd_path
        incr wave_data(wnd_cnt)
    }
    return $wnd_path
}

#********************************************************************
#* lookup_wave
#********************************************************************
proc lookup_wave args {
    set optspec [prv_get_wave_optspec]

    eval prv_wave_parse_args optspec opts $args

    return [prv_lookup_wave opts]
}

#********************************************************************
#* prv_get_wave_optspec
#********************************************************************
proc prv_get_wave_optspec {} {
    set optspec { 
        -win          ""
    }

    return $optspec
}

#********************************************************************
#* wave_zoom
#* Set the zoom the specified window
#********************************************************************
proc wave_zoom args {

    set optspec [prv_get_wave_optspec]
    eval prv_wave_parse_args optspec opts $args
    set wnd_path [prv_lookup_wave opts]

    if {[lindex $opts(args) 0] == "full"} {
        $wnd_path WaveZoom full
    } else {
        eval $wnd_path WaveZoom $opts(args)
    }
}

#********************************************************************
#* prv_wave_parse_args
#********************************************************************
proc prv_wave_parse_args {opt_tmpl_v opts_v args} {
    upvar $opts_v opts
    upvar $opt_tmpl_v opt_tmpl

    array set opts $opt_tmpl

    set opts(args) ""

    while {[llength $args] > 0} {
        if {[regexp {^-[^-]} [lindex $args 0]]} {
            set shift 2
            set key [lindex $args 0]
            #**** This is an option...
            set vval [array get opts $key]
            if {$vval == ""} {
                error "invalid option \"$key\""
            } else {
                set opts($key) [lindex $args 1]
            }
        } else {
            set shift 1
            lappend opts(args) [lindex $args 0]
        }
        set args [lrange $args $shift end]
    }
}

#********************************************************************
#* prv_get_specified_opts
#********************************************************************
proc prv_get_specified_opts {opt_tmpl_v opts_v} {
    upvar $opt_tmpl_v opt_tmpl
    upvar $opts_v     opts

    set ret ""

    foreach {key value} $opt_tmpl {
        if {$opts($key) != ""} {
            lappend ret $key $opts($key)
        }
    }
    return $ret
}

#********************************************************************
#* wave_add
#* Add signals to a specified wave window...
#*
#* wave add ?options? signals
#*
#* Options are:
#*     -radix [bin|hex|dec|str|oct]
#*     -visel        <num_vis_elements>
#*     -sig_dispname <name>
#*     -expanded     [0|1]
#*     -sdb          <sdb_name>
#********************************************************************
proc wave_add args {
    set wave_optspec {
        -radix        ""
        -visel        ""
        -sig_dispname ""
        -expanded     ""
    }

    set waveadd_optspec {
        -sdb          ""
    }

    set total_optspec $wave_optspec

    eval lappend total_optspec [prv_get_wave_optspec] $waveadd_optspec

    eval prv_wave_parse_args total_optspec opts $args

    #**** Get the specified window
    set wnd_path [prv_lookup_wave opts]

    #**** Add the specified signals. Pass on the information about
    #**** the SDB to use (if specified)
    set add_opts [prv_get_specified_opts waveadd_optspec opts]
    set result [eval $wnd_path AddToWave $add_opts $opts(args)]

    #**** Pull out the appropriate config-options
    set sig_conf [prv_get_specified_opts wave_optspec opts]
    if {[llength $sig_conf] > 0} {
        foreach sig $result {
            eval $sig config $sig_conf
        }
    }

    return $result
}

#********************************************************************
#* wave_add_sep
#********************************************************************
proc wave_add_sep args {
    set optspec [prv_get_wave_optspec]
    eval prv_wave_parse_args optspec opts $args
    set wnd_path [prv_lookup_wave opts]

    set result [$wnd_path AddSeparator "---- [lindex $opts(args) end] ----" ]

    return $result
}

#********************************************************************
#* wave_config
#********************************************************************
proc wave_config args {
    set optspec [prv_get_wave_optspec]
    eval prv_wave_parse_args optspec opts $args
    set wnd_path [prv_lookup_wave opts]

    set sdbr [$wnd_path subwidget SdbWaveReader]

    foreach arg $args {
        if {$arg == "-expanded"} {
        }
    }
}

#********************************************************************
#* wave_add_cursor
#********************************************************************
proc wave_add_cursor args {
    set optspec [prv_get_wave_optspec]
    eval prv_wave_parse_args optspec opts $args
    set wnd [prv_lookup_wave opts]

    set cmd [lindex $opts(args) 0]

    if {$wnd == ""} {
        error "Cannot locate the specified wave window"
    }

    set sdbr [$wnd subwidget SdbWaveReader]

    set cursor [$sdbr add_cursor]

    set wave_window [$wnd subwidget waveWinPane]
    set wave_widget [$wave_window subwidget wave_widget]

    set wave_time [$wave_widget get_limits]
    set mid [expr ([lindex $wave_time 1] - [lindex $wave_time 0]) / 2 \
             + [lindex $wave_time 0]]
    $cursor set_time $mid

    return $cursor
}

#********************************************************************
#* wave_list_cursors
#********************************************************************
proc wave_list_cursors args {
    set sdbr [eval wave_get_sdbr $args]

    return [$sdbr get_cursor_list]
}

#********************************************************************
#* wave_get_sdbr
#********************************************************************
proc wave_get_sdbr args {
    set wnd [eval lookup_wave $args]

    set sdbr [$wnd subwidget SdbWaveReader]

    return $sdbr
}

#********************************************************************
#* wave_close
#********************************************************************
proc wave_close args {
    global wave_data

    set optspec [prv_get_wave_optspec]
    eval prv_wave_parse_args optspec opts $args

    set wnd_path ""

    if {$opts(-win) == ""} {
        set wnd_path [lindex $wave_data(win_list) 0]
    } else {
        set wnd_path [lindex [array get signal_windows $wnd_name] 1]
    }

    if {$wnd_path == ""} {
        error "wave close failed - no window to close"
    }

    destroy $wnd_path
}

#********************************************************************
#* wave_save_setup
#********************************************************************
proc wave_save_setup args {
    global wave_data

    set optspec [prv_get_wave_optspec]
    eval prv_wave_parse_args optspec opts $args

    set wnd_path ""

    if {$opts(-win) == ""} {
        set wnd_path [lindex $wave_data(win_list) 0]
    } else {
        set wnd_path [lindex [array get signal_windows $wnd_name] 1]
    }

    if {$wnd_path == ""} {
        return
    }

#    set ww [$wnd_path subwidget waveWinPane]

    SigWindow::SaveSetup $wnd_path [lindex $opts(args) 0]
}

#********************************************************************
#* prv_wave_close
#********************************************************************
proc prv_wave_close {path} {
    global wave_data

    set new_list ""
    foreach win $wave_data(win_list) {
        if {$path != $win} {
            lappend new_list $win
        } 
    }

    set wave_data(win_list) $new_list
}

#********************************************************************
#* wave_save_fmt_fp
#********************************************************************
proc wave_save_fmt_fp {fp args} {
    global signal_windows

    foreach name [array names signal_windows] {
        if {$name == "__dummy" || $name == "base"} {
            continue
        }

        set pair [array get signal_windows $name]

        set win [lindex $pair 1]

        SigWindow::SaveSetup $win $fp $args
    }
}

#********************************************************************
#* wave_clear
#********************************************************************
proc wave_clear args {
    global signal_windows
    global CallbackTypes

    set optspec [prv_get_wave_optspec]
    eval prv_wave_parse_args optspec opts $args

    set wnd_path ""

    if {$opts(-win) == ""} {
        set wnd_path [lindex $wave_data(win_list) 0]
    } elseif {$opts(-win) == "all"} {
        set wnd_path ""

        foreach name [array names signal_windows] {
            if {$name == "__dummy" || $name == "base"} {
                continue
            }
            set pair [array get signal_windows $name]
            set win [lindex $pair 1]
            lappend wnd_path $win
        }
    } else {
        set wnd_path [lindex [array get signal_windows $wnd_name] 1]
    }

    foreach wnd $wnd_path  {
        set sdbr [$wnd subwidget SdbWaveReader]
        callback disable $CallbackTypes(SDBR_SIGLIST_UPDATE) $sdbr

        foreach sig [$sdbr list] {
            $sig delete
        }

        callback enable $CallbackTypes(SDBR_SIGLIST_UPDATE) $sdbr
        callback invoke $CallbackTypes(SDBR_SIGLIST_UPDATE) $sdbr
    }
}

#********************************************************************
#* wave
#********************************************************************
proc wave args {
    set cmd [lindex $args 0]

    if {[regexp {^-[^-]} [lindex $args 0]]} {
        error "pasing option \"[lindex $args 0]\" as wave sub-cmd"
    }

    switch $cmd {
         add {
	     return [eval wave_add [lrange $args 1 end]]
         }

         add_cursor {
             eval wave_add_cursor [lrange $args 1 end]
         }

         add_sep -
         add_separator {
             eval wave_add_sep [lrange $args 1 end]
         }

         cursors {
             eval wave_list_cursors [lrange $args 1 end]
         }

         get_sdbr {
             eval wave_get_sdbr [lrange $args 1 end]
         }

         zoom {
             eval wave_zoom [lrange $args 1 end]
         }

         lookup {
             eval lookup_wave [lrange $args 1 end]
         }

         close {
             eval wave_close [lrange $args 1 end]
         }

         save_setup {
             eval wave_save_setup [lrange $args 1 end]
         }


         config -
         configure {
             eval wave_config [lrange $args 1 end]
         }

         default {
             error "unknown wave sub-cmd \"$cmd\""
         }
     }
}

#********************************************************************
#* init_wave_cmds
#********************************************************************
proc init_wave_cmds { } {
    uplevel #0 {
        set wave_data(wnd_cnt)      0
        set wave_data(win_list)     ""
        set signal_windows(__dummy) 0
    }
}


